--***********************************************************************
--									*
--	COPYRIGHT 1992		DIGITAL EQUIPMENT CORPORATION		*
--									*
--   This software was written by Bevin Brett, of Digital Equipment	*
--   Corporation.							*
--									*
--   Digital assumes no responsibility AT ALL for the use or reliability*
--   of this software.							*
--									*
--   Redistribution and use in source and binary forms are permitted	*
--   provided that this entire heading from --*** to --*** are          *
--   duplicated in all such forms and that any documentation,		*
--   advertising materials, and other materials related to such		*
--   distribution and use acknowledge that the software was developed	*
--   by Digital Equipment Corporation. The name of Digital Equipment	*
--   Corporation may not be used to endorse or promote products derived	*
--   from this software without specific prior written permission.	*
--									*
--   THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR	*
--   IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
--   WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.*
--									*
--***********************************************************************


with SCALE_TYPE_MATH_LIB;
package body STANDARD_ATMOSPHERE is

    P0 : constant KILOGRAMS_PER_METRE_CUBED := 1.225;
    T0 : constant KELVIN		    := 288.0;

    type NVECTOR_OF_SCALE_TYPE is array(NATURAL range <>) of SCALE_TYPE;

    METRES_PER_TEN_THOUSAND_FEET : constant SCALE_TYPE
    := 3048.0;

    DENSITY_RATIO_PER_TEN_THOUSAND_FEET : constant NVECTOR_OF_SCALE_TYPE
    :=	(00 => 1.0000,
	 01 => 0.7385,
	 02 => 0.5328,
	 03 => 0.3741,
	 04 => 0.2462,
	 05 => 0.1522,
	 06 => 0.0941,
	 07 => 0.0582,
	 08 => 0.0360,
	 09 => 0.0217,
	 10 => 0.0132);

    TEMPERATURE_RATIO_PER_TEN_THOUSAND_FEET : constant NVECTOR_OF_SCALE_TYPE
    :=	(00 => 1.0000,
	 01 => 0.9312,
	 02 => 0.8625,
	 03 => 0.7937,
	 04 => 0.7519,
	 05 => 0.7519,
	 06 => 0.7519,
	 07 => 0.7519,
	 08 => 0.7519,
	 09 => 0.7772,
	 10 => 0.8089);

    generic
	TABLE : in NVECTOR_OF_SCALE_TYPE;
	with function MODIFY_ENTRY(X : SCALE_TYPE) return SCALE_TYPE;
    package G is
	function F(H : ALTITUDE) return SCALE_TYPE; pragma INLINE(F);
    end;

    package body G is
	A, B : NVECTOR_OF_SCALE_TYPE(0..TABLE'last-1);

	function F(H : ALTITUDE) return SCALE_TYPE is
	    N	: INTEGER
		:= INTEGER(SCALE_TYPE(H)/METRES_PER_TEN_THOUSAND_FEET-0.5);
	begin
	    if N < A'first then
		N := A'first;
	    elsif N > A'last then
		N := A'last;
	    end if;
	    return A(N)*SCALE_TYPE(H) + B(N);
	end;

	function T(I : NATURAL) return SCALE_TYPE is
	begin
	    return MODIFY_ENTRY(TABLE(I));
	end;

    begin
	for I in 0 .. TABLE'last-1 loop
	    A(I) := (T(I+1)-T(I))/METRES_PER_TEN_THOUSAND_FEET;
	    B(I) := T(I)-
		    SCALE_TYPE(I)*METRES_PER_TEN_THOUSAND_FEET*A(I);
	end loop;   
    end;

    function DENSITY_MODIFIER(X : SCALE_TYPE) return SCALE_TYPE is
    begin
	return SCALE_TYPE(P0)*X;
    end;

    function TEMPERATURE_MODIFIER(X : SCALE_TYPE) return SCALE_TYPE is
    begin
	return SCALE_TYPE(T0)*X;
    end;

    function MACH_MODIFIER(X : SCALE_TYPE) return SCALE_TYPE is
    begin
	-- From MCAD page 72
	--
	return SCALE_TYPE_MATH_LIB.SQRT(SCALE_TYPE(T0*X)*1.4*287.0);
    end;

    package P_DENSITY is
	new G(DENSITY_RATIO_PER_TEN_THOUSAND_FEET, DENSITY_MODIFIER);

    package P_TEMPERATURE is
	new G(TEMPERATURE_RATIO_PER_TEN_THOUSAND_FEET, TEMPERATURE_MODIFIER);

    package P_MACH is
	new G(TEMPERATURE_RATIO_PER_TEN_THOUSAND_FEET, MACH_MODIFIER);

    package P_SQRT_DENSITY_RATIO is
	new G(DENSITY_RATIO_PER_TEN_THOUSAND_FEET, SCALE_TYPE_MATH_LIB.SQRT);


    function TO_DENSITY(H : ALTITUDE) return KILOGRAMS_PER_METRE_CUBED is
    begin
	return KILOGRAMS_PER_METRE_CUBED(P_DENSITY.F(H));
    end;

    function TO_TEMPERATURE(H : ALTITUDE) return KELVIN is
    begin
	return KELVIN(P_TEMPERATURE.F(H));
    end;


    function TO_MACH(V : METRES_PER_SECOND; H : ALTITUDE) return MACH_TYPE is
    begin
	return MACH_TYPE(V/P_MACH.F(H));
    end;


    function TO_SQRT_DENSITY_RATIO(H : ALTITUDE) return SCALE_TYPE is
    begin
	return P_SQRT_DENSITY_RATIO.F(H);
    end;

end;
