--***********************************************************************
--									*
--	COPYRIGHT 1992		DIGITAL EQUIPMENT CORPORATION		*
--									*
--   This software was written by Bevin Brett, of Digital Equipment	*
--   Corporation.							*
--									*
--   Digital assumes no responsibility AT ALL for the use or reliability*
--   of this software.							*
--									*
--   Redistribution and use in source and binary forms are permitted	*
--   provided that this entire heading from --*** to --*** are          *
--   duplicated in all such forms and that any documentation,		*
--   advertising materials, and other materials related to such		*
--   distribution and use acknowledge that the software was developed	*
--   by Digital Equipment Corporation. The name of Digital Equipment	*
--   Corporation may not be used to endorse or promote products derived	*
--   from this software without specific prior written permission.	*
--									*
--   THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR	*
--   IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
--   WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.*
--									*
--***********************************************************************


with
    MISSILE_AERODYNAMICS_MOVE_CONTEXT, TEXT_IO, FLOAT_TEXT_IO,
    MISSILE_UTILITIES, MOVE_UTILITIES, PHYSICAL_UNITS;

pragma ELABORATE(
    MISSILE_AERODYNAMICS_MOVE_CONTEXT, TEXT_IO, FLOAT_TEXT_IO,
    MISSILE_UTILITIES, MOVE_UTILITIES, PHYSICAL_UNITS);

separate(MISSILE_AERODYNAMICS)
    procedure MOVE_DELTA(
		    TIME_DELTA		    : SECONDS;
		    O			    : in out OBJECTS.OBJECT_TYPE;
		    MISSILE_STATUS	    : in out MISSILE_STATUS_TYPE
		    ) is

	use OBJECTS, WORLD_PHYSICS;

	M_T	: MISSILE_TYPE renames MISSILE_STATUS.MISSILE.all;

	DENSITY : constant SCALE_TYPE
		:= SCALE_TYPE(MISSILE_STATUS.DENSITY);

	MASS	: constant KILOGRAMS
		:= (MISSILE_STATUS.FUEL_MASS+M_T.EMPTY_MASS);

	V	: constant METRES_PER_SECOND
		:= LENGTH(O.VELOCITY);

	DENSITY_V2 : constant SCALE_TYPE
		:= DENSITY * SCALE_TYPE(V)**2;

	-- Where is the target now?
	--
	NEW_TRACKING_TAN_I  : SCALE_TYPE;
	NEW_TRACKING_TAN_K  : SCALE_TYPE;

	-- Acceleration along UA
	--
	    --  sideways acceleration needed
	    --
	A_ON_I	: METRES_PER_SECOND_SQUARED;
	A_ON_K	: METRES_PER_SECOND_SQUARED;

	    --  the total sideways acceleration
	    --
	A_ON_IK : METRES_PER_SECOND_SQUARED;

	-- Coefficient of lift
	--
	CL	: SCALE_TYPE;

	-- Thrust and drag along UV = O.ORIENTATION.J
	--
	A_ON_J	: METRES_PER_SECOND_SQUARED;


    begin
	if MISSILE_STATUS.TARGET = null then

	    -- If temporarily lost track, just keep turning the same amount,
	    -- this is usually caused by an ethernet partner hiccupping.
	    --
	    NEW_TRACKING_TAN_I := MISSILE_STATUS.TRACKING_TAN_I;
	    NEW_TRACKING_TAN_K := MISSILE_STATUS.TRACKING_TAN_K;

	else
	    -- Difference in position of the missile and the target
	    --
	    declare
		T	: OBJECT_TYPE renames MISSILE_STATUS.TARGET.all;
		WP_DP   : WORLD_PHYSICS.POSITION;

		-- Difference in position of the missile and the target
		--
		DP_I	: METRES;
		DP_J	: METRES;
		DP_K	: METRES;

	    begin
		WP_DP.I := METRES(T.LOCATION.I - O.LOCATION.I);
		WP_DP.J := METRES(T.LOCATION.J - O.LOCATION.J);
		WP_DP.K := METRES(T.LOCATION.K - O.LOCATION.K);
		WORLD_PHYSICS.PROJECT_SAME_ORIGIN(
		    O.ORIENTATION.I,
		    O.ORIENTATION.J,
		    O.ORIENTATION.K,
		    WP_DP,
		    DP_I, DP_J, DP_K);

		-- Where is the target now?
		--
		NEW_TRACKING_TAN_I := SCALE_TYPE(DP_I)/SCALE_TYPE(DP_J);
		NEW_TRACKING_TAN_K := SCALE_TYPE(DP_K)/SCALE_TYPE(DP_J);
	    end;

	end if;

	-- Start/Continue tracking the target.
	--
	if not MISSILE_STATUS.TRACKING then
	    MISSILE_STATUS.TRACKING := TRUE;
	    MISSILE_STATUS.TRACKING_TAN_I := NEW_TRACKING_TAN_I;
	    MISSILE_STATUS.TRACKING_TAN_K := NEW_TRACKING_TAN_K;

	    A_ON_I  := 0.0;
	    A_ON_K  := 0.0;
	    A_ON_IK := 0.0;

	    CL	    := 0.0;

	else
	    declare
		-- Change in target position, try to regain all this.
		--
		DELTA_TAN_I : constant SCALE_TYPE
		    := (NEW_TRACKING_TAN_I - MISSILE_STATUS.TRACKING_TAN_I);

		DELTA_TAN_K : constant SCALE_TYPE
		    := (NEW_TRACKING_TAN_K - MISSILE_STATUS.TRACKING_TAN_K);

		-- maximum sideways acceleration
		--
		MAX_A_ON_IK : METRES_PER_SECOND_SQUARED;

	    begin

		-- Now pick acceleration along I and K to maintain lead on
		-- target.
		--
		A_ON_I := (V*DELTA_TAN_I) / TIME_DELTA;
		A_ON_K := (V*DELTA_TAN_K) / TIME_DELTA;

		A_ON_IK :=
		    METRES_PER_SECOND_SQUARED(
			SQRT(SCALE_TYPE(A_ON_I)**2+SCALE_TYPE(A_ON_K)**2));

		-- work out what the maximum sideways acceleration is
		--
		MAX_A_ON_IK := 
		    NEWTONS(DENSITY_V2*M_T.MAX_IK_NEWTONS_PER_DENSITY_V2)/
		    MASS;

		-- Decide the CL.  This hack is based on CL=1.0 being the max,
		-- which is ball-park...
		--
		CL := SCALE_TYPE(A_ON_IK)/SCALE_TYPE(MAX_A_ON_IK);

		-- make sure not stalled, by scaling down if necessary
		--
		if CL > 1.0 then
		    declare
			FRACTION : constant SCALE_TYPE := 1.0/CL;
		    begin
			A_ON_I  := A_ON_I*FRACTION;
			A_ON_K  := A_ON_K*FRACTION;
			A_ON_IK := MAX_A_ON_IK;
			CL	:= 1.0;
		    end;
		end if;
	    end;

	    -- A little damping in the lead to smooth network delays etc...
	    --
	    MISSILE_STATUS.TRACKING_TAN_I :=
		(NEW_TRACKING_TAN_I + 
		 MISSILE_STATUS.TRACKING_TAN_I * 5.0)/6.0;
	    MISSILE_STATUS.TRACKING_TAN_K :=
		(NEW_TRACKING_TAN_K + 
		 MISSILE_STATUS.TRACKING_TAN_K * 5.0)/6.0;
	end if;

	-- Calculate acceleration along J caused by
	--
	A_ON_J :=

	    (-- (a) engines

		MISSILE_STATUS.THRUST

		-- (b) drag due to straight flight

		-NEWTONS(M_T.ZERO_DRAG_MULTIPLIER*DENSITY_V2) 

	    )/MASS

	    -- (c) drag due to turning

	    -A_ON_IK*(M_T.IK_ACCELERATION_DRAG_MULTIPLIER*CL);

	-- Debugging info
	--
	if MISSILE_AERODYNAMICS_MOVE_CONTEXT.DEBUGGING then
	    declare
		use MISSILE_AERODYNAMICS_MOVE_CONTEXT, TEXT_IO, FLOAT_TEXT_IO;
	    begin
		PUT(F, "  Tan I  = "); PUT(F, FLOAT(MISSILE_STATUS.TRACKING_TAN_I));
		PUT(F, "  Tan K  = "); PUT(F, FLOAT(MISSILE_STATUS.TRACKING_TAN_K));
		PUT(F, "  A_ON_J = "); PUT(F, FLOAT(A_ON_J));
		PUT(F, "  A_ON_I = "); PUT(F, FLOAT(A_ON_I));
		PUT(F, "  A_ON_K = "); PUT(F, FLOAT(A_ON_K));
		NEW_LINE(F);
	    end;
	end if;

	-- Calculate new velocity
	--
	O.VELOCITY.I := O.VELOCITY.I +
	    (A_ON_I * SCALE_TYPE(O.ORIENTATION.I.I) +
	     A_ON_J * SCALE_TYPE(O.ORIENTATION.J.I) +
	     A_ON_K * SCALE_TYPE(O.ORIENTATION.K.I)
	    )*TIME_DELTA;

	O.VELOCITY.J := O.VELOCITY.J +
	    (A_ON_I * SCALE_TYPE(O.ORIENTATION.I.J) +
	     A_ON_J * SCALE_TYPE(O.ORIENTATION.J.J) +
	     A_ON_K * SCALE_TYPE(O.ORIENTATION.K.J)
	    )*TIME_DELTA;

	O.VELOCITY.K := O.VELOCITY.K +
	    (A_ON_I * SCALE_TYPE(O.ORIENTATION.I.K) +
	     A_ON_J * SCALE_TYPE(O.ORIENTATION.J.K) +
	     A_ON_K * SCALE_TYPE(O.ORIENTATION.K.K)
	    -METRES_PER_SECOND_SQUARED(PHYSICAL_UNITS.G)
	    )*TIME_DELTA;

	-- Add velocity*time_delta to the missile's position
	--
	MOVE_UTILITIES.ADD_MOVEMENT_TO_LOCATION(
	    O.VELOCITY*TIME_DELTA,
	    O.LOCATION);

	-- Check for ground contact
	--
	if O.LOCATION.K < 0.0 then
	    O.DELETE_PENDING := TRUE;
	    O.LOCATION.K := 0.0;
	end if;

	-- Set the new orientation
	--
	MISSILE_UTILITIES.SET_BASIS_FROM_VELOCITY(O,
	    MISSILE_STATUS.STABILIZE_ON_I_NOT_K);

    exception
	when others =>
	    O.DELETE_PENDING := TRUE;

    end;
