--***********************************************************************
--									*
--	COPYRIGHT 1992		DIGITAL EQUIPMENT CORPORATION		*
--									*
--   This software was written by Bevin Brett, of Digital Equipment	*
--   Corporation.							*
--									*
--   Digital assumes no responsibility AT ALL for the use or reliability*
--   of this software.							*
--									*
--   Redistribution and use in source and binary forms are permitted	*
--   provided that this entire heading from --*** to --*** are          *
--   duplicated in all such forms and that any documentation,		*
--   advertising materials, and other materials related to such		*
--   distribution and use acknowledge that the software was developed	*
--   by Digital Equipment Corporation. The name of Digital Equipment	*
--   Corporation may not be used to endorse or promote products derived	*
--   from this software without specific prior written permission.	*
--									*
--   THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR	*
--   IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
--   WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.*
--									*
--***********************************************************************


with MISSILE, MISSILE_GUIDANCE, MISSILE_AERODYNAMICS_MOVE_CONTEXT;

with TEXT_IO, FLOAT_TEXT_IO;

separate(MISSILE_AERODYNAMICS)
    procedure MOVE( O			    : in out OBJECTS.OBJECT_TYPE;
		    ELAPSED_SINCE_LAST_MOVE : SCALAR_PHYSICS.SECONDS
		    ) is

	MISSILE_STATUS 	: MISSILE_STATUS_TYPE
	    renames SUBCLASS_INFO_TO_ACCESS_MISSILE_STATUS(
			O.SUBCLASS_INFO).all;

	M_T	    : MISSILE_TYPE renames MISSILE_STATUS.MISSILE.all;

	ALTITUDE    : constant METRES := METRES(O.LOCATION.K);

	TIME_DELTA  : SECONDS
		    := ELAPSED_SINCE_LAST_MOVE +
			MISSILE_STATUS.TIME_DELTA_CARRIED_FORWARD;

    begin

	-- Air density
	--
	MISSILE_STATUS.DENSITY := STANDARD_ATMOSPHERE.TO_DENSITY(ALTITUDE);

	-- Engine thrust
	--
	declare
	    FUEL_REMAINING : KILOGRAMS renames MISSILE_STATUS.FUEL_MASS;
	    BURNT : KILOGRAMS;
	begin

	    MISSILE_STATUS.THRUST := 0.0;
	    if FUEL_REMAINING > 0.0 then

		if FUEL_REMAINING > M_T.FUEL_MASS_FOR_SUSTAIN then
		    MISSILE_STATUS.THRUST := M_T.ENGINE_BOOST_THRUST;
		else
		    MISSILE_STATUS.THRUST := M_T.ENGINE_SUSTAIN_THRUST;
		end if;

		BURNT := KILOGRAMS(
			    M_T.ENGINE_CONSUMPTION*
			    SCALE_TYPE(MISSILE_STATUS.THRUST)*
			    SCALE_TYPE(TIME_DELTA));

		if BURNT < FUEL_REMAINING then
		    FUEL_REMAINING := FUEL_REMAINING - BURNT;
		else
		    FUEL_REMAINING := 0.0;
		end if;

	    end if;
	end;

	-- Grow to kill radius when armed
	--
	MISSILE_STATUS.TIME_IN_FLIGHT :=
	    MISSILE_STATUS.TIME_IN_FLIGHT+TIME_DELTA;

	if  not MISSILE_STATUS.ARMED and then
	    MISSILE_STATUS.TIME_IN_FLIGHT > M_T.TIME_TO_ARM
	then
	    MISSILE_STATUS.ARMED := TRUE;
	    O.RADIUS := M_T.KILL_RADIUS;
	    O.EXTRA_RADII(1).RADIUS := M_T.KILL_RADIUS;
	end if;

	-- Select a target
	--
	declare
	    SIGNAL_STRENGTH : MISSILE.SIGNAL_STRENGTH_TYPE;
	    OLD_TARGET : constant OBJECTS.ACCESS_OBJECT_TYPE
		:= MISSILE_STATUS.TARGET;
	begin
	    MISSILE_GUIDANCE.GUIDE(
		O,
		SIGNAL_STRENGTH,
		MISSILE_STATUS.TARGET);

	    -- If none, give up after a while
	    --
	    if  OBJECTS."/="(MISSILE_STATUS.TARGET, null)
	    then
		MISSILE_STATUS.TIME_WITHOUT_TARGET := 0.0;
	    else
		MISSILE_STATUS.TIME_WITHOUT_TARGET :=
		    MISSILE_STATUS.TIME_WITHOUT_TARGET + TIME_DELTA;
		if MISSILE_STATUS.TIME_WITHOUT_TARGET > 2.0 then
		    O.DELETE_PENDING := TRUE;
		    return;
		end if;
	    end if;

	end;

	-- Step thru time
	--
	MOVE_DELTA(TIME_DELTA, O, MISSILE_STATUS);
	MISSILE_STATUS.TIME_DELTA_CARRIED_FORWARD := 0.0;

	-- Debugging info
	--
	if MISSILE_AERODYNAMICS_MOVE_CONTEXT.DEBUGGING then
	    declare
		use MISSILE_AERODYNAMICS_MOVE_CONTEXT, TEXT_IO, FLOAT_TEXT_IO;
	    begin
		PUT(F, " T = "); PUT(F, FLOAT(MISSILE_STATUS.TIME_IN_FLIGHT));
		PUT(F, " V = "); PUT(F, FLOAT(WORLD_PHYSICS.LENGTH(O.VELOCITY)));
		PUT(F, " LOCATION.J = ");   PUT(F, FLOAT(O.LOCATION.J));
		NEW_LINE(F,2);
	    end;
	end if;

    end;
