--***********************************************************************
--									*
--	COPYRIGHT 1992		DIGITAL EQUIPMENT CORPORATION		*
--									*
--   This software was written by Bevin Brett, of Digital Equipment	*
--   Corporation.							*
--									*
--   Digital assumes no responsibility AT ALL for the use or reliability*
--   of this software.							*
--									*
--   Redistribution and use in source and binary forms are permitted	*
--   provided that this entire heading from --*** to --*** are          *
--   duplicated in all such forms and that any documentation,		*
--   advertising materials, and other materials related to such		*
--   distribution and use acknowledge that the software was developed	*
--   by Digital Equipment Corporation. The name of Digital Equipment	*
--   Corporation may not be used to endorse or promote products derived	*
--   from this software without specific prior written permission.	*
--									*
--   THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR	*
--   IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
--   WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.*
--									*
--***********************************************************************


with
    LOGICAL_TO_STRING, SCALAR_PHYSICS,
    SCALE_TYPE_MATH_LIB, SCALE_TYPE_TRIG,
    STANDARD_ATMOSPHERE, TRIG,
    GENERIC_TABLE_LOOKUP;

with TEXT_IO, INTEGER_TEXT_IO, FLOAT_TEXT_IO;

pragma ELABORATE(
    SCALAR_PHYSICS,
    SCALE_TYPE_MATH_LIB, SCALE_TYPE_TRIG,
    STANDARD_ATMOSPHERE, TRIG,
    GENERIC_TABLE_LOOKUP);

package body LIFT_DRAG_CURVE_MANAGER is
    use SCALAR_PHYSICS, SCALE_TYPE_TRIG;


    NO_SUCH_AEROFOIL : exception;

    type ANGLE_INDEX_TYPE is range -360..360;
    subtype ANGLE_INDEX is ANGLE_INDEX_TYPE range -180..180;

    type ANGLE_VALUE is digits 4;

    type ANGLE_VALUE_ARRAY is array(ANGLE_INDEX range <>) of ANGLE_VALUE;
    subtype ALL_ANGLES_VALUE_ARRAY is ANGLE_VALUE_ARRAY(ANGLE_INDEX);

    type VECTOR_OF_ANGLE_INDEXES is array(POSITIVE range <>) of ANGLE_INDEX;
    type VECTOR_OF_ANGLE_VALUES  is array(POSITIVE range <>) of ANGLE_VALUE;


    package TABLE_LOOKUP is new GENERIC_TABLE_LOOKUP(
	DOMAIN_TYPE		    => TRIG.RADIANS,
	RANGE_TYPE		    => SCALE_TYPE,
	INTERMEDIATE_RESULTS_TYPE   => SCALE_TYPE,
	DOMAIN_FIRST		    => TRIG.RADIANS(-TRIG.PI),
	DOMAIN_LAST		    => TRIG.RADIANS(+TRIG.PI),
	NUMBER_OF_SUBDOMAINS	    => 360);


    type ACCESS_ALL_ANGLES_VALUE_ARRAY is access ALL_ANGLES_VALUE_ARRAY;

    type LIFT_AND_DRAG_CURVE_TYPE is
	record
	    LIFT_TABLE,
	    DRAG_TABLE		: ACCESS_ALL_ANGLES_VALUE_ARRAY;

	    LIFT_LOOKUP_TABLE,
	    DRAG_LOOKUP_TABLE	: TABLE_LOOKUP.TABLE_TYPE;
	    
	    -- The curve of ZERO_LIFT_DRAG against MACH looks like
	    --
	    --		/\
	    --	    ___/  \---  for a subpersonic jet,
	    --
	    -- and, more rounded on the way up, like
	    --
	    --		     |
	    --		    /
	    --		 __/
	    --	    ____/	for a subsonic plane with compression problems
	    --
	    --
	    -- so we model it with the points for the three angles, named after
	    -- where they occur on the jet, but letting us cut a chamfer on the
	    -- corner for the compression curves.
	    --
	    SUBSONIC_END_X,
	    SUBSONIC_END_Y,
	    TRANSONIC_MIDDLE_X,
	    TRANSONIC_MIDDLE_Y,
	    SUPERSONIC_BEGIN_X,
	    SUPERSONIC_BEGIN_Y	: SCALE_TYPE;

	    SLOPE_1,
	    SLOPE_2		: SCALE_TYPE;

	end record;


    procedure FILL_IN_ZERO_DRAG_INFO(
	LDC		    : in out LIFT_AND_DRAG_CURVE_TYPE;
	SUBSONIC_END_X,
	SUBSONIC_END_Y,
	TRANSONIC_MIDDLE_X,
	TRANSONIC_MIDDLE_Y,
	SUPERSONIC_BEGIN_X,
	SUPERSONIC_BEGIN_Y  : SCALE_TYPE) is
    begin
	LDC.SUBSONIC_END_X	:= SUBSONIC_END_X;
	LDC.SUBSONIC_END_Y	:= SUBSONIC_END_Y;
	LDC.TRANSONIC_MIDDLE_X  := TRANSONIC_MIDDLE_X;
	LDC.TRANSONIC_MIDDLE_Y  := TRANSONIC_MIDDLE_Y;
	LDC.SUPERSONIC_BEGIN_X  := SUPERSONIC_BEGIN_X;
	LDC.SUPERSONIC_BEGIN_Y  := SUPERSONIC_BEGIN_Y;

	begin
	    LDC.SLOPE_1 :=
		(LDC.TRANSONIC_MIDDLE_Y-LDC.SUBSONIC_END_Y)/
		(LDC.TRANSONIC_MIDDLE_X-LDC.SUBSONIC_END_X);
	exception
	    when CONSTRAINT_ERROR|NUMERIC_ERROR => LDC.SLOPE_1 := 0.0;
	end;

	begin
	    LDC.SLOPE_2 :=
		(LDC.SUPERSONIC_BEGIN_Y-LDC.TRANSONIC_MIDDLE_Y)/
		(LDC.SUPERSONIC_BEGIN_X-LDC.TRANSONIC_MIDDLE_X);
	exception
	    when CONSTRAINT_ERROR|NUMERIC_ERROR => LDC.SLOPE_2 := 0.0;
	end;
    end;


    procedure FILL_IN_ANGLE_DATA(
	SYMMETRY_MULTIPLIER : in ANGLE_VALUE;
	INDEX_DATA	    : in VECTOR_OF_ANGLE_INDEXES;
	VALUE_DATA	    : in VECTOR_OF_ANGLE_VALUES;
	CURVE		    : in out ALL_ANGLES_VALUE_ARRAY)
    is
	FIRST_ANGLE_INDEX   : constant ANGLE_INDEX
			    := INDEX_DATA(INDEX_DATA'first);

	LAST_ANGLE_INDEX    : constant ANGLE_INDEX
			    := INDEX_DATA(INDEX_DATA'last);

	N		    : POSITIVE := INDEX_DATA'first;
	BASE_INDEX	    : ANGLE_INDEX;
	BASE_VALUE	    : ANGLE_VALUE;
	SLOPE		    : FLOAT;
    begin

	-- Interpolate additional points
	--
	for I in FIRST_ANGLE_INDEX..LAST_ANGLE_INDEX loop
	    if I = INDEX_DATA(N) then
		BASE_INDEX := I;
		BASE_VALUE := VALUE_DATA(N);
		N := N + 1;
		SLOPE := 1.0;
		if N <= INDEX_DATA'last then
		    SLOPE := FLOAT(VALUE_DATA(N)-BASE_VALUE) /
			     FLOAT(INDEX_DATA(N)-BASE_INDEX);
		end if;
	    end if;
	    CURVE(I) := BASE_VALUE + ANGLE_VALUE(SLOPE*FLOAT(I-BASE_INDEX));
	end loop;

	-- Continue it to the end of the data
	--
	for I in LAST_ANGLE_INDEX+1..CURVE'last loop
	    CURVE(I) := VALUE_DATA(VALUE_DATA'last);
	end loop;

	-- Fill in the first portion by symmetry
	--
	for I in CURVE'first..FIRST_ANGLE_INDEX-1 loop
	    CURVE(I) := CURVE(CURVE'last);
	    declare
		SYMMETRY_POINT : constant ANGLE_INDEX_TYPE
		    := FIRST_ANGLE_INDEX+(FIRST_ANGLE_INDEX-I);
	    begin
		if SYMMETRY_POINT < LAST_ANGLE_INDEX then
		    CURVE(I) := CURVE(SYMMETRY_POINT)*SYMMETRY_MULTIPLIER;
		end if;
	    end;
	end loop;

    end;

    procedure DRAW_ANGLE_CURVE( INDEX_DATA  : in VECTOR_OF_ANGLE_INDEXES;
				VALUE_DATA  : in VECTOR_OF_ANGLE_VALUES;
				CURVE	: in ALL_ANGLES_VALUE_ARRAY) is

	use  TEXT_IO, INTEGER_TEXT_IO, FLOAT_TEXT_IO;

	subtype A_LINE is STRING(1..72);

	RULER, LINE : A_LINE;
	FIRST_ANGLE, LAST_ANGLE : ANGLE_INDEX;

	HIGHEST_VALUE	: ANGLE_VALUE := CURVE(CURVE'first);
	LOWEST_VALUE	: ANGLE_VALUE := CURVE(CURVE'first);

	BIAS, SCALE	: FLOAT;

	procedure PLOT(LINE : in out A_LINE; VALUE : FLOAT; CH : CHARACTER) is
	    I : INTEGER :=
		    1+INTEGER(FLOAT(A_LINE'last-1)*(SCALE*(VALUE-BIAS)));
	begin
	    if I < LINE'first
	    or I > LINE'last
	    then
		return;
	    end if;
	    LINE(I) := CH;
	end;

    begin
	for I in CURVE'range loop
	    if CURVE(I) < LOWEST_VALUE  then LOWEST_VALUE  := CURVE(I); end if;
	    if CURVE(I) > HIGHEST_VALUE then HIGHEST_VALUE := CURVE(I); end if;
	end loop;

	BIAS := 0.0;
	if LOWEST_VALUE < 0.0 then BIAS := FLOAT(LOWEST_VALUE); end if;

	SCALE := 1.0 / (FLOAT(HIGHEST_VALUE)-BIAS);

	FIRST_ANGLE := CURVE'first;
	while CURVE(FIRST_ANGLE) = CURVE(FIRST_ANGLE+1) loop
	    FIRST_ANGLE := FIRST_ANGLE + 1;
	end loop;

	LAST_ANGLE := CURVE'last;
	while CURVE(LAST_ANGLE) = CURVE(LAST_ANGLE-1) loop
	    LAST_ANGLE := LAST_ANGLE - 1;
	end loop;

	RULER := (others=>' ');
	for I in 1..4 loop
	    PLOT(RULER, FLOAT(I)/10.0, '|');
	end loop;
	PLOT(RULER, 0.0, '0');

	for I in FIRST_ANGLE..LAST_ANGLE loop
	    if I rem 2 = 0 then
		LINE := RULER;
		PUT("!");
		PUT(INTEGER(I), width=>3);
		SET_COL(8);
		PLOT(LINE, FLOAT(CURVE(I)), '*');
		PUT_LINE(LINE);
	    end if;
	end loop;
    end;

    procedure FILL_IN_AND_DRAW_ANGLE_DATA(
	AEROFOIL_NAME	    : in STRING;
	CURVE_NAME	    : in STRING;
	SYMMETRY_MULTIPLIER : in ANGLE_VALUE;
	INDEX_DATA	    : in VECTOR_OF_ANGLE_INDEXES;
	VALUE_DATA	    : in VECTOR_OF_ANGLE_VALUES;
	CURVE		    : in out ALL_ANGLES_VALUE_ARRAY)
    is
	use  TEXT_IO, INTEGER_TEXT_IO, FLOAT_TEXT_IO;
    begin

	FILL_IN_ANGLE_DATA(SYMMETRY_MULTIPLIER, INDEX_DATA, VALUE_DATA, CURVE);

	if not BOOLEAN'value(LOGICAL_TO_STRING("FCTM_DISPLAY_LD", "FALSE"))
	then
	    return;
	end if;

	PUT_LINE("!");
	PUT("! ");
	PUT_LINE(AEROFOIL_NAME);
	PUT_LINE("!");

	DRAW_ANGLE_CURVE  (INDEX_DATA, VALUE_DATA, CURVE);

	NEW_LINE(2);

	declare
	    subtype S is ANGLE_INDEX range -180/2 .. 180/2;
	    C : NATURAL := 0;
	begin
	    for I in S loop
		if C = 0 then
		    PUT("AERODYNAMICS/");
		    PUT(CURVE_NAME);
		    PUT("=(");
		end if;
		PUT(FLOAT(CURVE(I*2)), exp=>0, aft=>4);
		C := C + 1;
		if (C < 6) and (I /= S'last) then
		    PUT(", ");
		else
		    PUT_LINE(")");
		    C := 0;
		end if;
	    end loop;
	    NEW_LINE;
	end;
    end;


    procedure LOAD_LIFT_AND_DRAG_INFO(
	AEROFOIL_NAME	    : STRING;
	LIFT_AND_DRAG	    : in out LIFT_AND_DRAG_CURVE_TYPE) is

	LIFT_TABLE : ACCESS_ALL_ANGLES_VALUE_ARRAY renames
			LIFT_AND_DRAG.LIFT_TABLE;
	DRAG_TABLE : ACCESS_ALL_ANGLES_VALUE_ARRAY renames
			LIFT_AND_DRAG.DRAG_TABLE;

	procedure MAKE_TABLES is
	begin
	    LIFT_TABLE := new ALL_ANGLES_VALUE_ARRAY;
	    DRAG_TABLE := new ALL_ANGLES_VALUE_ARRAY;
	end;

	procedure MAKE_LOOKUP_TABLES is
	    use TRIG;

	    T : ACCESS_ALL_ANGLES_VALUE_ARRAY;

	    procedure GET_PAIR(
		I	    : POSITIVE;
		CONTINUE    : in out BOOLEAN;  -- set FALSE when I is last available
		D	    : out RADIANS; -- I1 < I2 must imply D1 < D2
		R	    : out SCALE_TYPE) is

		N : ANGLE_INDEX := ANGLE_INDEX(INTEGER(T'first)+(I-1));

	    begin
		CONTINUE := (N < T'last);
		D := RADIANS(RADIANS_PER_CIRCLE)*
		     (SCALE_TYPE(N)/SCALE_TYPE(360.0));
		R := SCALE_TYPE(T(N));
	    end;

	    procedure CREATE is new TABLE_LOOKUP.GENERIC_CREATE(GET_PAIR);

	begin
	    T := LIFT_AND_DRAG.LIFT_TABLE;
	    CREATE(LIFT_AND_DRAG.LIFT_LOOKUP_TABLE);
	    T := LIFT_AND_DRAG.DRAG_TABLE;
	    CREATE(LIFT_AND_DRAG.DRAG_LOOKUP_TABLE);
	end;

    begin

	-- lift v. drag
	--
	if AEROFOIL_NAME = "Clark-y (1922)"
	or AEROFOIL_NAME = "Spit"
	or AEROFOIL_NAME = "P47"
	or AEROFOIL_NAME = "P51"
	or AEROFOIL_NAME = "BF109"
	or AEROFOIL_NAME = "FW190"
	then
	    MAKE_TABLES;

	    declare
		DRAG_INDEX_DATA : VECTOR_OF_ANGLE_INDEXES(1..6)
			    := (  -5,    0,    4,   12,   17,   25);
		DRAG_VALUE_DATA : constant VECTOR_OF_ANGLE_VALUES 
			    := (0.01, 0.02, 0.04, 0.09, 0.15, 0.40);

		LIFT_INDEX_DATA : VECTOR_OF_ANGLE_INDEXES(1..7)
			    := (  -5,   18,   30,   35,   45,   70,   90);
		LIFT_VALUE_DATA : constant VECTOR_OF_ANGLE_VALUES 
			    := (0.00,  1.5,  1.0, 0.75, 0.70, 0.40,  0.0);

		INDEX_BIAS : constant ANGLE_INDEX_TYPE := -5;

	    begin
		-- bias it for the installation angle
		--
		for I in DRAG_INDEX_DATA'range loop
		    DRAG_INDEX_DATA(I) := DRAG_INDEX_DATA(I) - INDEX_BIAS;
		end loop;

		for I in LIFT_INDEX_DATA'range loop
		    LIFT_INDEX_DATA(I) := LIFT_INDEX_DATA(I) - INDEX_BIAS;
		end loop;

		-- do the curves
		--
		FILL_IN_AND_DRAW_ANGLE_DATA(AEROFOIL_NAME, "DRAG",
		    1.0, DRAG_INDEX_DATA, DRAG_VALUE_DATA, DRAG_TABLE.all);
		FILL_IN_AND_DRAW_ANGLE_DATA(AEROFOIL_NAME, "LIFT",
		    -0.75, LIFT_INDEX_DATA, LIFT_VALUE_DATA, LIFT_TABLE.all);

		-- build the lookup tables
		--
		MAKE_LOOKUP_TABLES;
	    end;

	elsif AEROFOIL_NAME = "NACA64" then
	    MAKE_TABLES;

	    declare
		DRAG_INDEX_DATA : VECTOR_OF_ANGLE_INDEXES(1..5)
			    := (  -2,    2,    6,   12,   20);
		DRAG_VALUE_DATA : constant VECTOR_OF_ANGLE_VALUES 
			    := (0.035, 0.04, 0.06, 0.12, 0.26);

		LIFT_INDEX_DATA : VECTOR_OF_ANGLE_INDEXES(1..9)
			    := (  -8,   -2,    0,     2,  12,   16,
				  30,   45,   90);  -- -8, and 30+ are guesses based on MISES text
		LIFT_VALUE_DATA : constant VECTOR_OF_ANGLE_VALUES 
			    := (0.00, 0.35,  0.45,  0.6, 1.2,  1.4,
				1.0,  0.70,  0.0);

		INDEX_BIAS : constant ANGLE_INDEX_TYPE := -8;

	    begin
		-- bias it for the installation angle
		--
		for I in DRAG_INDEX_DATA'range loop
		    DRAG_INDEX_DATA(I) := DRAG_INDEX_DATA(I) - INDEX_BIAS;
		end loop;

		for I in LIFT_INDEX_DATA'range loop
		    LIFT_INDEX_DATA(I) := LIFT_INDEX_DATA(I) - INDEX_BIAS;
		end loop;

		-- do the curves
		--
		FILL_IN_AND_DRAW_ANGLE_DATA(AEROFOIL_NAME, "DRAG",
		    1.0, DRAG_INDEX_DATA, DRAG_VALUE_DATA, DRAG_TABLE.all);
		FILL_IN_AND_DRAW_ANGLE_DATA(AEROFOIL_NAME, "LIFT",
		    -0.75, LIFT_INDEX_DATA, LIFT_VALUE_DATA, LIFT_TABLE.all);

		-- build the lookup tables
		--
		MAKE_LOOKUP_TABLES;
	    end;

	elsif AEROFOIL_NAME = "F-4" then
		FILL_IN_ZERO_DRAG_INFO(LIFT_AND_DRAG,
		    0.9,	0.02,	-- From MCAD pg 87
		    1.2,	0.045,
		    2.5,	0.025);

	elsif AEROFOIL_NAME = "MiG-21" then
		FILL_IN_ZERO_DRAG_INFO(LIFT_AND_DRAG,
		    0.95,	0.012,	-- Guesses
		    1.3,	0.030,
		    3.0,	0.016);

	else
	    raise NO_SUCH_AEROFOIL;

	end if;

	if AEROFOIL_NAME = "Clark-y (1922)"
	then
	    FILL_IN_ZERO_DRAG_INFO(LIFT_AND_DRAG,
		SCALE_TYPE'last, 0.0,
		SCALE_TYPE'last, 0.0,
		SCALE_TYPE'last, 0.0);

	elsif AEROFOIL_NAME = "Spit"
	then
	    FILL_IN_ZERO_DRAG_INFO(LIFT_AND_DRAG,
		0.60, 0.0200 - 0.01,	-- Fluid Dynamic Drag, 15-34
		0.73, 0.0225 - 0.01,	-- biased by Clark-y Cdz
		0.80, 1.0000 - 0.01);
	
	elsif AEROFOIL_NAME = "P47"
	then
	    FILL_IN_ZERO_DRAG_INFO(LIFT_AND_DRAG,
		0.60, 0.0275 - 0.01,	-- guesses based on Spitfire etc.
		0.70, 0.0290 - 0.01,
		0.80, 1.0000 - 0.01);

	elsif AEROFOIL_NAME = "P51"
	then
	    FILL_IN_ZERO_DRAG_INFO(LIFT_AND_DRAG,
		0.60, 0.0215 - 0.01,	-- Fluid Dynamic Drag, 15-34
		0.71, 0.0235 - 0.01,	-- biased by Clark-y Cdz
		0.80, 1.0000 - 0.01);

	elsif AEROFOIL_NAME = "BF109"
	then
	    FILL_IN_ZERO_DRAG_INFO(LIFT_AND_DRAG,
		0.60, 0.0250 - 0.01,	-- Fluid Dynamic Drag, 15-34
		0.70, 0.0275 - 0.01,	-- biased by Clark-y Cdz
		0.80, 1.0000 - 0.01);

	elsif AEROFOIL_NAME = "FW190"
	then
	    FILL_IN_ZERO_DRAG_INFO(LIFT_AND_DRAG,
		0.60, 0.0215 - 0.01,	-- guesses based on Spitfire etc.
		0.71, 0.0235 - 0.01,
		0.80, 1.0000 - 0.01);

	elsif AEROFOIL_NAME = "F-4" then
	    FILL_IN_ZERO_DRAG_INFO(LIFT_AND_DRAG,
		0.9,	0.02,	-- From MCAD pg 87
		1.2,	0.045,
		2.5,	0.025);

	elsif AEROFOIL_NAME = "MiG-21" then
	    FILL_IN_ZERO_DRAG_INFO(LIFT_AND_DRAG,
		0.95,	0.012,	-- Guesses
		1.3,	0.030,
		3.0,	0.016);

	else
	    raise NO_SUCH_AEROFOIL;

	end if;

    end;


    function ZERO_LIFT_DRAG(
	MACH		: SCALE_TYPE;
	LIFT_DRAG_CURVE : LIFT_AND_DRAG_CURVE_TYPE) return SCALE_TYPE is

	LDC : LIFT_AND_DRAG_CURVE_TYPE renames LIFT_DRAG_CURVE;

	function INTERPOLATE(X1, Y1, SLOPE, X : SCALE_TYPE) return SCALE_TYPE is
	begin
	    return Y1 + SLOPE*(X-X1);
	end;

    begin
	-- From MCAD pg 87
	--
	if MACH < LDC.SUBSONIC_END_X then
	    return LDC.SUBSONIC_END_Y;
	elsif MACH < LDC.TRANSONIC_MIDDLE_X then
	    return INTERPOLATE(
		LDC.SUBSONIC_END_X,	LDC.SUBSONIC_END_Y,
		LDC.SLOPE_1,
		MACH);
	elsif MACH < LDC.SUPERSONIC_BEGIN_X then
	    return INTERPOLATE(
		LDC.TRANSONIC_MIDDLE_X, LDC.TRANSONIC_MIDDLE_Y,
		LDC.SLOPE_2,
		MACH);
	else
	    return LDC.SUPERSONIC_BEGIN_Y;
	end if;
    end;


    function GET_LIFT_DRAG_CURVE(NAME : STRING)
	return ACCESS_LIFT_AND_DRAG_CURVE_TYPE is

	A   : ACCESS_LIFT_AND_DRAG_CURVE_TYPE := null;

    begin
	if NAME /= "" then
	    A := new LIFT_AND_DRAG_CURVE_TYPE;
	    LOAD_LIFT_AND_DRAG_INFO(NAME, A.all);
	end if;
	return A;
    end;


    procedure COMPUTE_CL_ALPHAS_AND_CD_PER_CL2(
	ASPECT_RATIO	    : SCALE_TYPE;
	SIN_SWEEP_ANGLE     : SCALE_TYPE;
	COS_SWEEP_ANGLE     : SCALE_TYPE;
	PARTIAL_LAMBDA_50   : SCALE_TYPE;
	MACH		    : STANDARD_ATMOSPHERE.MACH_TYPE;
	ORV_I, ORV_J	    : METRES_PER_SECOND;

	CL_ALPHA_LEFT	    : out SCALE_TYPE;
	CL_ALPHA_RIGHT	    : out SCALE_TYPE;
	CL_MAX		    : out SCALE_TYPE;
	CD_PER_CL2	    : out SCALE_TYPE) is

	use SCALE_TYPE_MATH_LIB;

    -- CL_ALPHA is the slope of the lift curve near zero AoA
    --
	-- see MCAD pg 44 for the following formulae
	--
	A   : SCALE_TYPE renames ASPECT_RATIO;
	SSA : SCALE_TYPE renames SIN_SWEEP_ANGLE;
	CSA : SCALE_TYPE renames COS_SWEEP_ANGLE;
	M   : SCALE_TYPE;

	-- unit vector in direction of V
	--
	VI  : SCALE_TYPE;
	VJ  : SCALE_TYPE;
	V   : SCALE_TYPE;

	-- unit vector perpendicular to V
	--
	UI  : SCALE_TYPE;
	UJ  : SCALE_TYPE;


	function COMPUTE_CL_ALPHA(
	    TAN_SWEEP : SCALE_TYPE) return SCALE_TYPE is

	    F2  : SCALE_TYPE;
	    PI  : constant SCALE_TYPE := 3.1416;

	begin

	    F2 := 1.0-M*M+
		(TAN_SWEEP-PARTIAL_LAMBDA_50)**2;

	    return
		(PI*A)/(1.0+SQRT(1.0+F2*(A/2.0)**2));

	end;

    begin
	-- Decide on the relevant mach number
	--
	M := SCALE_TYPE(MACH);
	if M > 0.7 then M := 0.7; end if;

	-- Compute the unit vector in the direction of the velocity
	--
	VI := SCALE_TYPE(ORV_I);
	VJ := SCALE_TYPE(ORV_J);

	if VJ <= 0.0 then
	    -- a very bad wing indeed, if you try to go backwards!
	    --
	    CL_ALPHA_LEFT  := 0.5;
	    CL_ALPHA_RIGHT := 0.5;
	else
	    V := SQRT(VI**2+VJ**2);
	    VI := VI/V;
	    VJ := VJ/V;

	    -- The unit vector at right angles to the velocity
	    --
	    UI :=  VJ;
	    UJ := -VI;

	    -- Compute the CL_ALPHA for each wing
	    --
	    CL_ALPHA_LEFT  :=
		COMPUTE_CL_ALPHA((CSA*VI+SSA*VJ)/(CSA*UI+SSA*UJ));

	    CL_ALPHA_RIGHT :=
		COMPUTE_CL_ALPHA((CSA*VI-SSA*VJ)/(SSA*UJ-CSA*UI));
	end if;

	-- See MCAD page 198 for an example of this curve.  I haven't yet
	-- found a reference for estimating this curve.  This 1.0 is a
	-- very bad approximation...
	--
	CL_MAX := 1.0;

	-- See "Aerodynamics of Wings and Bodies", Ashley and Landahl, 1965
	-- leads to the following approximate formulation of this value.
	--
	declare
	    TMP : SCALE_TYPE := M;
	    function INTERPOLATE(X1, Y1, X2, Y2, X : SCALE_TYPE)
		return SCALE_TYPE is
	    begin
		return Y1+(Y2-Y1)/(X2-X1)*(X-X1);
	    end;
	begin
	    if TMP > 1.0 then TMP := 1.0; end if;
	    if TMP < 0.2 then
		TMP := 0.3;
	    elsif TMP < 0.9 then
		TMP := INTERPOLATE(0.2, 0.3, 0.9, 0.75, TMP);
	    else
		TMP := INTERPOLATE(0.9, 0.75, 1.0, 0.95, TMP);
	    end if;
	    CD_PER_CL2 := TMP/A;
	end;

    end;


    procedure GET_LIFT_AND_DRAG(
	INPUTS	: in  LIFT_DRAG_INPUTS;
	OUTPUTS	: out LIFT_DRAG_OUTPUTS) is

	LDC		: LIFT_AND_DRAG_CURVE_TYPE  renames INPUTS.LIFT_DRAG_CURVE.all;
	ANGLE_OF_ATTACK : TRIG.RADIANS		    renames INPUTS.ANGLE_OF_ATTACK;
	CL_ALPHA	: SCALE_TYPE		    renames INPUTS.CL_ALPHA;
	EXTRA_ZERO_DRAG : SCALE_TYPE		    renames INPUTS.EXTRA_ZERO_DRAG;
	MACH		: MACH_TYPE		    renames INPUTS.MACH;
	AEROLONS	: AEROLON_SUBTYPE	    renames INPUTS.AEROLONS;
	HIGH_LIFT	: HIGH_LIFT_SUBTYPE	    renames INPUTS.HIGH_LIFT;

	LIFT		: LIFT_SUBTYPE		    renames OUTPUTS.CL;
	DRAG		: DRAG_SUBTYPE		    renames OUTPUTS.CD;

	D		: DRAG_SUBTYPE;
	L		: LIFT_SUBTYPE;
    begin
	-- Table lookup
	--
	if LDC.LIFT_TABLE /= null then
	    declare
		use TRIG;

		X   : RADIANS := ANGLE_OF_ATTACK;

	    begin
		OUTPUTS.AOA_MAX := 0.0;	-- NYI

		if abs X > PI then
		    while X >  PI loop X := X-PI; end loop;
		    while X < -PI loop X := X+PI; end loop;
		end if;

		-- HACK, no good justification for EXTRA_LIFT
		--
		declare
		    EXTRA_LIFT : SCALE_TYPE := 0.5*HIGH_LIFT + 0.3*AEROLONS;
		begin
		    L := TABLE_LOOKUP.F(LDC.LIFT_LOOKUP_TABLE,X) +
			 EXTRA_LIFT;
		    D := TABLE_LOOKUP.F(LDC.DRAG_LOOKUP_TABLE,X) +
			 ZERO_LIFT_DRAG(SCALE_TYPE(MACH), LDC) +
			 EXTRA_ZERO_DRAG +
			 0.02*EXTRA_LIFT**3;
		end;
	    end;
	else
	    declare
		use TRIG;

		NINETY_DEGREES	: constant SCALE_TYPE
				:= SCALE_TYPE(RADIANS_PER_CIRCLE)/4.0;

		A, PEAK_A	: SCALE_TYPE;
		LD		: SCALE_TYPE;

	    begin
		-- Lift devices.  0.7 is about right for slotted flaps + slots
		--
		LD := (0.7*HIGH_LIFT + 0.3*AEROLONS);

		-- From MCAD pg 43,44,45,87,92
		--
		A := SCALE_TYPE(ANGLE_OF_ATTACK);

		PEAK_A := INPUTS.CL_MAX / CL_ALPHA;

		OUTPUTS.AOA_MAX := TRIG.RADIANS(PEAK_A);

		if abs(A) > PEAK_A then
		    if A > 0.0 then
			if A > NINETY_DEGREES then
			    A := NINETY_DEGREES;
			end if;
			A := PEAK_A*(NINETY_DEGREES-A)/(NINETY_DEGREES-PEAK_A);
		    else
			if A < -NINETY_DEGREES then
			    A := -NINETY_DEGREES;
			end if;
			A := -PEAK_A*(-NINETY_DEGREES-A)/(-NINETY_DEGREES+PEAK_A);
		    end if;
		    L  := LD + A*CL_ALPHA;
		    LD := LD + PEAK_A*CL_ALPHA;
		else
		    L  := LD + A*CL_ALPHA;
		    LD := L;
		end if;


		-- Drag, see Aerodynamics for Naval Aviators, fig 1.17, page 40
		--
		D := ZERO_LIFT_DRAG(SCALE_TYPE(MACH), LDC) +
		    EXTRA_ZERO_DRAG +
		    (0.2*(HIGH_LIFT**3)) +
		    INPUTS.CD_PER_CL2*LD**2;

	    end;
	end if;

	LIFT := L;
	DRAG := D;
    end;

end;
