--***********************************************************************
--									*
--	COPYRIGHT 1992		DIGITAL EQUIPMENT CORPORATION		*
--									*
--   This software was written by Bevin Brett, of Digital Equipment	*
--   Corporation.							*
--									*
--   Digital assumes no responsibility AT ALL for the use or reliability*
--   of this software.							*
--									*
--   Redistribution and use in source and binary forms are permitted	*
--   provided that this entire heading from --*** to --*** are          *
--   duplicated in all such forms and that any documentation,		*
--   advertising materials, and other materials related to such		*
--   distribution and use acknowledge that the software was developed	*
--   by Digital Equipment Corporation. The name of Digital Equipment	*
--   Corporation may not be used to endorse or promote products derived	*
--   from this software without specific prior written permission.	*
--									*
--   THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR	*
--   IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
--   WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.*
--									*
--***********************************************************************


with
    IS_DEMO_PKG,
    CALENDAR, TRIG, LOGICAL_TO_INTEGER;
use 
    CALENDAR;

pragma ELABORATE(
    IS_DEMO_PKG,
    CALENDAR, TRIG, LOGICAL_TO_INTEGER);

separate(GRAPHICS_WINDOW_MANAGER)
    package body GW_INSTRUMENT_MANAGER is

	type SCALE_TYPE is new FLOAT;
	type ACCESS_STRING is access STRING;


	POWERS_OF_TEN : constant array(0..5) of INSTRUMENT_VALUE_TYPE
	    := (10.0**0, 10.0**1, 10.0**2, 10.0**3, 10.0**4, 10.0**5);


	-- Because drawing hands is so expensive in CPU time, the following
	-- structure is used to cache as much of the calculation as possible.
	--
	type HAND_LINE_TYPE is
	    record
		X1 : WIDTH_PIXEL_COUNT_SUBTYPE;
		Y1 : HEIGHT_PIXEL_COUNT_SUBTYPE;
		X2 : WIDTH_PIXEL_COUNT_SUBTYPE;
		Y2 : HEIGHT_PIXEL_COUNT_SUBTYPE;
	    end record;

	type VECTOR_OF_HAND_LINE_TYPE is
	    array(INTEGER range <>) of HAND_LINE_TYPE;

	type HAND_LINES_TYPE(NUMBER_OF_LINES : POSITIVE) is
	    record
		LINES : VECTOR_OF_HAND_LINE_TYPE(1..NUMBER_OF_LINES);
	    end record;

	type ACCESS_HAND_LINES_TYPE is access HAND_LINES_TYPE;

	type HAND_TICK_TYPE is new INTEGER range -5..365;
	HAND_TICK_PER_CIRCLE : constant HAND_TICK_TYPE := 360;

	type HAND_TICK_TO_ACCESS_HAND_LINES_TYPE is
	    array(HAND_TICK_TYPE) of ACCESS_HAND_LINES_TYPE;

	type VECTOR_OF_HAND_TICK_TO_ACCESS_HAND_LINES_TYPE is
	    array(POSITIVE range <>) of HAND_TICK_TO_ACCESS_HAND_LINES_TYPE;

	type ACCESS_VECTOR_OF_HAND_TICK_TO_ACCESS_HAND_LINES_TYPE is
	    access VECTOR_OF_HAND_TICK_TO_ACCESS_HAND_LINES_TYPE;


	package SCALE_TYPE_TRIG is new TRIG.TRIG_FUNCTIONS(SCALE_TYPE);

	type GW_INSTRUMENT_TYPE(
		APPEARANCE  : INSTRUMENT_APPEARANCE_TYPE;
		SCALE	    : INSTRUMENT_SCALE_TYPE)
	is
	    record
		NEXT	    : ACCESS_GW_INSTRUMENT_TYPE;
		NAME	    : INSTRUMENT_NAME_TYPE;

		POSITION_X  : WIDTH_PIXEL_COUNT_SUBTYPE;
		POSITION_Y  : HEIGHT_PIXEL_COUNT_SUBTYPE;

		WIDTH	    : WIDTH_PIXEL_COUNT_SUBTYPE;
		HEIGHT	    : HEIGHT_PIXEL_COUNT_SUBTYPE;

		LO, TOP, HI : INSTRUMENT_VALUE_TYPE;
		VALUE       : INSTRUMENT_VALUE_TYPE;

		INSTRUMENT_VALUE_PER_RADIAN : INSTRUMENT_VALUE_TYPE;
		RADIANS_PER_NUMBERED_TICK   : TRIG.RADIANS;
		TICKS_PER_NUMBERED_TICK     : POSITIVE;

		case SCALE is
		    when LINEAR|CLOCK =>
			NUMBER_OF_HANDS : POSITIVE;
			HAND_TO_LINES	:
			    ACCESS_VECTOR_OF_HAND_TICK_TO_ACCESS_HAND_LINES_TYPE;
		    when TEXT =>
			CHANGED 	: BOOLEAN;
			TEXT_VALUE	: ACCESS_STRING;
		end case;
	    end record;

	GW   : GRAPHICS_WINDOW_TYPE;

	BELL_TIME   : TIME := CLOCK;
	BELL_VOLUME : constant KEYBOARD_PERCENT_TYPE
		    :=	KEYBOARD_PERCENT_TYPE(
			    LOGICAL_TO_INTEGER("FCTM_BELL_VOLUME", -99));


	function "*"(
	    LEFT	: PIXEL_COUNT_TYPE;
	    RIGHT	: SCALE_TYPE)
	    return PIXEL_COUNT_TYPE is
	begin
	    return PIXEL_COUNT_TYPE(SCALE_TYPE(LEFT)*RIGHT);
	end;


	procedure DRAW(WINDOW : WINDOW_INFO_TYPE; I : in out GW_INSTRUMENT_TYPE) is

	    procedure PUT_CENTRED_TEXT(
		X	: WIDTH_PIXEL_COUNT_SUBTYPE;
		Y	: HEIGHT_PIXEL_COUNT_SUBTYPE;
		S	: STRING) is

		LO,HI	: NATURAL;

	    begin
		HI := S'last;
		while S'first < HI and S(HI) = ' ' loop
		    HI:= HI-1;
		end loop;

		LO := S'first;
		while LO < HI and S(LO) = ' ' loop
		    LO := LO+1;
		end loop;

		DRAW_IMAGE_STRING(
		    DISPLAY,
		    DRAWABLE_TYPE(WINDOW.WINDOW),
		    WINDOW.GC_ALWAYS,
		    X - (WIDTH_PER_CHAR*WIDTH_PIXEL_COUNT_SUBTYPE(HI-LO+1))/2,
		    Y,
		    S(LO..HI));
	    end;


	    procedure DRAW_TICKS is
		use TRIG, SCALE_TYPE_TRIG;

		N	: INTEGER;
		R	: RADIANS := 0.0;
		R_STEP	: RADIANS := RADIANS(RADIANS_PER_CIRCLE/10.0);

		procedure DRAW_TICK(
		    R		: RADIANS;
		    LENGTH	: SCALE_TYPE;
		    DRAW_NUMBER : BOOLEAN) is

		    ONE : constant SCALE_TYPE := 1.0;
		    TWO : constant SCALE_TYPE := 2.0;

		    SIN_R,COS_R : SCALE_TYPE;

		begin
		    SIN_R :=  SIN(R);
		    COS_R := -COS(R);

		    DRAW_SEGMENTS(
			DISPLAY,
			DRAWABLE_TYPE(WINDOW.WINDOW),
			WINDOW.GC_ALWAYS,
			(1=>
			    (I.POSITION_X + I.WIDTH  * ((ONE+SIN_R)/TWO),
			     I.POSITION_Y + I.HEIGHT * ((ONE+COS_R)/TWO),
			     I.POSITION_X + I.WIDTH  * ((ONE+SIN_R*LENGTH)/TWO),
			     I.POSITION_Y + I.HEIGHT * ((ONE+COS_R*LENGTH)/TWO)
			    )));

		    if DRAW_NUMBER then
			PUT_CENTRED_TEXT(
			    I.POSITION_X + I.WIDTH  * ((ONE+SIN_R*1.1)/TWO),
			    I.POSITION_Y + I.HEIGHT * ((ONE+COS_R*1.1)/TWO),
			    INTEGER'image(N));
		    end if;
		end;

	    begin

		if I.SCALE = LINEAR then
		    R := RADIANS((I.LO-I.TOP)/I.INSTRUMENT_VALUE_PER_RADIAN);
		    R_STEP := I.RADIANS_PER_NUMBERED_TICK;
		else
		    R := 0.0;
		    R_STEP := RADIANS(RADIANS_PER_CIRCLE/10.0);
		    N := -1;
		end if;

		loop
		    if I.SCALE = LINEAR then
			declare
			    V : INSTRUMENT_VALUE_TYPE;
			begin
			    V := INSTRUMENT_VALUE_TYPE(R)*
				    I.INSTRUMENT_VALUE_PER_RADIAN
				+I.TOP;
			    exit when V > I.HI*1.001;
			    N := INTEGER(V);
			end;
		    else
			exit when N = 9;
			N := N+1;
		    end if;
		    DRAW_TICK(R, 0.925, TRUE);
		    for J in 1..I.TICKS_PER_NUMBERED_TICK loop
			DRAW_TICK(R+R_STEP*
				    (RADIANS(J)/
				     RADIANS(I.TICKS_PER_NUMBERED_TICK+1)),
			    0.975, FALSE);
		    end loop;
		    R := R + R_STEP;
		end loop;
	    end;


	begin
	    case I.APPEARANCE is

		when DIAL =>
		    DRAW_ARC(
			DISPLAY,
			DRAWABLE_TYPE(WINDOW.WINDOW),
			WINDOW.GC_ALWAYS,
			I.POSITION_X,
			I.POSITION_Y,
			I.WIDTH,
			I.HEIGHT,
			0.0,
			360.0);

		    PUT_CENTRED_TEXT(
			I.POSITION_X + I.WIDTH/2,
			I.POSITION_Y + I.HEIGHT/4,
			STRING(I.NAME));

		    DRAW_TICKS;

		when CLICKER | BEEPER =>
		    null;

		when BOX =>
		    declare
			X1 : WIDTH_PIXEL_COUNT_SUBTYPE
			     renames I.POSITION_X;
			Y1 : HEIGHT_PIXEL_COUNT_SUBTYPE
			     renames I.POSITION_Y;
			X2 : constant WIDTH_PIXEL_COUNT_SUBTYPE
			     := X1 + I.WIDTH;
			Y2 : constant HEIGHT_PIXEL_COUNT_SUBTYPE
			     := Y1 + I.HEIGHT;
		    begin
			DRAW_LINES(
			    DISPLAY,
			    DRAWABLE_TYPE(WINDOW.WINDOW),
			    WINDOW.GC_ALWAYS,
			    ((X1, Y1),
			     (X2, Y1),
			     (X2, Y2),
			     (X1, Y2),
			     (X1, Y1)),
			    ORIGIN);
			PUT_CENTRED_TEXT(
			    I.POSITION_X + I.WIDTH/2,
			    I.POSITION_Y,
			    STRING(I.NAME));
		    end;

		when others =>
		    NYI("GWM.INSTRUMENTS.DRAW appearance", TRUE);

	    end case;

	    if I.SCALE = TEXT then I.CHANGED := TRUE; end if;
	end;


	procedure DRAW(GW : GRAPHICS_WINDOW_TYPE) is
	    A : ACCESS_GW_INSTRUMENT_TYPE := GW.INSTRUMENTS;
	begin
	    while A /= null loop
		DRAW(GW.all, A.all);
		A := A.next;
	    end loop;
	end;


	function CREATE_INSTRUMENT(
	    NAME	    : INSTRUMENT_NAME_TYPE;
	    X, Y	    : SCREEN_MEASUREMENT;
	    HEIGHT,WIDTH    : SCREEN_MEASUREMENT;
	    APPEARANCE	    : INSTRUMENT_APPEARANCE_TYPE;
	    SCALE	    : INSTRUMENT_SCALE_TYPE
	    ) return ACCESS_GW_INSTRUMENT_TYPE is

	    A	: ACCESS_GW_INSTRUMENT_TYPE
		:= new GW_INSTRUMENT_TYPE(APPEARANCE, SCALE);

	    I	: GW_INSTRUMENT_TYPE renames A.all;

	    W	: WINDOW_INFO_TYPE renames GW.all;

	begin
	    I.NEXT := W.INSTRUMENTS;

	    I.NAME	    := NAME;

	    I.WIDTH	    :=  TO_WIDTH_PIXEL_COUNT_SUBTYPE(WIDTH);
	    I.HEIGHT	    := -TO_HEIGHT_PIXEL_COUNT_SUBTYPE(HEIGHT);

	    I.POSITION_X    := TO_WIDTH_PIXEL_COUNT_SUBTYPE(X);
	    I.POSITION_Y    := W.HEIGHT +
				TO_HEIGHT_PIXEL_COUNT_SUBTYPE(Y) -
				I.HEIGHT;

	    W.INSTRUMENTS := A;

	    return A;
	end;


	function CREATE_NUMERIC(
	    NAME	    : INSTRUMENT_NAME_TYPE;
	    X, Y	    : SCREEN_MEASUREMENT;
	    HEIGHT,WIDTH    : SCREEN_MEASUREMENT;
	    APPEARANCE	    : INSTRUMENT_APPEARANCE_TYPE;
	    SCALE	    : NUMERIC_INSTRUMENT_SCALE_TYPE;

	    LO, TOP, HI     : INSTRUMENT_VALUE_TYPE := 0.0;

	    INSTRUMENT_VALUE_PER_RADIAN : INSTRUMENT_VALUE_TYPE;
	    RADIANS_PER_NUMBERED_TICK	: TRIG.RADIANS;
	    TICKS_PER_NUMBERED_TICK	: POSITIVE

	    ) return ACCESS_GW_INSTRUMENT_TYPE is


	    A	: constant ACCESS_GW_INSTRUMENT_TYPE
		:= CREATE_INSTRUMENT(NAME, X,Y, HEIGHT,WIDTH, APPEARANCE,SCALE);

	    I	: GW_INSTRUMENT_TYPE renames A.all;

	    LO_NOT_ZERO, TOP_NOT_ZERO, HI_LEQ_LO : exception;


	begin
	    I.LO    := LO;
	    I.TOP   := TOP;
	    I.HI    := HI;

	    I.VALUE := LO;

	    I.INSTRUMENT_VALUE_PER_RADIAN   := INSTRUMENT_VALUE_PER_RADIAN;
	    I.RADIANS_PER_NUMBERED_TICK     := RADIANS_PER_NUMBERED_TICK;
	    I.TICKS_PER_NUMBERED_TICK	    := TICKS_PER_NUMBERED_TICK;

	    case SCALE is

		when LINEAR =>
		    I.NUMBER_OF_HANDS := 1;
		    I.HAND_TO_LINES :=
			new VECTOR_OF_HAND_TICK_TO_ACCESS_HAND_LINES_TYPE(
			    1..I.NUMBER_OF_HANDS);

		when CLOCK =>
		    if LO  /= 0.0 then raise LO_NOT_ZERO;  end if;
		    if TOP /= 0.0 then raise TOP_NOT_ZERO; end if;
		    if HI  <= LO  then raise HI_LEQ_LO;    end if;

		    declare
			T : INSTRUMENT_VALUE_TYPE := HI;
		    begin
			I.NUMBER_OF_HANDS := 1;
			T := T/1000.0;
			while T > 1.0 loop
			    I.NUMBER_OF_HANDS := I.NUMBER_OF_HANDS + 1;
			    T := T/10.0;
			end loop;
			I.HAND_TO_LINES :=
			    new VECTOR_OF_HAND_TICK_TO_ACCESS_HAND_LINES_TYPE(
				    1..I.NUMBER_OF_HANDS);
		    end;

	    end case;

	    return A;
	end;


	function CREATE_SIGHT(
	    STYLE	    : SIGHT_STYLE_TYPE;
	    SIZE	    : SCREEN_MEASUREMENT;
	    HOLE_SIZE	    : SCREEN_MEASUREMENT
	    ) return ACCESS_GW_INSTRUMENT_TYPE is
	begin
	    return null;
	end;


	function CREATE_TEXT(
	    NAME	    : INSTRUMENT_NAME_TYPE;
	    X, Y	    : GRAPHICS_WINDOW_MANAGER.SCREEN_MEASUREMENT;
	    HEIGHT,WIDTH    : SCREEN_MEASUREMENT;
	    MAX_LENGTH	    : POSITIVE
	    ) return ACCESS_GW_INSTRUMENT_TYPE is

	    A	: ACCESS_GW_INSTRUMENT_TYPE
		:= CREATE_INSTRUMENT(NAME, X,Y, HEIGHT,WIDTH, BOX, TEXT);

	begin
	    A.CHANGED    := TRUE;
	    A.TEXT_VALUE := new STRING'(1..MAX_LENGTH => ' ');
	    return A;
	end;


	procedure SET_NUMERIC(
	    INSTRUMENT	    : ACCESS_GW_INSTRUMENT_TYPE;
	    VALUE	    : INSTRUMENT_VALUE_TYPE) is
	begin
	    INSTRUMENT.VALUE := VALUE;
	end;


	procedure SET_SIGHT(
	    SIGHT_ON	    : BOOLEAN;
	    X, Y	    : SCREEN_MEASUREMENT) is
	begin
	    null;
	end;


	procedure SET_TEXT(
	    INSTRUMENT	    : ACCESS_GW_INSTRUMENT_TYPE;
	    VALUE	    : STRING) is

	    TEXT : STRING renames INSTRUMENT.TEXT_VALUE.all;
	begin
	    INSTRUMENT.CHANGED := TRUE;
	    TEXT := VALUE & (1..TEXT'length-VALUE'length=>' ');
	end;


	procedure UPDATE_NUMERIC(
	    I		    : GW_INSTRUMENT_TYPE) is

	    procedure DO_HAND_HARD_WAY(
		HAND	: POSITIVE;
		VALUE	: INSTRUMENT_VALUE_TYPE;
		HL	: out HAND_LINES_TYPE) is

		use TRIG, SCALE_TYPE_TRIG;

		R   : constant RADIANS
		    := RADIANS(VALUE)*RADIANS(RADIANS_PER_CIRCLE);

		HAND_LENGTH : constant SCALE_TYPE
		    := 0.95*(0.6**(HAND-1));

		SIN_R	: constant SCALE_TYPE := SIN(R);
		COS_R	: constant SCALE_TYPE := COS(R);
		DX	: constant SCALE_TYPE := HAND_LENGTH*SIN_R;
		DY	: constant SCALE_TYPE := HAND_LENGTH*COS_R;

		X_STEP	: SCALE_TYPE;
		Y_STEP	: SCALE_TYPE;

		function TO_X_PIXEL(
		    S : SCALE_TYPE
		    ) return WIDTH_PIXEL_COUNT_SUBTYPE is
		begin
		    return I.POSITION_X + I.WIDTH*((1.0+S)/2.0);
		end;
		
		function TO_Y_PIXEL(
		    S : SCALE_TYPE
		    ) return HEIGHT_PIXEL_COUNT_SUBTYPE is
		begin
		    return I.POSITION_Y + I.HEIGHT*((1.0-S)/2.0);
		end;
		
	    begin
		for H in -(HAND-1)..HAND-1 loop
		    X_STEP := (+COS_R)*SCALE_TYPE(H)*0.02;
		    Y_STEP := (-SIN_R)*SCALE_TYPE(H)*0.02;
		    declare
			L : HAND_LINE_TYPE renames HL.LINES(H+HAND);
		    begin
			L.X1 := TO_X_PIXEL(0.0 + X_STEP);
			L.Y1 := TO_Y_PIXEL(0.0 + Y_STEP);
			L.X2 := TO_X_PIXEL(DX  + X_STEP);
			L.Y2 := TO_Y_PIXEL(DY  + Y_STEP);
		    end;
		end loop;
	    end;

	    procedure DO_HAND(
		HAND	: POSITIVE;
		VALUE	: INSTRUMENT_VALUE_TYPE) is

		V : INSTRUMENT_VALUE_TYPE := VALUE;

	    begin
		-- Quickly get V into [0.0..1.0)
		--
		while V < 0.0 loop
		    V := V - INSTRUMENT_VALUE_TYPE(INTEGER(V)) + 1.0;
		end loop;

		while V >= 1.0 loop
		    V := V - INSTRUMENT_VALUE_TYPE(INTEGER(V));
		end loop;

		-- Look up that hand at that position
		--
		declare
		    HAND_TICK	: constant HAND_TICK_TYPE
				:= HAND_TICK_TYPE(V*INSTRUMENT_VALUE_TYPE(
					    HAND_TICK_PER_CIRCLE));

		    AHL 	: ACCESS_HAND_LINES_TYPE
				renames I.HAND_TO_LINES(HAND)(HAND_TICK);
		begin
		    -- If necessary, load it
		    --
		    if AHL = null then
			AHL := new HAND_LINES_TYPE(2*HAND-1);
			DO_HAND_HARD_WAY(HAND, VALUE, AHL.all);
		    end if;

		    -- Draw it
		    --
		    declare
			HL : HAND_LINES_TYPE renames AHL.all;
		    begin
			for LINE in HL.LINES'range loop
			    declare
				L : HAND_LINE_TYPE renames HL.LINES(LINE);
			    begin
				DRAW_LINE(
				    L.X1,
				    L.Y1,
				    L.X2,
				    L.Y2,
				    GW);
			    end;
			end loop;
		    end;
		end;
	    end;

	begin
	    case I.APPEARANCE is
		when DIAL =>

		    case I.SCALE is

			when LINEAR =>
			    declare
				V : INSTRUMENT_VALUE_TYPE := I.VALUE;
			    begin
				if	V < I.LO then V := I.LO;
				elsif	V > I.HI then V := I.HI;
				end if;

				DO_HAND(1, 0.9*(V-I.TOP)/(I.HI-I.LO));
			    end;

			when CLOCK =>
			    declare
				F : INSTRUMENT_VALUE_TYPE;
				V : INSTRUMENT_VALUE_TYPE := abs I.VALUE/1000.0;
			    begin
				F := POWERS_OF_TEN(I.NUMBER_OF_HANDS-1);
				for HAND in reverse 1..I.NUMBER_OF_HANDS loop
				    V := V - F*INSTRUMENT_VALUE_TYPE(
						    INTEGER(V/F));
				    if V < 0.0 then
					V := V + F;
				    elsif V >= F then
					V := V - F;
				    end if;
				    DO_HAND(HAND, V/F);
				    F := F/10.0;
				end loop;
			    end;

			when others =>
			    NYI("GRAPHICS_WINDOW_MANAGER." &
				"INSTRUMENTS.UPDATE_NUMERIC : " &
				"DIAL scale", TRUE);

		    end case;

		when BEEPER|CLICKER =>

		    if I.VALUE = 0.0
		    or IS_DEMO_PKG.IS_DEMO
		    then
			return;
		    end if;

		    declare
			INTENSITY : INSTRUMENT_VALUE_TYPE;
			CURR_TIME : TIME := CLOCK;
			BELL_ELAPSED_TIME : DURATION := CURR_TIME - BELL_TIME;
		    begin

			case I.SCALE is

			    when LINEAR =>
				INTENSITY := abs(I.VALUE/I.HI);

			    when CLOCK =>
				declare
				    GRAIN : constant INTEGER := 20;
				    LOG_2 : NATURAL := 0;
				begin
				    INTENSITY := abs(I.VALUE/I.HI);
				    while INTENSITY < 1.0 and LOG_2 < GRAIN loop
					LOG_2 := LOG_2+1;
					INTENSITY := INTENSITY*2.0;
				    end loop;
				    INTENSITY :=
					1.0-INSTRUMENT_VALUE_TYPE(LOG_2)/
					    INSTRUMENT_VALUE_TYPE(GRAIN);
				end;

			    when others =>
				NYI("GRAPHICS_WINDOW_MANAGER." &
				    "INSTRUMENTS.UPDATE_NUMERIC : " &
				    "BEEPER|CLICKER scale", TRUE);

			end case;

			if INTENSITY = 0.0 then return; end if;

			-- Bell's faster than every 0.3 seconds don't work
			-- for LK201AA terminals (using X?).  I've not got a
			-- clue why not, but the symptom is the volume used 
			-- is the default, not that specified!
			--
			if INSTRUMENT_VALUE_TYPE(BELL_ELAPSED_TIME)
			> 2.0 - 1.7*INTENSITY
			then
			    BELL(DISPLAY, BELL_VOLUME);
			    BELL_TIME := CURR_TIME;
			end if;
		    end;

		when others =>
		    NYI("GRAPHICS_WINDOW_MANAGER." &
			"INSTRUMENTS.UPDATE_NUMERIC : " &
			"appearance", TRUE);
		    
	    end case;
	end;



	procedure UPDATE_TEXT(
	    I		    : in out GW_INSTRUMENT_TYPE) is

	    W	: WINDOW_INFO_TYPE renames GW.all;

	begin
	    if not I.CHANGED then return; end if;
	    DRAW_IMAGE_STRING(
		DISPLAY,
		DRAWABLE_TYPE(W.WINDOW),
		W.GC_ALWAYS,
		I.POSITION_X + 1,
		I.POSITION_Y + HEIGHT_PER_CHAR,
		I.TEXT_VALUE.all);
	    I.CHANGED := FALSE;
	end;


	procedure UPDATE is
	    INSTRUMENT : ACCESS_GW_INSTRUMENT_TYPE := GW.INSTRUMENTS;
	begin
	    while INSTRUMENT /= null loop
		declare
		    I : GW_INSTRUMENT_TYPE renames INSTRUMENT.all;
		begin
		    case I.SCALE is
			when NUMERIC_INSTRUMENT_SCALE_TYPE => UPDATE_NUMERIC(I);
			when TEXT => UPDATE_TEXT(I);
		    end case;
		    INSTRUMENT := I.NEXT;
		end;
	    end loop;
	end;


	procedure INIT_INSTRUMENTS(
	    GRAPHICS_WINDOW : in out GRAPHICS_WINDOW_TYPE) is
	begin
	    GW := GRAPHICS_WINDOW;
	end;


    begin
	null;
    end;
