--***********************************************************************
--									*
--	COPYRIGHT 1992		DIGITAL EQUIPMENT CORPORATION		*
--									*
--   This software was written by Bevin Brett, of Digital Equipment	*
--   Corporation.							*
--									*
--   Digital assumes no responsibility AT ALL for the use or reliability*
--   of this software.							*
--									*
--   Redistribution and use in source and binary forms are permitted	*
--   provided that this entire heading from --*** to --*** are          *
--   duplicated in all such forms and that any documentation,		*
--   advertising materials, and other materials related to such		*
--   distribution and use acknowledge that the software was developed	*
--   by Digital Equipment Corporation. The name of Digital Equipment	*
--   Corporation may not be used to endorse or promote products derived	*
--   from this software without specific prior written permission.	*
--									*
--   THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR	*
--   IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
--   WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.*
--									*
--***********************************************************************


with
    VIEW_DIRECTION_CONTROL,
    X, X_LIB,
    FLIGHT_RECORDER, RIBBON_MANAGER,
    AOA_INDICATOR,
    LOGICAL_TO_STRING, TEXT_IO;

use 
    X, X_LIB,
    FLIGHT_RECORDER,
    TEXT_IO;

pragma ELABORATE(
    X, X_LIB,
    FLIGHT_RECORDER,
    AOA_INDICATOR,
    LOGICAL_TO_STRING, TEXT_IO);

separate(GRAPHICS_WINDOW_MANAGER)

package body GW_CONTROL_MANAGER is

    type GW_CONTROL_TYPE is
	record
	    GW	    : GRAPHICS_WINDOW_TYPE;
	    X1, X2  : WIDTH_PIXEL_COUNT_SUBTYPE;
	    Y1, Y2  : HEIGHT_PIXEL_COUNT_SUBTYPE;
	    HEIGHT  : HEIGHT_PIXEL_COUNT_SUBTYPE;
	    WIDTH   : WIDTH_PIXEL_COUNT_SUBTYPE;
	end record;

    CONTROL_JAMMED  : BOOLEAN := FALSE;
    CONTROL_SETTING : CONTROL_SETTING_TYPE;

    GW_CONTROL_MADE : BOOLEAN := FALSE;
    GW_CONTROL	    : GW_CONTROL_TYPE;

    STICK_LEFT_RIGHT	: STICK_LEFT_RIGHT_TYPE     renames CONTROL_SETTING.STICK_LEFT_RIGHT;
    STICK_FORWARD_BACK	: STICK_FORWARD_BACK_TYPE   renames CONTROL_SETTING.STICK_FORWARD_BACK;
    FLAP		: FLAP_TYPE		    renames CONTROL_SETTING.FLAP;
    RUDDER		: RUDDER_TYPE		    renames CONTROL_SETTING.RUDDER;
    THROTTLE		: THROTTLE_TYPE 	    renames CONTROL_SETTING.THROTTLE;
    AFTERBURNER_ON	: BOOLEAN		    renames CONTROL_SETTING.AFTERBURNER_ON;
    SIGHT_ON		: BOOLEAN		    renames CONTROL_SETTING.SIGHT_ON;
    MODE		: MODE_TYPE		    renames CONTROL_SETTING.MODE;

    TRIGGER1_PRESSED	: BOOLEAN		    renames CONTROL_SETTING.TRIGGER1_PRESSED;
    TRIGGER2_PRESSED	: BOOLEAN		    renames CONTROL_SETTING.TRIGGER2_PRESSED;
    TRIGGER1_RELEASED	: BOOLEAN		    renames CONTROL_SETTING.TRIGGER1_RELEASED;
    TRIGGER2_RELEASED	: BOOLEAN		    renames CONTROL_SETTING.TRIGGER2_RELEASED;

    CHANGE_PRESSED	: BOOLEAN		    renames CONTROL_SETTING.CHANGE_PRESSED;
    QUIT_PRESSED	: BOOLEAN		    renames CONTROL_SETTING.QUIT_PRESSED;
    MESSAGE_LENGTH	: MESSAGE_LENGTH_SUBTYPE    renames CONTROL_SETTING.MESSAGE_LENGTH;
    MESSAGE_TEXT	: MESSAGE_SUBTYPE	    renames CONTROL_SETTING.MESSAGE_TEXT;

    DUMP_KEYSYM 	: constant BOOLEAN
	:= BOOLEAN'value(LOGICAL_TO_STRING("FCTM_DUMP_KEYSYM", "FALSE"));

    CROSS_FRACTION_W	: constant WIDTH_PIXEL_COUNT_SUBTYPE  := 20;
    CROSS_FRACTION_H	: constant HEIGHT_PIXEL_COUNT_SUBTYPE := 20;

    INDICATORS_DRAWN		: BOOLEAN := FALSE;
    AFTERBURNER_INDICATOR	: BOOLEAN := FALSE;
    MODE_INDICATOR		: MODE_TYPE;


    function TO_HEIGHT_PIXEL_COUNT_SUBTYPE(F : FLOAT)
	return HEIGHT_PIXEL_COUNT_SUBTYPE is
    begin
	return HEIGHT_PIXEL_COUNT_SUBTYPE(FLOAT(GW_CONTROL.HEIGHT)*F);
    end;
	
    function TO_WIDTH_PIXEL_COUNT_SUBTYPE(F : FLOAT)
	return WIDTH_PIXEL_COUNT_SUBTYPE is
    begin
	return WIDTH_PIXEL_COUNT_SUBTYPE(FLOAT(GW_CONTROL.WIDTH)*F);
    end;
	

    procedure DRAW(GW : GRAPHICS_WINDOW_TYPE) is
	C : GW_CONTROL_TYPE renames GW_CONTROL;

	CENTRE_X : constant WIDTH_PIXEL_COUNT_SUBTYPE
	    := C.X1 + TO_WIDTH_PIXEL_COUNT_SUBTYPE(FLOAT'(0.5));
	
	CENTRE_Y : constant HEIGHT_PIXEL_COUNT_SUBTYPE
	    := C.Y1 + TO_HEIGHT_PIXEL_COUNT_SUBTYPE(FLOAT'(0.5));

	LENGTH_X : constant WIDTH_PIXEL_COUNT_SUBTYPE
	    := TO_WIDTH_PIXEL_COUNT_SUBTYPE(1.0/FLOAT(CROSS_FRACTION_W));

	LENGTH_Y : constant HEIGHT_PIXEL_COUNT_SUBTYPE
	    := TO_HEIGHT_PIXEL_COUNT_SUBTYPE(1.0/FLOAT(CROSS_FRACTION_H));
	
    begin
	if not GW_CONTROL_MADE then return; end if;

	-- The box around the controls
	--
	DRAW_LINES(
	    DISPLAY,
	    DRAWABLE_TYPE(C.GW.WINDOW),
	    C.GW.GC_ALWAYS,
	    ((C.X1, C.Y1),
	     (C.X2, C.Y1),
	     (C.X2, C.Y2),
	     (C.X1, C.Y2),
	     (C.X1, C.Y1)),
	    ORIGIN);

	-- centre
	--
	DRAW_SEGMENTS(
	    DISPLAY,
	    DRAWABLE_TYPE(C.GW.WINDOW),
	    C.GW.GC_ALWAYS,
	    ( (CENTRE_X-LENGTH_X*2, CENTRE_Y,
	       CENTRE_X-LENGTH_X*1, CENTRE_Y),

	      (CENTRE_X+LENGTH_X*2, CENTRE_Y,
	       CENTRE_X+LENGTH_X*1, CENTRE_Y),

	      (CENTRE_X, CENTRE_Y-LENGTH_Y*2,
	       CENTRE_X, CENTRE_Y-LENGTH_Y*1),

	      (CENTRE_X, CENTRE_Y+LENGTH_Y*2,
	       CENTRE_X, CENTRE_Y+LENGTH_Y*1)
	    ));

	-- Haven't drawn the indicators
	--
	INDICATORS_DRAWN := FALSE;
    end;


    procedure PROCESS_KEYSYM(KEYSYM : KEYSYM_TYPE;
	RELEASED : BOOLEAN := FALSE;
	SHIFTED  : in out BOOLEAN)
    is
	use KEYSYMDEF;
    begin
	if KEYSYM.CLASS /= KEYSYM_CLASS_ASCII then return; end if;
	if not RELEASED then
	    case KEYSYM.CLASSCODE is

		when KS_LEFT =>
		    if CONTROL_JAMMED then
			null;
		    elsif RUDDER > RUDDER_TYPE'first then
			RUDDER := RUDDER - 1;
		    end if;

		when KS_RIGHT =>
		    if CONTROL_JAMMED then
			null;
		    elsif RUDDER < RUDDER_TYPE'last then
			RUDDER := RUDDER + 1;
		    end if;

		when KS_DOWN =>
		    if CONTROL_JAMMED then
			null;
		    elsif THROTTLE > THROTTLE_TYPE'first then
			THROTTLE := THROTTLE - 1;
		    end if;

		when KS_UP =>
		    if CONTROL_JAMMED then
			null;
		    elsif THROTTLE < THROTTLE_TYPE'last then
			THROTTLE := THROTTLE + 1;
		    end if;

		when KS_F16 | KS_MENU | KS_DELETE | KS_KP_ENTER =>
		    if CONTROL_JAMMED then
			null;
		    else
			AFTERBURNER_ON := not AFTERBURNER_ON;
		    end if;

		when KS_KP_SUBTRACT =>
		    CHANGE_PRESSED := TRUE;

		when KS_KP_DECIMAL =>
		    SIGHT_ON := not SIGHT_ON;

		when KS_SELECT =>
		    if CONTROL_JAMMED then
			null;
		    elsif MODE = MODE_TYPE'last then
			MODE := MODE_TYPE'first;
		    else
			MODE := MODE_TYPE'succ(MODE);
		    end if;

		when KS_KP_0 =>
		    if CONTROL_JAMMED then
			null;
		    else
			TRIGGER1_PRESSED := TRUE;
		    end if;

		when KS_SPACE =>
		    if CONTROL_JAMMED then
			null;
		    else
			TRIGGER2_PRESSED := TRUE;
		    end if;

		when KS_KP_1 =>
		    if CONTROL_JAMMED then
			null;
		    elsif FLAP > FLAP_TYPE'first then
			FLAP := FLAP - 1;
		    end if;

		when KS_KP_4 =>
		    if CONTROL_JAMMED then
			null;
		    elsif FLAP < FLAP_TYPE'last then
			FLAP := FLAP + 1;
		    end if;

		-- Flight recorder
		--
		when KS_L_B | KS_U_B =>
		     FLIGHT_RECORDER.SPEED := FLIGHT_RECORDER.SPEED - 1;

		when KS_L_F | KS_U_F =>
		     FLIGHT_RECORDER.SPEED := FLIGHT_RECORDER.SPEED + 1;

		when KS_L_M | KS_U_M =>
		     FLIGHT_RECORDER.MARK_PRESSED := TRUE;

		when KS_L_P | KS_U_P =>
		     FLIGHT_RECORDER.SPEED := 0;
		     FLIGHT_RECORDER.STATE := FLIGHT_RECORDER.ON_PLAY;

		when KS_L_R | KS_U_R =>
		     FLIGHT_RECORDER.SPEED := 0;
		     FLIGHT_RECORDER.STATE := FLIGHT_RECORDER.ON_RECORD;

		when KS_L_S | KS_U_S =>
		     FLIGHT_RECORDER.SPEED := 0;
		     FLIGHT_RECORDER.STATE := FLIGHT_RECORDER.OFF;

		when KS_L_C | KS_U_C =>
		    RIBBON_MANAGER.LENGTH := 0;

		when KS_L_T | KS_U_T =>
		    declare
			use RIBBON_MANAGER;
		    begin
			if LENGTH < 10 then
			    LENGTH := LENGTH + 1;
			end if;
		    end;

		when KS_L_U | KS_U_U =>
		    declare
			use RIBBON_MANAGER;
		    begin
			if LENGTH > 0 then
			    LENGTH := LENGTH - 1;
			end if;
		    end;

		-- View
		--
		when KS_RETURN =>
		    GW_CONTROL_MANAGER.CONTROL_SETTING.VD_CONTROL :=
			(CONTROL_STATE => VIEW_DIRECTION_CONTROL.TO_FRONT);

		-- General
		--
		when KS_U_Q | KS_L_Q =>
		    QUIT_PRESSED := TRUE;

		when KS_U_Z | KS_L_Z =>
		    -- Since the CONTROL gets ignored, CONTROL Z also exits :-)
		    --
		    STANDARD.CONTROLS.EXIT_PRESSED := TRUE;

		when KS_SHIFT_L | KS_SHIFT_R | KS_CAPS_LOCK |
                     KS_META_L  | KS_META_R  |
                     KS_ALT_L   | KS_ALT_R   |
		     KS_MULTI_KEY =>
  		    SHIFTED := TRUE;

		when others =>
		    PUT_LINE("Unrecognised : " &
			KEYSYM_CLASSCODE_TYPE'image(KEYSYM.CLASSCODE) &
			" pressed");

	    end case;

	else
	    case KEYSYM.CLASSCODE is

		when KS_KP_0 =>
		    if CONTROL_JAMMED then
			null;
		    else
			TRIGGER1_RELEASED := TRUE;
		    end if;

		when KS_SHIFT_L | KS_SHIFT_R | KS_CAPS_LOCK |
                     KS_META_L  | KS_META_R  |
                     KS_ALT_L   | KS_ALT_R   |
		     KS_MULTI_KEY =>
		    SHIFTED := FALSE;

		when others =>
		    null;

	    end case;

	end if;
    end;


    procedure DRAW_CONTROL_LINE(
	X	    : WIDTH_PIXEL_COUNT_SUBTYPE;
	Y	    : HEIGHT_PIXEL_COUNT_SUBTYPE;
	X_LENGTH,
	Y_LENGTH    : FLOAT) is
    begin
	DRAW_LINE(
	    X, Y,
	    X+TO_WIDTH_PIXEL_COUNT_SUBTYPE(X_LENGTH),
	    Y+TO_HEIGHT_PIXEL_COUNT_SUBTYPE(Y_LENGTH),
	    GW_CONTROL.GW);
    end;


    procedure UPDATE_NON_VOLATILE_AND_COPY(
	CONTROL_SETTING : out CONTROL_SETTING_TYPE) is

	STICK_LEFT_RIGHT_TYPE_LENGTH : constant FLOAT :=
	    FLOAT(STICK_LEFT_RIGHT_TYPE'last) - 
	    FLOAT(STICK_LEFT_RIGHT_TYPE'first);
	STICK_LEFT_RIGHT_TYPE_FIRST  : constant FLOAT :=
	    FLOAT(STICK_LEFT_RIGHT_TYPE'first);
	STICK_FORWARD_BACK_TYPE_LENGTH : constant FLOAT :=
	    FLOAT(STICK_FORWARD_BACK_TYPE'last) - 
	    FLOAT(STICK_FORWARD_BACK_TYPE'first);
	STICK_FORWARD_BACK_TYPE_FIRST  : constant FLOAT :=
	    FLOAT(STICK_FORWARD_BACK_TYPE'first);

    begin
	SYNC_X.SCAN_EVENTS;

	declare
	    W : WINDOW_INFO_TYPE renames GW_CONTROL.GW.all;
	begin
	    declare
		use VIEW_DIRECTION_CONTROL;
		VDC : CONTROL_TYPE renames
			    GW_CONTROL_MANAGER.CONTROL_SETTING.VD_CONTROL;
		SHIFTED : BOOLEAN renames W.SHIFT_DEPRESSED;
		MB1 : BOOLEAN renames W.BUTTONS_DEPRESSED(BUTTON1);
		MB2 : BOOLEAN renames W.BUTTONS_DEPRESSED(BUTTON2);
		MB3 : BOOLEAN renames W.BUTTONS_DEPRESSED(BUTTON3);
	    begin
		if W.BUTTONS_RELEASED then
		    W.BUTTONS_RELEASED := FALSE;
		    VDC := (CONTROL_STATE => STOP);
		elsif not W.BUTTONS_PRESSED
		then
		    VDC := (CONTROL_STATE => UNCHANGED);
		else
		    W.BUTTONS_PRESSED := FALSE;
		    VDC := (TO_ARBITRARY, 0.0, 0.0);
		    if MB1 then
			VDC.SWIVEL_RIGHT_LEFT	:= -1.0;
		    elsif MB2 then
			if SHIFTED then
			    VDC.SWIVEL_UP_DOWN	:= -1.0;
			else
			    VDC.SWIVEL_UP_DOWN  :=  1.0;
			end if;
		    elsif MB3 then
			VDC.SWIVEL_RIGHT_LEFT	:=  1.0;
		    end if;
		end if;
	    end;

	    for I in 1..W.KEY_PRESSED loop
		declare
		    use KEYSYMDEF;
		    KEYSYM  : KEYSYM_TYPE;
		begin
		    KEYSYM := KEYCODE_TO_KEYSYM(
				DISPLAY, W.KEYCODES_PRESSED(I), 0);
		    PROCESS_KEYSYM(KEYSYM, FALSE, W.SHIFT_DEPRESSED);
		end;
	    end loop;
	    W.KEY_PRESSED := 0;

	    for I in 1..W.KEY_RELEASED loop
		declare
		    use KEYSYMDEF;
		    KEYSYM  : KEYSYM_TYPE;
		begin
		    KEYSYM := KEYCODE_TO_KEYSYM(
				DISPLAY, W.KEYCODES_RELEASED(I), 0);
		    PROCESS_KEYSYM(KEYSYM, TRUE, W.SHIFT_DEPRESSED);
		end;
	    end loop;
	    W.KEY_RELEASED := 0;

	    if	W.POINTER_X > GW_CONTROL.X1  and then
		W.POINTER_X < GW_CONTROL.X2  and then
		W.POINTER_Y > GW_CONTROL.Y1  and then
		W.POINTER_Y < GW_CONTROL.Y2  and then
		not CONTROL_JAMMED
	    then
		STICK_LEFT_RIGHT :=
		    STICK_LEFT_RIGHT_TYPE(
			FLOAT(W.POINTER_X-GW_CONTROL.X1)/
			FLOAT(GW_CONTROL.WIDTH)*STICK_LEFT_RIGHT_TYPE_LENGTH+
			STICK_LEFT_RIGHT_TYPE_FIRST);
		STICK_FORWARD_BACK :=
		    -STICK_FORWARD_BACK_TYPE(
			FLOAT(W.POINTER_Y-GW_CONTROL.Y1)/
			FLOAT(GW_CONTROL.HEIGHT)*STICK_FORWARD_BACK_TYPE_LENGTH+
			STICK_FORWARD_BACK_TYPE_FIRST);
	    end if;

	end;

	-- draw the lines showing where each of the controls is
	--
	-- stick
	--
	declare
	    CENTRE_X : constant WIDTH_PIXEL_COUNT_SUBTYPE :=
		WIDTH_PIXEL_COUNT_SUBTYPE(
		    (FLOAT(STICK_LEFT_RIGHT)-STICK_LEFT_RIGHT_TYPE_FIRST)/
		    STICK_LEFT_RIGHT_TYPE_LENGTH*
		    FLOAT(GW_CONTROL.WIDTH)
		    ) + GW_CONTROL.X1;

	    CENTRE_Y : constant HEIGHT_PIXEL_COUNT_SUBTYPE :=
		HEIGHT_PIXEL_COUNT_SUBTYPE(
		    (-FLOAT(STICK_FORWARD_BACK)-STICK_FORWARD_BACK_TYPE_FIRST)/
		    STICK_FORWARD_BACK_TYPE_LENGTH*
		    FLOAT(GW_CONTROL.HEIGHT)
		    ) + GW_CONTROL.Y1;
		    
	    LO_Y : HEIGHT_PIXEL_COUNT_SUBTYPE;
	begin
	    -- cross
	    --
	    DRAW_LINE(
		CENTRE_X - GW_CONTROL.WIDTH/CROSS_FRACTION_W, CENTRE_Y,
		CENTRE_X + GW_CONTROL.WIDTH/CROSS_FRACTION_W, CENTRE_Y,
		GW_CONTROL.GW);

	    -- down
	    --
	    LO_Y := CENTRE_Y;
	    if FLOAT(AOA_INDICATOR.AOA_MAX_RATIO) <= 1.0 then
		LO_Y := CENTRE_Y - GW_CONTROL.HEIGHT/CROSS_FRACTION_H;
	    end if;

	    DRAW_LINE(
		CENTRE_X, LO_Y,
		CENTRE_X, CENTRE_Y + GW_CONTROL.HEIGHT/CROSS_FRACTION_H,
		GW_CONTROL.GW);
	end;

	-- throttle
	--
	DRAW_CONTROL_LINE(
	    GW_CONTROL.X1,
	    GW_CONTROL.Y2-1-TO_HEIGHT_PIXEL_COUNT_SUBTYPE(
			    FLOAT(THROTTLE)/FLOAT(THROTTLE_TYPE'last)*0.9),
	    0.05, 0.00);


	-- rudder
	--
	DRAW_CONTROL_LINE(
	    GW_CONTROL.X1+TO_WIDTH_PIXEL_COUNT_SUBTYPE(
			    FLOAT(RUDDER)/FLOAT(RUDDER_TYPE'last)/2.0+0.5),
	    GW_CONTROL.Y1,
	    0.00, 0.05);

	-- flaps
	--
	DRAW_CONTROL_LINE(
	    GW_CONTROL.X2,
	    GW_CONTROL.Y2-1-TO_HEIGHT_PIXEL_COUNT_SUBTYPE(
			    FLOAT(FLAP)/FLOAT(FLAP_TYPE'last)*0.9),
	    -0.05, 0.00);

	-- Show whether the afterburner is on or off
	--
	if not INDICATORS_DRAWN 
	or AFTERBURNER_INDICATOR  /= AFTERBURNER_ON
	then
	    declare
		C : GW_CONTROL_TYPE renames GW_CONTROL;
		S : constant array(BOOLEAN) of STRING(1..3) := ("Off", "On ");
	    begin

		AFTERBURNER_INDICATOR := AFTERBURNER_ON;

		DRAW_IMAGE_STRING(
		    DISPLAY,
		    DRAWABLE_TYPE(C.GW.WINDOW),
		    C.GW.GC_ALWAYS,
		    C.X1+10,
		    C.Y1+10,
		    S(AFTERBURNER_INDICATOR));
	    end;
	end if;

	-- Show the selected MODE
	--
	if not INDICATORS_DRAWN 
	or MODE_INDICATOR  /= MODE
	then
	    declare
		C : GW_CONTROL_TYPE renames GW_CONTROL;
	    begin

		MODE_INDICATOR := MODE;

		DRAW_IMAGE_STRING(
		    DISPLAY,
		    DRAWABLE_TYPE(C.GW.WINDOW),
		    C.GW.GC_ALWAYS,
		    C.X2-10-2*WIDTH_PER_CHAR,
		    C.Y1+10,
		    STRING'(1=>
			CHARACTER'val(
			    CHARACTER'pos('0') + 
			    INTEGER(MODE_INDICATOR))));
	    end;
	end if;

	-- Note that all the indicators have been drawn
	--
	INDICATORS_DRAWN := TRUE;

	-- return the result
	--
	CONTROL_SETTING := GW_CONTROL_MANAGER.CONTROL_SETTING;

	-- turn off one-shots
	--
	TRIGGER1_PRESSED    := FALSE;
	TRIGGER2_PRESSED    := FALSE;
	TRIGGER1_RELEASED   := FALSE;
	TRIGGER2_RELEASED   := FALSE;

	CHANGE_PRESSED	    := FALSE;
	QUIT_PRESSED	    := FALSE;

    end;


    procedure SET_CONTROLS(
	CONTROL_SETTING : in CONTROL_SETTING_TYPE;
	JAMMED		: BOOLEAN) is
    begin
	GW_CONTROL_MANAGER.CONTROL_JAMMED  := JAMMED;
	GW_CONTROL_MANAGER.CONTROL_SETTING := CONTROL_SETTING;
    end;

    procedure SET_CONTROLS(
	CONTROL_SETTING : in CONTROL_SETTING_TYPE) is
    begin
	GW_CONTROL_MANAGER.CONTROL_SETTING := CONTROL_SETTING;
    end;

    procedure SET_CONTROLS(
	JAMMED		: BOOLEAN) is
    begin
	GW_CONTROL_MANAGER.CONTROL_JAMMED  := JAMMED;
	STICK_LEFT_RIGHT    := 0.0;
	STICK_FORWARD_BACK  := 0.0;
	RUDDER		    := 0;
	TRIGGER1_PRESSED    := FALSE;
	TRIGGER2_PRESSED    := FALSE;
	TRIGGER1_RELEASED   := FALSE;
	TRIGGER2_RELEASED   := FALSE;
    end;


    procedure INIT_CONTROLS(
	GRAPHICS_WINDOW : in out GRAPHICS_WINDOW_TYPE;
	LOW_LEFT_X,
	LOW_LEFT_Y      : SCREEN_MEASUREMENT;
	HEIGHT,WIDTH    : SCREEN_MEASUREMENT) is

	W : WINDOW_INFO_TYPE renames GRAPHICS_WINDOW.all;

    begin
	GW_CONTROL.GW := GRAPHICS_WINDOW;

	GW_CONTROL.HEIGHT := -TO_HEIGHT_PIXEL_COUNT_SUBTYPE(HEIGHT);
	GW_CONTROL.WIDTH  :=  TO_WIDTH_PIXEL_COUNT_SUBTYPE(WIDTH);

	GW_CONTROL.X1 :=  TO_WIDTH_PIXEL_COUNT_SUBTYPE(LOW_LEFT_X);
	GW_CONTROL.Y2 :=  W.HEIGHT + TO_HEIGHT_PIXEL_COUNT_SUBTYPE(LOW_LEFT_Y);

	GW_CONTROL.X2 :=  GW_CONTROL.X1 + GW_CONTROL.WIDTH;
	GW_CONTROL.Y1 :=  GW_CONTROL.Y2 - GW_CONTROL.HEIGHT;

	GW_CONTROL_MADE := TRUE;
    end;

end;    
