--***********************************************************************
--									*
--	COPYRIGHT 1992		DIGITAL EQUIPMENT CORPORATION		*
--									*
--   This software was written by Bevin Brett, of Digital Equipment	*
--   Corporation.							*
--									*
--   Digital assumes no responsibility AT ALL for the use or reliability*
--   of this software.							*
--									*
--   Redistribution and use in source and binary forms are permitted	*
--   provided that this entire heading from --*** to --*** are          *
--   duplicated in all such forms and that any documentation,		*
--   advertising materials, and other materials related to such		*
--   distribution and use acknowledge that the software was developed	*
--   by Digital Equipment Corporation. The name of Digital Equipment	*
--   Corporation may not be used to endorse or promote products derived	*
--   from this software without specific prior written permission.	*
--									*
--   THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR	*
--   IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
--   WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.*
--									*
--***********************************************************************


with
    BUG, NYI,
    LOGICAL_TO_BOOLEAN, LOGICAL_TO_STRING, LOGICAL_TO_FLOAT,
    UNCHECKED_DEALLOCATION;

with C_TYPES, X, X_LIB, TEXT_IO;
use	      X, X_LIB, TEXT_IO;

pragma ELABORATE(
    BUG, NYI,
    LOGICAL_TO_BOOLEAN, LOGICAL_TO_STRING,
    UNCHECKED_DEALLOCATION,
    X, X_LIB, TEXT_IO);

package body GRAPHICS_WINDOW_MANAGER is

    DOES_DRAW_LAST_POINT : constant BOOLEAN
	:= LOGICAL_TO_BOOLEAN("FCTM_DOES_DRAW_LAST_POINT", TRUE);


    type LIST_OF_SEGMENTS_TYPE is
	record
	    N		: NATURAL := 0;
	    SEGMENTS	: VECTOR_OF_SEGMENT_TYPE(1..3_000);
	end record;

    type ACCESS_LIST_OF_SEGMENTS_TYPE is
	access LIST_OF_SEGMENTS_TYPE;

    procedure DEALLOCATE is
	new UNCHECKED_DEALLOCATION(
		LIST_OF_SEGMENTS_TYPE,
		ACCESS_LIST_OF_SEGMENTS_TYPE);


    procedure DRAW_LINE(
	X1		: WIDTH_PIXEL_COUNT_SUBTYPE;
	Y1		: HEIGHT_PIXEL_COUNT_SUBTYPE;
	X2		: WIDTH_PIXEL_COUNT_SUBTYPE;
	Y2		: HEIGHT_PIXEL_COUNT_SUBTYPE;
	GRAPHICS_WINDOW	: GRAPHICS_WINDOW_TYPE);


    package GW_CONTROL_MANAGER is
	use STANDARD.CONTROLS;

	type ACCESS_GW_CONTROL_TYPE is private;

	procedure DRAW(GW : GRAPHICS_WINDOW_TYPE);

	procedure UPDATE_NON_VOLATILE_AND_COPY(
	    CONTROL_SETTING : out CONTROL_SETTING_TYPE);

	procedure SET_CONTROLS(
	    CONTROL_SETTING : in CONTROL_SETTING_TYPE;
	    JAMMED	    : BOOLEAN);

	procedure SET_CONTROLS(
	    CONTROL_SETTING : in CONTROL_SETTING_TYPE);

	procedure SET_CONTROLS(
	    JAMMED	    : BOOLEAN);

	procedure INIT_CONTROLS(
	    GRAPHICS_WINDOW : in out GRAPHICS_WINDOW_TYPE;
	    LOW_LEFT_X,
	    LOW_LEFT_Y      : SCREEN_MEASUREMENT;
	    HEIGHT,WIDTH    : SCREEN_MEASUREMENT);

    private
	type GW_CONTROL_TYPE;
	type ACCESS_GW_CONTROL_TYPE is access GW_CONTROL_TYPE;
    end;


    package GW_INSTRUMENT_MANAGER is
	use INSTRUMENTS;

	type ACCESS_GW_INSTRUMENT_TYPE is private;

	procedure DRAW(GW : GRAPHICS_WINDOW_TYPE);

	function CREATE_NUMERIC(
	    NAME	    : INSTRUMENT_NAME_TYPE;
	    X, Y	    : SCREEN_MEASUREMENT;
	    HEIGHT,WIDTH    : SCREEN_MEASUREMENT;
	    APPEARANCE	    : INSTRUMENT_APPEARANCE_TYPE;
	    SCALE	    : NUMERIC_INSTRUMENT_SCALE_TYPE;

	    LO, TOP, HI     : INSTRUMENT_VALUE_TYPE := 0.0;

	    INSTRUMENT_VALUE_PER_RADIAN : INSTRUMENT_VALUE_TYPE;
	    RADIANS_PER_NUMBERED_TICK	: TRIG.RADIANS;
	    TICKS_PER_NUMBERED_TICK	: POSITIVE

	    ) return ACCESS_GW_INSTRUMENT_TYPE;

	function CREATE_SIGHT(
	    STYLE	    : SIGHT_STYLE_TYPE;
	    SIZE	    : SCREEN_MEASUREMENT;
	    HOLE_SIZE	    : SCREEN_MEASUREMENT
	    ) return ACCESS_GW_INSTRUMENT_TYPE;

	function CREATE_TEXT(
	    NAME	    : INSTRUMENT_NAME_TYPE;
	    X, Y	    : SCREEN_MEASUREMENT;
	    HEIGHT,WIDTH    : SCREEN_MEASUREMENT;
	    MAX_LENGTH	    : POSITIVE
	    ) return ACCESS_GW_INSTRUMENT_TYPE;

	procedure SET_NUMERIC(
	    INSTRUMENT	    : ACCESS_GW_INSTRUMENT_TYPE;
	    VALUE	    : INSTRUMENT_VALUE_TYPE);

	procedure SET_SIGHT(
	    SIGHT_ON	    : BOOLEAN;
	    X, Y	    : SCREEN_MEASUREMENT);		-- centre

	procedure SET_TEXT(
	    INSTRUMENT	    : ACCESS_GW_INSTRUMENT_TYPE;
	    VALUE	    : STRING);

	procedure UPDATE;

	procedure INIT_INSTRUMENTS(
	    GRAPHICS_WINDOW : in out GRAPHICS_WINDOW_TYPE);

    private
	type GW_INSTRUMENT_TYPE(
		APPEARANCE  : INSTRUMENT_APPEARANCE_TYPE;
		SCALE	    : INSTRUMENT_SCALE_TYPE);
	type ACCESS_GW_INSTRUMENT_TYPE is access GW_INSTRUMENT_TYPE;

    end;


    package GW_MASK_MANAGER is
	type ACCESS_GW_MASK_TYPE is private;

	procedure DRAW(GW : GRAPHICS_WINDOW_TYPE);

	procedure ERASE_RECTANGLE(
	    BLX, BLY	    : SCREEN_MEASUREMENT;	-- bottom left corner
	    WIDTH, HEIGHT   : SCREEN_MEASUREMENT;	-- size
	    GRAPHICS_WINDOW : in out GRAPHICS_WINDOW_TYPE);

	procedure FILL_RECTANGLE(
	    BLX, BLY	    : SCREEN_MEASUREMENT;	-- bottom left corner
	    WIDTH, HEIGHT   : SCREEN_MEASUREMENT;	-- size
	    GRAPHICS_WINDOW : in out GRAPHICS_WINDOW_TYPE);

    private
	type GW_MASK_TYPE;
	type ACCESS_GW_MASK_TYPE is access GW_MASK_TYPE;
    end;


    package GW_MENU_MANAGER is

	type ACCESS_GW_MENU_TYPE is private;

	procedure DRAW(GW : GRAPHICS_WINDOW_TYPE);

	function CREATE_MENU(
	    MENU    : in MENU_MANAGER.MENU_TYPE) return ACCESS_GW_MENU_TYPE;

	function SELECT_FROM_MENU(
	    MENU    : in ACCESS_GW_MENU_TYPE;
	    CHOICE  : STRING := "") return NATURAL;

	procedure DELETE_MENU(X : in out ACCESS_GW_MENU_TYPE);

    private
	type GW_MENU_TYPE;
	type ACCESS_GW_MENU_TYPE is access GW_MENU_TYPE;
    end;


    type SET_OF_BUTTON_TYPE is
	array(BUTTON_TYPE) of BOOLEAN;

    type VECTOR_OF_KEYCODES is
	array (POSITIVE range <>) of KEYCODE_TYPE;

    type WINDOW_INFO_TYPE is
	record
	    NEXT		    : GRAPHICS_WINDOW_TYPE;

	    WINDOW		    : WINDOW_TYPE;

	    HEIGHT		    : HEIGHT_PIXEL_COUNT_SUBTYPE;
	    WIDTH		    : WIDTH_PIXEL_COUNT_SUBTYPE;

	    GC_ALWAYS,
	    GC_ALWAYS_INVERSE,

	    GC_P0_SET,
	    GC_P1_SET,
	    GC_P0_CLEAR,
	    GC_P1_CLEAR 	    : GC_TYPE;

	    -- Segments
	    --
	    DRAWING_LIST_OF_SEGMENTS,
	    DRAWN_LIST_OF_SEGMENTS,
	    FREE_LIST_OF_SEGMENTS   : ACCESS_LIST_OF_SEGMENTS_TYPE
				    := new LIST_OF_SEGMENTS_TYPE;

	    -- Other things that might be in the window
	    --
	    CONTROLS		    : GW_CONTROL_MANAGER.ACCESS_GW_CONTROL_TYPE;
	    INSTRUMENTS 	    : GW_INSTRUMENT_MANAGER.ACCESS_GW_INSTRUMENT_TYPE;
	    MASKS		    : GW_MASK_MANAGER.ACCESS_GW_MASK_TYPE;
	    MENUS		    : GW_MENU_MANAGER.ACCESS_GW_MENU_TYPE;

	    -- effects of events
	    --
	    POINTER_X		    : WIDTH_PIXEL_COUNT_SUBTYPE	:= 0;
	    POINTER_Y		    : HEIGHT_PIXEL_COUNT_SUBTYPE	:= 0;

	    SHIFT_DEPRESSED	    : BOOLEAN			:= FALSE;

	    BUTTONS_PRESSED	    : BOOLEAN			:= FALSE;
	    BUTTONS_RELEASED	    : BOOLEAN			:= FALSE;
	    BUTTONS_DEPRESSED	    : SET_OF_BUTTON_TYPE	:= (others=>FALSE);

	    KEY_PRESSED 	    : NATURAL			:= 0;
	    KEYCODES_PRESSED	    : VECTOR_OF_KEYCODES(1..4);
	    KEY_RELEASED	    : NATURAL			:= 0;
	    KEYCODES_RELEASED	    : VECTOR_OF_KEYCODES(1..4);
	end record;

    procedure DEALLOCATE is
	new UNCHECKED_DEALLOCATION(
		WINDOW_INFO_TYPE,
		GRAPHICS_WINDOW_TYPE);



    DISPLAY	    : DISPLAY_TYPE;
    COLORMAP	    : COLORMAP_TYPE;
    FONT	    : FONT_TYPE;
    HEIGHT_PER_CHAR : HEIGHT_PIXEL_COUNT_SUBTYPE;
    WIDTH_PER_CHAR  : WIDTH_PIXEL_COUNT_SUBTYPE;
    SCREEN	    : ACCESS_SCREEN_TYPE;
    VISUAL	    : VISUAL_TYPE;

    DEPTH	    : DEPTH_TYPE;
    HEIGHT	    : HEIGHT_PIXEL_COUNT_SUBTYPE;
    HEIGHT_MM	    : MILLIMETRES_TYPE;
    SWA 	    : SET_WINDOW_ATTRIBUTES_TYPE;
    WIDTH	    : WIDTH_PIXEL_COUNT_SUBTYPE;
    WIDTH_MM	    : MILLIMETRES_TYPE;
    SWA_MASK	    : SET_WINDOW_ATTRIBUTES_MASK_TYPE;

    X_PIXELS_PER_SCREEN_MEASUREMENT_K,
    Y_PIXELS_PER_SCREEN_MEASUREMENT_K : PIXELS_PER_SCREEN_MEASUREMENT_TYPE;


    type COLORMAP_SLOTS_TYPE is
	(ALWAYS_BG,   BG_FG,	   FG_BG,	ALWAYS_FG,
	 FORCED_FG_1, FORCED_FG_2, FORCED_FG_3, FORCED_FG_4);

    NUMBER_OF_COLORMAP_SLOTS : constant NATURAL
	:=  COLORMAP_SLOTS_TYPE'pos(COLORMAP_SLOTS_TYPE'last) -
	    COLORMAP_SLOTS_TYPE'pos(COLORMAP_SLOTS_TYPE'first) + 1;

    subtype COLORMAP_INDEXS_RANGE is POSITIVE range 1..NUMBER_OF_COLORMAP_SLOTS;

    COLORMAP_INDEXS : VECTOR_OF_COLORMAP_INDEX_TYPE(COLORMAP_INDEXS_RANGE);
    COLORS	    : VECTOR_OF_COLOR_TYPE(COLORMAP_INDEXS_RANGE);
    SWITCHED_COLORS : VECTOR_OF_COLOR_TYPE renames COLORS(2..3);
    PLANE_MASKS     : VECTOR_OF_PLANE_MASK_TYPE(1..3);
    PLANE_TO_WRITE  : INTEGER range 0..1;


    ALREADY_A_GRAPHICS_WINDOW	: exception;
    NOT_A_GRAPHICS_WINDOW	: exception;


    function TO_HEIGHT_PIXEL_COUNT_SUBTYPE(Y : SCREEN_MEASUREMENT)
	return HEIGHT_PIXEL_COUNT_SUBTYPE is
    begin
	return HEIGHT_PIXEL_COUNT_SUBTYPE(
	    PIXELS_PER_SCREEN_MEASUREMENT_TYPE(Y)*
	    Y_PIXELS_PER_SCREEN_MEASUREMENT_K);
    end;


    function TO_WIDTH_PIXEL_COUNT_SUBTYPE(X : SCREEN_MEASUREMENT)
	return WIDTH_PIXEL_COUNT_SUBTYPE is
    begin
	return WIDTH_PIXEL_COUNT_SUBTYPE(
	    PIXELS_PER_SCREEN_MEASUREMENT_TYPE(X)*
	    X_PIXELS_PER_SCREEN_MEASUREMENT_K);
    end;


    function TO_COLORMAP_INDEXS_RANGE(COLORMAP_SLOT : COLORMAP_SLOTS_TYPE)
	return COLORMAP_INDEXS_RANGE is
    begin
	return COLORMAP_SLOTS_TYPE'pos(COLORMAP_SLOT) + 1;
    end;


    procedure DRAW_WINDOW(GW : GRAPHICS_WINDOW_TYPE) is
    begin
	GW_CONTROL_MANAGER.DRAW(GW);
	GW_INSTRUMENT_MANAGER.DRAW(GW);
	GW_MASK_MANAGER.DRAW(GW);
	GW_MENU_MANAGER.DRAW(GW);
    end;


    task SYNC_X is
	pragma PRIORITY(8);	-- One larger than the default.

	entry CLEAR_WINDOW	   (GW : in GRAPHICS_WINDOW_TYPE);

	entry CREATE_WINDOW(
		    GW	    : in GRAPHICS_WINDOW_TYPE;
		    X_COORD : WIDTH_PIXEL_COUNT_NATURAL_SUBTYPE;
		    Y_COORD : HEIGHT_PIXEL_COUNT_NATURAL_SUBTYPE;
		    WIDTH   : WIDTH_PIXEL_COUNT_NATURAL_SUBTYPE;
		    HEIGHT  : HEIGHT_PIXEL_COUNT_NATURAL_SUBTYPE);

	entry DESTROY_WINDOW(
		    GW	    : in GRAPHICS_WINDOW_TYPE);

	entry MAKE_SEGMENTS_VISIBLE(GW : in GRAPHICS_WINDOW_TYPE);

	entry SCAN_EVENTS;

    end;


    procedure CREATE(
	BLX		: WIDTH_PIXEL_COUNT_SUBTYPE;	-- top left
	BLY		: HEIGHT_PIXEL_COUNT_SUBTYPE;
	WIDTH		: WIDTH_PIXEL_COUNT_NATURAL_SUBTYPE;
	HEIGHT		: HEIGHT_PIXEL_COUNT_NATURAL_SUBTYPE;
	GRAPHICS_WINDOW : in out GRAPHICS_WINDOW_TYPE) is

	GW : GRAPHICS_WINDOW_TYPE renames GRAPHICS_WINDOW;

    begin
	if GW /= null then raise ALREADY_A_GRAPHICS_WINDOW; end if;
	GW := new WINDOW_INFO_TYPE;

	GW.HEIGHT := HEIGHT;
	GW.WIDTH  :=  WIDTH;

	SYNC_X.CREATE_WINDOW(GW,
	    BLX,
	    BLY,
	    WIDTH,
	    HEIGHT);
    end;


    procedure CREATE(
	BLX, BLY	: SCREEN_MEASUREMENT;		-- bottom left corner
	WIDTH, HEIGHT   : SCREEN_MEASUREMENT;		--  of where to place
	GRAPHICS_WINDOW : in out GRAPHICS_WINDOW_TYPE) is

	BLX_PC		: WIDTH_PIXEL_COUNT_SUBTYPE := 0;
	BLY_PC		: HEIGHT_PIXEL_COUNT_SUBTYPE := 0;
	WIDTH_PC	: WIDTH_PIXEL_COUNT_SUBTYPE := 0;
	HEIGHT_PC	: HEIGHT_PIXEL_COUNT_SUBTYPE := 0;
	
    begin
	BLX_PC		:=  TO_WIDTH_PIXEL_COUNT_SUBTYPE(BLX);
	BLY_PC		:= -TO_HEIGHT_PIXEL_COUNT_SUBTYPE(BLY);
	WIDTH_PC	:=  TO_WIDTH_PIXEL_COUNT_SUBTYPE(WIDTH);
	HEIGHT_PC	:= -TO_HEIGHT_PIXEL_COUNT_SUBTYPE(HEIGHT);

	CREATE(
	    BLX_PC,
	    GRAPHICS_WINDOW_MANAGER.HEIGHT - BLY_PC - HEIGHT_PC - 1,
	    WIDTH_PC,
	    HEIGHT_PC,
	    GRAPHICS_WINDOW);

    exception
	when others =>
	    PUT_LINE("CONSTRAINT_ERROR while trying to create window");
	    PUT_LINE("GRAPHICS_WINDOW_MANAGER.HEIGHT => "&
		INTEGER'IMAGE(INTEGER(GRAPHICS_WINDOW_MANAGER.HEIGHT)));
	    PUT_LINE("BLX_PC => "&
		INTEGER'IMAGE(INTEGER(BLX_PC)));
	    PUT_LINE("BLY_PC => "&
		INTEGER'IMAGE(INTEGER(BLY_PC)));
	    PUT_LINE("WIDTH_PC => "&
		INTEGER'IMAGE(INTEGER(WIDTH_PC)));
	    PUT_LINE("HEIGHT_PC => "&
		INTEGER'IMAGE(INTEGER(HEIGHT_PC)));
	    raise;
    end;


    procedure DELETE(GRAPHICS_WINDOW : in out GRAPHICS_WINDOW_TYPE) is
	GW : GRAPHICS_WINDOW_TYPE renames GRAPHICS_WINDOW;
	W  : WINDOW_INFO_TYPE renames GW.all;
    begin
	SYNC_X.DESTROY_WINDOW(GW);
	DEALLOCATE(W.DRAWING_LIST_OF_SEGMENTS);
	DEALLOCATE(W.DRAWN_LIST_OF_SEGMENTS);
	DEALLOCATE(W.FREE_LIST_OF_SEGMENTS);
	DEALLOCATE(GW);
    end;


    package body GW_CONTROL_MANAGER is separate;

    package body GW_INSTRUMENT_MANAGER is separate;

    package body GW_MASK_MANAGER is separate;

    package body GW_MENU_MANAGER is separate;

    procedure INIT_FONT is separate;


    package body CONTROLS is

	procedure UPDATE_NON_VOLATILE_AND_COPY(
	    CONTROL_SETTING : out CONTROL_SETTING_TYPE) is
	begin
	    GW_CONTROL_MANAGER.UPDATE_NON_VOLATILE_AND_COPY(CONTROL_SETTING);
	end;

	procedure SET_CONTROLS(
	    CONTROL_SETTING : in CONTROL_SETTING_TYPE;
	    JAMMED	    : BOOLEAN) is
	begin
	    GW_CONTROL_MANAGER.SET_CONTROLS(CONTROL_SETTING,JAMMED);
	end;

	procedure SET_CONTROLS(
	    CONTROL_SETTING : in CONTROL_SETTING_TYPE) is
	begin
	    GW_CONTROL_MANAGER.SET_CONTROLS(CONTROL_SETTING);
	end;

	procedure SET_CONTROLS(
	    JAMMED	    : BOOLEAN) is
	begin
	    GW_CONTROL_MANAGER.SET_CONTROLS(JAMMED);
	end;

	procedure INIT_CONTROLS(
	    GRAPHICS_WINDOW : in out GRAPHICS_WINDOW_TYPE;
	    LOW_LEFT_X,
	    LOW_LEFT_Y      : SCREEN_MEASUREMENT;
	    HEIGHT,WIDTH    : SCREEN_MEASUREMENT) is
	begin
	    GW_CONTROL_MANAGER.INIT_CONTROLS(
		GRAPHICS_WINDOW,
		LOW_LEFT_X, LOW_LEFT_Y,
		HEIGHT, WIDTH);
	end;

    end;


    package body INSTRUMENTS is

	DUMMY_APPEARANCE : constant INSTRUMENT_APPEARANCE_TYPE
	    := INSTRUMENT_APPEARANCE_TYPE'first;

	DUMMY_SCALE : constant INSTRUMENT_SCALE_TYPE
	    := INSTRUMENT_SCALE_TYPE'first;

	type INSTRUMENT_TYPE(
		APPEARANCE  : INSTRUMENT_APPEARANCE_TYPE;
		SCALE	    : INSTRUMENT_SCALE_TYPE) is
	    record
		GW_INSTRUMENT : GW_INSTRUMENT_MANAGER.ACCESS_GW_INSTRUMENT_TYPE;
	    end record;

	function CREATE_NUMERIC(
	    NAME	    : INSTRUMENT_NAME_TYPE;
	    X, Y	    : SCREEN_MEASUREMENT;
	    HEIGHT,WIDTH    : SCREEN_MEASUREMENT;
	    APPEARANCE	    : INSTRUMENT_APPEARANCE_TYPE;
	    SCALE	    : NUMERIC_INSTRUMENT_SCALE_TYPE;

	    LO, TOP, HI     : INSTRUMENT_VALUE_TYPE := 0.0;

	    INSTRUMENT_VALUE_PER_RADIAN : INSTRUMENT_VALUE_TYPE;
	    RADIANS_PER_NUMBERED_TICK	: TRIG.RADIANS;
	    TICKS_PER_NUMBERED_TICK	: POSITIVE

	    ) return ACCESS_INSTRUMENT_TYPE is

	    GW_INSTRUMENT
		: constant GW_INSTRUMENT_MANAGER.ACCESS_GW_INSTRUMENT_TYPE
		:=  GW_INSTRUMENT_MANAGER.CREATE_NUMERIC(
			NAME,
			X, Y, HEIGHT, WIDTH,
			APPEARANCE, SCALE,
			LO, TOP, HI,
			INSTRUMENT_VALUE_PER_RADIAN, 
			RADIANS_PER_NUMBERED_TICK,
			TICKS_PER_NUMBERED_TICK);
	begin
	    return 
		new INSTRUMENT_TYPE'(
			DUMMY_APPEARANCE, DUMMY_SCALE,
			GW_INSTRUMENT); 
	end;

	function CREATE_SIGHT(
	    STYLE	    : SIGHT_STYLE_TYPE;
	    SIZE	    : SCREEN_MEASUREMENT;
	    HOLE_SIZE	    : SCREEN_MEASUREMENT
	    ) return ACCESS_INSTRUMENT_TYPE is

	    GW_INSTRUMENT
		: constant GW_INSTRUMENT_MANAGER.ACCESS_GW_INSTRUMENT_TYPE
		:=  GW_INSTRUMENT_MANAGER.CREATE_SIGHT(
			STYLE, SIZE, HOLE_SIZE);

	begin
	    return 
		new INSTRUMENT_TYPE'(
			DUMMY_APPEARANCE, DUMMY_SCALE,
			GW_INSTRUMENT); 
	end;

	function CREATE_TEXT(
	    NAME	    : INSTRUMENT_NAME_TYPE;
	    X, Y	    : SCREEN_MEASUREMENT;
	    HEIGHT,WIDTH    : SCREEN_MEASUREMENT;
	    MAX_LENGTH	    : POSITIVE
	    ) return ACCESS_INSTRUMENT_TYPE is

	    GW_INSTRUMENT
		: constant GW_INSTRUMENT_MANAGER.ACCESS_GW_INSTRUMENT_TYPE
		:=  GW_INSTRUMENT_MANAGER.CREATE_TEXT(
			NAME, X, Y, HEIGHT, WIDTH, MAX_LENGTH);

	begin
	    return 
		new INSTRUMENT_TYPE'(
			DUMMY_APPEARANCE, DUMMY_SCALE,
			GW_INSTRUMENT); 
	end;


	procedure SET_NUMERIC(
	    INSTRUMENT	    : ACCESS_INSTRUMENT_TYPE;
	    VALUE	    : INSTRUMENT_VALUE_TYPE) is
	begin
	    GW_INSTRUMENT_MANAGER.SET_NUMERIC(
		INSTRUMENT.GW_INSTRUMENT,
		VALUE);
	end;

	procedure SET_SIGHT(
	    SIGHT_ON	    : BOOLEAN;
	    X, Y	    : SCREEN_MEASUREMENT) is
	begin
	    GW_INSTRUMENT_MANAGER.SET_SIGHT(
		SIGHT_ON,
		X, Y);
	end;

	procedure SET_TEXT(
	    INSTRUMENT	    : ACCESS_INSTRUMENT_TYPE;
	    VALUE	    : STRING) is
	begin
	    GW_INSTRUMENT_MANAGER.SET_TEXT(
		INSTRUMENT.GW_INSTRUMENT,
		VALUE);
	end;

	procedure UPDATE is
	begin
	    GW_INSTRUMENT_MANAGER.UPDATE;
	end;

	procedure INIT_INSTRUMENTS(
	    GRAPHICS_WINDOW : in out GRAPHICS_WINDOW_TYPE) is
	begin
	    GW_INSTRUMENT_MANAGER.INIT_INSTRUMENTS(GRAPHICS_WINDOW);
	end;

    end;


    package body MASK is

	procedure ERASE_RECTANGLE(
	    BLX, BLY	    : SCREEN_MEASUREMENT;	-- bottom left corner
	    WIDTH, HEIGHT   : SCREEN_MEASUREMENT;	-- size
	    GRAPHICS_WINDOW : in out GRAPHICS_WINDOW_TYPE) is
	begin
	    GW_MASK_MANAGER.ERASE_RECTANGLE(
		BLX, BLY, WIDTH, HEIGHT, GRAPHICS_WINDOW);
	end;

	procedure FILL_RECTANGLE(
	    BLX, BLY	    : SCREEN_MEASUREMENT;	-- bottom left corner
	    WIDTH, HEIGHT   : SCREEN_MEASUREMENT;	-- size
	    GRAPHICS_WINDOW : in out GRAPHICS_WINDOW_TYPE) is
	begin
	    GW_MASK_MANAGER.FILL_RECTANGLE(
		BLX, BLY, WIDTH, HEIGHT, GRAPHICS_WINDOW);
	end;

    end;


    function SELECT_FROM_MENU(
	MENU	: in MENU_MANAGER.MENU_TYPE;
	CHOICE	: STRING := "") return NATURAL is

	use GW_MENU_MANAGER;

	M	    : ACCESS_GW_MENU_TYPE := CREATE_MENU(MENU);
	SELECTION   : NATURAL := SELECT_FROM_MENU(M, CHOICE);

    begin
	DELETE_MENU(M);
	return SELECTION;
    end;


    procedure ERASE(
	GRAPHICS_WINDOW : in out GRAPHICS_WINDOW_TYPE) is

	GW : GRAPHICS_WINDOW_TYPE renames GRAPHICS_WINDOW;

    begin
	SYNC_X.CLEAR_WINDOW(GW);
    end;


    function X_PIXELS_PER_SCREEN_MEASUREMENT
	return PIXELS_PER_SCREEN_MEASUREMENT_TYPE is
    begin
	return X_PIXELS_PER_SCREEN_MEASUREMENT_K;
    end;


    function Y_PIXELS_PER_SCREEN_MEASUREMENT
	return PIXELS_PER_SCREEN_MEASUREMENT_TYPE is
    begin
	return Y_PIXELS_PER_SCREEN_MEASUREMENT_K;
    end;


    procedure MAKE_VISIBLE(
	GRAPHICS_WINDOW : in out GRAPHICS_WINDOW_TYPE) is
    begin
	SYNC_X.MAKE_SEGMENTS_VISIBLE(GRAPHICS_WINDOW);
    end;


    procedure DRAW_LINE(
	X1		: WIDTH_PIXEL_COUNT_SUBTYPE;
	Y1		: HEIGHT_PIXEL_COUNT_SUBTYPE;
	X2		: WIDTH_PIXEL_COUNT_SUBTYPE;
	Y2		: HEIGHT_PIXEL_COUNT_SUBTYPE;
	GRAPHICS_WINDOW	: GRAPHICS_WINDOW_TYPE) is

	S   : LIST_OF_SEGMENTS_TYPE
		    renames GRAPHICS_WINDOW.DRAWING_LIST_OF_SEGMENTS.all;

	L   : SEGMENT_TYPE renames S.SEGMENTS(S.N + 1);

    begin
	L.X1 := X1;
	L.Y1 := Y1;
	L.X2 := X2;
	L.Y2 := Y2;

	if not DOES_DRAW_LAST_POINT then
	    if X1=X2 and Y1=Y2 then
		if X1 > 0 then L.X1 := X1-1; else L.X1 := 1; end if;
	    end if;
	end if;

	S.N := S.N + 1;
    end;


    procedure DRAW_LINE(
	X1		: X_PIXEL_COUNT;
	Y1		: Y_PIXEL_COUNT;
	X2		: X_PIXEL_COUNT;
	Y2		: Y_PIXEL_COUNT;
	GRAPHICS_WINDOW : in out GRAPHICS_WINDOW_TYPE) is

	W  : WINDOW_INFO_TYPE renames GRAPHICS_WINDOW.all;

    begin
	DRAW_LINE(
	    WIDTH_PIXEL_COUNT_SUBTYPE(X1),
	    HEIGHT_PIXEL_COUNT_SUBTYPE(Y1) + W.HEIGHT,
	    WIDTH_PIXEL_COUNT_SUBTYPE(X2),
	    HEIGHT_PIXEL_COUNT_SUBTYPE(Y2) + W.HEIGHT,
	    GRAPHICS_WINDOW);
    end;


    task body SYNC_X is separate;


begin
    DISPLAY :=
	OPEN_DISPLAY(
	    C_TYPES.NULL_TERMINATED.TO_STRING(
		LOGICAL_TO_STRING("FCTM_DECWINDOWS_DISPLAY","")));

    if DISPLAY = NULL_DISPLAY then
	BUG("Unable to open FCTM_DECWINDOWS_DISPLAY");
    end if;

    SCREEN	:= DEFAULT_SCREEN_OF_DISPLAY(DISPLAY);
    COLORMAP	:= DEFAULT_COLORMAP_OF_SCREEN(SCREEN);
    HEIGHT	:= HEIGHT_OF_SCREEN(SCREEN);
    HEIGHT_MM	:= HEIGHT_MM_OF_SCREEN(SCREEN);
    WIDTH	:= WIDTH_OF_SCREEN(SCREEN);
    WIDTH_MM	:= WIDTH_MM_OF_SCREEN(SCREEN);
    DEPTH	:= DEFAULT_DEPTH_OF_SCREEN(SCREEN);
    VISUAL	:= DEFAULT_VISUAL_OF_SCREEN(SCREEN).all;

    -- Try to cope with TRUE_COLOR displays
    --
    if LOGICAL_TO_BOOLEAN("FCTM_TRUE_COLOR", FALSE) then
	declare
	    TEMPLATE	    : VISUAL_INFO_TYPE;
	    VISUAL_INFOS    : ACCESS_UNCHECKED_VECTOR_OF_VISUAL_INFO_TYPE;
	    LENGTH	    : NATURAL;
	begin
	    TEMPLATE.SCREEN := DEFAULT_SCREEN(DISPLAY);
	    TEMPLATE.DEPTH  := 8;
	    TEMPLATE.CLASS  := PSEUDO_COLOR;

	    GET_VISUAL_INFO(
		VISUAL_INFOS,
		DISPLAY,
		VISUAL_INFO_MASK_TYPE'(CLASS|SCREEN|DEPTH=>TRUE, others=>FALSE),
		TEMPLATE,
		LENGTH);

	    if LENGTH = 0 then
		BUG("Unable to find any visual infos");
	    end if;

	    COLORMAP :=
		CREATE_COLORMAP(
		    DISPLAY,
		    ROOT_WINDOW_OF_SCREEN(SCREEN),
		    VISUAL_INFOS(1).VISUAL.all,
		    NONE_OF_THEM);

	    if COLORMAP = COLORMAP_INIT then
		BUG("Unable to create colormap");
	    end if;

	    --HACK: FREE(VISUAL_INFOS.all'address);
	end;
    end if;

    -- Tricky, in that SCREEN_MEASUREMENTS are (a) cm's, (b) positive is
    -- left to right, bottom to top...
    --
    declare
	SCALE_FACTOR : PIXELS_PER_SCREEN_MEASUREMENT_TYPE := 1.0;

	procedure RESCALE(WHICH : in STRING; MAX : MILLIMETRES_TYPE) is
	    MM : MILLIMETRES_TYPE;
	begin
	    MM := MILLIMETRES_TYPE(LOGICAL_TO_FLOAT("FCTM_"&WHICH,100.0))*10;
	    if MM>MAX then
		declare
		    NSF : PIXELS_PER_SCREEN_MEASUREMENT_TYPE
			:=  PIXELS_PER_SCREEN_MEASUREMENT_TYPE(MAX)/
			    PIXELS_PER_SCREEN_MEASUREMENT_TYPE(MM);
		begin
		    if NSF < SCALE_FACTOR then
			SCALE_FACTOR := NSF;
		    end if;
		end;
	    end if;
	end;

    begin
	-- Reduce to fit the actual monitor you find
	--
	RESCALE("WIDTH",  WIDTH_MM);
	RESCALE("HEIGHT", HEIGHT_MM);

	-- Calculate the complete scale factor
	--
        X_PIXELS_PER_SCREEN_MEASUREMENT_K :=
	    PIXELS_PER_SCREEN_MEASUREMENT_TYPE(WIDTH)/
	    PIXELS_PER_SCREEN_MEASUREMENT_TYPE(WIDTH_MM)*
	    10.0*
	    SCALE_FACTOR;

	Y_PIXELS_PER_SCREEN_MEASUREMENT_K :=
	    (-1.0)*
	    PIXELS_PER_SCREEN_MEASUREMENT_TYPE(HEIGHT)/
	    PIXELS_PER_SCREEN_MEASUREMENT_TYPE(HEIGHT_MM)*
	    10.0*
	    SCALE_FACTOR;
    end;

    -- Allocate enough colormap entries for animation, and fill in the
    -- COLORMAP_INDEXS array.
    --
    declare
	CI		: NATURAL;
	ALLOCATED_CELLS : VECTOR_OF_COLORMAP_INDEX_TYPE(1..1);
	SUCCEEDED	: BOOLEAN;

	procedure FOR_ALL_PLANE_MASKS(
	    PMI : NATURAL; CMI : COLORMAP_INDEX_TYPE) is
	begin
	    if PMI < PLANE_MASKS'first then
		CI := CI+1;
		COLORMAP_INDEXS(CI) := CMI;
	    else
		FOR_ALL_PLANE_MASKS(PMI-1, CMI);
		FOR_ALL_PLANE_MASKS(PMI-1,
		    CMI + COLORMAP_INDEX_TYPE(TO_COLORMAP_INDEX(
			    PLANE_MASKS(PMI))));
	    end if;
	end;

    begin
	ALLOC_COLOR_CELLS( SUCCEEDED	    => SUCCEEDED,
	    DISPLAY	    => DISPLAY,
	    COLORMAP	    => COLORMAP,
	    CONTIG	    => TRUE,
	    PLANE_MASKS     => PLANE_MASKS,
	    PIXELS	    => ALLOCATED_CELLS);
	if not SUCCEEDED then
	    BUG("Unable to get enough color planes to do animation");
	end if;
	CI := COLORMAP_INDEXS'first-1;
	for I in ALLOCATED_CELLS'range loop
	    FOR_ALL_PLANE_MASKS(PLANE_MASKS'last, ALLOCATED_CELLS(I));
	end loop;
    end;

    -- Set the allocated colors in the colormap
    --
    declare
	FG_CI   : COLORMAP_INDEX_TYPE := WHITE_PIXEL_OF_SCREEN(SCREEN);
	BG_CI   : COLORMAP_INDEX_TYPE := BLACK_PIXEL_OF_SCREEN(SCREEN);
	TMP_CI  : COLORMAP_INDEX_TYPE;
    begin

	if LOGICAL_TO_STRING("FCTM_BG_COLOR","BLACK") = "WHITE" then
	    TMP_CI := BG_CI; BG_CI := FG_CI; FG_CI := TMP_CI;
	end if;

	for I in COLORS'range loop
	    COLORS(I).PIXEL := FG_CI;
	end loop;
	COLORS(TO_COLORMAP_INDEXS_RANGE(ALWAYS_BG)).PIXEL   := BG_CI;
	COLORS(TO_COLORMAP_INDEXS_RANGE(FG_BG)).PIXEL	:= BG_CI;

	QUERY_COLORS(DISPLAY, COLORMAP, COLORS);

	for I in COLORS'range loop
	    COLORS(I).PIXEL := COLORMAP_INDEXS(I);
	end loop;

	STORE_COLORS(DISPLAY, COLORMAP, COLORS);

    end;

    -- Set which plane to write in next
    --
    PLANE_TO_WRITE := 1;

    -- Specify all the non-default attributes for the window
    --
    SWA_MASK :=
	(EVENT_MASK|BACKGROUND_PIXEL|COLORMAP|BACKING_STORE => TRUE,
	 others => FALSE);

    SWA.EVENT_MASK := 
	(KEY_PRESS|KEY_RELEASE|
	 BUTTON_PRESS|BUTTON_RELEASE|
	 POINTER_MOTION|EXPOSURE => TRUE,
	 BUTTONS_MOTION => (others => FALSE), others => FALSE);

    SWA.BACKGROUND_PIXEL :=
	COLORMAP_INDEXS(TO_COLORMAP_INDEXS_RANGE(ALWAYS_BG));

    SWA.COLORMAP := COLORMAP;

    SWA.BACKING_STORE := NOT_USEFUL;

    -- Load the font for text writing
    --
    INIT_FONT;

end;
