--***********************************************************************
--									*
--	COPYRIGHT 1992		DIGITAL EQUIPMENT CORPORATION		*
--									*
--   This software was written by Bevin Brett, of Digital Equipment	*
--   Corporation.							*
--									*
--   Digital assumes no responsibility AT ALL for the use or reliability*
--   of this software.							*
--									*
--   Redistribution and use in source and binary forms are permitted	*
--   provided that this entire heading from --*** to --*** are          *
--   duplicated in all such forms and that any documentation,		*
--   advertising materials, and other materials related to such		*
--   distribution and use acknowledge that the software was developed	*
--   by Digital Equipment Corporation. The name of Digital Equipment	*
--   Corporation may not be used to endorse or promote products derived	*
--   from this software without specific prior written permission.	*
--									*
--   THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR	*
--   IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
--   WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.*
--									*
--***********************************************************************


with SYSTEM_INTERFACE;

pragma ELABORATE(SYSTEM_INTERFACE);

separate(GENERATE_NEXT_VIEW)
    package body CLIPPING_INFO_PKG is

	function SQRT is new SYSTEM_INTERFACE.DIGITS_SQRT(SCALE_TYPE);


-- Theory
--
--	Given a line described as   K(AX-BY)=0
--
--	it is possible, by defining K as 1.0/SQRT(A*A+B*B), to compute how far
--	away any (X,Y) point is from this line simply by evaluating
--
--				    K(AX-BY) = distance!
--
--	Furthermore this gives a negative value on one side of the line, and
--	as positive value on the other...
--
--	For example,	x/2 = y gets rewritten as K((1/2)x-y) = distance,
--				     where K = 1/SQRT(1.25),
--			and this is positive below the line and negative above
--			it.
--

	type ABK_TYPE is
	    record
		A,B,K	: SCALE_TYPE;
	    end record;

	LEFT_A : constant SCALE_TYPE := SCALE_TYPE( SCREEN_PLACEMENT.DISTANCE);
	LEFT_B : constant SCALE_TYPE := SCALE_TYPE(-SCREEN_PLACEMENT.LEFT);
	LEFT_K : constant SCALE_TYPE := SCALE_TYPE(1.0)/SQRT(LEFT_A*LEFT_A+LEFT_B*LEFT_B);

	LEFT_ABK
	    : constant ABK_TYPE
	    :=	ABK_TYPE'(LEFT_A, LEFT_B, LEFT_K);

	RIGHT_A : constant SCALE_TYPE := SCALE_TYPE(-SCREEN_PLACEMENT.DISTANCE);
	RIGHT_B : constant SCALE_TYPE := SCALE_TYPE(-SCREEN_PLACEMENT.RIGHT);
	RIGHT_K : constant SCALE_TYPE := SCALE_TYPE(1.0)/SQRT(RIGHT_A*RIGHT_A+RIGHT_B*RIGHT_B);

	RIGHT_ABK
	    : constant ABK_TYPE
	    :=  ABK_TYPE'(RIGHT_A, RIGHT_B, RIGHT_K);

	TOP_A : constant SCALE_TYPE := SCALE_TYPE( SCREEN_PLACEMENT.UP);
	TOP_B : constant SCALE_TYPE := SCALE_TYPE( SCREEN_PLACEMENT.DISTANCE);
	TOP_K : constant SCALE_TYPE := SCALE_TYPE(1.0)/SQRT(TOP_A*TOP_A+TOP_B*TOP_B);

	TOP_ABK
	    : constant ABK_TYPE
	    :=  ABK_TYPE'(TOP_A, TOP_B, TOP_K);

	BOTTOM_A : constant SCALE_TYPE := SCALE_TYPE( SCREEN_PLACEMENT.DOWN);
	BOTTOM_B : constant SCALE_TYPE := SCALE_TYPE(-SCREEN_PLACEMENT.DISTANCE);
	BOTTOM_K : constant SCALE_TYPE := SCALE_TYPE(1.0)/SQRT(BOTTOM_A*BOTTOM_A+BOTTOM_B*BOTTOM_B);

	BOTTOM_ABK
	    : constant ABK_TYPE
	    :=  ABK_TYPE'(BOTTOM_A, BOTTOM_B, BOTTOM_K);


	function DISTANCE( X, Y : SCALE_TYPE; ABK : ABK_TYPE) return SCALE_TYPE is
	begin
	    return ABK.K*(ABK.A*X-ABK.B*Y);
	end;

	procedure GET_CLIPPING_INFO(
	    O		    : OBJECTS.OBJECT_TYPE;
	    OBJECT_ORIGIN   : VIEW_PHYSICS.POSITION;
	    CLIPPING_INFO   : out CLIPPING_INFO_TYPE) is

	    M	: constant SCALE_TYPE
		:= SCALE_TYPE(APPEARANCE_MANAGER.APPEARANCE_TO_MAX_RADIUS(
				APPEARANCE_MANAGER.TO_ACCESS_APPEARANCE_TYPE(
				    O.APPEARANCE_INFO)));

	    J	: constant SCALE_TYPE
		:= SCALE_TYPE(OBJECT_ORIGIN.J);

	begin

	    -- Default, visible and no clipping needed
	    --
	    CLIPPING_INFO := (NOT_VISIBLE=>FALSE, others=>FALSE);
	    CLIPPING_INFO.ZERO_RADIUS := (M=0.0);

	    -- See if is simply behind the viewer
	    --
	    if J < SCREEN_PLACEMENT.DISTANCE - M then
		CLIPPING_INFO.NOT_VISIBLE := true;
		return;
	    end if;

	    -- Or in front, but close enough to require clipping
	    --
	    if J < SCREEN_PLACEMENT.DISTANCE + M then
		CLIPPING_INFO.SOME_CLIPPING := true;
		CLIPPING_INFO.FRONT_OR_BACK := true;
	    end if;


	    -- Try to improve, by considering the four lines, two each from
	    -- projecting down onto the IJ plane and along onto the JK plane.
	    --
	    declare
		I : constant SCALE_TYPE := SCALE_TYPE(OBJECT_ORIGIN.I);
		K : constant SCALE_TYPE := SCALE_TYPE(OBJECT_ORIGIN.K);

		function CHECK(
		    LEFT_RIGHT	: BOOLEAN;
		    X,Y 	: SCALE_TYPE;
		    ABK 	: ABK_TYPE)
		return BOOLEAN is
		    D : constant SCALE_TYPE := DISTANCE(X,Y,ABK);
		begin
		    if D < -M then return TRUE; end if;
		    if D < M then
			if LEFT_RIGHT then
			    CLIPPING_INFO.LEFT_OR_RIGHT := TRUE;
			else
			    CLIPPING_INFO.UP_OR_DOWN := TRUE;
			end if;
			CLIPPING_INFO.SOME_CLIPPING := TRUE;
		    end if;
		    return FALSE;
		end;	pragma INLINE(CHECK);

	    begin
		if	CHECK(TRUE,  I,J,LEFT_ABK)
		or else CHECK(TRUE,  I,J,RIGHT_ABK)
		or else CHECK(FALSE, J,K,TOP_ABK)
		or else CHECK(FALSE, J,K,BOTTOM_ABK)
		then
		    CLIPPING_INFO.NOT_VISIBLE := true;
		end if;
	    end;
	end;

    end;

