--***********************************************************************
--									*
--	COPYRIGHT 1992		DIGITAL EQUIPMENT CORPORATION		*
--									*
--   This software was written by Bevin Brett, of Digital Equipment	*
--   Corporation.							*
--									*
--   Digital assumes no responsibility AT ALL for the use or reliability*
--   of this software.							*
--									*
--   Redistribution and use in source and binary forms are permitted	*
--   provided that this entire heading from --*** to --*** are          *
--   duplicated in all such forms and that any documentation,		*
--   advertising materials, and other materials related to such		*
--   distribution and use acknowledge that the software was developed	*
--   by Digital Equipment Corporation. The name of Digital Equipment	*
--   Corporation may not be used to endorse or promote products derived	*
--   from this software without specific prior written permission.	*
--									*
--   THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR	*
--   IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
--   WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.*
--									*
--***********************************************************************


with IS_DEMO_PKG, TEXT_IO, LOGICAL_TO_BOOLEAN;

pragma ELABORATE(IS_DEMO_PKG, TEXT_IO, LOGICAL_TO_BOOLEAN);

package body FLIGHT_RECORDER is

    type COMPRESSED_SECONDS is
	delta 0.01 range 0.0..1.0;
	for COMPRESSED_SECONDS'small use 1.0/128.0;
	for COMPRESSED_SECONDS'size  use 8;

    type SUBFRAME_TYPE(
	    NUMBER_OF_ITEMS : NUMBER_OF_ITEMS_RANGE
			    := NUMBER_OF_ITEMS_RANGE'last)
    is
	record
	    ELAPSED_SINCE_LAST_NEW_FRAME : COMPRESSED_SECONDS;
	    NEW_FRAME	    : BOOLEAN;
	    MARK_FRAME	    : BOOLEAN;
	    REMOTE_NUMBER   : REMOTE_NUMBER_TYPE range 1..2**6-1;
	    MESSAGE	    : MESSAGE_TYPE(NUMBER_OF_ITEMS);
	end record;

    for SUBFRAME_TYPE use
	record
	    ELAPSED_SINCE_LAST_NEW_FRAME at 0 range 0..7;
	    NEW_FRAME		at 1 range 0..0;
	    MARK_FRAME		at 1 range 1..1;
	    REMOTE_NUMBER	at 1 range 2..7;
	end record;

    type TAPE_TYPE(NUMBER_OF_ITEMS : NUMBER_OF_ITEMS_RANGE);
    type ACCESS_TAPE_TYPE is access TAPE_TYPE;
    type TAPE_TYPE(NUMBER_OF_ITEMS : NUMBER_OF_ITEMS_RANGE) is
	record
	    NEXT, PREV  : ACCESS_TAPE_TYPE;
	    SUBFRAME	: SUBFRAME_TYPE(NUMBER_OF_ITEMS);
	end record;

    -- The actual tape
    --
    TAPE_FIRST		: ACCESS_TAPE_TYPE;
    TAPE_LAST		: ACCESS_TAPE_TYPE;

    -- Where-abouts on the tape is being played
    --
    TAPE_PLAY_HEAD	: ACCESS_TAPE_TYPE;
    TAPE_PLAY_ELAPSED_SINCE_LAST_NEW_FRAME : SCALAR_PHYSICS.SECONDS := 0.0;

    -- Info for recording more
    --
    NEW_FRAME		: BOOLEAN := TRUE;  -- Is indeed a new frame
    ELAPSED_SINCE_LAST_NEW_FRAME : COMPRESSED_SECONDS := 0.0;
    MARK_FRAME		: BOOLEAN := TRUE;  -- Place mark at beginning of tape


    package DISK is

	WRITING		    : BOOLEAN := FALSE;
	READING 	    : BOOLEAN := FALSE;
	LAST_FRAME_TO_WRITE : ACCESS_TAPE_TYPE;
	FRAMES_READ_FIRST,
	FRAMES_READ_LAST    : ACCESS_TAPE_TYPE;

	pragma SHARED(WRITING);
	pragma SHARED(READING);
	pragma SHARED(LAST_FRAME_TO_WRITE);
	pragma SHARED(FRAMES_READ_FIRST);
	pragma SHARED(FRAMES_READ_LAST);

	READER_STARTED	    : BOOLEAN := FALSE;
	WRITER_STARTED	    : BOOLEAN := FALSE;

	procedure START_READER;
	procedure START_WRITER;

    end;


    procedure START_NEW_FRAME(
	ELAPSED : SCALAR_PHYSICS.SECONDS;
	MARK	: BOOLEAN := FALSE) is
    begin
	NEW_FRAME    := TRUE;

	declare
	    use SCALAR_PHYSICS;
	    E : COMPRESSED_SECONDS := COMPRESSED_SECONDS'last;
	begin
	    if ELAPSED < SECONDS(
			    COMPRESSED_SECONDS'last-COMPRESSED_SECONDS'small)
	    then
		E := COMPRESSED_SECONDS(ELAPSED);
	    end if;
	    ELAPSED_SINCE_LAST_NEW_FRAME := E;
	end;

	MARK_FRAME   := MARK or MARK_PRESSED;
	MARK_PRESSED := FALSE;
    end;


    package FILE_MANAGER is

	function ALLOCATE(NUMBER_OF_ITEMS : NUMBER_OF_ITEMS_RANGE)
	    return ACCESS_TAPE_TYPE;

    end;

    package body FILE_MANAGER is separate;


    package body DISK is separate;


    procedure RECORD_MESSAGE(
	MESSAGE : MESSAGE_TYPE;
	FROM	: REMOTE_NUMBER_TYPE) is

	NEXT	: constant ACCESS_TAPE_TYPE
		:= FILE_MANAGER.ALLOCATE(MESSAGE.NUMBER_OF_ITEMS);

    begin

	if NEXT=null then
	    TEXT_IO.PUT_LINE("Out of tape");
	    STATE := OFF;
	    return;
	end if;

	-- Create (partial) frame
	--
	declare
	    SF : SUBFRAME_TYPE renames NEXT.SUBFRAME;
	begin
	    SF.NEW_FRAME    := NEW_FRAME;
	    SF.ELAPSED_SINCE_LAST_NEW_FRAME := ELAPSED_SINCE_LAST_NEW_FRAME;
	    SF.MARK_FRAME   := MARK_FRAME;
	    SF.REMOTE_NUMBER:= FROM;
	    SF.MESSAGE	    := MESSAGE;
	end;

	-- Reset flags, etc.
	--
	NEW_FRAME  := FALSE;
	ELAPSED_SINCE_LAST_NEW_FRAME := 0.0;
	MARK_FRAME := FALSE;

	-- Sew onto end of tape
	--
	NEXT.PREV := TAPE_LAST;
	if TAPE_FIRST = null then
	    TAPE_FIRST := NEXT;
	else
	    TAPE_LAST.NEXT := NEXT;
	end if;
	TAPE_LAST := NEXT;

	-- Sew ends of tape together
	--
	TAPE_LAST.NEXT	:= TAPE_FIRST;
	TAPE_FIRST.PREV := TAPE_LAST;

	-- Add this to the list of frames to write
	--
	DISK.LAST_FRAME_TO_WRITE := TAPE_LAST;

	-- Start the writer if this is the first time
	--
	if not DISK.WRITER_STARTED then
	    DISK.START_WRITER;
	end if;
    end;


    procedure PLAY(ELAPSED : SCALAR_PHYSICS.SECONDS) is

	procedure RECEIVE(
	    ITEMS	: in out VECTOR_OF_MESSAGE_ITEM_TYPE;
	    ORIGINAL_RN : REMOTE_NUMBER_TYPE) is
	begin
	    for I in ITEMS'range loop
		RECEIVE_AN_ITEM(
		    ITEMS(I),
		    FLIGHT_RECORDER_REMOTE_NUMBER,
		    ORIGINAL_RN);
	    end loop;
	end;

	procedure PLAY_ONE_FRAME is
	    FIRST : BOOLEAN := TRUE;
	begin
	    loop
		declare
		    TAPE : TAPE_TYPE renames TAPE_PLAY_HEAD.all;
		begin
		    if TAPE.SUBFRAME.NEW_FRAME then
			if FIRST then FIRST := FALSE; else exit; end if;
		    end if;
		    RECEIVE(
			TAPE.SUBFRAME.MESSAGE.ITEMS,
			TAPE.SUBFRAME.REMOTE_NUMBER);
		    TAPE_PLAY_HEAD := TAPE.NEXT;
		end;
	    end loop;
	end;

	procedure FORWARD_ONE_FRAME is
	    FIRST : BOOLEAN := TRUE;
	begin
	    loop
		declare
		    TAPE : TAPE_TYPE renames TAPE_PLAY_HEAD.all;
		begin
		    if TAPE.SUBFRAME.NEW_FRAME then
			if FIRST then FIRST := FALSE; else exit; end if;
		    end if;
		    TAPE_PLAY_HEAD := TAPE.NEXT;
		end;
	    end loop;
	end;

	procedure BACK_ONE_FRAME is
	    FIRST : BOOLEAN := TRUE;
	begin
	    loop
		declare
		    TAPE : TAPE_TYPE renames TAPE_PLAY_HEAD.all;
		begin
		    if TAPE.SUBFRAME.NEW_FRAME then
			if FIRST then FIRST := FALSE; else exit; end if;
		    end if;
		    TAPE_PLAY_HEAD := TAPE.PREV;
		end;
	    end loop;
	end;


    begin

	-- Start the reader if this is the first time
	--
	if not DISK.READER_STARTED then
	    DISK.START_READER;
	end if;

	-- If any new frames available from the READER then
	-- glue those onto the end of the tape
	--
	if DISK.FRAMES_READ_FIRST /= null then

	    if TAPE_LAST = null then
		TAPE_FIRST	:= DISK.FRAMES_READ_FIRST;
	    else
		TAPE_LAST.NEXT	:= DISK.FRAMES_READ_FIRST;
		DISK.FRAMES_READ_FIRST.PREV := TAPE_LAST;
	    end if;
	    TAPE_LAST	    := DISK.FRAMES_READ_LAST;

	    TAPE_LAST.NEXT  := TAPE_FIRST;
	    TAPE_FIRST.PREV := TAPE_LAST;

	    -- clear the pointer, so the reader knows it can put more stuff
	    -- there.  The _FIRST must change second, because it is the flag
	    -- the reader is waiting for.
	    --
	    DISK.FRAMES_READ_LAST  := null;
	    DISK.FRAMES_READ_FIRST := null;
	end if;

	-- Start tape if was none before
	--
	if TAPE_PLAY_HEAD = null then
	    TAPE_PLAY_HEAD := TAPE_FIRST;
	    if TAPE_PLAY_HEAD = null then return; end if;
	end if;

	-- Play frames
	--
	if MARK_PRESSED then
	    MARK_PRESSED := FALSE;
	    declare
		INIT_TAPE_PLAY_HEAD : ACCESS_TAPE_TYPE := TAPE_PLAY_HEAD;
	    begin
		loop
		    if SPEED < 0 then
			TAPE_PLAY_HEAD := TAPE_PLAY_HEAD.PREV;
		    else
			TAPE_PLAY_HEAD := TAPE_PLAY_HEAD.NEXT;
		    end if;
		    exit when	TAPE_PLAY_HEAD.SUBFRAME.MARK_FRAME or
				TAPE_PLAY_HEAD = INIT_TAPE_PLAY_HEAD;
		end loop;
		SPEED := 0;
	    end;
	end if;

	declare
	    use SCALAR_PHYSICS;
	    E	    : SECONDS := ELAPSED*SCALE_TYPE(SPEED) +
				  TAPE_PLAY_ELAPSED_SINCE_LAST_NEW_FRAME;
	    PLAYED  : BOOLEAN := FALSE;
	begin
	    if E > 0.0 then
		loop
		    declare
			S   : constant SECONDS
			    := SECONDS(TAPE_PLAY_HEAD.SUBFRAME.
				ELAPSED_SINCE_LAST_NEW_FRAME);
		    begin
			exit when E < S;
			E := E-S;
			PLAY_ONE_FRAME; PLAYED := TRUE;
		    end;
		end loop;

	    else
		loop
		    BACK_ONE_FRAME;
		    declare
			S   : constant SECONDS
			    := SECONDS(TAPE_PLAY_HEAD.SUBFRAME.
				ELAPSED_SINCE_LAST_NEW_FRAME);
		    begin
			E := E+S;
			PLAY_ONE_FRAME; PLAYED := TRUE;
			BACK_ONE_FRAME;
			exit when E >= 0.0;
		    end;
		end loop;
	    end if;

	    TAPE_PLAY_ELAPSED_SINCE_LAST_NEW_FRAME := E;

	    if not PLAYED then
		if TAPE_PLAY_HEAD = TAPE_FIRST then
		    PLAY_ONE_FRAME;
		    BACK_ONE_FRAME;
		else
		    BACK_ONE_FRAME;
		    PLAY_ONE_FRAME;
		end if;
	    end if;

	end;
    end;


begin
    if IS_DEMO_PKG.IS_DEMO then
	STATE := ON_PLAY;
	SPEED := 1;
    end if;
    if LOGICAL_TO_BOOLEAN("FCTM_RECORD", FALSE) then
	STATE := ON_RECORD;
    end if;
end;
