--***********************************************************************
--									*
--	COPYRIGHT 1992		DIGITAL EQUIPMENT CORPORATION		*
--									*
--   This software was written by Bevin Brett, of Digital Equipment	*
--   Corporation.							*
--									*
--   Digital assumes no responsibility AT ALL for the use or reliability*
--   of this software.							*
--									*
--   Redistribution and use in source and binary forms are permitted	*
--   provided that this entire heading from --*** to --*** are          *
--   duplicated in all such forms and that any documentation,		*
--   advertising materials, and other materials related to such		*
--   distribution and use acknowledge that the software was developed	*
--   by Digital Equipment Corporation. The name of Digital Equipment	*
--   Corporation may not be used to endorse or promote products derived	*
--   from this software without specific prior written permission.	*
--									*
--   THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR	*
--   IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
--   WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.*
--									*
--***********************************************************************


with
    BUG, ETHERNET_IO,
    UNCHECKED_DEALLOCATION, SYSTEM;

pragma ELABORATE(
    BUG, ETHERNET_IO);

package body ETHERNET is
    use ETHERNET_IO;

    CHANNEL	    : CHANNEL_TYPE;
    SYNCH_IOSB	    : IO_STATUS_TYPE;
    ASYNCH_IOSBS    : array(1..4) of IO_STATUS_TYPE;
    NEXT_ASYNCH_IOSB: INTEGER range ASYNCH_IOSBS'range := ASYNCH_IOSBS'first;

    type INPUT_CHANNEL_TYPE is
	record
	    PKT_HEAD	: PACKET_HEADER_INFORMATION_TYPE;
	end record;

    type OUTPUT_CHANNEL_TYPE is
	record
	    DESTINATION : ETHERNET_PHYSICAL_ADDRESS_TYPE;
	end record;

    -- The ETHERNET_ADDRESS_TYPE is a string "HL-HL-HL-HL-HL-HL"
    -- of ascending bytes of the physical address.
    --
    BAD_ETHERNET_ADDRESS : exception;


    ALREADY_INIT : BOOLEAN := FALSE;

    procedure INIT_IF_NEEDED is
    begin
	if ALREADY_INIT then return; end if;
	ALREADY_INIT := TRUE;
	ASSIGN(CHANNEL);
	INIT_AS_OKAY(SYNCH_IOSB);
	for I in ASYNCH_IOSBS'range loop
	    INIT_AS_OKAY(ASYNCH_IOSBS(I));
	end loop;
	STARTUP(CHANNEL);
    end;

    function TO_ETHERNET_PHYSICAL_ADDRESS_TYPE(
	FROM : ETHERNET_ADDRESS_TYPE)
	return ETHERNET_PHYSICAL_ADDRESS_TYPE is

	P : ETHERNET_PHYSICAL_ADDRESS_TYPE;

	function TO_NIBBLE(C : CHARACTER) return NATURAL is
	begin
	    if C in '0'..'9' then
		return CHARACTER'pos(C)-CHARACTER'pos('0');
	    elsif C in 'A'..'F' then
		return CHARACTER'pos(C)-CHARACTER'pos('A') + 10;
	    elsif C in 'a'..'f' then
		return CHARACTER'pos(C)-CHARACTER'pos('a') + 10;
	    else
		raise BAD_ETHERNET_ADDRESS;
	    end if;
	end;

    begin
	if FROM'first /= 1
	or FROM'last  /= 17
	then
	    raise BAD_ETHERNET_ADDRESS;
	end if;

	for I in 1..5 loop
	    if FROM(I*3) /= '-' then raise BAD_ETHERNET_ADDRESS; end if;
	end loop;

	for I in P'range loop
	    P(I) := TO_NIBBLE(FROM(3*I-2))*16 + TO_NIBBLE(FROM(3*I-1));
	end loop;

	return P;
    end;


    function TO_ETHERNET_ADDRESS_TYPE(
	FROM : ETHERNET_PHYSICAL_ADDRESS_TYPE)
	return ETHERNET_ADDRESS_TYPE is

	S : ETHERNET_ADDRESS_TYPE(1..17);

	TO_CHARACTER
	    :	constant array(0..15) of CHARACTER
	    :=	"0123456789ABCDEF";

    begin
	for I in 1..5 loop
	    S(I*3) := '-';
	end loop;

	for I in FROM'range loop
	    S(3*I-2) := TO_CHARACTER(FROM(I)  /  16);
	    S(3*I-1) := TO_CHARACTER(FROM(I) rem 16);
	end loop;

	return S;
    end;


    package REMOTE_TO_PHYSICAL is

	MAX_REMOTE_NUMBER_SO_FAR : NATURAL := 0;

	type LIST_OF_REMOTE_TYPE;
	type ACCESS_LIST_OF_REMOTE_TYPE is access LIST_OF_REMOTE_TYPE;
	type LIST_OF_REMOTE_TYPE is
	    record
		NEXT			    : ACCESS_LIST_OF_REMOTE_TYPE;
		REMOTE_NUMBER		    : REMOTE_NUMBER_TYPE;
		ETHERNET_PHYSICAL_ADDRESS   : ETHERNET_PHYSICAL_ADDRESS_TYPE;
	    end record;

	LIST_OF_REMOTE : ACCESS_LIST_OF_REMOTE_TYPE;
	    pragma SHARED(LIST_OF_REMOTE);

    end;

    procedure TO_REMOTE_NUMBER_TYPE(
	ETHERNET_PHYSICAL_ADDRESS : ETHERNET_PHYSICAL_ADDRESS_TYPE;
	ADD_IF_NOT_DEFINED  : in BOOLEAN;
	REMOTE_NUMBER	    : out REMOTE_NUMBER_TYPE;
	WAS_DEFINED	    : out BOOLEAN) is

	use REMOTE_TO_PHYSICAL;

	L : ACCESS_LIST_OF_REMOTE_TYPE := LIST_OF_REMOTE;

    begin
	-- search
	--
	while L /= null loop
	    if L.ETHERNET_PHYSICAL_ADDRESS = ETHERNET_PHYSICAL_ADDRESS then
		REMOTE_NUMBER := L.REMOTE_NUMBER;
		WAS_DEFINED   := TRUE;
		return;
	    end if;
	    L := L.NEXT;
	end loop;

	WAS_DEFINED := FALSE;

	if not ADD_IF_NOT_DEFINED then
	    REMOTE_NUMBER := REMOTE_NUMBER_TYPE'last;
	    return;
	end if;

	-- not there, add
	--
	MAX_REMOTE_NUMBER_SO_FAR := MAX_REMOTE_NUMBER_SO_FAR + 1;
	LIST_OF_REMOTE :=
	    new LIST_OF_REMOTE_TYPE'(
		    LIST_OF_REMOTE,
		    REMOTE_NUMBER_TYPE(MAX_REMOTE_NUMBER_SO_FAR),
		    ETHERNET_PHYSICAL_ADDRESS);

	REMOTE_NUMBER := LIST_OF_REMOTE.REMOTE_NUMBER;
    end;


    function TO_ETHERNET_PHYSICAL_ADDRESS_TYPE(
	REMOTE_NUMBER : REMOTE_NUMBER_TYPE)
	return ETHERNET_PHYSICAL_ADDRESS_TYPE is
	use REMOTE_TO_PHYSICAL;

	L : ACCESS_LIST_OF_REMOTE_TYPE := LIST_OF_REMOTE;

    begin
	-- search
	--
	while L /= null loop
	    if L.REMOTE_NUMBER = REMOTE_NUMBER then
		return L.ETHERNET_PHYSICAL_ADDRESS;
	    end if;
	    L := L.NEXT;
	end loop;

	-- not there
	--
	raise REMOTE_NUMBER_ERROR;
    end;


    procedure TO_REMOTE_NUMBER_TYPE(
	ETHERNET_ADDRESS    : ETHERNET_ADDRESS_TYPE;
	ADD_IF_NOT_DEFINED  : in BOOLEAN;
	REMOTE_NUMBER	    : out REMOTE_NUMBER_TYPE;
	WAS_DEFINED	    : out BOOLEAN) is
    begin
	TO_REMOTE_NUMBER_TYPE(
	    TO_ETHERNET_PHYSICAL_ADDRESS_TYPE(ETHERNET_ADDRESS),
	    ADD_IF_NOT_DEFINED,
	    REMOTE_NUMBER,
	    WAS_DEFINED);
    end;


    function TO_ETHERNET_ADDRESS_TYPE(
	REMOTE_NUMBER : REMOTE_NUMBER_TYPE)
	return ETHERNET_ADDRESS_TYPE is
    begin
	return TO_ETHERNET_ADDRESS_TYPE(
		    TO_ETHERNET_PHYSICAL_ADDRESS_TYPE(REMOTE_NUMBER));
    end;

    function ALLOCATE_REMOTE_NUMBER
	return REMOTE_NUMBER_TYPE is

	use REMOTE_TO_PHYSICAL;
    begin
	MAX_REMOTE_NUMBER_SO_FAR := MAX_REMOTE_NUMBER_SO_FAR + 1;
	return REMOTE_NUMBER_TYPE(MAX_REMOTE_NUMBER_SO_FAR);
    end;


    procedure DEALLOCATE is
	new UNCHECKED_DEALLOCATION(
		INPUT_CHANNEL_TYPE, ACCESS_INPUT_CHANNEL_TYPE);

    procedure DEALLOCATE is
	new UNCHECKED_DEALLOCATION(
		OUTPUT_CHANNEL_TYPE, ACCESS_OUTPUT_CHANNEL_TYPE);


    function OPEN
	return ACCESS_INPUT_CHANNEL_TYPE is separate;

    function CREATE(
	ETHERNET_ADDRESS    : ETHERNET_ADDRESS_TYPE)
	return ACCESS_OUTPUT_CHANNEL_TYPE is separate;


    package body GENERIC_TRANSCEIVER is separate;


    procedure CLOSE(
	INPUT_CHANNEL	    : in out ACCESS_INPUT_CHANNEL_TYPE) is
    begin
	DEALLOCATE(INPUT_CHANNEL);
    end;
	
    procedure CLOSE(
	OUTPUT_CHANNEL	: in out ACCESS_OUTPUT_CHANNEL_TYPE) is
    begin
	DEALLOCATE(OUTPUT_CHANNEL);
    end;


end;
