--***********************************************************************
--									*
--	COPYRIGHT 1992		DIGITAL EQUIPMENT CORPORATION		*
--									*
--   This software was written by Bevin Brett, of Digital Equipment	*
--   Corporation.							*
--									*
--   Digital assumes no responsibility AT ALL for the use or reliability*
--   of this software.							*
--									*
--   Redistribution and use in source and binary forms are permitted	*
--   provided that this entire heading from --*** to --*** are          *
--   duplicated in all such forms and that any documentation,		*
--   advertising materials, and other materials related to such		*
--   distribution and use acknowledge that the software was developed	*
--   by Digital Equipment Corporation. The name of Digital Equipment	*
--   Corporation may not be used to endorse or promote products derived	*
--   from this software without specific prior written permission.	*
--									*
--   THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR	*
--   IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
--   WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.*
--									*
--***********************************************************************

with MAIN_SIMULATION;

with
    LOGICAL_TO_FLOAT,
    CONTROLS, INTEGER_MIN, MAIN_SIMULATION, MOVE_UTILITIES,
    WORLD_PHYSICS, SCALAR_PHYSICS, ANGULAR_PHYSICS, TRIG, SCALE_TYPE_TRIG,
    OBJECT_VELOCITY_TO_WORLD_VELOCITY;

pragma ELABORATE(
    LOGICAL_TO_FLOAT,
    CONTROLS, INTEGER_MIN, MAIN_SIMULATION, MOVE_UTILITIES,
    WORLD_PHYSICS, SCALAR_PHYSICS, ANGULAR_PHYSICS, TRIG, SCALE_TYPE_TRIG,
    OBJECT_VELOCITY_TO_WORLD_VELOCITY);

package body AUTOMATIC_MAGIC_CARPET is
    use OBJECTS, SCALAR_PHYSICS;

    type SCORE_TYPE is digits 6;

    type VIEW_TYPE is
	record
	    SEQUENCE_NUMBER : INTEGER;
	    ORIGIN	    : OBJECT_LOCATION;
	    ORIENTATION	    : WORLD_PHYSICS.POSITION_BASIS;
	end record;

    VIEWING_SEQUENCE_NUMBER : INTEGER := 0;

    CURRENT_VIEW    : VIEW_TYPE
		    := (SEQUENCE_NUMBER => 0,
			ORIGIN	    => (0.0, 0.0, 0.0),
			ORIENTATION => WORLD_PHYSICS.USUAL_POSITION_BASIS);

    MAX_CURRENT_VIEW_BIAS : constant SCORE_TYPE
	:= SCORE_TYPE(LOGICAL_TO_FLOAT("FCTM_CURRENT_VIEW_BIAS", 1.0));

    MIN_RANGE		: METRES	:= 200.0;
    CURRENT_VIEW_BIAS	: SCORE_TYPE	:= MAX_CURRENT_VIEW_BIAS;
    CURRENT_VIEW_CLOCK	: NATURAL	:= 1;

    -- Inverse of the tan of the angle we want all the objects in front of
    -- us to be within for viewing.
    --
    TAN_ALPHA_VISIBLE
		    : constant SCALE_TYPE := SCALE_TYPE'(1.8);

    INV_TAN_ALPHA_VISIBLE
		    : constant SCALE_TYPE := 1.0/TAN_ALPHA_VISIBLE;

    TAN_ALPHA_DESIRABLE
		    : constant SCALE_TYPE := SCALE_TYPE'(0.5);

    INV_TAN_ALPHA_DESIRABLE
		    : constant SCALE_TYPE := 1.0/TAN_ALPHA_DESIRABLE;


    procedure PROJECT_TO_VIEW_RELATIVE(
	O		    : in OBJECTS.OBJECT_TYPE;
	VIEW		    : VIEW_TYPE;
	OP_J, OP_IK	    : out METRES) is

	WP  : constant WORLD_PHYSICS.POSITION
	    := WORLD_PHYSICS.POSITION'(
		METRES(OBJECTS."-"(O.LOCATION.I, VIEW.ORIGIN.I)),
		METRES(OBJECTS."-"(O.LOCATION.J, VIEW.ORIGIN.J)),
		METRES(OBJECTS."-"(O.LOCATION.K, VIEW.ORIGIN.K)));

	IK, K : METRES;

    begin
	WORLD_PHYSICS.PROJECT_SAME_ORIGIN(
	    VIEW.ORIENTATION.I,
	    VIEW.ORIENTATION.J,
	    VIEW.ORIENTATION.K,
	    WP,
	    IK, OP_J, K);

	IK := abs(IK);
	if K < 0.0 then K := K*SCALE_TYPE'(2.0); end if;
		-- view down is only half view up
	K  := abs(K);
	if K > IK then IK := K; end if;
	OP_IK := IK;
    end;


    procedure FIX_FLAT_VIEW_J_ETC(VIEW : in out VIEW_TYPE) is
	J : WORLD_PHYSICS.POSITION renames VIEW.ORIENTATION.J;
    begin
	WORLD_PHYSICS.MAKE_UNIT_VECTOR(J);
	VIEW.ORIENTATION.I := (J.J, -J.I, 0.0);
	VIEW.ORIENTATION.K := (0.0, 0.0, 1.0);
    end;


    procedure MOVE(
	O   : in out OBJECTS.OBJECT_TYPE;
	CS  : CONTROLS.CONTROL_SETTING_TYPE) is

	MC_SEQUENCE_NUMBER : constant INTEGER := O.SEQUENCE_NUMBER;

	BEST_VIEW   : VIEW_TYPE;
	BEST_SCORE  : SCORE_TYPE;
	NO_SCORE    : BOOLEAN := TRUE;


	generic
	    with procedure PROCESS_OBJECT(O : in OBJECTS.OBJECT_TYPE);
	    INCLUDING_MISSILES : in BOOLEAN := FALSE;
	procedure SCAN_TARGETS;

	procedure SCAN_TARGETS is
	    A : ACCESS_OBJECT_TYPE;
	begin
	    A := MAIN_SIMULATION.MAINTAINER_TO_ACCESS_OBJECT(ANOTHER_SIMULATOR);
	    while A /= null loop
		declare
		    O : OBJECT_TYPE renames A.all;
		begin
		    if not INCLUDING_MISSILES then
			exit when (O.CLASS not in TARGET_CLASS_SUBTYPE);
		    end if;

		    if (O.CLASS in TARGET_CLASS_SUBTYPE)
		    or (INCLUDING_MISSILES and O.CLASS=OBJECTS.MISSILE)
		    then
			if O.SEQUENCE_NUMBER /= MC_SEQUENCE_NUMBER then
			    PROCESS_OBJECT(O);
			end if;
		    end if;

		    A := O.NEXT;
		end;
	    end loop;
	end;


	procedure MOVE_0(
	    O   : in out OBJECTS.OBJECT_TYPE;
	    CS  : CONTROLS.CONTROL_SETTING_TYPE) is

	    L_I, L_J, L_K   : OBJECT_LOCATION_COORDINATE := 0.0;
	    COUNT	    : INTEGER := 0;


	    procedure TRY_VIEW(
		VIEW	    : VIEW_TYPE;
		BASE_BIAS   : SCORE_TYPE;
		DIST_BIAS   : SCORE_TYPE;
		TAN_BIAS    : SCORE_TYPE)
	    is
		SCORE	: SCORE_TYPE    := BASE_BIAS;
		IGNORE	: BOOLEAN	:= FALSE;

		procedure SCORE_ONE(O : in OBJECTS.OBJECT_TYPE) is
		    OP_J, OP_IK : METRES;
		begin
		    -- work out position relative to view
		    --
		    PROJECT_TO_VIEW_RELATIVE(
			O, VIEW, OP_J, OP_IK);

		    -- too close or not visible?
		    --
		    if OP_J <= 20.0
		    or else OP_IK*(INV_TAN_ALPHA_VISIBLE) > OP_J
		    then
			IGNORE := TRUE;
		    else
			SCORE := SCORE +
			    DIST_BIAS*SCORE_TYPE(MIN_RANGE)/SCORE_TYPE(OP_J) +
			    TAN_BIAS *SCORE_TYPE(OP_IK)/SCORE_TYPE(OP_J);
		    end if;

		end;

		procedure ADD_SCORE_OF_EACH is
		    new SCAN_TARGETS(SCORE_ONE);

	    begin
		-- Add bias for following same plane
		--
		if CURRENT_VIEW.SEQUENCE_NUMBER /= 0
		and then CURRENT_VIEW.SEQUENCE_NUMBER = VIEW.SEQUENCE_NUMBER
		then
		    SCORE := SCORE + MAX_CURRENT_VIEW_BIAS;
		end if;

		-- Work out this view's score
		--
		ADD_SCORE_OF_EACH;

		-- Decide if it is better
		--
		if IGNORE then
		    null;
		elsif NO_SCORE or else SCORE >= BEST_SCORE then

		    declare
			use WORLD_PHYSICS;
		    begin
			SCORE := SCORE + CURRENT_VIEW_BIAS*SCORE_TYPE(
			    (4.0 - SCALE_TYPE(LENGTH(
					VIEW.ORIENTATION.I -
					CURRENT_VIEW.ORIENTATION.I))
				 - SCALE_TYPE(LENGTH(
					VIEW.ORIENTATION.J -
					CURRENT_VIEW.ORIENTATION.J))
			    )/4.0);
		    end;

		    NO_SCORE    := FALSE;
		    BEST_VIEW   := VIEW;
		    BEST_SCORE  := SCORE;
		end if;
	    end;


	    procedure PICK_DISTANCE_AND_TRY_VIEW(
		VIEW		: in out VIEW_TYPE;
		MIN_DISTANCE    : METRES;
		BASE_BIAS	: SCORE_TYPE;
		DIST_BIAS	: SCORE_TYPE;
		TAN_BIAS	: SCORE_TYPE) is

		DISTANCE_BEHIND : SCALE_TYPE := SCALE_TYPE(MIN_DISTANCE);

		procedure UPDATE_DISTANCE_BEHIND_ONE(O : in OBJECTS.OBJECT_TYPE) is
		    OP_J, OP_IK : METRES;

		    D   : SCALE_TYPE;

		begin
		    -- where is this object?
		    --
		    PROJECT_TO_VIEW_RELATIVE(
			O, VIEW, OP_J, OP_IK);

		    -- how far back do we need to go?
		    --
		    D := SCALE_TYPE(OP_IK*INV_TAN_ALPHA_DESIRABLE-OP_J);

		    -- is this more?
		    --
		    if D > DISTANCE_BEHIND then
			DISTANCE_BEHIND := D;
		    end if;
		end;

		procedure UPDATE_DISTANCE_BEHIND is
		    new SCAN_TARGETS(UPDATE_DISTANCE_BEHIND_ONE);

	    begin

		UPDATE_DISTANCE_BEHIND;

		declare
		    use WORLD_PHYSICS, MOVE_UTILITIES;
		begin
		    ADD_MOVEMENT_TO_LOCATION(
			VIEW.ORIENTATION.J*(-DISTANCE_BEHIND),
			VIEW.ORIGIN);
		end;

		TRY_VIEW(VIEW, BASE_BIAS, DIST_BIAS, TAN_BIAS);
	    end;


	    procedure SUM_AND_TRY_BEHIND_ONE(O : in OBJECTS.OBJECT_TYPE) is
	    begin
		-- Try a place behind each object,
		-- provided it isn't going up or down too fast
		--
--		for FLAT_VIEW in FALSE..TRUE loop
		for FLAT_VIEW in FALSE..FALSE loop
		    if (not FLAT_VIEW)
		    or else 
			SCALE_TYPE(abs(O.ORIENTATION.J.K)) < TAN_ALPHA_DESIRABLE
		    then
			declare
			    VIEW : VIEW_TYPE;
			begin
			    VIEW.SEQUENCE_NUMBER := O.SEQUENCE_NUMBER;

			    if FLAT_VIEW then
				VIEW.ORIENTATION.J := O.ORIENTATION.J;
				VIEW.ORIENTATION.J.K := 0.0;
				FIX_FLAT_VIEW_J_ETC(VIEW);
			    else
				VIEW.ORIENTATION := O.ORIENTATION;
			    end if;

			    VIEW.ORIGIN := O.LOCATION;

			    declare
				use WORLD_PHYSICS, MOVE_UTILITIES;
			    begin
				ADD_MOVEMENT_TO_LOCATION(
				    VIEW.ORIENTATION.J*SCALE_TYPE(-MIN_RANGE),
				    VIEW.ORIGIN);
			    end;

			    VIEW.ORIGIN.K := VIEW.ORIGIN.K + 100.0;
			    TRY_VIEW(VIEW, MAX_CURRENT_VIEW_BIAS, 1.5, 0.0);

			exception
			    when others => null;
			end;
		    end if;
		end loop;

		-- include in average
		--
		COUNT := COUNT+1;
		L_I := L_I+O.LOCATION.I;
		L_J := L_J+O.LOCATION.J;
		L_K := L_K+O.LOCATION.K;

	    end;

	    procedure SUM_LOCATIONS_AND_TRY_BEHIND_EACH is
		new SCAN_TARGETS(SUM_AND_TRY_BEHIND_ONE);
	    
	begin
	    -- try the old viewpoint, unless it was behind an aeroplane
	    --
	    if CURRENT_VIEW.SEQUENCE_NUMBER = 0 then
		TRY_VIEW(CURRENT_VIEW, CURRENT_VIEW_BIAS, 1.0, 0.0);
	    end if;

	    -- if lost view, or bored, reduce bias; else prefer to stay in
	    -- same place.
	    --
	    CURRENT_VIEW_CLOCK := CURRENT_VIEW_CLOCK - 1;
	    if  NO_SCORE
	    or  CURRENT_VIEW_CLOCK = 0
	    or  CURRENT_VIEW.SEQUENCE_NUMBER /= 0
	    then
		CURRENT_VIEW_BIAS  := 0.0;
		CURRENT_VIEW_CLOCK := 50;
	    else
		CURRENT_VIEW_BIAS  := MAX_CURRENT_VIEW_BIAS;
	    end if;

	    -- Work out the average location of all the relevant objects.
	    -- Also try being behind each object.
	    --
	    SUM_LOCATIONS_AND_TRY_BEHIND_EACH;

	    -- If none of these views is any good, then try various side-on views
	    --
	    if NO_SCORE and COUNT /= 0 then
		L_I := L_I/COUNT;
		L_J := L_J/COUNT;
		L_K := L_K/COUNT;

		-- try a variety of orientations to the average location
		--
		for I in 1..4 loop  -- N, S, E, and W although not necessarily in that
		    declare	    -- order...
			VIEW : VIEW_TYPE;
		    begin
			VIEW.SEQUENCE_NUMBER := 0;
			VIEW.ORIENTATION.J := ((-1.0)**I, (-1.0)**((I+1)/2), 0.0);
			FIX_FLAT_VIEW_J_ETC(VIEW);

			VIEW.ORIGIN.I := L_I;
			VIEW.ORIGIN.J := L_J;
			VIEW.ORIGIN.K := L_K;

			PICK_DISTANCE_AND_TRY_VIEW(VIEW,
			    MIN_RANGE*SCALE_TYPE'(5.0),
			    0.0, 0.5, 0.5);
		    end;
		end loop;
	    end if;

	end;


	procedure MOVE_1(
	    O   : in out OBJECTS.OBJECT_TYPE;
	    CS  : CONTROLS.CONTROL_SETTING_TYPE) is

	    MC_VIEW : VIEW_TYPE
		:= (O.SEQUENCE_NUMBER,
		    O.LOCATION,
		    O.ORIENTATION);

	    SAME_SEEN : BOOLEAN := FALSE;

	    procedure TRY_ONE(O : in OBJECTS.OBJECT_TYPE) is
		OP_J, OP_IK : METRES;
		SCORE : SCORE_TYPE;
	    begin
		if SAME_SEEN then
		    return;
		elsif O.SEQUENCE_NUMBER = CURRENT_VIEW.SEQUENCE_NUMBER then
		    SAME_SEEN := TRUE;
		else
		    PROJECT_TO_VIEW_RELATIVE(O, MC_VIEW, OP_J, OP_IK);
		    if OP_J < 0.0 then return; end if;

		    begin
			if OP_IK = 0.0 then
			    SCORE := 0.0;
			else
			    SCORE := SCORE_TYPE(OP_IK)/SCORE_TYPE(OP_J);
			end if;
		    exception
			when others => return;
		    end;

		    if NO_SCORE then
			null;
		    elsif BEST_SCORE < SCORE then
			return;
		    end if;
		end if;

		NO_SCORE   := FALSE;
		BEST_VIEW  := (O.SEQUENCE_NUMBER,
				O.LOCATION,
				O.ORIENTATION);
		BEST_SCORE := SCORE;
		
	    end;

	    procedure TRY_EACH is new SCAN_TARGETS(TRY_ONE, TRUE);

	begin

	    -- pick the closest target and "join" it, but if we find our current
	    -- one we will use it
	    --
	    TRY_EACH;

	end;


    begin

	-- see if MC has changed
	--
	if VIEWING_SEQUENCE_NUMBER /= MC_SEQUENCE_NUMBER then
	    VIEWING_SEQUENCE_NUMBER := MC_SEQUENCE_NUMBER;
	    CURRENT_VIEW.SEQUENCE_NUMBER := 0;
	end if;

	-- find a good view
	--
	case CS.MODE is
	    when 0 => MOVE_0(O,CS);
	    when 1 => MOVE_1(O,CS);
	    when others => CLEAR;
	end case;

	-- use the one that scored best
	--
	if not NO_SCORE then

	    -- try to look as if moving in sync with the object, unless
	    -- we 'warp' to a new location
	    --
	    if BEST_VIEW.SEQUENCE_NUMBER /= 0
	    and BEST_VIEW.SEQUENCE_NUMBER /= CURRENT_VIEW.SEQUENCE_NUMBER
	    then
		O.OLD_LOCATION  := BEST_VIEW.ORIGIN;
	    end if;

	    O.LOCATION	    := BEST_VIEW.ORIGIN;
	    O.ORIENTATION   := BEST_VIEW.ORIENTATION;

	    CURRENT_VIEW    := BEST_VIEW;
	end if;
    end;


    procedure CLEAR is
    begin
	CURRENT_VIEW.SEQUENCE_NUMBER := 0;
    end;

end;
