--***********************************************************************
--									*
--	COPYRIGHT 1992		DIGITAL EQUIPMENT CORPORATION		*
--									*
--   This software was written by Bevin Brett, of Digital Equipment	*
--   Corporation.							*
--									*
--   Digital assumes no responsibility AT ALL for the use or reliability*
--   of this software.							*
--									*
--   Redistribution and use in source and binary forms are permitted	*
--   provided that this entire heading from --*** to --*** are          *
--   duplicated in all such forms and that any documentation,		*
--   advertising materials, and other materials related to such		*
--   distribution and use acknowledge that the software was developed	*
--   by Digital Equipment Corporation. The name of Digital Equipment	*
--   Corporation may not be used to endorse or promote products derived	*
--   from this software without specific prior written permission.	*
--									*
--   THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR	*
--   IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
--   WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.*
--									*
--***********************************************************************


with UNCHECKED_DEALLOCATION, INFO_MANAGER;

pragma ELABORATE(INFO_MANAGER);

package body APPEARANCE_MANAGER is
    use PREALLOCATED_APPEARANCES_PKG;

    MAX_APPEARANCES : constant APPEARANCE_INFO_TYPE := 255;

    APPEARANCE_INFO_TO_ACCESS_APPEARANCE :
	array(1..MAX_APPEARANCES) of ACCESS_APPEARANCE_TYPE;

    HAVE_FINISHED_CREATING_SUBPICTURES : exception;
    STILL_CREATING_SUBPICTURES : exception;

    type APPEARANCE_TYPE is
	record
	    APPEARANCE_INFO : APPEARANCE_INFO_TYPE;
	    FINISHED_CREATING_SUBPICTURES : BOOLEAN := FALSE;
	    FIRST_SUBPICTURE,
	    LAST_SUBPICTURE : ACCESS_PICTURE_TYPE   := null;
	    MAX_RADIUS	    : METRES		    := 0.0;
	end record;

    type VECTOR_OF_APPEARANCE_INFO_TYPE is
	array(POSITIVE range <>) of APPEARANCE_INFO_TYPE;

    type ACCESS_VECTOR_OF_APPEARANCE_INFO_TYPE is
	access VECTOR_OF_APPEARANCE_INFO_TYPE;

    PREALLOCATED_APPEARANCES
	: constant array(PREALLOCATED_APPEARANCES_TYPE) of
	    ACCESS_VECTOR_OF_APPEARANCE_INFO_TYPE
	:= (BULLETS		=> new VECTOR_OF_APPEARANCE_INFO_TYPE(1..2),
	    MISSILES		=> new VECTOR_OF_APPEARANCE_INFO_TYPE(1..3),
	    AEROPLANES		=> new VECTOR_OF_APPEARANCE_INFO_TYPE(1..20),
	    MAGIC_CARPETS	=> new VECTOR_OF_APPEARANCE_INFO_TYPE(1..1),
	    NOT_PREALLOCATED	=> new VECTOR_OF_APPEARANCE_INFO_TYPE(1..0));

    PREALLOCATED_APPEARANCES_ALLOCATED
	: array(PREALLOCATED_APPEARANCES_TYPE) of NATURAL
	:= (others => 0);


    function TO_ACCESS_APPEARANCE_TYPE(
	APPEARANCE_INFO : APPEARANCE_INFO_TYPE
	) return ACCESS_APPEARANCE_TYPE is
    begin
	return APPEARANCE_INFO_TO_ACCESS_APPEARANCE(APPEARANCE_INFO);
    end;

    function CREATE(
	PREALLOCATED_APPEARANCE : PREALLOCATED_APPEARANCES_PKG.PREALLOCATED_APPEARANCES_TYPE
				:= PREALLOCATED_APPEARANCES_PKG.NOT_PREALLOCATED
	) return APPEARANCE_INFO_TYPE
    is
	APPEARANCES : VECTOR_OF_APPEARANCE_INFO_TYPE renames
			PREALLOCATED_APPEARANCES(PREALLOCATED_APPEARANCE).all;
	ALLOCATED   : NATURAL renames
			PREALLOCATED_APPEARANCES_ALLOCATED(PREALLOCATED_APPEARANCE);
	A	    : constant ACCESS_APPEARANCE_TYPE := new APPEARANCE_TYPE;
    begin
	if ALLOCATED < APPEARANCES'last then
	    ALLOCATED := ALLOCATED+1;
	    A.APPEARANCE_INFO := APPEARANCES(ALLOCATED);
	else
	    A.APPEARANCE_INFO := INFO_MANAGER.ASSIGN;
	end if;
	APPEARANCE_INFO_TO_ACCESS_APPEARANCE(A.APPEARANCE_INFO) := A;
	return A.APPEARANCE_INFO;
    end;

    function CREATE return ACCESS_APPEARANCE_TYPE is
    begin
	return TO_ACCESS_APPEARANCE_TYPE(CREATE);
    end;


    procedure ADD_POINT(APPEARANCE  : ACCESS_APPEARANCE_TYPE;
			POINT	    : POSITION) is
    begin
	if APPEARANCE.FINISHED_CREATING_SUBPICTURES then
	    raise HAVE_FINISHED_CREATING_SUBPICTURES;
	end if;

	-- Make sure room for this point, if neccessary by starting a new
	-- subpicture where the current one leaves off.
	--
	if APPEARANCE.LAST_SUBPICTURE = null then
	    ADD_SUBPICTURE(APPEARANCE);
	elsif APPEARANCE.LAST_SUBPICTURE.COUNT = NATURAL_POINTS_RANGE'last then
	    declare
		PREV_L : PICTURE_TYPE renames APPEARANCE.LAST_SUBPICTURE.all;
	    begin
		ADD_SUBPICTURE(APPEARANCE,
		    PREV_L.CLOSEST_VISIBLE,
		    PREV_L.FURTHEREST_VISIBLE);
		ADD_POINT(APPEARANCE, PREV_L.POINTS(NATURAL_POINTS_RANGE'last));
	    end;
	end if;

	-- Add this point
	--
	declare
	    L : PICTURE_TYPE renames APPEARANCE.LAST_SUBPICTURE.all;
	    RADIUS : constant METRES := LENGTH(POINT);
	begin
	    L.COUNT := L.COUNT+1;
	    L.POINTS(L.COUNT) := POINT;
	    if RADIUS > APPEARANCE.MAX_RADIUS then
		APPEARANCE.MAX_RADIUS := RADIUS;
	    end if;
	end;

    end;

    procedure ADD_POINTS(APPEARANCE : ACCESS_APPEARANCE_TYPE;
			POINTS	    : VECTOR_OF_POSITION) is
    begin
	for I in POINTS'range loop
	    ADD_POINT(APPEARANCE, POINTS(I));
	end loop;
    end;

    procedure ADD_SUBPICTURE(APPEARANCE : ACCESS_APPEARANCE_TYPE;
	CLOSEST_VISIBLE     : METRES := 0.0;
	FURTHEREST_VISIBLE  : METRES := METRES'last) is

	F : ACCESS_PICTURE_TYPE renames APPEARANCE.FIRST_SUBPICTURE;
	L : ACCESS_PICTURE_TYPE renames APPEARANCE.LAST_SUBPICTURE;
	P : constant ACCESS_PICTURE_TYPE := new PICTURE_TYPE;
    begin
	if APPEARANCE.FINISHED_CREATING_SUBPICTURES then
	    raise HAVE_FINISHED_CREATING_SUBPICTURES;
	end if;

	P.CLOSEST_VISIBLE := CLOSEST_VISIBLE;
	P.FURTHEREST_VISIBLE :=	FURTHEREST_VISIBLE;

	if F = null then
	    F := P;
	else
	    L.NEXT_SUBPICTURE := P;
	end if;

	L := P;
    end;

    procedure FINISH_CREATING_SUBPICTURES
			    (APPEARANCE : ACCESS_APPEARANCE_TYPE) is
    begin
	APPEARANCE.FINISHED_CREATING_SUBPICTURES := TRUE;
    end;

    procedure REOPEN_CREATING_SUBPICTURES
			    (APPEARANCE : ACCESS_APPEARANCE_TYPE) is
    begin
	APPEARANCE.FINISHED_CREATING_SUBPICTURES := FALSE;
    end;

    procedure DELETE_FIRST_SUBPICTURES(APPEARANCE : ACCESS_APPEARANCE_TYPE;
	N : NATURAL := 1) is

	COUNT	: NATURAL := 0;
	P	: ACCESS_PICTURE_TYPE;

	procedure DEALLOCATE is
	    new UNCHECKED_DEALLOCATION(PICTURE_TYPE, ACCESS_PICTURE_TYPE);

    begin
	loop
	    P := APPEARANCE.FIRST_SUBPICTURE;
	    exit when (P = null) or (COUNT >= N);
	    APPEARANCE.FIRST_SUBPICTURE := P.NEXT_SUBPICTURE;
	    DEALLOCATE(P);
	    COUNT := COUNT + 1;
	end loop;
    end;

    procedure DELETE(APPEARANCE : in out ACCESS_APPEARANCE_TYPE) is

	procedure DEALLOCATE is
	    new UNCHECKED_DEALLOCATION(APPEARANCE_TYPE, ACCESS_APPEARANCE_TYPE);

    begin
	DELETE_FIRST_SUBPICTURES(APPEARANCE, INTEGER'last);
	APPEARANCE_INFO_TO_ACCESS_APPEARANCE(APPEARANCE.APPEARANCE_INFO):= null;
	DEASSIGN(APPEARANCE.APPEARANCE_INFO);
	DEALLOCATE(APPEARANCE);
    end;


    function APPEARANCE_TO_MAX_RADIUS(APPEARANCE : ACCESS_APPEARANCE_TYPE)
	return METRES is
    begin
	return APPEARANCE.MAX_RADIUS;
    end;

    -- A way of getting the PICTURE, given an APPEARANCE.  This picture should
    -- be treated as a constant by the caller, and also can only be done after
    -- FINISH_CREATING_SUBPICTURES has been called.
    --
    function APPEARANCE_TO_PICTURE(APPEARANCE : ACCESS_APPEARANCE_TYPE)
	return ACCESS_PICTURE_TYPE is
    begin
	if not APPEARANCE.FINISHED_CREATING_SUBPICTURES then
	    raise STILL_CREATING_SUBPICTURES;
	end if;

	return APPEARANCE.FIRST_SUBPICTURE;
    end;

begin
    for I in PREALLOCATED_APPEARANCES'range loop
	for J in PREALLOCATED_APPEARANCES(I)'range loop
	    PREALLOCATED_APPEARANCES(I)(J) := INFO_MANAGER.ASSIGN;
	end loop;
    end loop;
end;
