--***********************************************************************
--									*
--	COPYRIGHT 1992		DIGITAL EQUIPMENT CORPORATION		*
--									*
--   This software was written by Bevin Brett, of Digital Equipment	*
--   Corporation.							*
--									*
--   Digital assumes no responsibility AT ALL for the use or reliability*
--   of this software.							*
--									*
--   Redistribution and use in source and binary forms are permitted	*
--   provided that this entire heading from --*** to --*** are          *
--   duplicated in all such forms and that any documentation,		*
--   advertising materials, and other materials related to such		*
--   distribution and use acknowledge that the software was developed	*
--   by Digital Equipment Corporation. The name of Digital Equipment	*
--   Corporation may not be used to endorse or promote products derived	*
--   from this software without specific prior written permission.	*
--									*
--   THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR	*
--   IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
--   WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.*
--									*
--***********************************************************************


with TEXT_IO;

package body AEROPLANE_ENGINES is

    type ENGINE_DATA_TYPE is
	record
	    THRUST			: NEWTONS;
	    THRUST_AFTERBURNER		: NEWTONS;
	    CONSUMPTION 		: KILOGRAMS_PER_NEWTON_SECOND;
	    CONSUMPTION_AFTERBURNER	: KILOGRAMS_PER_NEWTON_SECOND;
	end record;

    type ENGINE_TYPE is
	(J79, F404, F100, TF30, RB199, ADOUR, M53_5,
	 TUMANSKY_R11F, TUMANSKY_R25, TUMANSKY_R33, TUMANSKY_RD_F,

	 -- added by me
	 --
	 WW1_A1, WW1_E1,
	 WW2_A1, WW2_E1,
	 GRIFFON);

    type VECTOR_OF_ENGINE_DATA_TYPE is
	array(ENGINE_TYPE) of ENGINE_DATA_TYPE;


    -- The following table is from	MODERN COMBAT AIRCRAFT DESIGN,
    --					ISBN 0-087021-426-8
    --					Naval Institute Press
    --
    R : constant := 36000.0;		-- KILOGRAMS_PER_daNEWTONS_PER_HOUR to
					-- KILOGRAMS_PER_SECOND_PER_NEWTON

    ENGINE_DATA : constant VECTOR_OF_ENGINE_DATA_TYPE := VECTOR_OF_ENGINE_DATA_TYPE'(
J79	=>(52550.0,  79700.0, KILOGRAMS_PER_NEWTON_SECOND(0.86/R), KILOGRAMS_PER_NEWTON_SECOND(2.00/R)),
F404	=>(45500.0,  71000.0, KILOGRAMS_PER_NEWTON_SECOND(0.70/R), KILOGRAMS_PER_NEWTON_SECOND(1.50/R)),  -- consumption unavailable
F100	=>(64000.0, 107000.0, KILOGRAMS_PER_NEWTON_SECOND(0.68/R), KILOGRAMS_PER_NEWTON_SECOND(2.55/R)),
TF30	=>(58000.0,  92000.0, KILOGRAMS_PER_NEWTON_SECOND(0.66/R), KILOGRAMS_PER_NEWTON_SECOND(2.78/R)),
RB199	=>(38000.0,  65000.0, KILOGRAMS_PER_NEWTON_SECOND(0.62/R), KILOGRAMS_PER_NEWTON_SECOND(2.25/R)),
ADOUR	=>(23700.0,  35800.0, KILOGRAMS_PER_NEWTON_SECOND(0.66/R), KILOGRAMS_PER_NEWTON_SECOND(2.68/R)),  -- consumption unavailable
M53_5	=>(54400.0,  88000.0, KILOGRAMS_PER_NEWTON_SECOND(0.89/R), KILOGRAMS_PER_NEWTON_SECOND(2.14/R)),

    -- From "MiGS" Modern Fighting Aircraft series
    --
TUMANSKY_R11F
	=>(42200.0,  56200.0, KILOGRAMS_PER_NEWTON_SECOND(0.86/R), KILOGRAMS_PER_NEWTON_SECOND(2.00/R)),  -- consumption unavailable
TUMANSKY_R25
	=>(57800.0,  88200.0, KILOGRAMS_PER_NEWTON_SECOND(0.86/R), KILOGRAMS_PER_NEWTON_SECOND(2.00/R)),  -- consumption unavailable
TUMANSKY_R33
	=>(50000.0,  81000.0, KILOGRAMS_PER_NEWTON_SECOND(0.68/R), KILOGRAMS_PER_NEWTON_SECOND(2.55/R)),  -- consumption unavailable
TUMANSKY_RD_F
	=>(98000.0, 142500.0, KILOGRAMS_PER_NEWTON_SECOND(0.68/R), KILOGRAMS_PER_NEWTON_SECOND(2.55/R)),  -- consumption unavailable

    -- added by me
    --
WW1_A1	=>( 2190.0,   0000.0, 0.0, 0.0),
WW1_E1	=>( 4000.0,   0000.0, 0.0, 0.0),
WW2_A1	=>(15000.0,   0000.0, 0.0, 0.0),
WW2_E1	=>(20000.0,   0000.0, 0.0, 0.0),
GRIFFON =>(20000.0,   0000.0, 0.0, 0.0)
);

    -- Max jet engine power output is a function of altitude and mach
    -- number that look approx like this...
    --
    --	    ____	    ________
    --Thrust	----____----	    \
    --				     \
    --		    Mach
    --
    -- Unfortunately these curves vary dramatically according to engine etc
    -- and the data is very hard to get.  I have only gotten data for the J79
    -- engines of an F-4, and then only fragmentary.
    --
    -- The curves will be modelled by three lines,
    --
    --	A. from Mach 0 to the low
    --	B. from the lo to the right high
    --	C. from the right high to zero
    --
    -- The data I have gives these values at sealevel and 35_000 ft.
    --
    -- Turbocharged piston engines driving propellers can be described by
    -- similar curves, but these tend (a) be more constant up to the turbocharge
    -- altitude limits, and thrust is based on propeller speed v. air speed
    -- considerations so look more like
    --
    --Thrust -----------------------
    --				    \
    --				    |
    --
    -- These are going to require tuning based on the turbocharge altitude
    -- because this can vary between engines.
    --
    type POINT is
	record
	    MACH	: SCALE_TYPE;
	    THRUST	: SCALE_TYPE;
	end record;

    type POINTS is
	array(1..4) of POINT;

    type KNOWN_ALTITUDES is
	(SEALEVEL, THIRTY_FIVE_THOUSAND_FEET, OUTER_SPACE);

    KNOWN_ALTITUDE_TO_DENSITY : constant array(KNOWN_ALTITUDES) of SCALE_TYPE
	:= (SEALEVEL			=> SCALE_TYPE(TO_DENSITY(METRES'(0.0))),
	    THIRTY_FIVE_THOUSAND_FEET	=> SCALE_TYPE(TO_DENSITY(METRES(35_000.0*0.3048))),
	    OUTER_SPACE 		=> 0.0);

    type DATA_TYPE is array(KNOWN_ALTITUDES) of POINTS;

    type ENGINE_TYPE_AND_MODE_TO_DATA_TYPE is
	array(ENGINE_TYPE, ENGINE_MODE_TYPE) of DATA_TYPE;

    ENGINE_TYPE_AND_MODE_TO_DATA : constant ENGINE_TYPE_AND_MODE_TO_DATA_TYPE :=
	ENGINE_TYPE_AND_MODE_TO_DATA_TYPE'(
	TUMANSKY_R11F =>			-- based on getting the MiG-21
	    (MILITARY =>			-- to do a little better than
		(SEALEVEL =>			-- the F_4
		    (( 0.00, 1.00),
		     ( 0.50, 0.75),
		     ( 0.90, 1.00),
		     ( 1.40, 0.00)
		    ),
		THIRTY_FIVE_THOUSAND_FEET =>
		    (( 0.00, 0.3),
		     ( 0.50, 0.3),
		     ( 0.96, 0.3),
		     ( 1.40, 0.00)
		    ),
		OUTER_SPACE =>
		    (( 0.00, 0.00),
		     ( 0.50, 0.00),
		     ( 0.96, 0.00),
		     ( 1.40, 0.00)
		    )),
	     MAXIMUM =>
		(SEALEVEL =>
		    (( 0.00, 1.00),
		     ( 0.50, 1.30),
		     ( 1.00, 1.30),
		     ( 1.40, 0.00)
		    ),
		THIRTY_FIVE_THOUSAND_FEET =>
		    (( 0.00, 0.5),
		     ( 1.50, 0.67),
		     ( 2.0,  0.84),
		     ( 2.50, 0.00)
		    ),
		OUTER_SPACE =>
		    (( 0.00, 0.00),
		     ( 1.50, 0.00),
		     ( 2.00, 0.00),
		     ( 2.50, 0.00)
		    ))
	    ),
	WW1_A1|WW1_E1 =>
	    (MILITARY|MAXIMUM =>
		(SEALEVEL =>
		    (( 0.00, 1.00),
		     ( 0.50, 1.00),
		     ( 0.90, 0.00),
		     ( 1.40, 0.00)
		    ),
		THIRTY_FIVE_THOUSAND_FEET =>
		    (( 0.00, 0.30),
		     ( 0.50, 0.30),
		     ( 0.90, 0.00),
		     ( 1.40, 0.00)
		    ),
		OUTER_SPACE =>
		    (( 0.00, 0.00),
		     ( 0.50, 0.00),
		     ( 0.90, 0.00),
		     ( 1.40, 0.00)
		    ))
	    ),

	WW2_A1|WW2_E1|GRIFFON =>
	    (MILITARY|MAXIMUM =>
		(SEALEVEL =>
		    (( 0.00, 1.00),
		     ( 0.50, 1.00),
		     ( 0.90, 0.00),
		     ( 1.40, 0.00)
		    ),
		THIRTY_FIVE_THOUSAND_FEET =>
		    (( 0.00, 1.00),
		     ( 0.50, 1.00),
		     ( 0.90, 0.00),
		     ( 1.40, 0.00)
		    ),
		OUTER_SPACE =>
		    (( 0.00, 0.00),
		     ( 0.50, 0.00),
		     ( 0.90, 0.00),
		     ( 1.40, 0.00)
		    ))
	    ),

--HACK	J79|
	others => -- HACK
	    (MILITARY =>
		(SEALEVEL =>
		    (( 0.00, 1.00),
		     ( 0.50, 0.80),
		     ( 0.90, 0.80),
		     ( 1.40, 0.00)
		    ),
		THIRTY_FIVE_THOUSAND_FEET =>
		    (( 0.00, 0.3),
		     ( 0.50, 0.3),
		     ( 0.96, 0.3),
		     ( 1.40, 0.00)
		    ),
		OUTER_SPACE =>
		    (( 0.00, 0.00),
		     ( 0.50, 0.00),
		     ( 0.96, 0.00),
		     ( 1.40, 0.00)
		    )),
	     MAXIMUM =>
		(SEALEVEL =>
		    (( 0.00, 1.00),
		     ( 0.50, 1.00),
		     ( 1.00, 1.25),
		     ( 1.40, 0.00)
		    ),
		THIRTY_FIVE_THOUSAND_FEET =>
		    (( 0.00, 0.52),
		     ( 1.00, 0.52),
		     ( 2.10, 0.90),
		     ( 2.50, 0.00)
		    ),
		OUTER_SPACE =>
		    (( 0.00, 0.00),
		     ( 1.50, 0.00),
		     ( 2.00, 0.00),
		     ( 2.50, 0.00)
		    ))
	    ));


    function ENGINE_TYPE_VALUE(
	ENGINE_NAME : STRING) return ENGINE_HANDLE_TYPE is
    begin
	return ENGINE_TYPE'pos(ENGINE_TYPE'value(ENGINE_NAME));
    exception
	when others =>
	    TEXT_IO.PUT_LINE("AEROPLANE_ENGINES.ENGINE_TYPE_VALUE, no '" &
		ENGINE_NAME & "' is defined");
	    return 0;
    end;

    function ENGINE_TYPE_IMAGE(ENGINE : ENGINE_HANDLE_TYPE) return STRING is
    begin
	return ENGINE_TYPE'image(ENGINE_TYPE'val(ENGINE));
    exception
	when others =>
	    TEXT_IO.PUT_LINE("AEROPLANE_ENGINES.ENGINE_TYPE_IMAGE, bad handle");
	    return ENGINE_TYPE'image(ENGINE_TYPE'first);
    end;


    function EXTRAPOLATE(X1, Y1, X2, Y2, X : SCALE_TYPE) return SCALE_TYPE is
    begin
	if X1 = X2 then
	    return Y1;
	else
	    return (X-X1)*((Y2-Y1)/(X2-X1))+Y1;
	end if;
    end;


    function THRUST_FACTOR(
	ET	    : ENGINE_TYPE;
	ENGINE_MODE : ENGINE_MODE_TYPE;
	MACH	    : MACH_TYPE;
	DENSITY     : KILOGRAMS_PER_METRE_CUBED)
	return SCALE_TYPE is

	M : constant SCALE_TYPE := SCALE_TYPE(MACH);
	D : constant SCALE_TYPE := SCALE_TYPE(DENSITY);

	P : POINTS;

	DATA : DATA_TYPE renames ENGINE_TYPE_AND_MODE_TO_DATA(ET, ENGINE_MODE);

	LO, HI : KNOWN_ALTITUDES;

    begin
	-- Find the bracketing range
	--
	if D > KNOWN_ALTITUDE_TO_DENSITY(THIRTY_FIVE_THOUSAND_FEET) then
	    LO := SEALEVEL;
	    HI := THIRTY_FIVE_THOUSAND_FEET;
	else
	    LO := THIRTY_FIVE_THOUSAND_FEET;
	    HI := OUTER_SPACE;
	end if;

	-- Build the points for this particular altitude
	--
	for I in P'range loop
	    P(I).MACH :=
		    EXTRAPOLATE(
			X1 => KNOWN_ALTITUDE_TO_DENSITY(LO),
			    Y1 => DATA(LO)(I).MACH,
			X2 => KNOWN_ALTITUDE_TO_DENSITY(HI),
			    Y2 => DATA(HI)(I).MACH,
			X  => D);
	    P(I).THRUST :=
		    EXTRAPOLATE(
			X1 => KNOWN_ALTITUDE_TO_DENSITY(LO),
			    Y1 => DATA(LO)(I).THRUST,
			X2 => KNOWN_ALTITUDE_TO_DENSITY(HI),
			    Y2 => DATA(HI)(I).THRUST,
			X  => D);
	end loop;

	-- Find the pair of points that bracket this mach
	--
	for I in P'first+1..P'last loop
	    if P(I).MACH > M then
		return
		    EXTRAPOLATE(
			X1 => P(I-1).MACH, Y1 => P(I-1).THRUST,
			X2 => P(I  ).MACH, Y2 => P(I  ).THRUST,
			X  => M);
	    end if;
	end loop;

	-- Engine must be fried to a crisp
	--
	return 0.0;
    end;


    procedure GET_ENGINE_INFO(
	ENGINE	    : ENGINE_HANDLE_TYPE;

	THROTTLE    : SCALE_TYPE;
	ENGINE_MODE : ENGINE_MODE_TYPE;

	MACH	    : MACH_TYPE;
	DENSITY     : KILOGRAMS_PER_METRE_CUBED;

	THRUST	    : out NEWTONS;
	CONSUMPTION : out KILOGRAMS_PER_SECOND

	) is

	ET  : constant ENGINE_TYPE := ENGINE_TYPE'val(ENGINE);
	E   : ENGINE_DATA_TYPE renames ENGINE_DATA(ET);

	SEALEVEL_THRUST 	: NEWTONS;
	SEALEVEL_CONSUMPTION	: KILOGRAMS_PER_NEWTON_SECOND;
	MODIFIER		: SCALE_TYPE;

    begin

	-- Because afterburners are totally differently affected by
	-- altitude and airspeed, and the proportion of the thrust
	-- generated in each differs with altitude and airspeed, they
	-- must be considered as two separate problems!
	--
	if ENGINE_MODE = MAXIMUM then
	    -- sealevel, static; assume full throttle
	    --
	    SEALEVEL_THRUST	    := E.THRUST_AFTERBURNER;
	    SEALEVEL_CONSUMPTION    := E.CONSUMPTION_AFTERBURNER;

	else
	    -- sealevel, static
	    --
	    SEALEVEL_THRUST	    := E.THRUST*THROTTLE;
	    SEALEVEL_CONSUMPTION    := E.CONSUMPTION;

	end if;

	-- modifier
	--
	MODIFIER := THRUST_FACTOR(ET, ENGINE_MODE, MACH, DENSITY);

	-- answers (hack, use SEALEVEL_THRUST since THRUST is mode out)
	--
	SEALEVEL_THRUST := SEALEVEL_THRUST*MODIFIER;
	THRUST	    := SEALEVEL_THRUST;
	CONSUMPTION := SEALEVEL_CONSUMPTION*SEALEVEL_THRUST;

    end;

end;
