--***********************************************************************
--									*
--	COPYRIGHT 1992		DIGITAL EQUIPMENT CORPORATION		*
--									*
--   This software was written by Bevin Brett, of Digital Equipment	*
--   Corporation.							*
--									*
--   Digital assumes no responsibility AT ALL for the use or reliability*
--   of this software.							*
--									*
--   Redistribution and use in source and binary forms are permitted	*
--   provided that this entire heading from --*** to --*** are          *
--   duplicated in all such forms and that any documentation,		*
--   advertising materials, and other materials related to such		*
--   distribution and use acknowledge that the software was developed	*
--   by Digital Equipment Corporation. The name of Digital Equipment	*
--   Corporation may not be used to endorse or promote products derived	*
--   from this software without specific prior written permission.	*
--									*
--   THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR	*
--   IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
--   WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.*
--									*
--***********************************************************************


with SCORE, CONTROLS, MOVE_UTILITIES, AOA_INDICATOR;

with SCALAR_PHYSICS, ANGULAR_PHYSICS,
     SCALE_TYPE_MATH_LIB, TRIG, SCALE_TYPE_TRIG;
use  SCALAR_PHYSICS, ANGULAR_PHYSICS,
     SCALE_TYPE_MATH_LIB, TRIG, SCALE_TYPE_TRIG;

with WORLD_PHYSICS, OBJECT_PHYSICS, PHYSICAL_UNITS, STANDARD_ATMOSPHERE;
use  WORLD_PHYSICS, OBJECT_PHYSICS;

with TEXT_IO, FLOAT_TEXT_IO;
use  TEXT_IO, FLOAT_TEXT_IO;

separate(AEROPLANE)
    procedure MOVE( O	: in out OBJECTS.OBJECT_TYPE;
		    ELAPSED_SINCE_LAST_MOVE	: SCALAR_PHYSICS.SECONDS;
		    CS	: in out CONTROLS.CONTROL_SETTING_TYPE) is

    begin
	-- Make sure have required info
	--
	if O.SUBCLASS_INFO = 0 then CREATE_SUBCLASS_INFO(O); end if;

	declare
	    AEROPLANE_STATUS 	: AEROPLANE_STATUS_TYPE
		renames SUBCLASS_INFO_PKG.SUBCLASS_INFO_TO_ACCESS_STATUS(
			    O.SUBCLASS_INFO).all;

	    A_T 	: AEROPLANE_TYPE renames AEROPLANE_STATUS.AEROPLANE.all;

	    ALTITUDE	: constant METRES := METRES(O.LOCATION.K);

	    TIME_DELTA  : SECONDS
			:=  ELAPSED_SINCE_LAST_MOVE +
			    AEROPLANE_STATUS.TIME_DELTA_CARRIED_FORWARD;

	    MAIN_WING_LIFT_DRAG_INPUTS,
	    BODY_LIFT_DRAG_INPUTS : LIFT_DRAG_CURVE_MANAGER.LIFT_DRAG_INPUTS;

	    INITIAL_SPEED : constant METRES_PER_SECOND
		:= LENGTH(O.VELOCITY);

	    CAS : constant METRES_PER_SECOND
		:= INITIAL_SPEED*
		    STANDARD_ATMOSPHERE.TO_SQRT_DENSITY_RATIO(ALTITUDE);

	begin

	    -- Changes to controls based on CAS
	    --	    Auto-flaps, primitive so far.
	    --		F-5E and others will require better.
	    --
	    if CAS > A_T.CAS_FLAP_LIMIT then
		CS.FLAP := 0;
	    end if;

	    -- Fill in various values
	    --
	    AEROPLANE_STATUS.FLAP_CONTROL
		:=  SCALE_TYPE(CS.FLAP)/
		    SCALE_TYPE(CONTROLS.FLAP_TYPE'last);

	    AEROPLANE_STATUS.PITCH_CONTROL
		:=  SKEW(SCALE_TYPE(CS.STICK_FORWARD_BACK)/
			SCALE_TYPE(CONTROLS.STICK_FORWARD_BACK_TYPE'last));

	    AEROPLANE_STATUS.ROLL_CONTROL
		:=  SKEW(SCALE_TYPE(CS.STICK_LEFT_RIGHT)/
			SCALE_TYPE(CONTROLS.STICK_LEFT_RIGHT_TYPE'last));

	    AEROPLANE_STATUS.THRUST_CONTROL
		:=  SCALE_TYPE(CS.THROTTLE)/
		    SCALE_TYPE(CONTROLS.THROTTLE_TYPE'last);

	    AEROPLANE_STATUS.YAW_CONTROL 
		:=  SKEW(SCALE_TYPE(CS.RUDDER)/
			 SCALE_TYPE(CONTROLS.RUDDER_TYPE'last));

	    -- Limits on controls effectiveness based on CAS
	    --
	    declare

		procedure LIMIT(L : METRES_PER_SECOND; C : in out SCALE_TYPE) is
		begin
		    if CAS > L then
			--
			-- AEROLON/PITCH limits are the result of pilot
			-- strength and wing strength limits.  Since the force
			-- on the surfaces is proportional to CAS**2 and RC,
			-- the following is used to limit C...
			--
			C := C*((SCALE_TYPE(L)/SCALE_TYPE(CAS))**2);
		    end if;
		end;
		
	    begin

		-- Limits on pitch and roll
		--
		LIMIT(A_T.CAS_AEROLON_LIMIT, AEROPLANE_STATUS.ROLL_CONTROL);
		LIMIT(A_T.CAS_PITCH_LIMIT,   AEROPLANE_STATUS.PITCH_CONTROL);

		-- Below rotate speed, the tail controls are ineffective.
		--
		if CAS > A_T.ROTATE_SPEED then
		    AEROPLANE_STATUS.TAIL_EFFECTIVENESS := 1.0;
		else
		    AEROPLANE_STATUS.TAIL_EFFECTIVENESS :=
			SCALE_TYPE(CAS/A_T.ROTATE_SPEED)**4;
		end if;

	    end;

	    -- Set the object relative velocity
	    --
	    SET_O_VELOCITY(O, AEROPLANE_STATUS);

	    -- Air density
	    --
	    AEROPLANE_STATUS.DENSITY
		:= STANDARD_ATMOSPHERE.TO_DENSITY(ALTITUDE);

	    -- Mach
	    --
	    AEROPLANE_STATUS.MACH 
		:=  STANDARD_ATMOSPHERE.TO_MACH(INITIAL_SPEED, ALTITUDE);

	    -- Engine thrust
	    --
	    declare
		FUEL_REMAINING : KILOGRAMS renames AEROPLANE_STATUS.FUEL_MASS;
		BURNT : KILOGRAMS;
	    begin

		AEROPLANE_STATUS.THRUST := 0.0;
		if FUEL_REMAINING > 0.0 then

		    ENGINE_THRUST_V_MACH_AND_ALT(
			AEROPLANE_STATUS,
			ALTITUDE,
			TIME_DELTA,
			CS.AFTERBURNER_ON,

			AEROPLANE_STATUS.THRUST,
			BURNT);

		    -- Multi-engine.
		    --
		    AEROPLANE_STATUS.THRUST :=
			AEROPLANE_STATUS.THRUST * A_T.ENGINE_COUNT;
		    BURNT := BURNT * A_T.ENGINE_COUNT;

		    -- Don't burn more than there is...
		    --
		    if BURNT < FUEL_REMAINING then
			FUEL_REMAINING := FUEL_REMAINING - BURNT;
		    else
			FUEL_REMAINING := 0.0;
		    end if;

		end if;
	    end;


	    -- Set up the LIFT_DRAG_INPUTS that are independent of the roll rate
	    --
	    MAIN_WING_LIFT_DRAG_INPUTS.
		LIFT_DRAG_CURVE := A_T.WING_LIFT_DRAG_CURVE;
--	    MAIN_WING_LIFT_DRAG_INPUTS.
--		ANGLE_OF_ATTACK := {depends on roll rate};
--	    MAIN_WING_LIFT_DRAG_INPUTS.
--		CL_ALPHA	:= {depends on which wing};
	    MAIN_WING_LIFT_DRAG_INPUTS.
		EXTRA_ZERO_DRAG := A_T.EXTRA_ZERO_DRAG;
	    MAIN_WING_LIFT_DRAG_INPUTS.
		MACH		:= AEROPLANE_STATUS.MACH;
--	    MAIN_WING_LIFT_DRAG_INPUTS.
--		AEROLONS	:= {depends on which wing};
	    MAIN_WING_LIFT_DRAG_INPUTS.
		HIGH_LIFT	:= AEROPLANE_STATUS.FLAP_CONTROL*A_T.FLAPS_AND_SLATS_EFFECT;

--####	    BODY_LIFT_DRAG_INPUTS



	    COMPUTE_CL_ALPHAS_AND_CD_PER_CL2(
		-- inputs
		--
		A_T.WING_ASPECT_RATIO,
		A_T.SIN_SWEEP_ANGLE,
		A_T.COS_SWEEP_ANGLE,
		A_T.PARTIAL_LAMBDA_50,
		AEROPLANE_STATUS.MACH,
		AEROPLANE_STATUS.OBJECT_RELATIVE_VELOCITY.I,
		AEROPLANE_STATUS.OBJECT_RELATIVE_VELOCITY.J,

		-- outputs
		--
		AEROPLANE_STATUS.CL_ALPHA_LEFT,
		AEROPLANE_STATUS.CL_ALPHA_RIGHT,
		MAIN_WING_LIFT_DRAG_INPUTS.CL_MAX,
		MAIN_WING_LIFT_DRAG_INPUTS.CD_PER_CL2
		);



	    -- Step thru time
	    --
	    declare
		FIRST_TIME : BOOLEAN := TRUE;
	    begin
		while TIME_DELTA >= PARTIAL_TIME_DELTA
		and not O.DELETE_PENDING
		loop

		    if FIRST_TIME then
			FIRST_TIME := FALSE;
		    else
			-- the first time through, this is done earlier...
			--
			SET_O_VELOCITY(O, AEROPLANE_STATUS);
		    end if;

		    MOVE_DELTA(PARTIAL_TIME_DELTA, CS,
			O,
			AEROPLANE_STATUS,
			MAIN_WING_LIFT_DRAG_INPUTS,
			BODY_LIFT_DRAG_INPUTS);

		    TIME_DELTA := TIME_DELTA - PARTIAL_TIME_DELTA;
		end loop;
	    end;

	    AEROPLANE_STATUS.TIME_DELTA_CARRIED_FORWARD := TIME_DELTA;

	    -- Check if plane is still intact
	    --
	    if AEROPLANE_STATUS.Gs_FELT not in
		A_T.NEGATIVE_G_LIMIT..A_T.POSITIVE_G_LIMIT
	    then
		SCORE.OTHER_DEATH("Wings broke off");
		O.DELETE_PENDING := TRUE;
	    end if;

	    if	(abs AEROPLANE_STATUS.Gs_FELT) > A_T.FLAP_G_LIMIT
	    and (AEROPLANE_STATUS.FLAP_CONTROL /= 0.0)
	    then
		SCORE.OTHER_DEATH("Flaps broke off");
		O.DELETE_PENDING := TRUE;
	    end if;

	    -- Remember the AoA indicator value
	    --
	    AOA_INDICATOR.AOA_MAX_RATIO := 0.0;
	    if AEROPLANE_STATUS.AOA_MAX /= 0.0 then
		begin
		    AOA_INDICATOR.AOA_MAX_RATIO :=
			SCALE_TYPE(AEROPLANE_STATUS.AOA)/
			SCALE_TYPE(AEROPLANE_STATUS.AOA_MAX);
		exception
		    when CONSTRAINT_ERROR|NUMERIC_ERROR =>
			AOA_INDICATOR.AOA_MAX_RATIO := 1.1;
		end;
	    end if;
	end;

	-- Make sure the basis stays "good", rather than slowly distorts
	--
	FIX_BASIS(O);

    end;
