/* Ygl: Run 2d-GL programs with standard X11 routines. (C) Fred Hucht 1993/94 */

#define EXTERN
#include "header.h"

static int        YglWmDel = False;
static Atom       wm_delete_window;
static Ulong      YglButtonMask = 0;

static int        YglPCM  = False,
		  YglBS   = False, /* YGL_BS (backing store ) set */
		  YglRGB  = False, /* flag to tell gconfig to switch to RGB mode */
                  YglSTAT = False;

static YglWindow  *YglWindows;
static XSizeHints YglNext;

static Ulong      GCValueMask;
static XGCValues  GCValues;

static short      YglActiveWindow =  0,
                  YglNextWindow   =  1;
static Int32      YglActiveColor;
static long       YglEventMask    = EnterLeaveMask; /* GL seems to set this... */
static Ulong     *YglColors,
                 *YglColorsInv; /* for lrectread */

static Colormap   CMapColormap, RGBColormap;
static Visual    *CMapVisual,  *RGBVisual;
static int        CMapDepth,    RGBDepth,
                  CMapSize,     RGBSize;
static short      RGBrs, RGBgs, RGBbs; /* bits to shift */
static short      RGBrb, RGBgb, RGBbb; /* width of mask */

#define YglVisual   	((W->rgb) ? RGBVisual   : CMapVisual  )
#define YglDepth    	((W->rgb) ? RGBDepth    : CMapDepth   )
#define YglColormap 	((W->rgb) ? RGBColormap : CMapColormap)

#define DoRGB 		(DoRed|DoGreen|DoBlue)
#define GLCOLORS(x) 	(YglPCM ? (x) : YglColors[x])
#define GLCOLORSINV(x) 	(YglPCM ? (x) : YglColorsInv[x])

/* Initialization */

void minsize(Int32 x, Int32 y) {
  YglNext.min_width  = x;
  YglNext.min_height = y;
  YglNext.flags |= PMinSize;
}

void maxsize(Int32 x, Int32 y) {
  YglNext.max_width  = x;
  YglNext.max_height = y;
  YglNext.flags |= PMaxSize;
}

void prefsize(Int32 x, Int32 y) {
  YglNext.width  = x;
  YglNext.height = y;
  YglNext.flags |= PSize;
  YglNext.flags |= USSize;
  minsize(x, y);
  maxsize(x, y);
}

void prefposition(Int32 x1, Int32 x2, Int32 y1, Int32 y2) {
  YglNext.x = x1;
  YglNext.y = y1;
  YglNext.flags |= PPosition;
  YglNext.flags |= USPosition;
  prefsize(x2-x1, y2-y1);
}

void stepunit(Int32 x, Int32 y) {
  YglNext.width_inc  = x;
  YglNext.height_inc = y;
  YglNext.flags |= PResizeInc;
}

void keepaspect(Int32 x, Int32 y) {
  YglNext.min_aspect.x = YglNext.max_aspect.x = x;
  YglNext.min_aspect.y = YglNext.max_aspect.y = y;
  YglNext.flags |= PAspect;
}

static int errorHandler(Display *dpy, XErrorEvent *error) {
  char errortext[1024];
  I;
  XGetErrorText(dpy, error->error_code, errortext, 1024);
  fprintf(stderr, "Ygl: X Error: %s on 0x%x\n", errortext, error->resourceid);
  return(0);
}

static int free_colormaps() { /* returns number of unused colormaps */
  int num;
  XListInstalledColormaps(D, YglParentWindow, &num);
#ifdef DEBUG
  fprintf(stderr,"%d of %d installed Colormaps\n",
	 num, MaxCmapsOfScreen(DefaultScreenOfDisplay(D)));
#endif
  return(MaxCmapsOfScreen(DefaultScreenOfDisplay(D)) - num);
}

#ifdef AUTOFLUSH
static void signal_handler(int sig_no) {
#if 0
  if(_YglAllowFlush) {	/* If not in graphics routine, flush buffer */
    XFlush(D);
  }
  else 			/* If in graphics routine, set flag to flush at end of routine */
#endif
    
  _YglDoFlush = True;
  
#ifdef DEBUG
  fprintf(stderr, "In signal_handler\n");
#endif
  
  signal(SIGVTALRM, signal_handler);
}
#endif

void ginit() {
  char *display = NULL, *env;
  const char *colors[] = {"black","white","green","yellow","blue","magenta","cyan","red"};
  const char *names[3][4] = {
    { "YGL_FLUSHTIME", 		"YGL_FT",	"XGL_FLUSHTIME", 	"XGL_FT"  },
    { "YGL_PRIVATECOLORMAP", 	"YGL_PCM",	"XGL_PRIVATECOLORMAP", 	"XGL_PCM" },
    { "YGL_BACKINGSTORE", 	"YGL_BS",	"XGL_BACKINGSTORE", 	"XGL_BS"  }
  };
  int i, msec = 100;
  
  for(i = 0; i < 4; i++)
    if ((env = getenv(names[0][i])) != NULL && env[0] != '\0') {
      msec = 1000 * atof(env);
      i = 4711; /* End of loop */
    }

  for(i = 0; i < 4; i++)
    if ((env = getenv(names[1][i])) != NULL && env[0] != '\0') {
      YglPCM = (strcmp(env,"1") == 0); /* Use private colormap */
      i = 4711; /* End of loop */
    }

  for(i = 0; i < 4; i++)
    if ((env = getenv(names[2][i])) != NULL && env[0] != '\0') {
      YglBS = (strcmp(env,"1") == 0); /* Use backing store */
      i = 4711; /* End of loop */
    }

  if ((D = XOpenDisplay(display)) == NULL) {
    fprintf(stderr, "Ygl: ginit: Can\'t open display \"%s\"\n", XDisplayName(display));
    exit(1);
  }

#ifdef DEBUG
/*  XSynchronize(D, True); /* XStoreColor fails in sync. mode... */
#endif
  
#ifndef DEBUG
  XSetErrorHandler(errorHandler);
#endif

  { /* find Visuals */
    XVisualInfo xvproto, *xv;
    int Pseudoi=-1, Grayi=-1, Truei=-1, Directi=-1, SGrayi=-1;
    int xvn, i = 0;
    xvproto.screen = YglScreen;
    xv = XGetVisualInfo(D, VisualScreenMask, &xvproto, &xvn);
    
    for(i=0; i < xvn; i++)
      switch(xv[i].class) {
      case StaticGray:  SGrayi  = i; break;
      case GrayScale:   Grayi   = i; break;
      case PseudoColor: Pseudoi = i; break;
      case TrueColor:   Truei   = i; break;
      case DirectColor: Directi = i; break;
      default: break;
      }
    
    if      (Pseudoi >= 0) i = Pseudoi;
    else if (Grayi >= 0)   i = Grayi;
    else if (SGrayi >= 0){ i = SGrayi; YglSTAT = True; }
    else {
      fprintf(stderr, "Ygl: ginit: no PseudoColor, GrayScale of StaticGray visual found.\n");
      exit(1);
    }
    CMapVisual = xv[i].visual;
    CMapDepth  = xv[i].depth;
    CMapSize   = xv[i].colormap_size;


    if      (Truei   >= 0) i = Truei;
    else if (Directi >= 0) i = Directi;
    else                   i = -1;
    
    if(i >= 0) {
      RGBVisual   = xv[i].visual;
      RGBDepth    = xv[i].depth;
      RGBSize     = xv[i].colormap_size;
      RGBColormap = XCreateColormap(D, YglParentWindow, RGBVisual, AllocNone);
      
      /* Determine shift value */
      RGBrs = 0; while (!((1<< RGBrs) & RGBVisual->red_mask   )) RGBrs++;
      RGBgs = 0; while (!((1<< RGBgs) & RGBVisual->green_mask )) RGBgs++;
      RGBbs = 0; while (!((1<< RGBbs) & RGBVisual->blue_mask  )) RGBbs++;
      
      /* Determine mask widths */
      RGBrb = 0; while ( ((1<< (RGBrs+RGBrb)) & RGBVisual->red_mask   )) RGBrb++;
      RGBgb = 0; while ( ((1<< (RGBgs+RGBgb)) & RGBVisual->green_mask )) RGBgb++;
      RGBbb = 0; while ( ((1<< (RGBbs+RGBbb)) & RGBVisual->blue_mask  )) RGBbb++;

#ifdef DEBUG
      fprintf(stderr,"Shift = (%d,%d,%d), width = (%d,%d,%d)\n",
	     RGBrs, RGBgs, RGBbs, RGBrb, RGBgb, RGBbb);
#endif
      /* We need it in this form in RGBcolor ... */
      RGBrb = 8-RGBrb; RGBgb = 8-RGBgb; RGBbb = 8-RGBbb;
    }
    
    XFree((char*) xv);

    if(CMapVisual != DefaultVisual(D, YglScreen) && !YglSTAT) {
      YglPCM = True; /* We must use a private colormap if we don't use the default visual */
#ifdef DEBUG
      fprintf(stderr,"Setting YglPCM = True\n");
#endif
    }
  }

#ifdef DEBUG
  fprintf(stderr,"CMap VisualID = 0x%x, Depth = %d, CMEntries = %d\n",
	  XVisualIDFromVisual(CMapVisual), CMapDepth, CMapSize);
  fprintf(stderr,"RGB  VisualID = 0x%x, Depth = %d, CMEntries = %d\n",
	  XVisualIDFromVisual(RGBVisual ), RGBDepth , RGBSize);
#endif

  if(!YglPCM) {
    CMapColormap = DefaultColormap(D, YglScreen);
    
    YglColors    = calloc( CMapSize , sizeof(Ulong));
    YglColorsInv = calloc( CMapSize , sizeof(Ulong));
    
    for(i=0; i<MIN(8,CMapSize); i++) { /* define only allocated colors,
					  others may change */
      XColor used, exact;
      if(XAllocNamedColor(D, CMapColormap, colors[i], &used, &exact)) {
	YglColors[i] = used.pixel;
	YglColorsInv[used.pixel] = i;
      } else {
	fprintf(stderr, "Ygl: ginit: can't allocate color: %s\n", colors[i]);
      }
    }
  }
  else {
    if(YglSTAT) {
      fprintf(stderr, "Ygl: ginit: Can't use private colormap on static visuals\n");
      exit(1);
    }
    CMapColormap = XCreateColormap(D, YglParentWindow, CMapVisual, AllocAll);
    
    if(free_colormaps() > 0) {
      XInstallColormap(D, CMapColormap);
#ifdef DEBUG
      fprintf(stderr,"Installing CMapColormap\n");
#endif
    }

    for(i=0; i<8; i++) {
      if(!XStoreNamedColor(D, CMapColormap, colors[i], i, DoRGB))
	fprintf(stderr, "Ygl: ginit: can't store color: %s\n", colors[i]);
    }
  }

  if(msec == -1000) {
    fprintf(stderr, "Ygl: ginit: Switching to synchronous mode.\n");
    XSynchronize(D, True);
  }
  else if(msec > 0) {
#ifdef AUTOFLUSH
    struct itimerval itimer, otimer;
    itimer.it_interval.tv_sec  =  msec / 1000;
    itimer.it_value.tv_sec     =  msec / 1000;
    itimer.it_interval.tv_usec = (msec % 1000) * 1000;
    itimer.it_value.tv_usec    = (msec % 1000) * 1000;
    
    /* _YglAllowFlush = False; */
    _YglDoFlush = False;
    
    setitimer(ITIMER_VIRTUAL, &itimer, &otimer);
    signal(SIGVTALRM, signal_handler);
#endif /* AUTOFLUSH */
  }
  
  loadXfont(0, "fixed");
  
  YglWindows = (YglWindow*) calloc(1, sizeof(YglWindow)); /* allocate one window structure. The first window(0) is never used...*/

  GCValueMask = GCGraphicsExposures;
  GCValues.graphics_exposures = False; /* so rectcopy won't generate {Graphics|No}Expose events */

#ifdef GCLIST
  { /* Create a tmp window to create GCs */
    Window tmpwin;
    XSetWindowAttributes sw;
    sw.background_pixel = WhitePixel(D,YglScreen);
    sw.border_pixel     = WhitePixel(D,YglScreen);
    sw.colormap         = CMapColormap; /* These three lines are nessesary
					   in a private visual */

    tmpwin = XCreateWindow(D, YglParentWindow, 0, 0, 10, 10, 2,
			   CMapDepth, InputOutput, CMapVisual,
			   CWBackPixel | CWBorderPixel | CWColormap,
			   &sw);

    _YglGCList = calloc( CMapSize , sizeof(GC));
    for(i = 0; i < CMapSize; i++) {
      _YglGCList[i] = XCreateGC(D, tmpwin, GCValueMask, &GCValues);
      XSetForeground(D, _YglGCList[i], i);
    }
    XDestroyWindow(D, tmpwin);
  }
#endif

  wm_delete_window = XInternAtom(D, "WM_DELETE_WINDOW", False);
#ifdef DEBUG
  fprintf(stderr, "wm_delete_window = %d\n", wm_delete_window);
#endif
}

Int32 winopen(char *Title) {
  XSetWindowAttributes sw;
  unsigned long mask = 0;

  if(D == NULL) ginit(); /* first window */

  YglActiveWindow = 1;
  while(YglActiveWindow < YglNextWindow && YglWindows[YglActiveWindow].win != 0)
    YglActiveWindow++;
  if(YglActiveWindow == YglNextWindow) {
    YglNextWindow++;
    YglWindows = (YglWindow*) realloc(YglWindows, YglNextWindow * sizeof(YglWindow));
  }
  W = YglWindows + YglActiveWindow;

  if(0 == (YglNext.flags & PMinSize)) minsize(40,30);
  if(0 == (YglNext.flags & PMaxSize)) maxsize(YglScreenWidth, YglScreenHeight);
  
  W->xf  = W->yf = 1.0;
  W->xd  = W->xm = YglNext.min_width;
  W->yd  = W->ym = YglNext.min_height;

#ifdef ORTHO
  ortho2(0, W->xm, 0, W->ym); /* set default viewport */
#endif

  W->xp  = W->yp = W->xc = W->yc = 0;
  W->rgb = False; /* New windows are always in cmode */

#ifdef MULTIBUF
  W->abuf= -1;    /* doublebuffering is not initialized */
#endif

  sw.background_pixel = WhitePixel(D,YglScreen);
  sw.border_pixel     = WhitePixel(D,YglScreen);
  sw.colormap         = YglColormap;
  mask = CWBackPixel | CWBorderPixel | CWColormap;
  
  if(YglBS) {sw.backing_store = Always; mask |= CWBackingStore; }
  
  W->win = XCreateWindow(D,
			 YglParentWindow,
			 YglNext.x, 
			 YglNext.y,
			 YglNext.min_width, 
			 YglNext.min_height,
			 2,			/* BorderWidth */
			 YglDepth,		/* Depth */
			 InputOutput,		/* Class */
			 YglVisual,
			 mask,
			 &sw);

  W->draw = W->win;
#ifdef GCLIST
  W->gc  = _YglGCList[0];
  W->chargc  = XCreateGC(D, W->win, GCValueMask, &GCValues);
#else
  W->gc  = XCreateGC(D, W->win, GCValueMask, &GCValues);
#endif

  font(0); /* Set default font to "fixed" */

  XSetStandardProperties(D, W->win, Title, Title, None, NULL, 0, &YglNext);
  
  memset(&YglNext, 0, sizeof(XSizeHints)); /* clear YglNext */

  if(YglPCM) XSetWindowColormap(D, W->win, YglColormap);

  XSelectInput(D, W->win, StructureNotifyMask|ExposureMask); /* So we get the MapNotify and
								  the first Exposure(REDRAW) event */
  XMapWindow(D, W->win);
  XSync(D, False);
  
  /* Await the MapNotify event */
  {
    XEvent e;
    e.type = 0;
    while(e.type != MapNotify)
      XWindowEvent(D, W->win, StructureNotifyMask, &e);
  }
  
  XSelectInput(D, W->win, YglEventMask);
  XSetWMProtocols(D, W->win, &wm_delete_window, YglWmDel?1:0);

  qenter(REDRAW, YglActiveWindow);

  return(YglActiveWindow);
}

void gconfig() {
  XWindowAttributes old;
  XSetWindowAttributes new;
  XSizeHints sh;
  unsigned long mask = 0;
  Window junkwin;
  int x, y;
  char *Title;

  I;
  if(YglRGB == W->rgb) return; /* no mode change */
  if(RGBVisual == NULL) {
    fprintf(stderr, "Ygl: gconfig: RGBmode requires a TrueColor visual\n");
    exit(1);
  }

  /* Close the active window... */
  
  XGetWindowAttributes(D, W->win, &old);
  XTranslateCoordinates(D, W->win, old.root,
			-old.border_width-old.x,
			-old.border_width-old.y,
			&x, &y, &junkwin);
  XFetchName(D, W->win, &Title);
  XGetNormalHints(D, W->win, &sh);

#ifdef DEBUG
  fprintf(stderr,"%s\n",Title);
  fprintf(stderr, "x = %d, y = %d\n", old.x, old.y);
  fprintf(stderr, "x = %d, y = %d\n", x,y);
#endif

  winclose(YglActiveWindow);

  XUninstallColormap(D, YglColormap);

  /* Switch to new mode */
  W->rgb = YglRGB;
  
  /* And open a new one with other visual at same place */
  
  if(free_colormaps() > 0) {
    XInstallColormap(D, YglColormap);
#ifdef DEBUG
    fprintf(stderr, "Installing own colormap\n");
#endif
  }
  
  new.background_pixel = WhitePixel(D,YglScreen);
  new.border_pixel     = WhitePixel(D,YglScreen);
  new.colormap         = YglColormap;
  mask = CWBackPixel | CWBorderPixel | CWColormap;
  
  if(YglBS) { new.backing_store = Always; mask |= CWBackingStore;}
  
  W->win = XCreateWindow(D, YglParentWindow, x, y, old.width, old.height,
			 old.border_width,	/* BorderWidth */
			 YglDepth,		/* Depth */
			 InputOutput,		/* Class */
			 YglVisual,
			 mask,
			 &new);
  
  W->draw = W->win;
  if(W->rgb) { /* RGB windows have only one GC */
    W->gc  = XCreateGC(D, W->win, GCValueMask, &GCValues);
  } else {
#ifdef GCLIST
    W->gc  = _YglGCList[0];
    W->chargc  = XCreateGC(D, W->win, GCValueMask, &GCValues);
#else
    W->gc  = XCreateGC(D, W->win, GCValueMask, &GCValues);
#endif
  }
  
  font(0); /* Set default font to "fixed" */

  sh.x = x;
  sh.y = y;
  sh.flags |= PPosition | USPosition;
  
  XSetStandardProperties(D, W->win, Title, Title, None, NULL, 0, &sh);
  XFree(Title);
  
  if(YglPCM) XSetWindowColormap(D, W->win, YglColormap);
  
  XSelectInput(D, W->win, StructureNotifyMask|ExposureMask); /* So we get the MapNotify and
								  the first Exposure(REDRAW) event */
  XMapWindow(D, W->win);
  XSync(D, False);
  
  /* Await the MapNotify event */
  {
    XEvent e;
    e.type = 0;
    while(e.type != MapNotify)
      XWindowEvent(D, W->win, StructureNotifyMask, &e);
  }
  
  XSelectInput(D, W->win, YglEventMask);
  XSetWMProtocols(D, W->win, &wm_delete_window, YglWmDel?1:0);

  qenter(REDRAW, YglActiveWindow);
}

void wintitle(Char8 *Title) {
  XStoreName(D, W->win, Title);
}

void winclose(Int32 id) {
  XEvent e;
  I;
  
  if(YglWindows[id].rgb) { /* RGB windows have only one GC */
    XFreeGC(D, YglWindows[id].gc);
  } else {
#ifdef GCLIST
    XFreeGC(D, YglWindows[id].chargc);
#else
    XFreeGC(D, YglWindows[id].gc);
#endif
  }
  
#ifndef DEBUG
  XGrabServer(D); /* So no additional events show up */
#endif

  XSelectInput(D, YglWindows[id].win, YglEventMask|ExposureMask);
  while(XCheckWindowEvent(D, YglWindows[id].win, YglEventMask|ExposureMask, &e))
    {} /* Discard all remaining events for this window */

  XDestroyWindow(D, YglWindows[id].win);

#ifndef DEBUG
  XUngrabServer(D);
#endif
  
  YglWindows[id].win = 0;
}

void gexit() {
  short i;
  I;
#ifdef GCLIST
  for(i = 0; i < CMapSize; i++) XFreeGC(D, _YglGCList[i]);
  free(_YglGCList);
#endif
  for(i=1; i<YglNextWindow; i++) if(YglWindows[i].win != 0) winclose(i);

  if(YglPCM) {
    XFreeColormap(D, CMapColormap);
  } else {
    free(YglColors);
    free(YglColorsInv);
  }

  if(RGBColormap != 0) XFreeColormap(D, RGBColormap);

  free(YglWindows);
  
  XCloseDisplay(D);
}

void winset(Int32 windowid) {
  if(YglWindows[windowid].win != 0)
    W = YglWindows + (YglActiveWindow = windowid);
  else
    fprintf(stderr, "Ygl: winset: invalid windowid: %d\n", windowid);
}

Int32 winget()    { return(YglActiveWindow); }
Int32 getplanes() { return(YglDepth       ); }

/* GC stuff */

void linewidth(Int16 w) {
  int i;
  I;
  GCValues.line_width = w;
  GCValueMask = GCLineWidth;
#ifdef GCLIST
  for(i = 0; i < CMapSize; i++) {
    XChangeGC(D, _YglGCList[i], GCValueMask, &GCValues);
  }
#endif

  /* additional loop over all windows, some may be rgb... */
  for(i=1; i<YglNextWindow; i++) if(YglWindows[i].win != 0) {
    XChangeGC(D, YglWindows[i].gc, GCValueMask, &GCValues);
  }
}

/* Colors */

void RGBmode() { /* switch the window active on next gconfig() to RGB mode */
  YglRGB = True;
}
void cmode() { /* switch the window active on next gconfig() to colormap mode */
  YglRGB = False;
}

void mapcolor(Colorindex index, Int16 r, Int16 g, Int16 b) {
  XColor color;
  I;
  if(YglSTAT) {
    fprintf(stderr, "Ygl: mapcolor: can't map colors on static visuals\n");
    return;
  }
  if(index > CMapSize) {
    fprintf(stderr, "Ygl: mapcolor: can't map cell %d (must be between 0 and %d).\n", index, CMapSize-1);
    return;
  }
  color.red   = r << 8;
  color.green = g << 8;
  color.blue  = b << 8;
  if(!YglPCM) {
    if (YglColors[index]) XFreeColors(D, CMapColormap, YglColors + index, 1, 0);

    if(XAllocColor(D, CMapColormap, &color)) {
      YglColors[index] = color.pixel;
      YglColorsInv[color.pixel] = index;
    } else {
      fprintf(stderr, "Ygl: mapcolor: can't allocate color ( %d, %d, %d) for cell %d\n", r, g, b, index);
    }
  }
  else {
    /* XFreeColors(D, CMapColormap, (Ulong*)&index, 1, 0); */
  
    color.pixel = index;
    color.flags = DoRGB;
    XStoreColor(D, CMapColormap, &color);
  }
#ifdef GCLIST
  XSetForeground(D, _YglGCList[index], index);
#endif
  F;
}

/* converts 24 bit RGB values ( 0xBBGGRR ) to value appropiate for RGBVisual */
#define RGB24_TO_RGBVisual(rgb24) ( \
		((((rgb24) >> (RGBrb +  0)) << RGBrs) & RGBVisual->red_mask  ) \
	      + ((((rgb24) >> (RGBgb +  8)) << RGBgs) & RGBVisual->green_mask) \
	      + ((((rgb24) >> (RGBbb + 16)) << RGBbs) & RGBVisual->blue_mask ) )

#define RGB_TO_RGBVisual(r,g,b) ( \
		((((r) >> RGBrb) << RGBrs) & RGBVisual->red_mask  ) \
	      + ((((g) >> RGBgb) << RGBgs) & RGBVisual->green_mask) \
	      + ((((b) >> RGBbb) << RGBbs) & RGBVisual->blue_mask ) )

void cpack(Uint32 rgb) {
  I; if(!W->rgb) {fprintf(stderr, "Ygl: cpack: not in RGB mode\n");return; }
  XSetForeground(D, W->gc, YglActiveColor = RGB24_TO_RGBVisual(rgb));
}

void c3s(Int16 c[3]) {
  I; if(!W->rgb) { fprintf(stderr, "Ygl: c3s: not in RGB mode\n"); return; }
  XSetForeground(D, W->gc, YglActiveColor = RGB_TO_RGBVisual(c[0], c[1], c[2]));
}

void c3i(Int32 c[3]) {
  I; if(!W->rgb) { fprintf(stderr, "Ygl: c3i: not in RGB mode\n"); return; }
  XSetForeground(D, W->gc, YglActiveColor = RGB_TO_RGBVisual(c[0], c[1], c[2]));
}
  
void c3f(float c[3]) {
  I; if(!W->rgb) { fprintf(stderr, "Ygl: c3f: not in RGB mode\n"); return; }
  YglActiveColor = RGB_TO_RGBVisual((Uint32)(c[0]*255.99),(Uint32)(c[1]*255.99),(Uint32)(c[2]*255.99));
  XSetForeground(D, W->gc, YglActiveColor);
}
				   
void RGBcolor (Int16 r, Int16 g, Int16 b) {
  I; if(!W->rgb) { fprintf(stderr, "Ygl: RGBcolor: not in RGB mode\n"); return; }
  XSetForeground(D, W->gc, YglActiveColor = RGB_TO_RGBVisual(r, g, b));
}

#if 0 /* This version is veeeryyy slooowwww !!! */
void RGBcolor (Int16 r, Int16 g, Int16 b) {
  XColor color;
  I;
  if(!W->rgb) {
    fprintf(stderr, "Ygl: RGBcolor: not in RGB mode\n");
    return;
  }
  color.red   = r << 8;
  color.green = g << 8;
  color.blue  = b << 8;
  if(XAllocColor(D, RGBColormap, &color))
    YglActiveColor = color.pixel;
  else
    fprintf(stderr, "Ygl: RGBcolor: can't allocate color ( %d, %d, %d)\n", r, g, b);
  XSetForeground(D, W->gc, YglActiveColor);
}
#endif

Int32 getcolor() { return(YglActiveColor);}

void getmcolor (Colorindex index, Int16 *r, Int16 *g, Int16 *b) {
  XColor color;
  I;
  color.pixel = index;
  XQueryColor(D, CMapColormap, &color);
  *r = color.red   >> 8;
  *g = color.green >> 8;
  *b = color.blue  >> 8;
}

void gRGBcolor(Int16 *r, Int16 *g, Int16 *b) {
  *r = ((YglActiveColor & RGBVisual->red_mask  ) >> RGBrs) << RGBrb;
  *g = ((YglActiveColor & RGBVisual->green_mask) >> RGBgs) << RGBgb;
  *b = ((YglActiveColor & RGBVisual->blue_mask ) >> RGBbs) << RGBbb;
  
#ifdef DEBUG
  fprintf(stderr, "YglActiveColor = %d, (r,g,b) = (%d,%d,%d)\n", YglActiveColor,*r,*g,*b);
#endif

}

#ifdef GCLIST
void color (Colorindex color) {
  if(color >= CMapSize)
    fprintf(stderr, "Ygl: color: invalid color %d (must be between 0 and %d).\n", color, CMapSize-1);
  else
    W->gc = _YglGCList[GLCOLORS(YglActiveColor = color)];
}

void Xcolor(Colorindex color) {
  if(color >= CMapSize)
    fprintf(stderr, "Ygl: Xcolor: invalid color %d (must be between 0 and %d).\n", color, CMapSize-1);
  else
    W->gc = _YglGCList[YglActiveColor = color];
}

void charstr(char *Text) {
  I;
  if(W->rgb) {
    XDrawString(D, W->draw, W->gc, X(W->xc), Y(W->yc), Text, strlen(Text));
  } else {
    XSetForeground(D, W->chargc, GLCOLORS(YglActiveColor));
    XDrawString(D, W->draw, W->chargc, X(W->xc), Y(W->yc), Text, strlen(Text));
  }
  W->xc += strwidth(Text);
  F;
}
#else /*GCLIST*/
void  color(Colorindex color) {
  I;
  if(color >= CMapSize)
    fprintf(stderr, "Ygl: color: invalid color %d (must be between 0 and %d).\n", index, CMapSize-1);
  else
    XSetForeground(D, W->gc, GLCOLORS(YglActiveColor = color));
}

void Xcolor(Colorindex color) {
  I;
  if(color >= CMapSize)
    fprintf(stderr, "Ygl: Xcolor: invalid color %d (must be between 0 and %d).\n", index, CMapSize-1);
  else
    XSetForeground(D, W->gc, YglActiveColor = color);
}

void charstr(char *Text) {
  I;
  XDrawString(D, W->draw, W->gc , X(W->xc), Y(W->yc), Text, strlen(Text));
  W->xc += strwidth(Text);
  F;
}
#endif /*GCLIST*/

void readsource(Int32 source) {
  switch(source) {
  case SRC_AUTO: break;
  case SRC_FRONT: 
  case SRC_BACK: 
    fprintf(stderr, "Ygl: readsource: mode %d not implemented.", source);
    break;
  default:
    fprintf(stderr, "Ygl: readsource: unknown mode %d.", source);
    break;
  }
}

static int rect_read_err;
static int rect_read_errh(Display *dpy, XErrorEvent *error) {
  rect_read_err = error->error_code;
  return(0);
}

static const char rect_first[] = "?c ?l"; /* first char for errors */

static Int32 rect_read(Screencoord x1, Screencoord y1,
		       Screencoord x2, Screencoord y2,
		       int size, void *data) {
  XImage *XI;
  register Ulong x, y;
  register Int32 *data32 = (Int32*) data;
  register Int16 *data16 = (Int16*) data;
  register Uint8 *data8  = (Uint8*) data;

  int (*err)();
  I;

  err = XSetErrorHandler(rect_read_errh);
  
  XI = XGetImage(D, W->draw,
		 x1, W->ym - 1 - y2, x2-x1+1, y2-y1+1,
		 (1 << YglDepth)-1,
		 ZPixmap);

  XSetErrorHandler(err); /* reset */

#ifdef DEBUG
  fprintf(stderr, "rect_read_err: %d\n", rect_read_err);
#endif
  if(rect_read_err == BadMatch) {
    fprintf(stderr, "Ygl: %crectread: window not completely visible\n", rect_first[size]);
    return(0);
  }
  
  for(y=XI->height-1; y != -1UL; y--) {
    for(x=0; x < XI->width; x++) {
      Ulong pixel;
      long val;

      pixel = XGetPixel(XI, x, y);

      if(W->rgb) {
	val = (((pixel & RGBVisual->red_mask  ) >> RGBrs) << (RGBrb +  0))
	    + (((pixel & RGBVisual->green_mask) >> RGBgs) << (RGBgb +  8))
	    + (((pixel & RGBVisual->blue_mask ) >> RGBbs) << (RGBbb + 16));
      } else {
	val = GLCOLORSINV(pixel);
      }
      
      switch(size) {
      case 1: *(data8++)  = val; break;
      case 2: *(data16++) = val; break;
      case 4: *(data32++) = val; break;
      }
#ifdef DEBUG
      printf("%x,%x ", pixel, val);
#endif
    }
#ifdef DEBUG
    puts("");
#endif
  }
  x = XI->width * XI->height;
  XDestroyImage(XI);
  return(x);
}

static void rect_write(Screencoord x1, Screencoord y1,
		       Screencoord x2, Screencoord y2,
		       int size, void *data) {
  XImage *XI;
  register Ulong x, y;

  register Int32 *data32 = (Int32*) data;
  register Int16 *data16 = (Int16*) data;
  register Uint8 *data8  = (Uint8*) data;

  char *d;
  
  I;
  d = malloc((x2-x1+1)*(y2-y1+1)*(YglDepth/8));
  
  XI = XCreateImage(D,
		    YglVisual,
		    YglDepth,
		    ZPixmap,	/* Format */
		    0,		/* Offset */
		    d,		/* Data */
		    x2-x1+1,	/* Width */
		    y2-y1+1,	/* Height */
		    8, 		/* BitmapPad */
		    0);		/* BytesPerLine */

  for(y=XI->height-1; y != -1UL; y--) {
    for(x=0; x < XI->width; x++) {
      Ulong pixel;
      long val;
      switch(size) {
      case 1: val = *(data8++);  break;
      case 2: val = *(data16++); break;
      case 4: val = *(data32++); break;
      }
      if(W->rgb) {
	pixel = RGB24_TO_RGBVisual(val);
	XPutPixel(XI, x, y, pixel); 
	/* *(d + y * XI->bytes_per_scanline + x) = pixel; */
      } else { /* CMap */
	if((val < 0) || (val > CMapSize)) {
	  fprintf(stderr, "Ygl: %crectwrite: invalid color %d at position (%d,%d) in cmode (must be between 0 and %d).\n",
		  rect_first[size], val, x, y, CMapSize-1);
	} else {
	  XPutPixel(XI, x, y, GLCOLORS(val));
	}
      }
    }
  }
  
  XPutImage(D, W->draw, W->gc, XI, 0, 0,
	    x1, W->ym - 1 - y2,
	    x2-x1+1, y2-y1+1);
  
  XDestroyImage(XI);
  F;
}

Int32 lrectread(Screencoord x1, Screencoord y1,	Screencoord x2, Screencoord y2,	Int32 *data) { return(rect_read(x1,y1,x2,y2, 4, data)); }
Int32  rectread(Screencoord x1, Screencoord y1,	Screencoord x2, Screencoord y2,	Int16 *data) { return(rect_read(x1,y1,x2,y2, 2, data)); }
Int32 crectread(Screencoord x1, Screencoord y1,	Screencoord x2, Screencoord y2,	Uint8 *data) { return(rect_read(x1,y1,x2,y2, 1, data)); }

void lrectwrite(Screencoord x1, Screencoord y1, Screencoord x2, Screencoord y2, Int32 *data) { rect_write(x1,y1,x2,y2, 4, data); }
void  rectwrite(Screencoord x1, Screencoord y1,	Screencoord x2, Screencoord y2,	Int16 *data) { rect_write(x1,y1,x2,y2, 2, data);}
void crectwrite(Screencoord x1, Screencoord y1,	Screencoord x2, Screencoord y2,	Uint8 *data) { rect_write(x1,y1,x2,y2, 1, data);}

void rectcopy(Screencoord x1, Screencoord y1,
	      Screencoord x2, Screencoord y2,
	      Screencoord xn, Screencoord yn) {
  XCopyArea(D, W->draw, W->draw, W->gc,
	    x1, W->ym - 1 - y2,
	    x2-x1+1, y2-y1+1,
	    xn, W->ym - 1 - yn - (y2-y1));
  F;
}

/* Queue */
static void set_YglEventMask() {
  short i;

#ifdef DEBUG
  fprintf(stderr, "Setting event mask to 0x%x\n", YglEventMask);
#endif

  for(i=1; i<YglNextWindow; i++) 
    if(YglWindows[i].win != 0) {
      XSelectInput(D, YglWindows[i].win, YglEventMask );
      XSetWMProtocols(D, YglWindows[i].win, &wm_delete_window, YglWmDel?1:0);
    }
}

void qdevice(Device dev) { 
  switch(dev) {
  case WINCLOSE    : YglWmDel = True; break;
  case REDRAW      : YglEventMask  |= ExposureMask;      break;
  case INPUTCHANGE : YglEventMask  |= EnterLeaveMask;    break;
  case KEYBD       : YglEventMask  |= KeyPressMask;      break;
  case MOUSEX      :
  case MOUSEY      : YglEventMask  |= PointerMotionMask; break;
  case LEFTMOUSE   : YglButtonMask |= Button1Mask;       break;
  case MIDDLEMOUSE : YglButtonMask |= Button2Mask;       break;
  case RIGHTMOUSE  : YglButtonMask |= Button3Mask;       break;
  default          : fprintf(stderr, "Ygl: qdevice: Unknown device: %d\n", dev); return;
  }
  if (YglButtonMask) YglEventMask |= ButtonPressMask|ButtonReleaseMask;
  
  set_YglEventMask();
}    

void unqdevice(Device dev) { 
  switch(dev) {
  case WINCLOSE    : YglWmDel = False; break;
  case REDRAW      : YglEventMask  &= ~ExposureMask;      break;
  case INPUTCHANGE : YglEventMask  &= ~EnterLeaveMask;    break;
  case KEYBD       : YglEventMask  &= ~KeyPressMask;      break;
  case MOUSEX      :
  case MOUSEY      : YglEventMask  &= ~PointerMotionMask; break;
  case LEFTMOUSE   : YglButtonMask &= ~Button1Mask;       break;
  case MIDDLEMOUSE : YglButtonMask &= ~Button2Mask;       break;
  case RIGHTMOUSE  : YglButtonMask &= ~Button3Mask;       break;
  default          : fprintf(stderr, "Ygl: unqdevice: Unknown device: %d\n", dev); return;
  }
  if (!YglButtonMask) YglEventMask &= ~(ButtonPressMask|ButtonReleaseMask);

  set_YglEventMask();
}    

void qreset() { I; XSync(D, True); }

static Int16 mousex = -1, mousey = -1, mouseychanged = False;

Int32 qtest() { 
  Int32 dev = 0;
  I;

  if(mouseychanged) { /* Hack for mouse events */
    return(MOUSEY);
  }

 next:
  
  if(XPending(D)) {
    XEvent e;
    XPeekEvent(D, &e);
    
#ifdef DEBUG
    fprintf(stderr, "e.type = %d, XPending returns %d\n", e.type, XPending(D));
#endif
    
    switch(e.type) {
    case Expose      : dev = REDRAW;      break;
    case EnterNotify :
    case LeaveNotify : dev = INPUTCHANGE; break;
    case MotionNotify: 
#if 0
      if(mousex != e.xmotion.x_root) { dev = MOUSEX; break; } /* if both values have changed, report x first... */
      if(mousey != e.xmotion.y_root) { dev = MOUSEY; break; }
#endif
      if(mousex != e.xmotion.x_root)
	dev = MOUSEX; /* if both values have changed, report x first... */
      else
	dev = MOUSEY;
      break;
    case KeyPress:
      {
	char key;
	if(1 == XLookupString(&e.xkey, &key, 1, NULL, NULL)) /* ignore modifier keys */
	  dev = KEYBD;
	break;
      }
    case ButtonPress :
    case ButtonRelease:
      switch(e.xbutton.button) {
      case Button1: if(YglButtonMask&Button1Mask) dev =   LEFTMOUSE; break;
      case Button2: if(YglButtonMask&Button2Mask) dev = MIDDLEMOUSE; break;
      case Button3: if(YglButtonMask&Button3Mask) dev =  RIGHTMOUSE; break;
      default:      fprintf(stderr, "Ygl: qtest: Ignoring unknown button: %d\n", e.xbutton.button); break;
      }
      break;
    case ClientMessage: 
#ifdef DEBUG
      fprintf(stderr, "e.xclient.message_type = %d\n", e.xclient.message_type);
#endif
      dev = WINCLOSE;
      break;
    default: fprintf(stderr, "Ygl: qtest: Ignoring unknown event: %d\n", e.type); break;
    }
    
    if(dev == 0) {
#ifdef DEBUG
      fprintf(stderr, "qtest: eating unknown event, e.type = %d\n", e.type);
#endif
      XNextEvent(D, &e); /* read and ignore unreported events so they don't block the queue */
      goto next; /* more events? */
    }
  }
  return(dev);
}

static Int32 x2gl_wid(Window win) { /* Computes Ygl wid from X wid */
  Int32 i = YglNextWindow - 1;
  while(i > 0 && win != YglWindows[i].win) i--;
  return(i); /* 0 if not found */
}

Int32 qread(Int16 *val) {
  XEvent e;
  Int32 dev;
  I;
 next:
  dev = 0;
  if(mouseychanged) { /* Hack for mouse events */
    *val = YglScreenHeight - mousey - 1;
    mouseychanged = False;
    return(MOUSEY);
  }
  XNextEvent(D, &e);
  
#ifdef DEBUG
  fprintf(stderr, "e.type = %d, XPending returns %d\n", e.type, XPending(D));
#endif
  
  switch(e.type) {

  case Expose:
    if(e.xexpose.count != 0) { 
#ifdef DEBUG
      fprintf(stderr, "e.xexpose.count = %d\n", e.xexpose.count);
#endif
      goto next;
    }
    
    if(XPending(D)) { /* next event also Expose event for the same window ? */
      XEvent p;
      XPeekEvent(D, &p);
      if(p.type == Expose && p.xexpose.window == e.xexpose.window) {
#ifdef DEBUG
	fprintf(stderr, "Next event is REDRAW for same window. Ignoring this one...\n");
#endif
	goto next;
      }
    }
    dev  = REDRAW;
    *val = x2gl_wid(e.xany.window);
    break;

  case EnterNotify:
    dev  = INPUTCHANGE;
    *val = x2gl_wid(e.xany.window);
    break;

  case LeaveNotify: dev = INPUTCHANGE; *val = 0; break;

  case MotionNotify:
#ifdef DEBUG
    	fprintf(stderr, " e.xmotion.x_root = %d, e.xmotion.y_root = %d\n",
		    	  e.xmotion.x_root,      e.xmotion.y_root);
#endif

    if(mousey != e.xmotion.y_root) {
      mousey = e.xmotion.y_root;
      mouseychanged = True;
    }
    if(mousex != e.xmotion.x_root) {
      dev   = MOUSEX;
      *val  = mousex = e.xmotion.x_root;
    } 
    else goto next;
    break;
    
  case KeyPress: dev = KEYBD;
    {
      char key;
      if(1 != XLookupString(&e.xkey, &key, 1, NULL, NULL)) goto next; /* ignore modifier keys */
      *val = (Int16)key;
    }
    break;

  case ButtonPress:
  case ButtonRelease:
    switch(e.xbutton.button) {
    case Button1: if(!(YglButtonMask&Button1Mask)) goto next; dev =   LEFTMOUSE; break;
    case Button2: if(!(YglButtonMask&Button2Mask)) goto next; dev = MIDDLEMOUSE; break;
    case Button3: if(!(YglButtonMask&Button3Mask)) goto next; dev =  RIGHTMOUSE; break;
    default:      fprintf(stderr, "Ygl: qread: Unknown button: %d\n", e.xbutton.button); break;
    }
    *val = (e.type == ButtonPress) ? 1 : 0;
    break;

  case ClientMessage: 
#ifdef DEBUG
    fprintf(stderr, "e.xclient.message_type = %d\n", e.xclient.message_type);
#endif
    dev   = WINCLOSE;
    *val = x2gl_wid(e.xany.window);
    break;
    
  default: fprintf(stderr, "Ygl: qread: Unknown event %d for window %d received.\n",
		   e.type, x2gl_wid(e.xany.window)); break;
  }
  return(dev);
}

void qenter(Int16 dev, Int16 val) {
  XEvent e;
  I;
  switch(dev) {
  case REDRAW      : e.type = Expose; 
                     e.xexpose.window = (val > 0 && val < YglNextWindow) ? YglWindows[val].win : 0;
                     e.xexpose.count = 0; break;

  case INPUTCHANGE : fprintf(stderr, "Ygl: qenter: Unsupported device: INPUTCHANGE\n") ; return; break;
    /* e.type = EnterNotify; e.xcrossing.window = val; break; */
  case KEYBD       : fprintf(stderr, "Ygl: qenter: Unsupported device: KEYBD\n")       ; return; break;
  case MENUBUTTON  : fprintf(stderr, "Ygl: qenter: Unsupported device: MENUBUTTON\n")  ; return; break;
  default          : fprintf(stderr, "Ygl: qenter: Unknown device: %d\n", dev)         ; return; break;
  }
  XPutBackEvent(D, &e);
}
