/*
** tagged.c
**	A program that uses the XeTextEd widget in tagged mode.
*/

#include <stdio.h>
#include <ctype.h>
#if SYSV_INCLUDES
#	include <memory.h>
#endif
#include <string.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Viewport.h>
#if X11R5
#include <X11/Xaw/Porthole.h>
#include <X11/Xaw/Panner.h>
#include <X11/Xaw/Tree.h>
#endif
#include <X11/Xew/TextEd.h>

#if SYSV_INCLUDES || ANSI_INCLUDES
#	include <limits.h>
#else
#ifndef LONG_MAX
#define LONG_MAX (~((unsigned long)0L)>>1)
#endif
#endif
#include "ui_tools.h"


static Widget root_widget;
static Widget edit_widget;	/* TextEd widget instance */
static char *edit_file;		/* Current primary file */
static Widget edit_title;	/* Label Widget for filename */

typedef enum
    {
	FileMenu_None,
	FileMenu_SaveAs,
	FileMenu_LoadFile
    } FileMenuAction;

typedef struct TagClass
    {
	char *name;
	struct TagClass *next;
    } TagClass;

typedef enum
    {
	TagOption_flatten,
	TagOption_inclusive,
	TagOption_add,
	TagOption_subtract,
	TagOption_override
    } TagOptions;

typedef enum
    {
	NewLineOption_inherit,	/* Inherit from enclosing tag */
	NewLineOption_hard,
	NewLineOption_enriched
    } NewLineOption;


typedef struct TagDef
    {
	char *name;
	struct TagDef *next;
	unsigned int newline:3;	/* Check against size of NewLineOption */
	unsigned int option:3;	/* Check against size of TagOptions */
	TagClass *class;
	char *rendition;
	char *begin, *line, *end;
    } TagDef;

typedef struct Tag
    {
	TagDef *tag;	/* Tag description */
	int down;	/* Previous tag */
#if X11R5
	Widget tree;	/* Tree Node widget */
#endif
    } Tag;

static TagDef default_tag = {"*",}; /* Associated with tags[0] entry */
static TagDef flatten_tag = {"*",}; /* Used for 'flatten' menu command */

static TagDef *known_tags, *known_last;
static TagClass *tag_classes, *last_class;


/*
** tags
**	vector is used to maintain father linked tree structure
**	which describes all used tag combinations. The text in
**	in text widget is actually tagged with indexes into
**	this vector. Following the father (down) links from this
**	element, one can find all enclosing tags.
*/
#define TAGS_INITIAL_SIZE 100
#define TAGS_INCREASE_SIZE 100

static Tag *tags;	/* Tag stack array */
static int tags_used;	/* Last used *INDEX* from tags */
static int tags_size;	/* Current allocated size of the tags */
static int tags_index;	/* Current Tag in effect */

#if X11R5
/*
** ************************************
** Visualize the current tags in a tree
** ************************************
** This is only available with X11R5 (Tree/Panner/Porthole widgets needed)
*/
static Widget tags_visual = None;	/* Tree of tags */

static void PortholeCallback(w, panner_ptr, report_ptr)
Widget w;
XtPointer panner_ptr, report_ptr;
    {
	Arg args[10];
	Cardinal n = 0;
	XawPannerReport *report = (XawPannerReport *)report_ptr;
	Widget panner = (Widget)panner_ptr;

	XtSetArg(args[n], XtNsliderX, report->slider_x); n++;
	XtSetArg(args[n], XtNsliderY, report->slider_y); n++;
	if (report->changed != (XawPRSliderX | XawPRSliderY))
	    {
		XtSetArg(args[n], XtNsliderWidth, report->slider_width); n++;
		XtSetArg(args[n], XtNsliderHeight, report->slider_height); n++;
		XtSetArg(args[n], XtNcanvasWidth, report->canvas_width); n++;
		XtSetArg(args[n], XtNcanvasHeight, report->canvas_height); n++;
	    }
	XtSetValues (panner, args, n);
    }

static void PannerCallback(w, closure, report_ptr)
Widget w;
XtPointer closure, report_ptr;
    {
	Arg args[2];
	XawPannerReport *report = (XawPannerReport *)report_ptr;

	if (tags_visual == None) 
		return;
	XtSetArg (args[0], XtNx, -report->slider_x);
	XtSetArg (args[1], XtNy, -report->slider_y);
	XtSetValues(tags_visual, args, 2);
    }

#if NeedFunctionPrototypes
static void ChangeContentTag(Widget, XtPointer, XtPointer);
#else
static void ChangeContentTag();
#endif

static void AddTreeNode(node)
int node;
    {
	int parent_node = tags[node].down;
	Widget parent = tags[parent_node].tree;
	Arg args[2];
	Cardinal n;

	if (!parent && node)
	    {
		AddTreeNode(parent_node);
		parent = tags[parent_node].tree;
	    }
	n = 0;
	XtSetArg(args[n], XtNtreeParent, parent); ++n;
	tags[node].tree = XtCreateManagedWidget
		(tags[node].tag->name,commandWidgetClass,tags_visual,args,n);
	XtAddCallback(tags[node].tree, XtNcallback, 
		      ChangeContentTag, (XtPointer)node);
    }

static Widget BuildVisualTagsTree()
    {
	int i;
	
	for (i = tags_used; ; --i)
	    {
		if (tags[i].tree == None)
			AddTreeNode(i);
		if (!i)
			break;
	    }
    }

static void ShowTagsTree(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
    {
	static Widget shell = None;
	Widget box, panner, porthole, pane;
	
	if (!shell)
	    {
		shell = XtCreatePopupShell
			("tree", transientShellWidgetClass,XtParent(w),NULL,0);
		pane = XtCreateManagedWidget
			("treePane", panedWidgetClass, shell, NULL, 0);
		box = XtCreateManagedWidget
			("treeBox", boxWidgetClass, pane, NULL, 0);
		panner = XtCreateManagedWidget
			("treePanner", pannerWidgetClass, box, NULL, 0);
		porthole = XtCreateManagedWidget
			("treePorthole", portholeWidgetClass, pane, NULL, 0);
		XtAddCallback(porthole, XtNreportCallback,
			      PortholeCallback, (XtPointer)panner);
		XtAddCallback(panner, XtNreportCallback,
			      PannerCallback, (XtPointer)porthole);
		tags_visual = XtCreateManagedWidget
			("tagsTree", treeWidgetClass, porthole, NULL, 0);
	    }
	BuildVisualTagsTree();
	if (XtIsRealized(shell))
		XtPopup(shell, XtGrabNone);
	else
		UI_PopupCentered(NULL, shell, XtGrabNone);
    }
#endif

/*
** GetInitialState
**	corresponding to the tag index.
**
**	Recursive implementation just for kicks... --msa (GO LISP!)
*/
static char *GetInitialState(tag, need)
int tag, need;
    {
	static char *state;
	static int size;

	if (tag == 0)
	    {
		if (default_tag.rendition)
			need += strlen(default_tag.rendition);
		if (size <= need)
		    {
			if (state)
				XtFree(state);
			size = need+1;
			state = (char *)XtMalloc(size);
		    }
		if (default_tag.rendition)
			strcpy(state, default_tag.rendition);
		return state;
	    }
	else if (tags[tag].tag->rendition &&
		 (need += strlen(tags[tag].tag->rendition)) > 0)
		return strcat(GetInitialState(tags[tag].down, need),
			      tags[tag].tag->rendition);
	else
		return GetInitialState(tags[tag].down, need);
    }

static void InsertText(w, text, length)
Widget w;
char *text;
int length;
    {
	if (length > 0)
	    {
		int tag = (int)XeTextGetInsertionTag(edit_widget);

		if (tag != tags_index)
		    {
			long p = XeTextGetInsertionPoint(edit_widget);
			char *m = GetInitialState(tags_index, 0);
			XeTextReplaceTagged
				(edit_widget, p, p, m,
				 m ? strlen(m) : 0, (XeTextTag)tags_index);
		    }
		XeTextInsert(w, text, length);
	    }
    }

static void InsertChar(w, c)
Widget w;
int c;
    {
	char b = c;
	InsertText(w, &b, 1);
    }

static void InsertString(w, s, e)
Widget w;
char *s, *e;
    {
	InsertText(w, s, e - s);
    }

/*
** ExpandTags
**	Expand (or create) tags vector.
*/
static void ExpandTags()
    {
	if (tags == NULL)
	    {
		static Tag init_tag;

		/* Initialize tags */
		tags_size = TAGS_INITIAL_SIZE;
		tags = (Tag *)XtMalloc(tags_size * sizeof(Tag));
		tags[0] = init_tag;
		tags[0].tag = &default_tag;
	    }
	else
	    {
		tags_size += TAGS_INCREASE_SIZE;
		tags = (Tag *)XtRealloc((char *)tags, sizeof(Tag)*tags_size);
	    }
    }

/*
** PushTag
**	Enter new tag level, create new if required
*/
static int PushTag(level, t)
int level;
TagDef *t;
    {
	int i;

	for (i = tags_used;; --i)
	    {
		if (i == 0)
		    {
			static Tag initial_tag;

			++tags_used;
			while (tags_used >= tags_size)
				ExpandTags();
			tags[tags_used] = initial_tag;
			tags[tags_used].tag = t;
			tags[tags_used].down = level;
			i = tags_used;
			break;
		    }
		else if (tags[i].tag == t && tags[i].down == level)
			break;
	    }
	return i;
    }

/*
** PopTag
**	Popup tags until matching start tag found
*/
static int PopTag(level, t)
int level;
TagDef *t;
    {
	TagDef *r;

	while (level)
	    {
		r = tags[level].tag;
		level = tags[level].down;
		if (r == t)
			break;
	    }
	return level;
    }

/*
** FindTagDef
**	Locate the tag description that matches the name. If not found,
**	create a new empty description for it and add it to the known
**	tags list.
**
**	A special feature:
**		If the name is NULL or empty string, then returned
**		tag definition will be the default_tag.
*/
static TagDef *FindTagDef(name)
char *name;
    {
	TagDef *t;

	if (name == NULL || *name == 0)
		return &default_tag;

	for (t = known_tags; ;t = t->next)
		if (t == NULL)
		    {
			t = (TagDef *)calloc(1, sizeof(TagDef));
			if (t == NULL || (t->name = strdup(name)) == NULL)
				exit(1);
			if (known_last == NULL)
				known_tags = t;
			else
				known_last->next = t;
			t->next = NULL;
			known_last = t;
			break;
		    }
		else if (t->name && strcmp(t->name, name) == 0)
			break;
	return t;
    }

/*
** FindTagClass
**	Locaate the tag class description that matches the name. If not
**	found, create a new empty description and add ti to the known
**	tags list.
*/
static TagClass *FindTagClass(name)
char *name;
    {
	TagClass *class;
	for (class = tag_classes; ; class = class->next)
		if (class == NULL)
		    {
			class = (TagClass *)calloc(1, sizeof(TagClass));
			if (class == NULL ||
			    (class->name = strdup(name)) == NULL)
				exit(1);
			if (last_class == NULL)
				tag_classes = class;
			else
				last_class->next = class;
			class->next = NULL;
			last_class = class;
			break;
		    }
		else if (class->name && strcmp(class->name, name) == 0)
			break;
	return class;
    }

static void ProcessTag(w, s, e)
Widget w;
char *s, *e;
    {
	TagDef *t;
	int endtag, i;

	*e = 0;
	if (*s == '/')
	    {
		endtag = 1;
		s++;
	    }
	else
		endtag = 0;
	t = FindTagDef(s);
	if (endtag)
		tags_index = PopTag(tags_index, t);
	else
		tags_index = PushTag(tags_index, t);
    }

/*
** ParseTag
*/
static char *ParseTag(w, p)
Widget w;
char *p;
    {
	char *tag = p;
	register int c;

	c = *p++;
	if (c == '<')
		InsertChar(w, '<');
	else
	    {
		for ( ;c != '>'; c = *p++)
		    {
			if (c == '\n')
				return p-1; /* Invalid tag, ignore! */
			p[-1] = isupper(c) ? tolower(c) : c;
		    }
		ProcessTag(w, tag, p-1);
	    }
	return p;
    }

/*
** InsertContent
**	Insert text/enriched content from the file to the TextEd
**	widget.
*/
static void InsertContent(w, fp)
Widget w;
FILE *fp;
    {
	char buf[200];
	char *p, *r;
	register int c, f;

	while (fgets(buf, sizeof(buf), fp) != NULL)
	    {
		buf[strlen(buf)-1] = '\n';
		for (r = p = buf, f = 0; (c = *p++) != '\n';)
		    {
			if (c == '<')
			    {
				InsertString(w, r, p - 1);
				r = p = ParseTag(w, p);
			    }
		    }
		if (p == buf+1)
			InsertChar(w, '\n');
		else
		    {
			InsertString(w, r, p - 1);
			InsertChar(w, ' ');
		    }
	    }
    }


/*
** SetFilenameToTitle
*/
static void SetFilenameToTitle(name)
char *name;
    {
	Arg args[1];

	if (name == NULL)
		name = "(noname)";

	if (edit_file)
		free(edit_file);
	edit_file = strdup(name);
	XtSetArg(args[0], XtNlabel, edit_file);
	XtSetValues(edit_title, args, (Cardinal)1);
    }

/*
** LoadContent
**	Reload new content to the edit widget from a file.
*/
static void LoadContent(name)
char *name;
    {
	Cardinal n;
	Arg args[4];
	FILE *fp;

	/*
	** A bit complex way to clean out the old content. does this mean
	** there should be a special function to clear the content? --msa
	*/
	n = 0;
	XtSetArg(args[n], XtNcontentFile, NULL); ++n;
	XtSetArg(args[n], XtNcontentStream, NULL); ++n;
	XtSetArg(args[n], XtNcontentString, NULL); ++n;
	XtSetArg(args[n], XtNcontentLoaded, FALSE); ++n;
	XtSetValues(edit_widget, args, n);
	/*
	** Load new content through the parser
	*/
	tags_used = 0;
	tags_index = 0;
	if ((fp = fopen(name, "r")) != NULL)
	    {
		InsertContent(edit_widget, fp);
		fclose(fp);
	    }
	else
		printf("Cannot open %s.\n", name);
	SetFilenameToTitle(name);
    }

#define LINE_WIDTH 80

typedef struct SaveState
    {
	FILE *fp;
	int tag;
	char *ptr;
	char line[LINE_WIDTH+1];
    } SaveState;

static void WrapLine(save)
SaveState *save;
    {
	char *r = save->ptr;
	int n;

	while (r > save->line)
		if (*--r == ' ')
		    {
			*r++ = '\n';
			n = r - save->line;
			fwrite(save->line, 1, n, save->fp);
			n = save->ptr - r;
			if (n > 0)
				memcpy(save->line, r, n);
			save->ptr = &save->line[n];
			return;
		    }
	/*
	** Line contained no SPACE character, just flush it all out and
	** hope for the best.
	*/
	n = save->ptr - save->line;
	fwrite(save->line, 1, n, save->fp);
	save->ptr = save->line;
    }


/*
** WriteTag
**	generate opening (end == 0) or closing (end != 0) tag
**	to the output file.
*/
static void WriteTag(save, name, end)
SaveState *save;
char *name;
int end;
    {
	int taglen = strlen(name);

	while (LINE_WIDTH <= (save->ptr-save->line) + taglen+2+(end != 0))
		WrapLine(save);
	*save->ptr++ = '<';
	if (end)
		*save->ptr++ = '/';
	strcpy(save->ptr, name);
	save->ptr += taglen;
	*save->ptr++ = '>';
    }

static void OpenTags(save, start, end)
SaveState *save;
int start, end;
    {
	if (start != end)
	    {
		OpenTags(save, start, tags[end].down);
		WriteTag(save, tags[end].tag->name, 0);
	    }
    }

static int SaveContentSegment(s, n, tag, client_data)
char *s;
int n;
int tag;
XtPointer client_data;
    {
	SaveState *save = (SaveState *)client_data;
	int t, p;
	char *q;
	/*
	** Locate the first common ancestor of the current tag
	** (save->tag) and the new tag. When found, generate
	** closing tags up to this ancestor from current tag, and
	** then generate opening tags up to the new tag.
	*/
	for (t = save->tag; t; t = tags[t].down)
		for (p = tag; p; p = tags[p].down)
			if (t == p)
				goto found;
	t = 0;
    found:
	/* t is the common ancestor (or ZERO) */
	for (p = save->tag; p != t; p = tags[p].down)
		WriteTag(save, tags[p].tag->name, 1);
	OpenTags(save, t, tag); /* ..generate in reverse order */
	save->tag = tag;
	while (n > 0)
	    {
		while ((save->ptr - save->line) >= (LINE_WIDTH-1))
			WrapLine(save);
		if (*s == '<' || (*s == '\n' && save->ptr > save->line))
			*save->ptr++ = *s;
		*save->ptr++ = *s;
		if (*s++ == '\n')
		    {
			fwrite(save->line,1,(save->ptr-save->line),save->fp);
			save->ptr = save->line;
		    }
		--n;
	    }
    }

/*
** SaveContent
**	Save the content of the widget into a file
*/
static void SaveContent(name)
char *name;
    {
	SaveState save;
	save.tag = 0;
	if ((save.fp = fopen(name, "w")) != NULL)
	    {
		save.ptr = save.line;
		XeTextExtract(edit_widget,
			      (long)0, (long)LONG_MAX,
			      SaveContentSegment,
			      (XtPointer)&save);
		/* Flush out tags to "base level" */
		SaveContentSegment((char *)NULL, 0, 0, (XtPointer)&save);
		if (save.ptr - save.line > 0)
			fwrite(save.line,1,(save.ptr-save.line),save.fp);
		fclose(save.fp);
	    }
	else
		printf("Cannot open %s for output.\n", name);
	SetFilenameToTitle(name);
    }

static void SaveToFile(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
    {
	SaveContent(edit_file);
    }

static Position AdjustCoord(x, y, a, b)
int x, a, b;
    {
	if (x < a)
		a = x;
	else if (x + y >= a + b)
		a = x + y - b;
	return a;
    }

/*
** Sample Notify skeleton..
*/
static void Notify(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
    {
	XeNotifyCallbackData *notify = (XeNotifyCallbackData *)call_data;
	XeNotifyAreaCallbackData *area;
	int i;
	Position x, y;

	switch (notify->reason)
	    {
	    case XeCR_NOTIFY:
		/* Quickhack: just assume it's a button event! */
		printf("Notify(");
		for (i = 0; i < notify->num_params; i++)
			printf("%s%s", i==0?"":",", notify->params[i]);
		printf(") at (%d,%d)\n",
		       (int)notify->event->xbutton.x,
		       (int)notify->event->xbutton.y);
		break;
	    case XeCR_NOTIFY_AREA:
		area = (XeNotifyAreaCallbackData *)call_data;
		x = AdjustCoord(area->area.x, area->area.width,
				area->visible.x, area->visible.width);
		y = AdjustCoord(area->area.y, area->area.height,
				area->visible.y, area->visible.height);
#if X11R5
		if (x != area->visible.x || y != area->visible.y)
			XawViewportSetCoordinates(XtParent(w), x, y);
#endif
		break;
	    default:
		printf("UNKNOWN: XeRC=%d\n");
		break;
	    }
    }

/*
** File dialog is copied and modified from X11R5 editres client code!
*/
static void  PopdownFileDialog(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
    {
	Widget dialog = XtParent(w);
	
	if ((FileMenuAction)client_data)
	    {
		String filename = XawDialogGetValueString(dialog);
		switch ((FileMenuAction)client_data)
		    {
		    case FileMenu_SaveAs:
			SaveContent(filename);
			break;
		    case FileMenu_LoadFile:
			LoadContent(filename);
			break;
		    default:
			break;
		    }
		SetFilenameToTitle(filename);
	    }
	UI_Popdown(w, client_data, call_data);
    }

static UI_DialogDef popup_dialog_saveas =
    {
	"File name: ",
	"",
	PopdownFileDialog,
	(XtPointer)FileMenu_SaveAs,
    };

static UI_DialogDef popup_dialog_loadfile =
    {
	"File name: ",
	"",
	PopdownFileDialog,
	(XtPointer)FileMenu_LoadFile,
    };

static UI_MenuItemDef file_menu_items[] =
    {
	{"Save", SaveToFile, (XtPointer)NULL,},
	{"Save as", UI_PopupDialog, (XtPointer)&popup_dialog_saveas,},
	{"Load File", UI_PopupDialog, (XtPointer)&popup_dialog_loadfile,},
	{"Quit", UI_Quit, NULL},
	{NULL},
    };

static UI_MenuDef file_menu =
    {
	"File",
	file_menu_items,
    };

static void WarningHandler(name, type, class, defaultp, params, num_params)
String name, type, class, defaultp, *params;
Cardinal *num_params;
    {
	static UI_DialogDef warning =
	    {
		NULL,
		NULL,
		UI_Popdown,
	    };

	UI_PopupMessage(root_widget, &warning,
			name, type, class, defaultp, params, num_params);
    }
/*
** Layout callback handling
*/

static int show_tags;

static void ToggleShowTags(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
    {
	show_tags = !show_tags;
	XeTextLayout(edit_widget);
    }


static UI_MenuItemDef view_menu_items[] =
    {
	{"Toggle show tags", ToggleShowTags, (XtPointer)NULL},
#if X11R5
	{"Show/Refresh tags tree", ShowTagsTree, (XtPointer)NULL},
#endif
	{NULL},
    };


static UI_MenuDef view_menu =
    {
	"View",
	view_menu_items,
    };


static void LayoutFeedBegin(layout, start, end)
XeTextLayoutCallbackData *layout;
int start, end;
    {
	char *s;

	if (start != end)
	    {
		LayoutFeedBegin(layout, start, tags[end].down);
		if (show_tags && (s = tags[end].tag->name) != NULL)
		    {
			(*layout->feed)(layout->context, "<", (long)1);
			(*layout->feed)(layout->context, s, (long)strlen(s));
			(*layout->feed)(layout->context, ">", (long)1);
		    }
		if ((s = tags[end].tag->begin) != NULL && *s)
			(*layout->feed)(layout->context, s, (long)strlen(s));
	    }
    }

static void LayoutFeedChange(layout, old, new)
XeTextLayoutCallbackData *layout;
int old, new;
    {
	char *s;
	int i, j;

	/*
	** Move 'old' to be an ancestor of 'new', each move indicates
	** a closing tag. Feed end string to the layout process.
	*/
	for (i = old; i; i = tags[i].down)
		for (j = new; j; j = tags[j].down)
			if (i == j)
				goto found;
	i = 0;
    found:
	/* i is the common ancestor (or zero) */
	for (j = old; j != i; j = tags[j].down)
	    {
		if ((s = tags[j].tag->end) != NULL && *s)
			(*layout->feed)(layout->context, s, (long)strlen(s));
		if (show_tags && (s = tags[j].tag->name) != NULL)
		    {
			(*layout->feed)(layout->context, "</", (long)2);
			(*layout->feed)(layout->context, s, (long)strlen(s));
			(*layout->feed)(layout->context, ">", (long)1);
		    }
	    }
	LayoutFeedBegin(layout, i, new);
    }

static void LayoutFeedLine(layout, i)
XeTextLayoutCallbackData *layout;
int i;
    {
	char *s;

	if (i > 0)
		LayoutFeedLine(layout, tags[i].down);
	if ((s = tags[i].tag->line) && *s)
		(*layout->feed)(layout->context, s, (long)strlen(s));
    }

static void Layout(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
    {
	XeTextLayoutCallbackData *layout =
		(XeTextLayoutCallbackData *)call_data;

	if (layout->reason == XeCR_LAYOUT_CHANGE_TAG)
		LayoutFeedChange(layout, (int)layout->old, (int)layout->new);
	else if (layout->reason == XeCR_LAYOUT_BEGIN_LINE)
		LayoutFeedLine(layout, (int)layout->new);
    }

/*
** ChangeState*
**	ChangeState data structures are used to store temporarily all
**	of the extracted content. This is currently necessary for two
**	reasons: 1) Doing inserts into TextEd widget while it is
**	extracting content may put extract into loop (if inserting
**	into extract area), 2) when inserting the text back, this
**	implementation uses logic that needs to recompute new tag
**	indexes using the lowest extracted tag index in the tree
**	as a base value.
*/
typedef struct ChangeStateSegment
    {
	int tag;	/* Tag for the segment */
	int length;	/* Length of the segment */
    } ChangeStateSegment;

typedef struct ChangeStateData
    {
	struct ChangeStateData *next;
	int size;		      /* allocated UNITS in data */
	ChangeStateSegment data[999]; /* ..actual space dynamically decided */
    } ChangeStateData;

typedef struct ChangeState
    {
	int ancestor;	/* Common ancestor index for all inluded segments */
	ChangeStateData *first, *last;
	int available;	/* Availabe units on current block */
	char *data_ptr;	/* Data loading pointer on current block */
	ChangeStateSegment *seg_ptr; /* Segment loading pointer */
    } ChangeState;

#define UNIT_SIZE sizeof(ChangeStateSegment)
#define DEFAULT_SIZE 100

static int ChangeContentSegment(s, n, tag, client_data)
char *s;
int n, tag;
XtPointer client_data;
    {
	ChangeState *change = (ChangeState *)client_data;
	int request, i;

	if (n <= 0)
		return 0;	/* Ignore empty segments (if any) */
	/*
	** Check if the current data block is sufficient for the
	** segment and allocate a new data block if not.
	** (The space allocation is tested without caring whether this
	** tag can be combined with previous or not. If concatenation
	** occurs, the extra space is available for the next segment).
	*/
	request = (n + 2 * UNIT_SIZE) / UNIT_SIZE;
	if (change->available < request)
	    {
		ChangeStateData *new;
		int size = request > DEFAULT_SIZE ? request : DEFAULT_SIZE;
		new = (ChangeStateData *)
			malloc(XtOffsetOf(ChangeStateData, data[0]) +
			       (size+1) * UNIT_SIZE);
		if (change->last)
		    {
			/* Add terminator segment. There is always room
			   for this (size+1 allocated!) */
			change->seg_ptr -= 1;
			change->seg_ptr->tag = 0;
			change->seg_ptr->length = 0;
			change->last->next = new;
		    }
		else
			change->first = new;
		new->next = NULL;
		new->size = size+1;
		change->last = new;
		change->seg_ptr = &new->data[size];
		change->data_ptr = (char *)&new->data[0];
		change->seg_ptr->tag = tag;
		change->seg_ptr->length = 0;
		change->available = size;
	    }
	if (change->seg_ptr->tag != tag)
	    {
		change->seg_ptr -= 1;
		change->seg_ptr->tag = tag;
		change->seg_ptr->length = 0;
	    }
	memcpy((void *)change->data_ptr, (void *)s, n);
	change->seg_ptr->length += n;
	change->data_ptr += n;
	change->available =
		((char *)change->seg_ptr - change->data_ptr) / UNIT_SIZE;
	/*
	** Maintain the common ancestor
	*/
	if (change->ancestor < 0 || tag == 0)
		change->ancestor = tag;
	else for (i = tag; change->ancestor;)
	    {
		if (i == 0)
		    {
			i = tag;
			change->ancestor = tags[change->ancestor].down;
		    }
		else if (i == change->ancestor)
			break;	/* Common found */
		i = tags[i].down;
	    }
    }

static int MakeTags(base, start, end)
int base, start, end;
    {
	while (start != end)
		if ((tags[base].tag->option == TagOption_override ||
		     tags[base].tag->option == TagOption_inclusive) &&
		    tags[base].tag->class == tags[end].tag->class)
			end = tags[end].down;
		else
			return PushTag(MakeTags(base, start, tags[end].down),
				       tags[end].tag);
	return base;
    }


static void ChangeContentTag(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
    {
	TagDef *tag = (TagDef *)client_data;
	unsigned int tag_index = (unsigned int)client_data;
	long start, end, length;
	int level, i;
	ChangeState change;
	ChangeStateData *data;
	char *m;

	XeTextGetSelectionPosition(edit_widget, &start, &end);
	if (end <= start)
	    {
		start = XeTextGetInsertionPoint(edit_widget);
		if (tag_index > tags_used)
		    {
			i = (int)XeTextGetInsertionTag(edit_widget);
			level = tag ? PushTag(i, tag) : i;
		    }
		else
			level = tag_index;
		m = GetInitialState(level, 0);
		XeTextReplaceTagged
			(edit_widget, start, start, m,
			 m ? strlen(m):0, (XeTextTag)level);
	    }
	else if (tag_index > tags_used) /* Kludgy test!! not portable --msa */
		/* ..the test assumes that the dynamically allocated addresses
		   converted to unsigned will always be something larger
		   than max number of used tags... */
	    {
		change.ancestor = -1;	/* None yet found */
		change.first = NULL;
		change.last = NULL;
		change.available = 0;
		change.data_ptr = NULL;
		change.seg_ptr = NULL;
		XeTextExtract(edit_widget, start, end,
			      ChangeContentSegment, (XtPointer)&change);
		if (change.last)
		    {
			/* Add last terminator */
			change.seg_ptr -= 1;
			change.seg_ptr->length = 0;
			change.seg_ptr->tag = 0;
		    }
		if (change.ancestor == -1)
			return;	/* Nothing extracted, all done */
		/*
		** Decide what the new base level tag is.
		*/
		level = change.ancestor;
		switch (tag->option)
		    {
		    case TagOption_flatten:
			break;
		    case TagOption_inclusive:
			for (i = level;; i = tags[i].down)
				if (i == 0)
				    {
					level = PushTag(level, tag);
					break;
				    }
				else if (tag->class == tags[i].tag->class)
					break;
			break;
		    case TagOption_override:
			if (tag->class == tags[level].tag->class)
				level = PushTag(tags[level].down, tag);
			else
				level = PushTag(level, tag);
			break;
		    default:
			level = PushTag(level, tag);
			break;
		    }
		for (data = change.first; data;)
		    {
			ChangeStateData *temp;
			char *s = (char *)&data->data[0];
			ChangeStateSegment *p = &data->data[data->size];

			while ((--p)->length)
			    {
				if (tag->option == TagOption_flatten)
					i = level;
				else
					i = MakeTags
						(level,change.ancestor,p->tag);
				m = GetInitialState(i, 0);
				XeTextReplaceTagged
					(edit_widget, start, end, m,
					 m ? strlen(m) : 0, (XeTextTag)i);
				XeTextInsert(edit_widget, s, p->length);
				s += p->length;
				start=end=XeTextGetInsertionPoint(edit_widget);
			    }
			temp = data->next;
			free((void *)data);
			data = temp;
		    }
	    }
	else
	    {
		String value;

		value = XeTextGetSubstring(edit_widget, &length, start, end);
		m = GetInitialState(tag_index, 0);
		XeTextReplaceTagged(edit_widget, start, end, m,
				    m ? strlen(m) : 0, (XeTextTag)tag_index);
		XeTextInsert(edit_widget, value, length);
		XtFree(value);
	    }
    }

/*
** Build a menu from the current known tags
*/
static UI_MenuDef *BuildTags()
    {
	TagDef *tag;
	int count, i;
	UI_MenuDef *menu;

#define FIXED_TAG_ENTRIES 2


	menu = (UI_MenuDef *)calloc(1, sizeof(UI_MenuDef));
	if (menu == NULL)
		return NULL;
	menu->name = "Tags";
	/*
	** Count the number of tags and allocate item definition
	** array accordingly. Note: There will always be some fixed
	** pre-allocated entries.
	*/
	for (count = FIXED_TAG_ENTRIES, tag = known_tags; tag; tag = tag->next)
		count++;
	menu->items = (UI_MenuItemDef *)calloc(count+1,sizeof(UI_MenuItemDef));
	if (menu->items == NULL)
		return menu;
	i = 0;
	/*
	** Generate fixed "None" entry.
	*/
	menu->items[i].name = "* None";
	menu->items[i].callback = ChangeContentTag;
	menu->items[i].client_data = NULL;
	++i;
	/*
	** Generate fixed "Flatten" entry.
	*/
	menu->items[i].name = "* Flatten";
	menu->items[i].callback = ChangeContentTag;
	menu->items[i].client_data = (XtPointer)&flatten_tag;
	flatten_tag.option = TagOption_flatten;
	++i;
	/*
	** Fill up the rest of menu elements. The item will directly
	** reference the TagDef entries--beware, do not delete or modify
	** TagDef without scrapping this generated menu also.
	*/
	for (tag = known_tags; i < count; ++i, tag = tag->next)
	    {
		menu->items[i].name = tag->name;
		menu->items[i].callback = ChangeContentTag;
		menu->items[i].client_data = (XtPointer)tag;
	    }
	menu->items[i].name = NULL; /* Item list Terminator */
	return menu;
    }

/*
** FieldNumber
**	defines the order and meaning of the fields in the
**	tag description line.
*/
typedef enum
    {
	Field_TAG,	/* Name of the tag */
	Field_OPTIONS,	/* Option flags */
	Field_CLASS,	/* Class, grouping of tags */
	Field_RENDITION,/* Rendition control string */
	Field_BEGIN,	/* Layout, begin tag control string */
	Field_LINE,	/* Layout, begin line control string */
	Field_END,	/* Layout, end tag control string */
	Field_MAX	/* Enumeration terminator, MAX # of fields */
    } FieldNumber;

	
/*
** ParseField
**	Parse single field from the configuration. This function returns
**	the TagDef pointer passed as an argument, except when the 'f'
**	parameter is Field_TAG: then the new TagDef is returned instead.
**
**	*Note*	The data delimited by [s, e) may get modified. The byte
**		pointed 'e' may get replaced by a NUL byte.
*/
static TagDef *ParseField(tag, f, ptr, e)
TagDef *tag;
int f;
char **ptr;
char *e;
    {
	char *s, *r, *p;
	int c;

	s = *ptr;

	/* Strip off leading and trailing white space */
	while (s < e && (*s == ' ' || *s == '\t'))
		s++;
	r = p = s;
	if (s < e && *s == '"')
	    {
		/* Parse quoted string parameter */
		for (p++; p < e && (c = *p++) != '"' ; *r++ = c)
		    {
			if (p < e && c == '\\')
				if ((c = *p++) == 'n')
					c = '\n';
				else if (c >= '0' && c < '8')
				    {
					int v, i;

					for (v = 0, i = 0;;)
					    {
						v = v * 8 + (c - '0');
						i += 1;
						if (p >= e || i == 3 ||
						    *p < '0' || *p >= '7')
							break;
						c = *p++;
					    }
					c = v;
				    }
				else if (c == 't')
					c = '\t';
				else if (c == 'r')
					c = '\r';
				else if (c == 'b')
					c = '\b';
		    }
		*r = 0;
		while (p < e && *p != ':')
			++p;
	    }
	else
	    {
		while (p < e && *p != ':')
			++p;
		for (r = p; s < r;)
			if (*--r != ' ' && *r != '\t')
			    {
				++r;
				break;
			    }
		*r = 0;
	    }
	*r = 0;
	*ptr = p < e ? p + 1 : e;
	switch (f)
	    {
	    case Field_TAG:
		tag = FindTagDef(s);
		break;
	    case Field_OPTIONS:
		while (*s)
			switch (*s++)
			    {
			    case 'a':
				tag->option = TagOption_add;
				break;
			    case 's':
				tag->option = TagOption_subtract;
				break;
			    case 'i':
				tag->option = TagOption_inclusive;
				break;
			    case 'o':
				tag->option = TagOption_override;
				break;
			    case 'E':
				tag->newline = NewLineOption_enriched;
				break;
			    case 'H':
				tag->newline = NewLineOption_hard;
				break;
			    default:
				break;
			    }
		break;
	    case Field_CLASS:
		tag->class = FindTagClass(s);
		break;
	    case Field_RENDITION:
		tag->rendition = *s ? strdup(s) : NULL;
		break;
	    case Field_BEGIN:
		tag->begin = *s ? strdup(s) : NULL;
		break;
	    case Field_LINE:
		tag->line = *s ? strdup(s) : NULL;
		break;
	    case Field_END:
		tag->end = *s ? strdup(s) : NULL;
		break;
	    default:
		break;
	    }
	return tag;
    }

/*
** LoadTags
**	Read in tag definitions file.
*/
static void LoadTags(name)
char *name;
    {
	char buf[200];
	TagDef *tag;

	FILE *fp = fopen(name, "r");
	if (fp == NULL)
	    {
		printf("Cannot open tag description file: %s\n", name);
		return;
	    }
	while (fgets(buf, sizeof(buf), fp) != NULL)
	    {
		int f, c;
		char *p;

		int n = strlen(buf);
		if (n < 2 || buf[0] == '#')
			continue;	/* Skip empty lines and comments */
		if (buf[n-1] == '\n')
			n -= 1;
		f = Field_TAG;
		tag = NULL;
		for (p = buf; f < Field_MAX ; f += 1)
			tag = ParseField(tag, f, &p, &buf[n]);
	    }
	fclose(fp);
    }

#include "fallback.c"

main(argc, argv)
int argc; char *argv[];
    {
	XtAppContext demo_application;
	Widget view, widget, pane, box, button;
	Arg args[10];
	int n;

	root_widget = XtAppInitialize(&demo_application, "Enriched",
				      NULL,0, &argc, argv,
				      fallback_resources,
				      NULL, 0);

	ExpandTags(); /* Initialize array */
	LoadTags("enriched.tag");
	n = 0;
	pane = XtCreateManagedWidget
		("pane", panedWidgetClass, root_widget, args, n);
	n = 0;
	box = XtCreateManagedWidget
		("box", boxWidgetClass, pane, args, n);

	UI_CreateMenu(box, &file_menu);
	UI_CreateMenu(box, BuildTags());
	UI_CreateMenu(box, &view_menu);
	
	n = 0;
	XtSetArg(args[n], XtNborderWidth, 0); ++n;
	edit_title = XtCreateManagedWidget("fileName", labelWidgetClass,
					  box, args, n);

	XtAppSetWarningMsgHandler(demo_application, WarningHandler);
	n = 0;
	XtSetArg(args[n], XtNallowHoriz, True); ++n;
	XtSetArg(args[n], XtNallowVert, True); ++n;
	XtSetArg(args[n], XtNshowGrip, False); ++n;
	view = XtCreateManagedWidget("demoEnriched", viewportWidgetClass,
				     pane,args, n);
	n = 0;
	/*
	** Request proportional spacing, because we are not
	** specifying the lineSpacing.
	*/
	XtSetArg(args[n], XtNproportional, True); ++n;
	XtSetArg(args[n], XtNcolumnWidth, 400); ++n;
	XtSetArg(args[n], XtNformat, True); ++n;
	XtSetArg(args[n], XtNexportFormat, XeTextExport_STRING); ++n;
	edit_widget = XtCreateManagedWidget
		("enrichedEdit", xeTextEdWidgetClass, view, args, n);
	XtAddCallback(edit_widget, XtNnotifyCallback, Notify, (XtPointer)NULL);
	XtAddCallback(edit_widget, XtNlayoutCallback, Layout, (XtPointer)NULL);
	if (argc > 1)
		LoadContent(argv[1]);
	else
		SetFilenameToTitle((char *)NULL);
	XtRealizeWidget(root_widget);
	XtAppMainLoop(demo_application);
	return 0;
    }
