/*
** Copyright 1992, 1993 by Markku Savela and
**	Technical Research Centre of Finland
*/
#include <stdio.h>
#if SYSV_INCLUDES
#	include <memory.h>

extern int atoi( /* No header for this in SYSV */
#if NeedFunctionPrototypes
		char *
#endif
		);

#endif
#if ANSI_INCLUDES
#	include <stddef.h>
#	include <stdlib.h>
#endif
#include <ctype.h>
#include <string.h>

#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xmu/CharSet.h>
#include <X11/Xew/XewInit.h>
#include <X11/Xew/TextP.h>
#include "TextImport.h"
#include "TextLayout.h"
#include "TextFont.h"


#define TEXT_MIN_WIDTH 1
#define TEXT_MIN_HEIGHT 1

#define offset(field) XtOffsetOf(XeTextRec, text.field)

static XtResource resources[] =
    {
	{
	    XtNalignment, XtCAlignment,
	    XtRXeAlignment, sizeof(XeAlignment), offset(alignment),
	    XtRImmediate, (XtPointer)XeAlignment_START,
	},
	{
	    XtNexportFormat, XtCExportFormat,
	    XtRXeTextExport, sizeof(XeTextExport), offset(export_format),
	    XtRImmediate, (XtPointer)XeTextExport_ODIF,
	},
	{XtNfirstLineOffset, XtCOffset, XtRInt, sizeof(int),
		 offset(first_line_offset), XtRImmediate, 0},
	{XtNfonts, XtCFonts, XtRXeFonts, sizeof(XeFont *),
	     offset(fonts), XtRImmediate, (XtPointer)NULL},
	{XtNformat, XtCFormat, XtRBoolean, sizeof(Boolean),
	     offset(format), XtRImmediate, (XtPointer)0},
	{XtNgraphicRendition, XtCRendition, XtRString, sizeof(String),
		 offset(graphic_rendition), XtRImmediate, (XtPointer)NULL},
	{XtNindentation, XtCIndentation, XtRDimension, sizeof(Dimension),
		 offset(indentation), XtRImmediate, (Dimension)0 },
	{XtNcolumnWidth, XtCColumnWidth, XtRDimension, sizeof(Dimension),
		 offset(column_width), XtRImmediate, (Dimension)0 },
	{XtNinitialState, XtCSequence,  XtRString, sizeof(String),
		 offset(initial_state), XtRImmediate, (XtPointer)NULL},
	{XtNitemization,XtCItemization, XtRXeItemization,
		 sizeof(XeItemization *),
		 offset(itemization), XtRImmediate, (XtPointer)NULL},
	{XtNkerningOffset, XtCKerningOffset, XtRKerningOffset,
		 sizeof(Dimension *),
		 offset(kerning_offset), XtRImmediate, (XtPointer)NULL},
	{XtNlayoutCallback, XtCLayoutCallback, XtRCallback, sizeof(XtPointer),
		 offset(layout_callbacks), XtRCallback, (XtPointer)NULL},
	{XtNlineLayoutTable, XtCLineLayoutTable, XtRXeLineLayoutTable,
		 sizeof(XeTabStop *),
		 offset(line_layout_table), XtRImmediate, (XtPointer)NULL},
	{XtNlineSpacing, XtCLineSpacing, XtRDimension, sizeof(Dimension),
		 offset(line_spacing), XtRImmediate, (Dimension)0},
	{XtNproportional, XtCProportional, XtRBoolean, sizeof(Boolean),
	     offset(proportional), XtRImmediate, (XtPointer)0},
    };
#undef offset

#if NeedFunctionPrototypes
static void FreeColors(XeTextWidget);
#else
static void FreeColors();
#endif

static void ClassPartInitialize(), ClassInitialize();
static void Initialize(), Redisplay(), Resize(), Destroy();
static Boolean SetValues();

XeTextClassRec xeTextClassRec = {
  { /* core fields */
    /* superclass		*/	(WidgetClass) &xeBasicClassRec,
    /* class_name		*/	"XeText",
    /* widget_size		*/	sizeof(XeTextRec),
    /* class_initialize		*/	ClassInitialize,
    /* class_part_initialize	*/	ClassPartInitialize,
    /* class_inited		*/	FALSE,
    /* initialize		*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	XtInheritRealize,
    /* actions			*/	NULL,
    /* num_actions		*/	0,
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	XtExposeCompressMultiple,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	Resize,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	XtInheritTranslations,
    /* query_geometry		*/	XtInheritQueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
    /* extension		*/	NULL
  },
  { /* XeBasic fields */
    /* not used			*/	0
  },
  { /* XeText fields */
    /* not used			*/	0
  }
};

WidgetClass xeTextWidgetClass = (WidgetClass)&xeTextClassRec;


static CvtEnumInfo cvtAlignment[] =
    {
	{"none",	XeAlignment_NONE },
	{"start",	XeAlignment_START },
	{"end",		XeAlignment_END },
	{"center",	XeAlignment_CENTER },
	{"justified",	XeAlignment_JUSTIFIED },
	{"around",	XeAlignment_AROUND },
	{NULL,		sizeof(XeAlignment) },
    };

static CvtEnumInfo cvtExportFormat[] =
    {
	{"string",	XeTextExport_STRING },
	{"string-f",	XeTextExport_STRING_F },
	{"odif",	XeTextExport_ODIF },
	{"odif-f",	XeTextExport_ODIF_F },
	{"odif-fp",	XeTextExport_ODIF_FP },
	{NULL,		sizeof(XeTextExport) },
    };


#define	CvtDone(type, value)				\
    do {						\
	if (toVal->addr != NULL)			\
	    {						\
		if (toVal->size < sizeof(type))		\
		    {					\
			toVal->size = sizeof(type);	\
			return False;			\
		    }					\
		*(type*)(toVal->addr) = (value);	\
	    }						\
	else						\
	    {						\
		static type static_val;			\
		static_val = (value);			\
		toVal->addr = (XPointer)&static_val;	\
	    }						\
	toVal->size = sizeof(type);			\
	return True;					\
    } while (0)


/*
** XeFontsDestructor
*/
static void XeFontsDestructor(app, to, converter_data, args, num_args)
XtAppContext app;
XrmValue *to;
XtPointer converter_data;
XrmValue *args;
Cardinal *num_args;
    {
	XtFree((char *)converter_data);
    }

/*
** XeCvtStringToFonts
**	Convert a string into Fonts (XeFont *)
**
** <fonts>	::= <font> [ ";" <fonts> ]
** <font>	::= <size> [ <fontlist> ]
** <fontlist>	::= <fontstring> [ "=" <characterset> ]
** <fontstring>	::= <Any X11 font specification string>
** <size>	::= <positive integer>
** <characterset> ::= <Any supported X11 characterset name>
*/
static Boolean XeCvtStringToFonts
	(dpy,args,num_args,fromVal,toVal,converter_data)
Display *dpy;
XrmValue *args;
Cardinal *num_args;
XrmValue *fromVal, *toVal;
XtPointer *converter_data;
    {
	XeFont *f;
	XeFont fonts[10];
	char *s = (char *) fromVal->addr;
	char *r;
	int i, j;
	XeFontList font_list = NULL;

	for (i = 0; i < XtNumber(fonts); )
	    {
		/* Extract size */

		while (*s == ' ')
			s += 1;
		fonts[i].font_size = atoi(s);
		while (isdigit(*s))
			s += 1;
		while (*s == ' ')
			s += 1;
		/*
		** Extract font list. If the font list portion is
	        ** omitted, use the same list as was specified to
		** the previous entry
		*/
		r = strchr(s, ';');
		j = r ? r - s : strlen(s);
		if (j > 0)
			font_list = XeMakeFontList(s, j);
		fonts[i].font_list = font_list;
		i += 1;
		if (r == NULL)
			break;
		s = r + 1;
	    }
	/*
	** Allocate space for all entries and for a sentinel. Initialize.
	*/
	f = (XeFont *)XtMalloc((i+1) * sizeof(XeFont));
	memcpy((char *)f, (char *)fonts, i * sizeof(XeFont));
	f[i].font_size = 0;
	f[i].font_list = 0;
	CvtDone(XeFont *, f);
    }

/*
** XeItemizationDestructor
*/
static void XeItemizationDestructor(app, to, converter_data, args, num_args)
XtAppContext app;
XrmValue *to;
XtPointer converter_data;
XrmValue *args;
Cardinal *num_args;
    {
	XtFree((char *)converter_data);
    }

/*
** XeCvtStringToItemization
**	Convert a string into Itemization (XeItemization).
**
** <itemization>::= <alignment> [ "," [ <start> ] ["," <end>]]
** <alignment>	::= "no" | "start" | "end"
** <start>	::= <integer>
** <end>	::= <integer>
*/
Boolean XeCvtStringToItemization
	(dpy,args,num_args,fromVal,toVal,converter_data)
Display *dpy;
XrmValue *args;
Cardinal *num_args;
XrmValue *fromVal, *toVal;
XtPointer *converter_data;
    {
	XeItemization itemization;
	int need_allocate = True;
	char *s = (char *) fromVal->addr;
	char *r;
	char buf[100];
	int len;
	XrmValue toTmp, fromTmp;
	String params[1];
	Cardinal num_params;

	if (*num_args != 1)
	    {
		XtAppWarningMsg(XtDisplayToApplicationContext(dpy),
				"wrongParameters","XeCvtStringToItemization",
				"XewError",
		       "String to XeItemization conversion needs one argument",
				(String *)NULL, (Cardinal *)NULL);
		return False;
	    }
	if (converter_data == NULL)
	    {
		XtAppWarningMsg(XtDisplayToApplicationContext(dpy),
				"noConverterData","XeCvtStringToItemization",
				"XewError",
		       "String to XeItemization needs converter_data",
				(String *)NULL, (Cardinal *)NULL);
		return False;
	    }
	/*
	** Initialize default values
	*/
	itemization.identifier_alignment = XeAlignment_START;
	itemization.identifier_start_offset = - *((Dimension *) args[0].addr);
	itemization.identifier_end_offset = 0;
	for ( ;; ) /* Just to enable us to use 'break' below */
	    {
		/*
		** Extract <Alignment>
		*/
		if (s == NULL)
			break;
		r = strchr(s, ',');
		len = r ? r - s : strlen(s);
		if (len >= sizeof(buf))
			len = sizeof(buf) - 1;
		memcpy(buf, s, len);
		buf[len] = '\0';
		if (strcmp("no",buf) == 0 || strcmp("none", buf) == 0)
		    {
			need_allocate = False;
			break;
		    }
		else if (strcmp(buf, "start") == 0)
			itemization.identifier_alignment = XeAlignment_START;
		else if (strcmp(buf, "end") == 0)
			itemization.identifier_alignment = XeAlignment_END;
		else
		    {
			params[0] = (String)buf;
			num_params = 1;
			XtAppWarningMsg(XtDisplayToApplicationContext(dpy),
					"XeCvtStringToItemization",
					"unknownKeyword",
					"XewError",
			       "Cannot convert \"%s\" to identifier alignment",
					params,
					&num_params);
			return False;
		    }
		if (r == NULL)
			break;
		/*
		** Extract <start>
		*/
		s = r + 1;
		r = strchr(s, ',');
		len = r ? r - s : strlen(s);
		if (len >= sizeof(buf))
			len = sizeof(buf) - 1;
		memcpy(buf, s, len);
		buf[len] = '\0';
		fromTmp.size = len + 1;
		fromTmp.addr = (XtPointer)buf;
		toTmp.size = sizeof(itemization.identifier_start_offset);
		toTmp.addr = (XtPointer)&itemization.identifier_start_offset;
		if (!XtCallConverter(dpy, XeCvtStringToHorizontalValue,
				     (XrmValue *)NULL, 0,
				     &fromTmp,
				     &toTmp,
				     (XtCacheRef *)NULL))
			return False;
		if (r == NULL)
			break;
		/*
		** Extract <end>
		*/
		s = r + 1;
		len = strlen(s);
		if (len >= sizeof(buf))
			len = sizeof(buf) - 1;
		memcpy(buf, s, len);
		buf[len] = '\0';
		fromTmp.size = len + 1;
		fromTmp.addr = (XtPointer)buf;
		toTmp.size = sizeof(itemization.identifier_end_offset);
		toTmp.addr = (XtPointer)&itemization.identifier_end_offset;
		if (!XtCallConverter(dpy, XeCvtStringToHorizontalValue,
				     (XrmValue *)NULL, 0,
				     &fromTmp,
				     &toTmp,
				     (XtCacheRef *)NULL))
			return False;
		break; /* This was not really a loop! Fall through.. */
	    }

	/* Cannot use macro CvtDone because the memory should be
	   allocated to Itemization only on succesful return... --msa */

	if (toVal->addr != NULL)
	    {
		if (toVal->size < sizeof(XeItemization *))
		    {
			toVal->size = sizeof(XeItemization *);
			return False;
		    }
	    }
	else
	    {
		static XeItemization *value;
		toVal->addr = (XtPointer)&value;
	    }
	if (need_allocate)
	    {
		*converter_data = XtMalloc(sizeof(XeItemization));
		**(XeItemization **)converter_data = itemization;
	    }
	else
		*converter_data = NULL;
	*(XeItemization **)(toVal->addr) = (XeItemization *)(*converter_data);
	toVal->size = sizeof(XeItemization *);
	return True;
    }


/*
** XeCvtStringToLineLayoutTable
**	Convert a string into LineLayoutTable (XeTabStop *).
**
** <table>	::= <tabstop> [ "," <table> ]
** <tabstop>	::= <referencestring> "-" <position>
**			[ "-" <alignment> [ '-"' <around> '"' ]]
** <referencestring> ::= <string of one to four digits>
** <positition>	::= <non-negative-integer>
** <alignment>	::= "start" | "end" | "centred" | "around"
** <around>	::= <string of characters>
*/
void XeCvtStringToLineLayoutTable(args, num_args, fromVal, toVal)
XrmValue *args;
Cardinal *num_args;
XrmValuePtr fromVal, toVal;
    {
    }

static void ClassInitialize()
    {

	static XtConvertArgRec convertArg[] =
	    {
		{
		    XtResourceString,
		    (XtPointer) XtNindentation,
		    sizeof(Dimension),
		}
	    };
	static XtConvertArgRec convertAlignmentArg[] =
	    {
		{
		    XtAddress,
		    (XtPointer) &cvtAlignment[0],
		    sizeof(XtPointer),
		}
	    };
	static XtConvertArgRec convertExportFormatArg[] =
	    {
		{
		    XtAddress,
		    (XtPointer) &cvtExportFormat[0],
		    sizeof(XtPointer),
		}
	    };


	XewInitializeWidgetSet();
	XtAddConverter(XtRString, XtRLong, XmuCvtStringToLong, NULL, 0);

	XtSetTypeConverter(XtRString, XtRXeAlignment,
			   XeCvtStringToEnum,
			   convertAlignmentArg, XtNumber(convertAlignmentArg),
			   XtCacheNone,
			   (XtDestructor)NULL);
	XtSetTypeConverter(XtRString, XtRXeTextExport,
			   XeCvtStringToEnum,
			   convertExportFormatArg,
			   XtNumber(convertExportFormatArg),
			   XtCacheNone,
			   (XtDestructor)NULL);
	XtSetTypeConverter(XtRString, XtRXeFonts,
			   XeCvtStringToFonts,
			   NULL, 0,
			   XtCacheNone,
			   (XtDestructor)NULL);
	/*
	** Note: Itemization needs to know the current indentation.
	** (? How do we guarantee, that indentation has been computed
	** at the time this converter is used? --msa
	*/
	XtSetTypeConverter(XtRString, XtRXeItemization,
			   XeCvtStringToItemization,
			   convertArg, 1,
			   XtCacheAll | XtCacheRefCount,
			   XeItemizationDestructor);
	XtAddConverter(XtRString, XtRXeLineLayoutTable,
		       XeCvtStringToLineLayoutTable, NULL, 0);
    }

static void ClassPartInitialize(class)
WidgetClass class;
    {
    }

static void LoadFromStream(w, f)
XeTextWidget w;
FILE *f;
    {
	char buf[2000];
	int len;
	long cnt = w->basic.content_offset;

	if (cnt > 0 && fseek(f, cnt, 0))
	    {
		/*
		** For some reason fseek failed, just skip by reading.
		*/
		fseek(f, 0L, 0);	/* Just make sure we restart */
		for (;cnt > 0; cnt -= len)
		    {
			len = fread(buf,1,cnt>sizeof(buf)?sizeof(buf):cnt,f);
			if (len <= 0)
				return;
		    }
	    }
	cnt = w->basic.content_length;
	if (cnt == 0)
		while ((len = fread(buf,1,sizeof(buf),f)) > 0)
			_XeTextFeedContent(w->text.inserting, buf, len);
	else
		for ( ;cnt > 0; cnt -= len)
		    {
			len = fread(buf,1,cnt>sizeof(buf)?sizeof(buf):cnt,f);
			if (len <= 0)
				break;
			_XeTextFeedContent(w->text.inserting, buf, len);
		    }
    }

/*
** SetupContent
**	initialize widget state from the current resources. Returns
**	true, if redisplay is required.
*/
static int SetupContent(w, dolayout)
XeTextWidget w;
int dolayout;
    {
	int num;
	XeFont *f;
	XeTabStop *t;
	/*
	** Initialize num_fonts, num_tabs
	*/
	num = 0;
	if ((f = w->text.fonts) != NULL)
		for ( ; f->font_size > 0; f++)
			num += 1;
	w->text.num_fonts = num;
	num = 0;
	if ((t = w->text.line_layout_table) != NULL)
		for ( ;  t->tabulation_reference != NULL; t++)
			num += 1;
	w->text.num_tabs = num;

	/*
	** Content initialization (note that the defaults for content
	** related fields have been already dealt in Basic.c)
	*/
	if (w->basic.content_loaded == False)
	    {
		if (w->text.inserting)
			_XeTextEndContent(w->text.inserting,(Region)0);
		w->text.inserting = _XeTextBeginContent(w);
		if (w->basic.content_string != NULL)
			_XeTextFeedContent
				(w->text.inserting,
				 w->basic.content_string
				 + w->basic.content_offset,
				 w->basic.content_length);
		else if (w->basic.content_stream != NULL)
			LoadFromStream(w, w->basic.content_stream);
		else if (w->basic.content_file != NULL)
		    {
			FILE *fl = fopen(w->basic.content_file, "rb");
			if (fl != NULL)
			    {
				LoadFromStream(w, fl);
				fclose(fl);
			    }
			else
			    {
				String params[1];
				params[0] = w->basic.content_file;
				XeWidgetWarningMsg((Widget)w,
						   "cannotOpen",
						   "Cannot open file '%s'",
						   params, 1);
			    }
		    }
		_XeTextEndContent(w->text.inserting,(Region)0);
		w->text.inserting = NULL;
		w->basic.content_loaded = True;
		dolayout = True;
	    }
	FreeColors(w);
	if (dolayout)
		_XeTextLayout(w, (Snip *)NULL, (Region)0);
	if (w->basic.resize || w->core.width == 0)
		w->core.width =  w->text.w;
	if (w->basic.resize || w->core.height == 0)
		w->core.height = w->text.h;
	if (w->core.height < TEXT_MIN_HEIGHT)
		w->core.height = TEXT_MIN_HEIGHT;
	if (w->core.width < TEXT_MIN_WIDTH)
		w->core.width = TEXT_MIN_WIDTH;
	XSetState(XtDisplay(w),w->text.mygc, w->basic.foreground_pixel,
		  w->core.background_pixel, GXcopy, ~0L);
	XSetState(XtDisplay(w),w->text.mygcXOR, ~0L, 0, GXxor,
		  w->basic.foreground_pixel ^ w->core.background_pixel);
	return dolayout;
    }

static void Initialize (request, new)
Widget request, new;
    {
	XeTextWidget w = (XeTextWidget)new;

	if (!w->text.alignment)
		w->text.alignment = XeAlignment_START;
	w->text.mygc  =
		XCreateGC(XtDisplay(w), DefaultRootWindow(XtDisplay(w)), 0, 0);
	w->text.mygcXOR =
		XCreateGC(XtDisplay(w), DefaultRootWindow(XtDisplay(w)), 0, 0);
	XSetLineAttributes(XtDisplay(w), w->text.mygc,
			   1, LineSolid, CapButt, JoinMiter);
	w->text.w = w->text.h = 0;
	w->text.first = NULL;
	w->text.colors = NULL;
	w->text.inserting = NULL;
	if (w->text.initial_state)
		w->text.initial_state = XtNewString(w->text.initial_state);
	SetupContent(w, True);
    }

/*
**	EXPOSE HANDLING
**	===============
*/

/*
** CornerMark
**	remembers one corner of a rectangular Area.
*/
typedef enum
    {
	MARK_None,
	MARK_Single,
	MARK_Double,
	MARK_Crossed,
	MARK_Overlined,
	MARK_Framed,
	MARK_Encircled
    } MarkerId;

typedef struct CornerMark
    {
	MarkerId id;
	int x, y, h;
    } CornerMark;

static CornerMark markers[10];

static void MarkArea(w, id, x, y, h)
XeTextWidget w;
MarkerId id;
int x, y, h;
    {
	register CornerMark *p, *q = NULL;

	for (p = markers; p < &markers[XtNumber(markers)]; ++p)
		if (p->id == MARK_None)
			q = p;
		else if (p->id == id)
		    {
			/*
			** Found closing Marker, Draw Area
			*/
			p->id = MARK_None;
			switch (id)
			    {
			    case MARK_Framed:
				XDrawRectangle(XtDisplay(w), XtWindow(w),
					       w->text.mygc,
					       p->x , p->y, x - p->x, p->h);
				break;
			    default:
				XFillRectangle(XtDisplay(w), XtWindow(w),
					       w->text.mygc,
					       p->x , p->y, x - p->x, p->h);
				break;
			    }
			return;
		    }
	/*
	** Opening Marker, Store for later use
	*/
	if (h <= 0)
	    {
		String params[1];
		params[0] = (String)h;
		XeWidgetWarningMsg
			((Widget)w, "zeroHeightMarker",
			 "Opening marker with zero height (%d) marker",
			 params, 1);
	    }
	if (q != NULL)
	    {
		q->id = id;
		q->x = x;
		q->y = y;
		q->h = h;
	    }
	else
		XeWidgetWarningMsg((Widget)w, "markerStackOverflow",
				   "Marker stack overflow", (String *)NULL, 0);
    }

/*
** FindFontExtents
**	Find font extents for the current Snip (s). If the current
**	Snip doesn't have any font associated, look forward in the
**	Snip chain and use the first defined font. If no such is
**	found, return with ascent=0 and descent=0.
**
**	Using precomputed extents Snip is not suitable.
**	If used, should compute max over all included
**	Snips. Consider this later, for now use font extents
**	... --msa
*/
static void FindFontExtents(s, ascent, descent)
Snip *s;
int *ascent;
int *descent;
    {
	XCharStruct XE;
	int dir;

	*ascent = 0;
	*descent = 0;
	for ( ; s; s = s->next)
	    {
		register SnipData *h = s->head;

		if (h && h->font)
		    {
			if (h->bytes == 2)
				XTextExtents16(h->font, (XChar2b *)NULL, 0,
					       &dir, ascent, descent, &XE);
			else
				XTextExtents(h->font, (char *)NULL, 0,
					     &dir, ascent, descent, &XE);
			break;
		    }
	    }
    }

static void ChangeUnderline(w, p, s, x, y)
XeTextWidget w;
SnipMode *p;
Snip *s;
int x, y;
    {
	switch (p->bits & Underline_MASK)
	    {
	    case Underline_DOUBLE:
		MarkArea(w, MARK_Double, x, y, 0);
	    case Underline_SINGLE:
		MarkArea(w, MARK_Single, x, y, 0);
		break;
	    default:
		break;
	    }
	if (s == NULL)
		return;
	x = s->x;
	y = s->y;
	switch (s->mode.bits & Underline_MASK)
	    {
	    case Underline_DOUBLE:
		MarkArea(w, MARK_Double, x, y + 3, 1);
	    case Underline_SINGLE:
		MarkArea(w, MARK_Single, x, y + 1, 1);
		break;
	    default:
		break;
	    }
    }

static void ChangeCrossedOut(w, p, s, x, y)
XeTextWidget w;
SnipMode *p;
Snip *s;
int x, y;
    {
	if (p->bits & CrossedOut_MASK)
		MarkArea(w, MARK_Crossed, x, y, 0);
	if (s == NULL)
		return;
	x = s->x;
	y = s->y;
	if (s->mode.bits & CrossedOut_MASK)
	    {
		int fa, fd;

		FindFontExtents(s, &fa, &fd);
		MarkArea(w, MARK_Crossed, x, y - fa / 2, 1);
	    }
    }

static void ChangeFraming(w, p, s, x, y)
XeTextWidget w;
SnipMode *p;
Snip *s;
int x, y;
    {

	/* Only FRAMED implemented, no ENCIRCLED yet */

	if (p->bits & Framing_MASK)
		MarkArea(w, MARK_Framed, x, y, 0);
	if (s == NULL)
		return;
	x = s->x;
	y = s->y;
	if (s->mode.bits & Framing_MASK)
	    {
		int fa, fd;

		FindFontExtents(s, &fa, &fd);
		MarkArea(w, MARK_Framed, x, y - fa, fa + fd);
	    }
    }

/*
** GetColors
**	Allocate array of 8 pixel values and allocate the 8 basic
**	colors that will be used for the text effects:
**
**	0=black, 1=red, 2=green, 3=yellow,
**	4=blue, 5=magenta, 6=cyan, 7=white
*/

#define NOPIX 0xffffffff /* Hopefully never used as a real pixel value --msa */

static void GetColors(w)
XeTextWidget w;
    {
	int i;
	XColor exact, color;

	static char *colorname[] =
	    {
		"black", "red", "green", "yellow",
		"blue", "magenta", "cyan", "white",
	    };

	w->text.colors = (unsigned long *)XtMalloc(8 * sizeof(unsigned long));
	for (i = 0; i < 8; i++)
	    {
		if (!XAllocNamedColor(XtDisplay(w), w->core.colormap,
				      colorname[i], &exact, &color))
		    {
			XeWidgetWarningMsg((Widget)w,
					   "colorAllocation",
					   "Failed allocating color '%s'",
					   (String *)&colorname[i], 1);
			color.pixel = NOPIX;
		    }
		w->text.colors[i] = color.pixel;
	    }
    }

static void FreeColors(w)
XeTextWidget w;
    {
	int i;

	if (w->text.colors == NULL)
		return;
	for (i = 0; i < 8; i++)
		if (w->text.colors[i] != NOPIX)
			XFreeColors(XtDisplay(w), w->core.colormap,
			    &w->text.colors[i], 1, (unsigned long)0);
	XtFree((void *)w->text.colors);
	w->text.colors = NULL;
    }

static void ChangeForeground(w, p, s, x, y)
XeTextWidget w;
SnipMode *p;
Snip *s;
int x, y;
    {
	unsigned long pixel;

	if (w->text.colors == NULL)
		GetColors(w);
	if (s && (s->mode.bits & Foreground_MASK) &&
	    w->text.colors[Foreground_COLOR(s->mode.bits)-1] != NOPIX)
		pixel = w->text.colors[Foreground_COLOR(s->mode.bits)-1];
	else
		pixel = w->basic.foreground_pixel;
	if (s && (s->mode.bits & ImageInversion_MASK))
		XSetBackground(XtDisplay(w), w->text.mygc, pixel);
	else
		XSetForeground(XtDisplay(w), w->text.mygc, pixel);
    }

static void ChangeBackground(w, p, s, x, y)
XeTextWidget w;
SnipMode *p;
Snip *s;
int x, y;
    {
	unsigned long pixel;

	if (w->text.colors == NULL)
		GetColors(w);
	if (s && (s->mode.bits & Background_MASK) &&
	    w->text.colors[Background_COLOR(s->mode.bits)-1] != NOPIX)
		pixel = w->text.colors[Background_COLOR(s->mode.bits)-1];
	else
		pixel = w->core.background_pixel;
	if (s && (s->mode.bits & ImageInversion_MASK))
		XSetForeground(XtDisplay(w), w->text.mygc, pixel);
	else
		XSetBackground(XtDisplay(w), w->text.mygc, pixel);
    }

static void FlushPendingGraphics(w, p, x, y)
XeTextWidget w;
SnipMode *p;
int x, y;
    {
	if (p->bits & Underline_MASK)
	    {
		ChangeUnderline(w, p, (Snip *)NULL, x, y);
		p->bits &= ~Underline_MASK;
	    }
	if (p->bits & CrossedOut_MASK)
	    {
		ChangeCrossedOut(w, p, (Snip *)NULL, x, y);
		p->bits &= ~CrossedOut_MASK;
	    }
	if (p->bits & Framing_MASK)
	    {
		ChangeFraming(w, p, (Snip *)NULL, x, y);
		p->bits &= ~Framing_MASK;
	    }
    }

static void Redisplay(w, e, r)
Widget w;
XExposeEvent *e;
Region r;
    {
	register XeTextWidget t = (XeTextWidget)w;
	Snip *s;
	int x, y;
	XRectangle clip;
	Region my_r = r;
	SnipMode p;	/* Current effective mode */
#if NeedFunctionPrototypes
#ifndef _XConst
#define _XConst const
#endif
	int (*f_draw16)(Display *,Drawable,GC,int,int,_XConst XChar2b *,int);
	int (*f_draw)(Display *,Drawable,GC,int,int,_XConst char *,int);
#else
	int (*f_draw16)();
	int (*f_draw)();
#endif	

	if (!XtIsRealized(w))
		return;
	if (my_r == NULL)
	    {
		/*
		** No Region, make one from exposed area. This should be
		** exceptional, widgets should have the compress exposures
		** mode set and always have a region!
		*/
		if (e == NULL)
			return; /* No event either--someone goofed! */
		clip.x = e->x;
		clip.y = e->y;
		clip.width = e->width;
		clip.height = e->height;
		my_r = XCreateRegion();
		XUnionRectWithRegion(&clip, my_r, my_r); 
	    }
	XSetRegion(XtDisplay(w), t->text.mygc, my_r);
	XSetRegion(XtDisplay(w), t->text.mygcXOR, my_r);
	XSetBackground(XtDisplay(w), t->text.mygc, t->core.background_pixel);
	if (e == NULL)
	    {
		/*
		** If the XEvent parameter is NULL, then assume this is
		** internally generated expose event and the covered
		** area is not necessarily initially cleared. Clear the
		** area covered by the expose region (my_r). [The following
		** code is the simplest I could invent for the task, if
		** anyone has better, let me know --msa].
		*/
		XSetForeground
			(XtDisplay(w), t->text.mygc, t->core.background_pixel);
		XClipBox(my_r, &clip);
		XFillRectangle(XtDisplay(w), XtWindow(w), t->text.mygc,
			       clip.x, clip.y, clip.width, clip.height);
	    }
	XSetForeground(XtDisplay(w), t->text.mygc, t->basic.foreground_pixel);
	memset((void *)markers,0,sizeof(markers));
	memset((void *)&p,0,sizeof(p));
	f_draw16 = XDrawString16;
	f_draw = XDrawString;
	y = t->text.y;
	x = t->text.x;
	for (s = t->text.first; s; s = s->next)
	    {
		SnipData *h = s->head;
		unsigned int m = p.bits ^ s->mode.bits;

		if (m) 
		    {
			if (m & Underline_MASK)
				ChangeUnderline(t, &p, s, x, y);
			if (m & CrossedOut_MASK)
				ChangeCrossedOut(t, &p, s, x, y);
			if (m & Framing_MASK)
				ChangeFraming(t, &p, s, x, y);
			if (m & (Foreground_MASK|ImageInversion_MASK))
				ChangeForeground(t, &p, s, x, y);
			if (m & (Background_MASK|ImageInversion_MASK))
				ChangeBackground(t, &p, s, x, y);
			if (s->mode.bits&(ImageInversion_MASK|Background_MASK))
			    {
				f_draw16 = XDrawImageString16;
				f_draw = XDrawImageString;
			    }
			else
			    {
				f_draw16 = &XDrawString16;
				f_draw = &XDrawString;
			    }
		    }
		p = s->mode;
		x = s->x;
		y = s->y;
		if (!s->space && s->xWidth > 0 &&
		    XRectInRegion(my_r, x, y - s->ascent, s->xWidth,
				  s->ascent + s->descent) != RectangleOut &&
		    s->data != NULL)
		    {
			XSetFont(XtDisplay(w), t->text.mygc, h->font->fid);
			if (h->bytes == 2)
				(*f_draw16)
					(XtDisplay(w), XtWindow(w),
					 t->text.mygc, (int)x, (int)y,
					 (XChar2b *)s->data, s->length);
			else
				(*f_draw)
					(XtDisplay(w), XtWindow(w),
					 t->text.mygc, (int)x, (int)y,
					 s->data, s->length);
		    }
		x += s->xWidth;
		if (s->endseq &&
		    (p.bits & (Underline_MASK|CrossedOut_MASK|Framing_MASK)))
			FlushPendingGraphics(t, &p, x, y);
	     }
	FlushPendingGraphics(t, &p, x, y); /* ..in case endseq was omitted! */
	if (my_r != r)
		XDestroyRegion(my_r);
	if (t->basic.expose_callbacks&&t->core.widget_class==xeTextWidgetClass)
	    {
		XeExposeCallbackData data;
		data.reason = XeCR_EXPOSE;
		data.event = e;
		data.region = r;
		XtCallCallbackList
			(w, t->basic.expose_callbacks, (XtPointer)&data);
	    }
    }

static Boolean SetValues(current, request, new, args, num_args)
Widget current, request, new;
ArgList args;
Cardinal *num_args;
    {
	XeTextWidget cw = (XeTextWidget)current;
	XeTextWidget nw = (XeTextWidget)new;
	int redisplay;

	redisplay =
		/* Only request full layout, if layout width has been
		   changed, ignore all height changes, and width change
		   if column_width is defined */
		(!nw->text.column_width && cw->core.width != nw->core.width) ||
		 cw->text.column_width != nw->text.column_width ||
		 /* Other parameters affecting layout */
		 cw->text.alignment != nw->text.alignment ||
		 cw->text.first_line_offset!=nw->text.first_line_offset ||
		 cw->text.fonts != nw->text.fonts ||
		 cw->text.format != nw->text.format ||
		 cw->text.indentation != nw->text.indentation ||
		 cw->text.itemization != nw->text.itemization ||
		 cw->text.line_spacing != nw->text.line_spacing ||
		 cw->text.line_layout_table!=nw->text.line_layout_table ||
		 cw->text.proportional != nw->text.proportional;

	if (cw->text.initial_state != nw->text.initial_state)
	    {
		/*
		** Make a local copy of the initial_state resource
		*/
		XtFree(cw->text.initial_state);
		if (nw->text.initial_state)
			nw->text.initial_state =
				XtNewString(nw->text.initial_state);
		if (nw->text.initial_state && nw->text.inserting)
			_XeTextFeedContent(nw->text.inserting,
					   nw->text.initial_state,
					   strlen(nw->text.initial_state));
	    }
	if (cw->text.itemization != nw->text.itemization)
	    {
		/*
		** Make a local copy of the itemization resource
		*/
		XeItemization *new = nw->text.itemization;
		XtFree((char *)cw->text.itemization);
		if (new)
		    {
			nw->text.itemization = XtNew(XeItemization);
			memcpy((void *)nw->text.itemization,
			       (void *)new, sizeof(XeItemization));
		    }
	    }
	redisplay = SetupContent(nw, redisplay);
	return redisplay;
    }

static void Resize(w)
Widget w;
    {
	/*
	** Ignore full re-layout, if column_width is non-zero (in that
	** case the change of size doesn't affect the layout. Should
	** also test whether only width changed, but that would need
	** remembering the old width from somewhere...
	*/
	if (((XeTextWidget)w)->text.column_width == 0)
	    {
		_XeTextLayout((XeTextWidget)w, (Snip *)NULL, (Region)0);
		XClearArea(XtDisplay(w), XtWindow(w),0,0,0,0,True); /* Ugh! */
	    }
    }

static void Destroy(w)
Widget w;
    {
	XeTextWidget tw = (XeTextWidget)w;

	/* Destroy the Snip chain */

	if (tw->text.inserting)
		(void)_XeTextEndContent(tw->text.inserting,(Region)0);
	while (tw->text.first)
		_XeDeleteSnip(&tw->text.first);

	/* Release GC's */

	XFreeGC(XtDisplay(w), tw->text.mygc);
	XFreeGC(XtDisplay(w), tw->text.mygcXOR);

	XtFree(tw->text.initial_state);
	XtFree((char *)tw->text.itemization);
	/* Release fonts -- not copied currently */
	/* Release kerning_offset -- not copied currently */
	/* Release line_layout_table -- not copied currently */

	/* Release text colors */
	FreeColors(tw);
    }

/*
** Progamming Interface functions. For documentation, see Text.h
*/
void XeTextLayout(w)
Widget w;
    {
	String params[2];

	if (XtIsSubclass(w, xeTextWidgetClass))
	    {
		_XeTextLayout((XeTextWidget)w,(Snip *)NULL, (Region)0);
		if (XtIsRealized(w))
			XClearArea(XtDisplay(w), XtWindow(w),0,0,0,0,True);
	    }
	else
	    {
		params[0] = w->core.name;
		params[1] = w->core.widget_class->core_class.class_name;
		XeWidgetWarningMsg
			(w, "bugTextNotText",
			 "Widget %s is class %s, and not xeTextWidget",
			 (String *)params, (Cardinal)2);
	    }
    }


