/***************************************************************************
 * SPDX-FileCopyrightText: 2024 S. MANKOWSKI stephane@mankowski.fr
 * SPDX-FileCopyrightText: 2024 G. DE BURE support@mankowski.fr
 * SPDX-License-Identifier: GPL-3.0-or-later
 ***************************************************************************/
/** @file
 * This file is a test script.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include <qfile.h>
#include <qsqldatabase.h>

#include "skgtestmacro.h"

/**
 * The main function of the unit test
 * @param argc the number of arguments
 * @param argv the list of arguments
 */
int main(int argc, char** argv)
{
    Q_UNUSED(argc)
    Q_UNUSED(argv)

    // Init test
    SKGINITTEST(true) {
        SKGTESTBOOL("QSQLITE:isDriverAvailable", QSqlDatabase::isDriverAvailable(QLatin1String("QSQLITE")), true)
        SKGTESTBOOL("SKGSQLCIPHER:isDriverAvailable", QSqlDatabase::isDriverAvailable(QLatin1String("SKGSQLCIPHER")), true)
    }
    {
        // test class SKGDocument / LOAD / SAVE
        QString filename1 = SKGTest::getTestPath(QLatin1String("IN")) % QLatin1String("/filename1_e.skg");
        qDebug("filename1=%s", filename1.toUtf8().constData());
        {
            SKGDocument document1;
            QFile(filename1).remove();
            SKGTESTERROR(QLatin1String("DOC:initialize"), document1.initialize(), true)
            SKGTESTERROR(QLatin1String("DOC:setLanguage"), document1.setLanguage(QLatin1String("en")), true)
            SKGTESTERROR(QLatin1String("DOC:setLanguage"), document1.setLanguage(QLatin1String("fr")), true)
            SKGTESTBOOL("DOC:isFileModified", document1.isFileModified(), false)

            SKGTESTERROR(QLatin1String("DOC:beginTransaction"), document1.beginTransaction(QLatin1String("tt")), true)
            SKGTESTERROR(QLatin1String("DOC:setParameter"), document1.setParameter(QLatin1String("ATT3"), QLatin1String("VAL3")), true)
            SKGTESTBOOL("DOC:isFileModified", document1.isFileModified(), true)
            SKGTESTERROR(QLatin1String("DOC:endTransaction"), document1.endTransaction(false), true)
            SKGTESTBOOL("DOC:isFileModified", document1.isFileModified(), false)

            SKGTESTERROR(QLatin1String("DOC:beginTransaction"), document1.beginTransaction(QLatin1String("tt")), true)
            SKGTESTERROR(QLatin1String("DOC:setParameter"), document1.setParameter(QLatin1String("ATT3"), QLatin1String("VAL3")), true)
            SKGTESTBOOL("DOC:isFileModified", document1.isFileModified(), true)
            SKGTESTERROR(QLatin1String("DOC:endTransaction"), document1.endTransaction(true), true)
            SKGTESTBOOL("DOC:isFileModified", document1.isFileModified(), true)

            document1.setFileNotModified();
            SKGTESTBOOL("DOC:isFileModified", document1.isFileModified(), false)

            SKGTEST(QLatin1String("DOC:getDisplay"), document1.getDisplay(QLatin1String("t_name")), QLatin1String("Name"))
            SKGTEST(QLatin1String("DOC:getDisplay"), document1.getDisplay(QLatin1String("node.t_name")), QLatin1String("Name"))
            SKGTEST(QLatin1String("DOC:getDisplay"), document1.getDisplay(QLatin1String("v_node.t_name")), QLatin1String("Name"))
            SKGTEST(QLatin1String("DOC:getDisplay"), document1.getDisplay(QLatin1String("v_node.unknown")), QLatin1String("v_node.unknown"))

            QString oResult;
            SKGTESTERROR(QLatin1String("DOC:isFileModified"), document1.dumpSelectSqliteOrder(QLatin1String("select * from parameters"), oResult), true)
        }
        {
            SKGDocument document1;
            QFile(filename1).remove();
            SKGTESTERROR(QLatin1String("DOC:initialize"), document1.initialize(), true)
            SKGTESTBOOL("DOC:isFileModified", document1.isFileModified(), false)

            SKGTESTERROR(QLatin1String("DOC:beginTransaction"), document1.beginTransaction(QLatin1String("tt")), true)
            SKGTESTERROR(QLatin1String("DOC:endTransaction"), document1.endTransaction(true), true)
            SKGTESTBOOL("DOC:isFileModified", document1.isFileModified(), false)  // because the transaction is empty
        }
        {
            SKGDocument document1;
            QFile(filename1).remove();
            SKGTESTERROR(QLatin1String("DOC:load"), document1.load(filename1, QLatin1String("")), false)
            SKGTESTERROR(QLatin1String("DOC:saveAs+invalid path"), document1.saveAs(QLatin1String("/notfound/file.skg"), true), false)
            SKGTEST(QLatin1String("DOC:getCurrentFileName"), document1.getCurrentFileName(), QLatin1String(""))
            SKGTESTBOOL("DOC:isFileModified", document1.isFileModified(), false)
            SKGTESTERROR(QLatin1String("DOC:initialize"), document1.initialize(), true)
            SKGTESTBOOL("DOC:isFileModified", document1.isFileModified(), false)
            SKGTESTERROR(QLatin1String("DOC:beginTransaction"), document1.beginTransaction(QLatin1String("t1")), true)
            SKGTEST(QLatin1String("DOC:getCurrentFileName"), document1.getCurrentFileName(), QLatin1String(""))
            SKGTESTERROR(QLatin1String("DOC:save"), document1.save(), false)
            SKGTEST(QLatin1String("DOC:getCurrentFileName"), document1.getCurrentFileName(), QLatin1String(""))
            SKGTESTBOOL("DOC:isFileModified", document1.isFileModified(), true)  // due to the beginTransaction
            SKGTEST(QLatin1String("DOC:getParameter"), document1.getParameter(QLatin1String("ATT1")), QLatin1String(""))
            SKGTESTERROR(QLatin1String("DOC:setParameter"), document1.setParameter(QLatin1String("ATT1"), QLatin1String("VAL1")), true)
            SKGTESTBOOL("DOC:isFileModified", document1.isFileModified(), true)
            SKGTEST(QLatin1String("DOC:getParameter"), document1.getParameter(QLatin1String("ATT1")), QLatin1String("VAL1"))
            SKGTESTERROR(QLatin1String("DOC:endTransaction"), document1.endTransaction(true), true)
            SKGTESTERROR(QLatin1String("DOC:saveAs+overwrite=false"), document1.saveAs(filename1, false), true)
            SKGTESTBOOL("DOC:exist", QFile(filename1).exists(), true)
            SKGTEST(QLatin1String("DOC:getCurrentFileName"), document1.getCurrentFileName(), filename1)
            SKGTESTERROR(QLatin1String("DOC:saveAs+overwrite=false"), document1.saveAs(filename1, false), false)
            SKGTESTBOOL("DOC:exist", QFile(filename1).exists(), true)
            SKGTEST(QLatin1String("DOC:getCurrentFileName"), document1.getCurrentFileName(), filename1)
            SKGTESTERROR(QLatin1String("DOC:save"), document1.save(), true)
            SKGTESTBOOL("DOC:exist", QFile(filename1).exists(), true)
            SKGTEST(QLatin1String("DOC:getCurrentFileName"), document1.getCurrentFileName(), filename1)

            SKGTESTERROR(QLatin1String("DOC:beginTransaction"), document1.beginTransaction(QLatin1String("t1")), true)
            SKGTESTBOOL("DOC:isFileModified", document1.isFileModified(), true)  // due to the beginTransaction
            SKGTEST(QLatin1String("DOC:getParameter"), document1.getParameter(QLatin1String("ATT1")), QLatin1String("VAL1"))
            SKGTESTERROR(QLatin1String("DOC:setParameter"), document1.setParameter(QLatin1String("ATT1"), QLatin1String("VAL2")), true)
            SKGTEST(QLatin1String("DOC:getParameter"), document1.getParameter(QLatin1String("ATT1")), QLatin1String("VAL2"))
            SKGTESTBOOL("DOC:isFileModified", document1.isFileModified(), true)
            SKGTESTERROR(QLatin1String("DOC:endTransaction"), document1.endTransaction(true), true)

            SKGTESTBOOL("DOC:exist", QFile(filename1).exists(), true)
            SKGTESTERROR(QLatin1String("DOC:load"), document1.load(filename1, QLatin1String("")), true)
            SKGTESTBOOL("DOC:isFileModified", document1.isFileModified(), false)
            SKGTESTBOOL("DOC:getCurrentFileName", (!document1.getCurrentFileName().isEmpty()), true)
            SKGTEST(QLatin1String("DOC:getParameter"), document1.getParameter(QLatin1String("ATT1")), QLatin1String("VAL1"))
        }

        {
            SKGDocument document1;
            SKGTESTERROR(QLatin1String("DOC:load"), document1.load(filename1, QLatin1String("")), true)
            SKGTESTERROR(QLatin1String("DOC:beginTransaction"), document1.beginTransaction(QLatin1String("t3")), true)
            SKGTESTERROR(QLatin1String("DOC:setParameter"), document1.setParameter(QLatin1String("ATT3"), QLatin1String("VAL3")), true)
            SKGTESTERROR(QLatin1String("DOC:endTransaction"), document1.endTransaction(true), true)
            SKGTESTERROR(QLatin1String("DOC:save"), document1.save(), true)

            SKGTESTERROR(QLatin1String("DOC:changePassword"), document1.changePassword(QLatin1String("pwd1")), true)
            SKGTESTERROR(QLatin1String("DOC:changePassword"), document1.changePassword(QLatin1String("pwd")), true)
            SKGTESTERROR(QLatin1String("DOC:beginTransaction"), document1.beginTransaction(QLatin1String("tt")), true)
            SKGTESTERROR(QLatin1String("DOC:setParameter"), document1.setParameter(QLatin1String("SAVE"), QLatin1String("1")), true)
            SKGTESTERROR(QLatin1String("DOC:endTransaction"), document1.endTransaction(true), true)

            SKGTEST(QLatin1String("DOC:getParameter"), document1.getParameter(QLatin1String("SAVE")), QLatin1String("1"))
            SKGTESTERROR(QLatin1String("DOC:save"), document1.save(), true)
        }
        {
            SKGDocument document1;
            SKGTESTERROR(QLatin1String("DOC:load"), document1.load(filename1, QLatin1String("")), false)
            SKGTEST(QLatin1String("DOC:load"), document1.load(filename1, QLatin1String("wrong pwd")).getReturnCode(), ERR_ENCRYPTION)
        }
        {
            SKGDocument document1;
            SKGTESTERROR(QLatin1String("DOC:load"), document1.load(filename1, QLatin1String("pwd")), true)
            SKGTESTERROR(QLatin1String("DOC:changePassword"), document1.changePassword(QLatin1String("pwd1")), true)
            SKGTESTERROR(QLatin1String("DOC:changePassword"), document1.changePassword(QLatin1String("")), true)
            SKGTESTERROR(QLatin1String("DOC:save"), document1.save(), true)
            SKGTESTERROR(QLatin1String("DOC:load"), document1.load(filename1, QLatin1String("")), true)
        }

        {
            SKGDocument document1;
            QFile(filename1).remove();
            SKGTESTERROR(QLatin1String("DOC:initialize"), document1.initialize(), true)
            document1.setBackupParameters(QLatin1String(""), QLatin1String("/invalid suffix"));
            SKGTESTERROR(QLatin1String("DOC:saveAs"), document1.saveAs(filename1, true), true)
            SKGTESTERROR(QLatin1String("DOC:saveAs"), document1.saveAs(filename1, true), true)

            document1.setBackupParameters(QLatin1String("."), QLatin1String(".old"));
            SKGTEST(QLatin1String("DOC:getBackupFile"), document1.getBackupFile(QLatin1String("/tmp/file.skg")), QLatin1String("/tmp/.file.skg.old"))

            document1.setBackupParameters(QLatin1String(""), QLatin1String("_<DATE>.backup"));
            SKGTEST(QLatin1String("DOC:getBackupFile"), document1.getBackupFile(QLatin1String("/tmp/file.skg")), QLatin1String("/tmp/file.skg_") % SKGServices::dateToSqlString(QDateTime::currentDateTime().date()) % ".backup")

            document1.setBackupParameters(QLatin1String(""), QLatin1String("_<DATE>.skg"));
            SKGTEST(QLatin1String("DOC:getBackupFile"), document1.getBackupFile(QLatin1String("/tmp/file.skg")), QLatin1String("/tmp/file_") % SKGServices::dateToSqlString(QDateTime::currentDateTime().date()) % ".skg")

            document1.setBackupParameters(QLatin1String(""), QLatin1String("_<TIME>.backup"));
            SKGTEST(QLatin1String("DOC:getBackupFile"), document1.getBackupFile(QLatin1String("/tmp/file.skg")), QLatin1String("/tmp/file.skg_") % SKGServices::timeToString(QDateTime::currentDateTime()) % ".backup")

            document1.setBackupParameters(QLatin1String("/tmp/"), QLatin1String(".old"));
            SKGTEST(QLatin1String("DOC:getBackupFile"), document1.getBackupFile(QLatin1String("/home/user1/file.skg")), QLatin1String("/tmp/file.skg.old"))
        }

        {
            SKGDocument document1;
            SKGTESTERROR(QLatin1String("DOC:load"), document1.load(filename1, QLatin1String("")), true)
            SKGTESTERROR(QLatin1String("DOC:beginTransaction"), document1.beginTransaction(QLatin1String("tt")), true)
            SKGTESTERROR(QLatin1String("DOC:setParameter"), document1.setParameter(QLatin1String("SAVE"), QLatin1String("1")), true)
            SKGTEST(QLatin1String("DOC:getParameter"), document1.getParameter(QLatin1String("SAVE")), QLatin1String("1"))
            SKGTESTERROR(QLatin1String("DOC:endTransaction"), document1.endTransaction(true), true)
            SKGTESTERROR(QLatin1String("DOC:save"), document1.save(), true)

            // Check save when file system is full
            qputenv("SKGFILESYSTEMFULL", "Y");
            SKGTEST(QLatin1String("DOC:getParameter"), static_cast<unsigned int>(QFile(filename1).exists()), static_cast<unsigned int>(true))
            SKGTESTERROR(QLatin1String("DOC:save"), document1.save(), false)
            SKGTEST(QLatin1String("DOC:getParameter"), static_cast<unsigned int>(QFile(filename1).exists()), static_cast<unsigned int>(true))
        }

        QFile(filename1).remove();
        QFile(filename1 % ".old").remove();
    }
    // End test
    SKGENDTEST()
}
