/***************************************************************************
 * SPDX-FileCopyrightText: 2024 S. MANKOWSKI stephane@mankowski.fr
 * SPDX-FileCopyrightText: 2024 G. DE BURE support@mankowski.fr
 * SPDX-License-Identifier: GPL-3.0-or-later
 ***************************************************************************/
/** @file
 * This file is a test script.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtestmacro.h"

/**
 * The main function of the unit test
 * @param argc the number of arguments
 * @param argv the list of arguments
 */
void test_getPeriodWhereClause(int& nberror, int& nbcheck, bool showonlyfailures)
{
    SKGTEST(QLatin1String("CONV:getPeriodWhereClause"), SKGServices::getPeriodWhereClause(QLatin1String("2014"), QLatin1String("my_date"), QLatin1String("<")), QLatin1String("(STRFTIME('%Y',my_date)<'2014' OR my_date='0000-00-00')"))
    SKGTEST(QLatin1String("CONV:getPeriodWhereClause"), SKGServices::getPeriodWhereClause(QLatin1String("2014")), QLatin1String("STRFTIME('%Y',d_date)='2014'"))
    SKGTEST(QLatin1String("CONV:getPeriodWhereClause"), SKGServices::getPeriodWhereClause(QLatin1String("2014-02")), QLatin1String("STRFTIME('%Y-%m',d_date)='2014-02'"))
    SKGTEST(QLatin1String("CONV:getPeriodWhereClause"), SKGServices::getPeriodWhereClause(QLatin1String("2014-10")), QLatin1String("STRFTIME('%Y-%m',d_date)='2014-10'"))
    SKGTEST(QLatin1String("CONV:getPeriodWhereClause"), SKGServices::getPeriodWhereClause(QLatin1String("2014-Q2")), QLatin1String("STRFTIME('%Y',d_date)||'-Q'||(CASE WHEN STRFTIME('%m',d_date)<='03' THEN '1' WHEN STRFTIME('%m',d_date)<='06' THEN '2' WHEN STRFTIME('%m',d_date)<='09' THEN '3' ELSE '4' END)='2014-Q2'"))
    SKGTEST(QLatin1String("CONV:getPeriodWhereClause"), SKGServices::getPeriodWhereClause(QLatin1String("2014-S2")), QLatin1String("STRFTIME('%Y',d_date)||'-S'||(CASE WHEN STRFTIME('%m',d_date)<='06' THEN '1' ELSE '2' END)='2014-S2'"))
    SKGTEST(QLatin1String("CONV:getPeriodWhereClause"), SKGServices::getPeriodWhereClause(QLatin1String("INVALID STRING")), QLatin1String("1=0"))
    SKGTEST(QLatin1String("CONV:getPeriodWhereClause"), SKGServices::getPeriodWhereClause(QLatin1String("ALL")), QLatin1String("1=1"))
}

void test_getNeighboringPeriod(int& nberror, int& nbcheck, bool showonlyfailures)
{
    SKGTEST(QLatin1String("CONV:getNeighboringPeriod"), SKGServices::getNeighboringPeriod(QLatin1String("2014")), QLatin1String("2013"))
    SKGTEST(QLatin1String("CONV:getNeighboringPeriod"), SKGServices::getNeighboringPeriod(QLatin1String("2014-10")), QLatin1String("2014-09"))
    SKGTEST(QLatin1String("CONV:getNeighboringPeriod"), SKGServices::getNeighboringPeriod(QLatin1String("2014-01")), QLatin1String("2013-12"))
    SKGTEST(QLatin1String("CONV:getNeighboringPeriod"), SKGServices::getNeighboringPeriod(QLatin1String("2014-Q2")), QLatin1String("2014-Q1"))
    SKGTEST(QLatin1String("CONV:getNeighboringPeriod"), SKGServices::getNeighboringPeriod(QLatin1String("2014-Q1")), QLatin1String("2013-Q4"))
    SKGTEST(QLatin1String("CONV:getNeighboringPeriod"), SKGServices::getNeighboringPeriod(QLatin1String("2014-S2")), QLatin1String("2014-S1"))
    SKGTEST(QLatin1String("CONV:getNeighboringPeriod"), SKGServices::getNeighboringPeriod(QLatin1String("2014-S1")), QLatin1String("2013-S2"))
    SKGTEST(QLatin1String("CONV:getNeighboringPeriod"), SKGServices::getNeighboringPeriod(QLatin1String("INVALID STRING")), QLatin1String("1=0"))
    SKGTEST(QLatin1String("CONV:getNeighboringPeriod"), SKGServices::getNeighboringPeriod(QLatin1String("ALL")), QLatin1String("1=0"))

    SKGTEST(QLatin1String("CONV:getNeighboringPeriod +1"), SKGServices::getNeighboringPeriod(QLatin1String("2014"), 1), QLatin1String("2015"))
    SKGTEST(QLatin1String("CONV:getNeighboringPeriod +1"), SKGServices::getNeighboringPeriod(QLatin1String("2014-10"), 1), QLatin1String("2014-11"))
    SKGTEST(QLatin1String("CONV:getNeighboringPeriod +1"), SKGServices::getNeighboringPeriod(QLatin1String("2014-01"), 1), QLatin1String("2014-02"))
    SKGTEST(QLatin1String("CONV:getNeighboringPeriod +1"), SKGServices::getNeighboringPeriod(QLatin1String("2014-Q4"), 1), QLatin1String("2015-Q1"))
    SKGTEST(QLatin1String("CONV:getNeighboringPeriod +1"), SKGServices::getNeighboringPeriod(QLatin1String("2014-Q2"), 1), QLatin1String("2014-Q3"))
    SKGTEST(QLatin1String("CONV:getNeighboringPeriod +1"), SKGServices::getNeighboringPeriod(QLatin1String("2014-Q1"), 1), QLatin1String("2014-Q2"))
    SKGTEST(QLatin1String("CONV:getNeighboringPeriod +1"), SKGServices::getNeighboringPeriod(QLatin1String("2014-S2"), 1), QLatin1String("2015-S1"))
    SKGTEST(QLatin1String("CONV:getNeighboringPeriod +1"), SKGServices::getNeighboringPeriod(QLatin1String("2014-S1"), 1), QLatin1String("2014-S2"))
}

void test_periodToDate(int& nberror, int& nbcheck, bool showonlyfailures)
{
    SKGTEST(QLatin1String("CONV:periodToDate"), SKGServices::periodToDate(QLatin1String("2014")).toString(), QDate(2014, 12, 31).toString())
    SKGTEST(QLatin1String("CONV:periodToDate"), SKGServices::periodToDate(QLatin1String("2014-S1")).toString(), QDate(2014, 6, 30).toString())
    SKGTEST(QLatin1String("CONV:periodToDate"), SKGServices::periodToDate(QLatin1String("2014-S2")).toString(), QDate(2014, 12, 31).toString())
    SKGTEST(QLatin1String("CONV:periodToDate"), SKGServices::periodToDate(QLatin1String("2014-Q1")).toString(), QDate(2014, 3, 31).toString())
    SKGTEST(QLatin1String("CONV:periodToDate"), SKGServices::periodToDate(QLatin1String("2014-Q2")).toString(), QDate(2014, 6, 30).toString())
    SKGTEST(QLatin1String("CONV:periodToDate"), SKGServices::periodToDate(QLatin1String("2014-Q3")).toString(), QDate(2014, 9, 30).toString())
    SKGTEST(QLatin1String("CONV:periodToDate"), SKGServices::periodToDate(QLatin1String("2014-Q4")).toString(), QDate(2014, 12, 31).toString())
    SKGTEST(QLatin1String("CONV:periodToDate"), SKGServices::periodToDate(QLatin1String("2014-01")).toString(), QDate(2014, 1, 31).toString())
    SKGTEST(QLatin1String("CONV:periodToDate"), SKGServices::periodToDate(QLatin1String("2014-07")).toString(), QDate(2014, 7, 31).toString())
    SKGTEST(QLatin1String("CONV:periodToDate"), SKGServices::periodToDate(QLatin1String("ALL")).toString(), QDate::currentDate().toString())
}

void test_partialStringToDate(int& nberror, int& nbcheck, bool showonlyfailures)
{
    SKGTEST(QLatin1String("CONV:partialStringToDate"), SKGServices::partialStringToDate(QLatin1String("INVALID"), true).toString(), QLatin1String(""))

    QDate currentMonth10 = QDate::currentDate();
    currentMonth10 = currentMonth10.addDays(10 - QDate::currentDate().day());
    QDate current0102(QDate::currentDate().year(), 2, 1);
    QDate current3112(QDate::currentDate().year(), 12, 31);
    SKGTEST(QLatin1String("CONV:partialStringToDate"), SKGServices::partialStringToDate(QLatin1String("10"), true).toString(), (currentMonth10 <= QDate::currentDate() ? currentMonth10 : currentMonth10.addMonths(-1)).toString())
    SKGTEST(QLatin1String("CONV:partialStringToDate"), SKGServices::partialStringToDate(QLatin1String("1/2"), true).toString(), (current0102 <= QDate::currentDate() ? current0102 : current0102.addYears(-1)).toString())
    SKGTEST(QLatin1String("CONV:partialStringToDate"), SKGServices::partialStringToDate(QLatin1String("31/12"), true).toString(), (current3112 <= QDate::currentDate() ? current3112 : current3112.addYears(-1)).toString())

    SKGTEST(QLatin1String("CONV:partialStringToDate"), SKGServices::partialStringToDate(QLatin1String("1/2/14"), true).toString(), QDate(2014, 2, 1).toString())
    SKGTEST(QLatin1String("CONV:partialStringToDate"), SKGServices::partialStringToDate(QLatin1String("1/2/99"), true).toString(), QDate(1999, 2, 1).toString())
    SKGTEST(QLatin1String("CONV:partialStringToDate"), SKGServices::partialStringToDate(QLatin1String("1/2/014"), true).toString(), QDate(2014, 2, 1).toString())
    SKGTEST(QLatin1String("CONV:partialStringToDate"), SKGServices::partialStringToDate(QLatin1String("1/2/199"), true).toString(), QDate(1199, 2, 1).toString())
    SKGTEST(QLatin1String("CONV:partialStringToDate"), SKGServices::partialStringToDate(QLatin1String("1/2/1014"), true).toString(), QDate(1014, 2, 1).toString())
    SKGTEST(QLatin1String("CONV:partialStringToDate"), SKGServices::partialStringToDate(QLatin1String("1/2/2222"), true).toString(), QDate(2222, 2, 1).toString())

    SKGTEST(QLatin1String("CONV:partialStringToDate"), SKGServices::partialStringToDate(QLatin1String("10"), false).toString(), (currentMonth10 >= QDate::currentDate() ? currentMonth10 : currentMonth10.addMonths(1)).toString())
    SKGTEST(QLatin1String("CONV:partialStringToDate"), SKGServices::partialStringToDate(QLatin1String("1/2"), false).toString(), (current0102 >= QDate::currentDate() ? current0102 : current0102.addYears(1)).toString())
    SKGTEST(QLatin1String("CONV:partialStringToDate"), SKGServices::partialStringToDate(QLatin1String("1/2/14"), false).toString(), QDate(2114, 2, 1).toString())
    SKGTEST(QLatin1String("CONV:partialStringToDate"), SKGServices::partialStringToDate(QLatin1String("1/2/99"), false).toString(), QDate(2099, 2, 1).toString())
    SKGTEST(QLatin1String("CONV:partialStringToDate"), SKGServices::partialStringToDate(QLatin1String("1/2/014"), false).toString(), QDate(3014, 2, 1).toString())
    SKGTEST(QLatin1String("CONV:partialStringToDate"), SKGServices::partialStringToDate(QLatin1String("1/2/199"), false).toString(), QDate(2199, 2, 1).toString())
    SKGTEST(QLatin1String("CONV:partialStringToDate"), SKGServices::partialStringToDate(QLatin1String("1/2/1014"), false).toString(), QDate(1014, 2, 1).toString())
    SKGTEST(QLatin1String("CONV:partialStringToDate"), SKGServices::partialStringToDate(QLatin1String("1/2/2222"), false).toString(), QDate(2222, 2, 1).toString())
}

void test_conversions(int& nberror, int& nbcheck, bool showonlyfailures)
{
    SKGTEST(QLatin1String("CONV:dateToPeriod"), SKGServices::dateToPeriod(QDate(2013, 03, 05), QLatin1String("D")), QLatin1String("2013-03-05"))
    SKGTEST(QLatin1String("CONV:dateToPeriod"), SKGServices::dateToPeriod(QDate(2013, 01, 01), QLatin1String("W")), QLatin1String("2013-W01"))
    SKGTEST(QLatin1String("CONV:dateToPeriod"), SKGServices::dateToPeriod(QDate(2013, 03, 05), QLatin1String("W")), QLatin1String("2013-W10"))
    SKGTEST(QLatin1String("CONV:dateToPeriod"), SKGServices::dateToPeriod(QDate(2013, 03, 05), QLatin1String("M")), QLatin1String("2013-03"))
    SKGTEST(QLatin1String("CONV:dateToPeriod"), SKGServices::dateToPeriod(QDate(2013, 03, 05), QLatin1String("Q")), QLatin1String("2013-Q1"))
    SKGTEST(QLatin1String("CONV:dateToPeriod"), SKGServices::dateToPeriod(QDate(2013, 03, 05), QLatin1String("S")), QLatin1String("2013-S1"))
    SKGTEST(QLatin1String("CONV:dateToPeriod"), SKGServices::dateToPeriod(QDate(2013, 03, 05), QLatin1String("Y")), QLatin1String("2013"))

    SKGTEST(QLatin1String("CONV:dateToPeriod"), SKGServices::dateToPeriod(QDate(2019, 12, 31), QLatin1String("W")), QLatin1String("2020-W01"))

    SKGTEST(QLatin1String("CONV:intToString"), SKGServices::intToString(10), QLatin1String("10"))
    SKGTEST(QLatin1String("CONV:intToString"), SKGServices::intToString(5490990004), QLatin1String("5490990004"))
    SKGTEST(QLatin1String("CONV:stringToInt"), SKGServices::stringToInt(QLatin1String("56")), 56)
    SKGTEST(QLatin1String("CONV:stringToInt"), SKGServices::stringToInt(QLatin1String("HELLO")), 0)
    SKGTEST(QLatin1String("CONV:stringToInt"), SKGServices::stringToInt(QLatin1String("5HELLO")), 0)
    SKGTEST(QLatin1String("CONV:stringToInt"), SKGServices::stringToInt(QLatin1String("5490990004")), 5490990004)
    SKGTEST(QLatin1String("CONV:doubleToString"), SKGServices::doubleToString(10), QLatin1String("10"))
    SKGTEST(QLatin1String("CONV:doubleToString"), SKGServices::doubleToString(5.3), QLatin1String("5.3"))
    SKGTEST(QLatin1String("CONV:doubleToString"), SKGServices::doubleToString(11111115.33), QLatin1String("11111115.33"))
    SKGTEST(QLatin1String("CONV:doubleToString"), SKGServices::doubleToString(119999.11), QLatin1String("119999.11"))
    SKGTEST(QLatin1String("CONV:stringToDouble"), SKGServices::stringToDouble(QLatin1String("10")) - 10, 0)
    SKGTEST(QLatin1String("CONV:stringToDouble"), SKGServices::stringToDouble(QLatin1String("5.3")) - 5.3, 0)
    SKGTEST(QLatin1String("CONV:stringToDouble"), SKGServices::stringToDouble(QLatin1String("11111115.33")) - 11111115.33, 0)
    SKGTEST(QLatin1String("CONV:stringToDouble"), SKGServices::stringToDouble(QLatin1String("25,000.00")) - 25000.00, 0)
    SKGTEST(QLatin1String("CONV:stringToDouble"), SKGServices::stringToDouble(QLatin1String("1.307,40")) - 1307.40, 0)
    SKGTEST(QLatin1String("CONV:stringToDouble"), SKGServices::stringToDouble(QLatin1String("2 150,10")) - 2150.10, 0)

    QLocale::setDefault(QLocale::German);
    SKGTEST(QLatin1String("CONV:stringToDouble"), SKGServices::stringToDouble(QLatin1String("11.000")), 11000)
    SKGTEST(QLatin1String("CONV:stringToDouble"), SKGServices::stringToDouble(QLatin1String("241,53")), 241.53)

    QLocale::setDefault(QLocale::C);
    SKGTEST(QLatin1String("CONV:stringToDouble"), SKGServices::stringToDouble(QLatin1String("-$8.35")) + 8.35, 0)
    SKGTEST(QLatin1String("CONV:stringToDouble"), SKGServices::stringToDouble(QStringLiteral("8.35€")) - 8.35, 0)
    SKGTEST(QLatin1String("CONV:stringToDouble"), SKGServices::stringToDouble(QLatin1String("1234.56e-02")) - 12.3456, 0)
    SKGTEST(QLatin1String("CONV:stringToDouble"), SKGServices::stringToDouble(QLatin1String("31238/100")) - 312.38, 0)
    SKGTEST(QLatin1String("CONV:stringToDouble"), SKGServices::stringToDouble(QLatin1String("31238/abc")), 0)
    SKGTEST(QLatin1String("CONV:stringToDouble"), SKGServices::stringToDouble(QLatin1String("nan")), 0)
    SKGTEST(QLatin1String("CONV:stringToDouble"), SKGServices::stringToDouble(QLatin1String("inf")), 1e300)
    SKGTEST(QLatin1String("CONV:stringToDouble"), SKGServices::stringToDouble(QLatin1String("-inf")), -1e300)
    SKGTEST(QLatin1String("CONV:stringToDouble"), SKGServices::stringToDouble(QLatin1String("00000000194065")), 194065)
    SKGServices::timeToString(QDateTime());
    SKGTEST(QLatin1String("CONV:stringToTime"), SKGServices::timeToString(SKGServices::stringToTime(QLatin1String("1970-07-16"))), QLatin1String("1970-07-16 00:00:00"))
    SKGTEST(QLatin1String("CONV:stringToTime"), SKGServices::timeToString(SKGServices::stringToTime(QLatin1String("2008-04-20"))), QLatin1String("2008-04-20 00:00:00"))
    SKGTEST(QLatin1String("CONV:stringToTime"), SKGServices::dateToSqlString(SKGServices::stringToTime(QLatin1String("1970-07-16"))), QLatin1String("1970-07-16"))
    SKGTEST(QLatin1String("CONV:stringToTime"), SKGServices::dateToSqlString(SKGServices::stringToTime(QLatin1String("2008-04-20"))), QLatin1String("2008-04-20"))
    SKGTEST(QLatin1String("CONV:stringToDate"), SKGServices::dateToSqlString(SKGServices::stringToDate(QLatin1String("1970-07-16"))), QLatin1String("1970-07-16"))
    SKGTEST(QLatin1String("CONV:stringToDate"), SKGServices::dateToSqlString(SKGServices::stringToDate(QLatin1String("2008-04-20"))), QLatin1String("2008-04-20"))

    SKGTESTBOOL("CONV:SKGServices::getMicroTime", (SKGServices::getMicroTime() > 0), true)

    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("20080525"), QLatin1String("YYYYMMDD")), QLatin1String("2008-05-25"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("2008-05-25"), QLatin1String("YYYY-MM-DD")), QLatin1String("2008-05-25"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("2008-05-25 00:00:00"), QLatin1String("YYYY-MM-DD")), QLatin1String("2008-05-25"))

    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("05/25/08"), QLatin1String("MM/DD/YY")), QLatin1String("2008-05-25"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("05/25/78"), QLatin1String("MM/DD/YY")), QLatin1String("1978-05-25"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("05-25-08"), QLatin1String("MM-DD-YY")), QLatin1String("2008-05-25"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("05-25-78"), QLatin1String("MM-DD-YY")), QLatin1String("1978-05-25"))

    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("5/25/08"), QLatin1String("MM/DD/YY")), QLatin1String("2008-05-25"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("5/25/78"), QLatin1String("MM/DD/YY")), QLatin1String("1978-05-25"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("5-25-08"), QLatin1String("MM-DD-YY")), QLatin1String("2008-05-25"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("5-6-08"), QLatin1String("MM-DD-YY")), QLatin1String("2008-05-06"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("5-25-78"), QLatin1String("MM-DD-YY")), QLatin1String("1978-05-25"))

    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("25/05/08"), QLatin1String("DD/MM/YY")), QLatin1String("2008-05-25"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("25/05/78"), QLatin1String("DD/MM/YY")), QLatin1String("1978-05-25"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("25-05-08"), QLatin1String("DD-MM-YY")), QLatin1String("2008-05-25"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("25-05-78"), QLatin1String("DD-MM-YY")), QLatin1String("1978-05-25"))

    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("25/05/2008"), QLatin1String("DD/MM/YYYY")), QLatin1String("2008-05-25"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("25-05-2008"), QLatin1String("DD-MM-YYYY")), QLatin1String("2008-05-25"))

    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("5/25/2008"), QLatin1String("MM/DD/YYYY")), QLatin1String("2008-05-25"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("5-25-2008"), QLatin1String("MM-DD-YYYY")), QLatin1String("2008-05-25"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("5-6-2008"), QLatin1String("MM-DD-YYYY")), QLatin1String("2008-05-06"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("5-6-8"), QLatin1String("MM-DD-YYYY")), QLatin1String("2008-05-06"))

    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("3.21.01"), QLatin1String("MM-DD-YY")), QLatin1String("2001-03-21"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("1/ 1' 6"), QLatin1String("DD-MM-YY")), QLatin1String("2006-01-01"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("6/ 1/94"), QLatin1String("DD-MM-YY")), QLatin1String("1994-01-06"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("21/12'2001"), QLatin1String("DD-MM-YYYY")), QLatin1String("2001-12-21"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("21122001"), QLatin1String("DDMMYYYY")), QLatin1String("2001-12-21"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("12212001"), QLatin1String("MMDDYYYY")), QLatin1String("2001-12-21"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("010203"), QLatin1String("MMDDYY")), QLatin1String("2003-01-02"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("010203"), QLatin1String("DDMMYY")), QLatin1String("2003-02-01"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("3/9/04"), QLatin1String("DD-MM-YY")), QLatin1String("2004-09-03"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("31Dec2005"), QLatin1String("DDMMMYYYY")), QLatin1String("2005-12-31"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("31-Dec-2005"), QLatin1String("DD-MMM-YYYY")), QLatin1String("2005-12-31"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("31/Dec/2005"), QLatin1String("DD/MMM/YYYY")), QLatin1String("2005-12-31"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("31Dec05"), QLatin1String("DDMMMYY")), QLatin1String("2005-12-31"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("31-Dec-05"), QLatin1String("DD-MMM-YY")), QLatin1String("2005-12-31"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("31/Dec/05"), QLatin1String("DD/MMM/YY")), QLatin1String("2005-12-31"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("31DEC2005"), QLatin1String("DDMMMYYYY")), QLatin1String("2005-12-31"))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("INVALIDDATE"), QLatin1String("DD-MM-YY")), SKGServices::dateToSqlString(QDateTime::currentDateTime()))
    SKGTEST(QLatin1String("STR:dateToSqlString"), SKGServices::dateToSqlString(QLatin1String("02.01.2015"), QLatin1String("DD-MM-YYYY")), QLatin1String("2015-01-02"))
}

void test_nbWorkingDays(int& nberror, int& nbcheck, bool showonlyfailures)
{
    SKGTEST(QLatin1String("SKGServices::nbWorkingDays"), SKGServices::nbWorkingDays(QDate(2010, 9, 3), QDate(2010, 9, 6)), 1)
    SKGTEST(QLatin1String("SKGServices::nbWorkingDays"), SKGServices::nbWorkingDays(QDate(2010, 9, 6), QDate(2010, 9, 3)), 1)
    SKGTEST(QLatin1String("SKGServices::nbWorkingDays"), SKGServices::nbWorkingDays(QDate(2010, 9, 3), QDate(2010, 9, 3)), 1)
    SKGTEST(QLatin1String("SKGServices::nbWorkingDays"), SKGServices::nbWorkingDays(QDate(2010, 9, 1), QDate(2010, 9, 3)), 2)
    SKGTEST(QLatin1String("SKGServices::nbWorkingDays"), SKGServices::nbWorkingDays(QDate(2010, 9, 1), QDate(2010, 9, 8)), 5)
}

void test_getnext(int& nberror, int& nbcheck, bool showonlyfailures)
{
    SKGTEST(QLatin1String("SKGServices::getNextString"), SKGServices::getNextString(QLatin1String("12345")), QLatin1String("12346"))
    SKGTEST(QLatin1String("SKGServices::getNextString"), SKGServices::getNextString(QLatin1String("9")), QLatin1String("10"))
}

void test_errors(int& nberror, int& nbcheck, bool showonlyfailures)
{
    {
        SKGTraces::cleanProfilingStatistics();

        SKGError err;
        SKGTEST(QLatin1String("ERR:Default RC"), err.getReturnCode(), 0)
        SKGTEST(QLatin1String("ERR:Default RC"), err.getReturnCode(), 0)
        SKGTESTBOOL("ERR:isWarning", err.isWarning(), false)
        SKGTESTBOOL("ERR:isSucceeded", err.isSucceeded(), true)
        SKGTESTBOOL("ERR:isFailed", err.isFailed(), false)
        SKGTESTBOOL("ERR:!", !err, true)
        SKGTESTBOOL("ERR:operator bool", err, false)
        SKGTESTBOOL("ERR:getPreviousError", (err.getPreviousError() == nullptr), true)

        SKGTEST(QLatin1String("ERR:Default message"), err.getMessage(), QLatin1String(""))
        err.setReturnCode(10);
        err.setMessage(QLatin1String("Invalid parameter"));
        SKGTEST(QLatin1String("ERR:getHistoricalSize"), err.getHistoricalSize(), 0)
        SKGTEST(QLatin1String("ERR:RC 10"), err.getReturnCode(), 10)
        SKGTEST(QLatin1String("ERR:MSG Invalid parameter"), err.getMessage(), QLatin1String("Invalid parameter"))
        err.addError(11, QLatin1String("Message 11"));
        SKGTESTBOOL("ERR:isWarning", err.isWarning(), false)
        SKGTESTBOOL("ERR:isSucceeded", err.isSucceeded(), false)
        SKGTESTBOOL("ERR:isFailed", err.isFailed(), true)
        SKGTESTBOOL("ERR:!", !err, false)
        SKGTESTBOOL("ERR:operator bool", err, true)

        SKGTEST(QLatin1String("ERR:getHistoricalSize"), err.getHistoricalSize(), 1)
        err.addError(-12, QLatin1String("Message 12"));
        SKGTESTBOOL("ERR:isWarning", err.isWarning(), true)
        SKGTESTBOOL("ERR:isSucceeded", err.isSucceeded(), true)
        SKGTESTBOOL("ERR:isFailed", err.isFailed(), false)
        SKGTESTBOOL("ERR:!", !err, true)
        SKGTESTBOOL("ERR:operator bool", err, false)
        SKGTEST(QLatin1String("ERR:getHistoricalSize"), err.getHistoricalSize(), 2)
        SKGTEST(QLatin1String("ERR:getFullMessageWithHistorical"), err.getFullMessageWithHistorical(), QLatin1String("[WAR--12]: Message 12\n[ERR-11]: Message 11\n[ERR-10]: Invalid parameter"))
    }

    {
        SKGTraces::cleanProfilingStatistics();

        SKGError err;
        err.setReturnCode(10).setMessage(QLatin1String("Invalid parameter")).addError(11, QLatin1String("Message 11"));
        SKGTEST(QLatin1String("ERR:getHistoricalSize"), err.getHistoricalSize(), 1)
        SKGTEST(QLatin1String("ERR:RC 10"), err.getReturnCode(), 11)
        SKGTEST(QLatin1String("ERR:MSG Message 11"), err.getMessage(), QLatin1String("Message 11"))
        err.setProperty(QLatin1String("ABC"));
        SKGTEST(QLatin1String("ERR:Property ABC"), err.getProperty(), QLatin1String("ABC"))
    }
}

void test_getDateFormat(int& nberror, int& nbcheck, bool showonlyfailures)
{
    {
        QStringList dates;
        dates << QLatin1String("19/08/2008");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("DD-MM-YYYY"))
    }
    {
        QStringList dates;
        dates << QLatin1String("20080819");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("YYYYMMDD"))
    }
    {
        QStringList dates;
        dates << QLatin1String("10141989");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("MMDDYYYY"))
    }
    {
        QStringList dates;
        dates << QLatin1String("14101989");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("DDMMYYYY"))
    }
    {
        QStringList dates;
        dates << QLatin1String("05/08/2008") << QLatin1String("19/08/2008");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("DD-MM-YYYY"))
    }
    {
        QStringList dates;
        dates << QLatin1String("19/ 1' 6");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("DD-MM-YYYY"))
    }
    {
        QStringList dates;
        dates << QLatin1String("21/12'2001");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("DD-MM-YYYY"))
    }
    {
        QStringList dates;
        dates << QLatin1String("6/ 1/94");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("DD-MM-YYYY"))
    }
    {
        QStringList dates;
        dates << QLatin1String("10.29.07");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("MM-DD-YY"))
    }
    {
        QStringList dates;
        dates << QLatin1String("10.05.07");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("DD-MM-YY"))
    }
    {
        QStringList dates;
        dates << QLatin1String("2001.10.25");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("YYYY-MM-DD"))
    }
    {
        QStringList dates;
        dates << QLatin1String("45450116");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("YYYYMMDD"))
    }
    {
        QStringList dates;
        dates << QLatin1String("7/14' 0") << QLatin1String("11/30/99");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("MM-DD-YYYY"))
    }
    {
        QStringList dates;
        dates << QLatin1String("10/ 8'10");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("DD-MM-YYYY"))
    }
    {
        QStringList dates;
        dates << QLatin1String("7/8/06");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("DD-MM-YY"))
    }
    {
        QStringList dates;
        dates << QLatin1String("7/8/06") << QLatin1String("11/30/99") << QLatin1String("7/14' 0");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("MM-DD-YYYY"))
    }
    {
        QStringList dates;
        dates << QLatin1String("3/9/04");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("DD-MM-YY"))
    }
    {
        QStringList dates;
        dates << QLatin1String("31Dec2005");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("DDMMMYYYY"))
    }
    {
        QStringList dates;
        dates << QLatin1String("31-Dec-05");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("DD-MMM-YY"))
    }
    {
        QStringList dates;
        dates << QLatin1String("2008-05-25 00:00:00");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("YYYY-MM-DD"))
    }
    {
        QStringList dates;
        dates << QLatin1String("2008-05-25 01:02:03");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("YYYY-MM-DD"))
    }
    {
        QStringList dates;
        QFile inputFile(SKGTest::getTestPath(QLatin1String("IN")) % "/dates.txt");
        if (inputFile.open(QIODevice::ReadOnly)) {
            QTextStream in(&inputFile);
            while (!in.atEnd()) {
                QString l = in.readLine().trimmed();
                if (!l.isEmpty()) {
                    dates << l;
                }
            }

            inputFile.close();
        }
        SKGTEST(QLatin1String("STR:dats.count"), dates.count(), 2364)
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("MM-DD-YY"))
    }
    {
        QStringList dates;
        dates << QLatin1String("  7/8/06    ") << QLatin1String("");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String("DD-MM-YY"))
    }
    {
        QStringList dates;
        dates << QLatin1String("99999999") << QLatin1String("9999999999");
        SKGTEST(QLatin1String("STR:getDateFormat"), SKGServices::getDateFormat(dates), QLatin1String(""))
    }
}

void test_csv(int& nberror, int& nbcheck, bool showonlyfailures)
{
    SKGTEST(QLatin1String("STR:stringToCsv"), SKGServices::stringToCsv(QLatin1String("ABC")), QLatin1String("\"ABC\""))
    SKGTEST(QLatin1String("STR:stringToCsv"), SKGServices::stringToCsv(QLatin1String("ABC;CDE")), QLatin1String("\"ABC;CDE\""))
    SKGTEST(QLatin1String("STR:stringToCsv"), SKGServices::stringToCsv(QLatin1String("AB \"C\" DE")), QLatin1String("\"AB \"\"C\"\" DE\""))
    SKGTEST(QLatin1String("STR:stringToCsv"), SKGServices::stringToCsv(QLatin1String("AB \"C\";DE")), QLatin1String("\"AB \"\"C\"\";DE\""))

    QStringList parameters = SKGServices::splitCSVLine(QLatin1String("52.33,\"9/28/2010\",52.36,231803,52.33,0.00,+0.15,-,0.00,-,0.00,0.00,0.00,0.00,\"- - -\",\"-\",-,\"n\",N/A,0,+15.82,+43.33%,9,672,-1.08,-2.02%,-,-,\"51.76 - 52.57\",0.00,0.00,N/A,N/A,N/A,N/A,\"-\",51.91,52.18,-,\"+0.29%\",N/A,N/A,\"N/A\",N/A,\"N/A\",N/A,N/A,N/A,-,N/A,\"11:35am\",N/A,211524,-,\"36.51 - 53.41\",\"- - +0.29%\",\"Paris\",N/A,\"DASSAULT SYST.\""), ',');
    SKGTEST(QLatin1String("STR:splitCSVLine count"), parameters.count(), 59)

    parameters = SKGServices::splitCSVLine(QLatin1String("2013-04-02;transfer;\"a2\";'some text 2';-20,13"), ';', true);
    SKGTEST(QLatin1String("STR:splitCSVLine count"), parameters.count(), 5)
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[0], QLatin1String("2013-04-02"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[1], QLatin1String("transfer"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[2], QLatin1String("a2"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[3], QLatin1String("some text 2"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[4], QLatin1String("-20,13"))

    parameters = SKGServices::splitCSVLine(QLatin1String("\"A;'B\";'A;\"B'"), ';', true);
    SKGTEST(QLatin1String("STR:splitCSVLine count"), parameters.count(), 2)
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[0], QLatin1String("A;'B"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[1], QLatin1String("A;\"B"))

    SKGTEST(QLatin1String("STR:stringToHtml"), SKGServices::stringToHtml(QLatin1String("<hello>&<world>")), QLatin1String("&lt;hello&gt;&amp;&lt;world&gt;"))
    SKGTEST(QLatin1String("STR:htmlToString"), SKGServices::htmlToString(QLatin1String("&lt;hello&gt;&amp;&lt;world&gt;")), QLatin1String("<hello>&<world>"))

    parameters = SKGServices::splitCSVLine(SKGServices::stringToCsv(QLatin1String("Y")) % QLatin1Char(';') % SKGServices::stringToCsv(QLatin1String("A;B")));
    SKGTEST(QLatin1String("STR:splitCSVLine count"), parameters.count(), 2)
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[0], QLatin1String("Y"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[1], QLatin1String("A;B"))

    parameters = SKGServices::splitCSVLine(QLatin1String("A;\"B;C\";D"));
    SKGTEST(QLatin1String("STR:splitCSVLine count"), parameters.count(), 3)
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[0], QLatin1String("A"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[1], QLatin1String("B;C"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[2], QLatin1String("D"))

    parameters = SKGServices::splitCSVLine(QLatin1String("'A';'B'"));
    SKGTEST(QLatin1String("STR:splitCSVLine count"), parameters.count(), 2)
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[0], QLatin1String("A"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[1], QLatin1String("B"))

    parameters = SKGServices::splitCSVLine(QLatin1String("\"A ' B\",\"C\""));
    SKGTEST(QLatin1String("STR:splitCSVLine count"), parameters.count(), 2)
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[0], QLatin1String("A ' B"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[1], QLatin1String("C"))

    parameters = SKGServices::splitCSVLine(QLatin1String("'A \" B','C'"));
    SKGTEST(QLatin1String("STR:splitCSVLine count"), parameters.count(), 2)
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[0], QLatin1String("A \" B"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[1], QLatin1String("C"))

    QChar realsep;
    parameters = SKGServices::splitCSVLine(QLatin1String("\"A\",18,\"B\""), ';', true, &realsep);
    SKGTEST(QLatin1String("STR:splitCSVLine count"), parameters.count(), 3)
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[0], QLatin1String("A"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[1], QLatin1String("18"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[2], QLatin1String("B"))
    SKGTEST(QLatin1String("STR:sep"), realsep, ',')

    parameters = SKGServices::splitCSVLine(QLatin1String("30/05/2008;RETRAIT ESPECES AGENCE;                                ;      100,00;DEBIT;"));
    SKGTEST(QLatin1String("STR:splitCSVLine count"), parameters.count(), 6)

    parameters = SKGServices::splitCSVLine(QLatin1String("A|\"B\";\"C\""), '|', false);
    SKGTEST(QLatin1String("STR:splitCSVLine count"), parameters.count(), 2)
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[0], QLatin1String("A"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[1], QLatin1String("\"B\";\"C\""))

    parameters = SKGServices::splitCSVLine(QLatin1String("+123 \"-abc def\" \"e:f\" e:f"), ' ', true);
    SKGTEST(QLatin1String("STR:splitCSVLine count"), parameters.count(), 4)
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[0], QLatin1String("+123"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[1], QLatin1String("-abc def"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[2], QLatin1String("e:f"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[3], QLatin1String("e:f"))

    parameters = SKGServices::splitCSVLine(QLatin1String("A; \"B\"; \"C\" ;\"D\" ;E"), ';', true);
    SKGTEST(QLatin1String("STR:splitCSVLine count"), parameters.count(), 5)
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[0], QLatin1String("A"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[1], QLatin1String("B"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[2], QLatin1String("C"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[3], QLatin1String("D"))
    SKGTEST(QLatin1String("STR:splitCSVLine"), parameters[4], QLatin1String("E"))

    SKGTEST(QLatin1String("STR:splitCSVLine words"), SKGServices::splitCSVLine(QLatin1String("w1 w2"), ' ', true).count(), 2)
    SKGTEST(QLatin1String("STR:splitCSVLine words"), SKGServices::splitCSVLine(QLatin1String("\"w1 w1\""), ' ', true).count(), 1)
    SKGTEST(QLatin1String("STR:splitCSVLine words"), SKGServices::splitCSVLine(QLatin1String("\"w1 w1\" w2 \"w3 w3 w3\" w4"), ' ', true).count(), 4)

    QChar realseparator;
    SKGTEST(QLatin1String("STR:splitCSVLine comma"), SKGServices::splitCSVLine(QLatin1String("date,payee,amount"), ';', true, &realseparator).count(), 1)
    SKGTEST(QLatin1String("STR:splitCSVLine comma"), realseparator, ';')

    QString t(QLatin1String("Export Format, Date (YYYY-MM-DD as UTC), Time (HH:MM:SS), Merchant, Txn Amount (Funding Card), Txn Currency (Funding Card), Txn Amount (Foreign Spend), Txn Currency (Foreign Spend), Card Name, Card Last 4 Digits, Type, Category, Notes"));
    SKGTEST(QLatin1String("STR:splitCSVLine comma"), SKGServices::splitCSVLine(t, ';', true, &realseparator).count(), 1)
    SKGTEST(QLatin1String("STR:splitCSVLine comma"), SKGServices::splitCSVLine(t, ',', true, &realseparator).count(), 13)
}

void test_stringToSearchCriterias(int& nberror, int& nbcheck, bool showonlyfailures)
{
    SKGServices::SKGSearchCriteriaList criterias = SKGServices::stringToSearchCriterias(QLatin1String("abc +def +ghi"));
    SKGTEST(QLatin1String("STR:stringToSearchCriterias count"), criterias.count(), 3)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias mode"), criterias[0].mode, '+')
    SKGTEST(QLatin1String("STR:stringToSearchCriterias words count"), criterias[0].words.count(), 1)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias word"), criterias[0].words[0], QLatin1String("ghi"))
    SKGTEST(QLatin1String("STR:stringToSearchCriterias mode"), criterias[1].mode, '+')
    SKGTEST(QLatin1String("STR:stringToSearchCriterias words count"), criterias[1].words.count(), 1)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias word"), criterias[1].words[0], QLatin1String("def"))
    SKGTEST(QLatin1String("STR:stringToSearchCriterias mode"), criterias[2].mode, '+')
    SKGTEST(QLatin1String("STR:stringToSearchCriterias words count"), criterias[2].words.count(), 1)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias word"), criterias[2].words[0], QLatin1String("abc"))

    criterias = SKGServices::stringToSearchCriterias(QLatin1String("-abc +def ghi"));
    SKGTEST(QLatin1String("STR:stringToSearchCriterias count"), criterias.count(), 2)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias mode"), criterias[0].mode, '+')
    SKGTEST(QLatin1String("STR:stringToSearchCriterias words count"), criterias[0].words.count(), 2)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias word"), criterias[0].words[0], QLatin1String("def"))
    SKGTEST(QLatin1String("STR:stringToSearchCriterias word"), criterias[0].words[1], QLatin1String("ghi"))
    SKGTEST(QLatin1String("STR:stringToSearchCriterias mode"), criterias[1].mode, '-')
    SKGTEST(QLatin1String("STR:stringToSearchCriterias words count"), criterias[1].words.count(), 1)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias word"), criterias[1].words[0], QLatin1String("abc"))

    criterias = SKGServices::stringToSearchCriterias(QLatin1String("-abc +def ghi -10"));
    SKGTEST(QLatin1String("STR:stringToSearchCriterias count"), criterias.count(), 2)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias mode"), criterias[0].mode, '+')
    SKGTEST(QLatin1String("STR:stringToSearchCriterias words count"), criterias[0].words.count(), 3)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias word"), criterias[0].words[0], QLatin1String("def"))
    SKGTEST(QLatin1String("STR:stringToSearchCriterias word"), criterias[0].words[1], QLatin1String("ghi"))
    SKGTEST(QLatin1String("STR:stringToSearchCriterias word"), criterias[0].words[2], QLatin1String("-10"))
    SKGTEST(QLatin1String("STR:stringToSearchCriterias mode"), criterias[1].mode, '-')
    SKGTEST(QLatin1String("STR:stringToSearchCriterias words count"), criterias[1].words.count(), 1)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias word"), criterias[1].words[0], QLatin1String("abc"))

    criterias = SKGServices::stringToSearchCriterias(QLatin1String("-10"));
    SKGTEST(QLatin1String("STR:stringToSearchCriterias count"), criterias.count(), 1)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias mode"), criterias[0].mode, '+')
    SKGTEST(QLatin1String("STR:stringToSearchCriterias words count"), criterias[0].words.count(), 1)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias word"), criterias[0].words[0], QLatin1String("-10"))

    criterias = SKGServices::stringToSearchCriterias(QLatin1String("-abc"));
    SKGTEST(QLatin1String("STR:stringToSearchCriterias count"), criterias.count(), 2)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias mode"), criterias[0].mode, '+')
    SKGTEST(QLatin1String("STR:stringToSearchCriterias words count"), criterias[0].words.count(), 1)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias word"), criterias[0].words[0], QLatin1String(""))
    SKGTEST(QLatin1String("STR:stringToSearchCriterias mode"), criterias[1].mode, '-')
    SKGTEST(QLatin1String("STR:stringToSearchCriterias words count"), criterias[1].words.count(), 1)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias word"), criterias[1].words[0], QLatin1String("abc"))

    criterias = SKGServices::stringToSearchCriterias(QLatin1String("\"abc def ghi\" \"123 456\""));
    SKGTEST(QLatin1String("STR:stringToSearchCriterias count"), criterias.count(), 1)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias mode"), criterias[0].mode, '+')
    SKGTEST(QLatin1String("STR:stringToSearchCriterias words count"), criterias[0].words.count(), 2)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias word"), criterias[0].words[0], QLatin1String("abc def ghi"))
    SKGTEST(QLatin1String("STR:stringToSearchCriterias word"), criterias[0].words[1], QLatin1String("123 456"))

    criterias = SKGServices::stringToSearchCriterias(QLatin1String("\"-payee:abc def : ghi\" +amount:25"));
    SKGTEST(QLatin1String("STR:stringToSearchCriterias count"), criterias.count(), 2)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias mode"), criterias[0].mode, '+')
    SKGTEST(QLatin1String("STR:stringToSearchCriterias words count"), criterias[0].words.count(), 1)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias word"), criterias[0].words[0], QLatin1String("amount:25"))
    SKGTEST(QLatin1String("STR:stringToSearchCriterias mode"), criterias[1].mode, '-')
    SKGTEST(QLatin1String("STR:stringToSearchCriterias words count"), criterias[1].words.count(), 1)
    SKGTEST(QLatin1String("STR:stringToSearchCriterias word"), criterias[1].words[0], QLatin1String("payee:abc def : ghi"))

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("ToTo")),
                    QStringList() << QLatin1String("d_DATEOP") << QLatin1String("i_number") << QLatin1String("t_PAYEE") << QLatin1String("t_bookmarked"), nullptr),
            QLatin1String("((lower(d_DATEOP) LIKE '%toto%' OR lower(i_number) LIKE '%toto%' OR lower(t_PAYEE) LIKE '%toto%' OR lower(t_bookmarked) LIKE '%toto%'))"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("i_num:ToTo")),
                    QStringList() << QLatin1String("d_DATEOP") << QLatin1String("i_number") << QLatin1String("t_PAYEE") << QLatin1String("t_bookmarked"), nullptr),
            QLatin1String("((lower(i_number) LIKE '%toto%'))"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("i_num=1234")),
                    QStringList() << QLatin1String("d_DATEOP") << QLatin1String("i_number") << QLatin1String("t_PAYEE") << QLatin1String("t_bookmarked"), nullptr),
            QLatin1String("((i_number=1234))"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("t_PAYEE>ToTo")),
                    QStringList() << QLatin1String("d_DATEOP") << QLatin1String("i_number") << QLatin1String("t_PAYEE") << QLatin1String("t_bookmarked"), nullptr),
            QLatin1String("((lower(t_PAYEE)>'toto'))"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("t_PAYEE<ToTo")),
                    QStringList() << QLatin1String("d_DATEOP") << QLatin1String("i_number") << QLatin1String("t_PAYEE") << QLatin1String("t_bookmarked"), nullptr),
            QLatin1String("((lower(t_PAYEE)<'toto'))"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("t_PAYEE>=ToTo")),
                    QStringList() << QLatin1String("d_DATEOP") << QLatin1String("i_number") << QLatin1String("t_PAYEE") << QLatin1String("t_bookmarked"), nullptr),
            QLatin1String("((lower(t_PAYEE)>='toto'))"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("t_PAYEE<=ToTo")),
                    QStringList() << QLatin1String("d_DATEOP") << QLatin1String("i_number") << QLatin1String("t_PAYEE") << QLatin1String("t_bookmarked"), nullptr),
            QLatin1String("((lower(t_PAYEE)<='toto'))"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("t_PAYEE=ToTo")),
                    QStringList() << QLatin1String("d_DATEOP") << QLatin1String("i_number") << QLatin1String("t_PAYEE") << QLatin1String("t_bookmarked"), nullptr),
            QLatin1String("((lower(t_PAYEE)='toto'))"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("t_PAYEE#^t[o|a]to$")),
                    QStringList() << QLatin1String("d_DATEOP") << QLatin1String("i_number") << QLatin1String("t_PAYEE") << QLatin1String("t_bookmarked"), nullptr),
            QLatin1String("((REGEXP('^t[o|a]to$',t_PAYEE)))"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("+i_number<20 +i_number>30")),
                    QStringList() << QLatin1String("d_DATEOP") << QLatin1String("i_number") << QLatin1String("t_PAYEE") << QLatin1String("t_bookmarked"), nullptr),
            QLatin1String("((i_number>30)) OR ((i_number<20))"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("d_DATEOP>2015-05-04")),
                    QStringList() << QLatin1String("d_DATEOP") << QLatin1String("i_number") << QLatin1String("t_PAYEE") << QLatin1String("t_bookmarked"), nullptr),
            QLatin1String("((lower(d_DATEOP)>'2015-05-04'))"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("notfound:ToTo")),
                    QStringList() << QLatin1String("d_DATEOP") << QLatin1String("i_number") << QLatin1String("t_PAYEE") << QLatin1String("t_bookmarked"), nullptr),
            QLatin1String("(1=0)"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("v1 v2 +v3 -v4 -v5")),
                    QStringList() << QLatin1String("t_comment"), nullptr),
            QLatin1String("((lower(t_comment) LIKE '%v3%')) OR ((lower(t_comment) LIKE '%v1%') AND (lower(t_comment) LIKE '%v2%')) AND NOT((lower(t_comment) LIKE '%v4%')) AND NOT((lower(t_comment) LIKE '%v5%'))"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("v1 v2 +v3 -v4 -v5")),
                    QStringList() << QLatin1String("t_comment"), nullptr, true),
            QLatin1String("(((t_comment:v3) and ((t_comment:v1) and (t_comment:v2))) and not (t_comment:v4)) and not (t_comment:v5)"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("-v5")),
                    QStringList() << QLatin1String("t_comment"), nullptr),
            QLatin1String("((lower(t_comment) LIKE '%%')) AND NOT((lower(t_comment) LIKE '%v5%'))"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("a'b")),
                    QStringList() << QLatin1String("t_comment"), nullptr),
            QLatin1String("((lower(t_comment) LIKE '%a''b%'))"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("-v5")),
                    QStringList() << QLatin1String("t_comment"), nullptr, true),
            QLatin1String("(t_comment:) and not (t_comment:v5)"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String(":ToTo")),
                    QStringList() << QLatin1String("p_prop1") << QLatin1String("p_prop2"), nullptr),
            QLatin1String("((i_PROPPNAME='prop1' AND (lower(i_PROPVALUE) LIKE '%toto%') OR i_PROPPNAME='prop2' AND (lower(i_PROPVALUE) LIKE '%toto%')))"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String(":ToTo")),
                    QStringList() << QLatin1String("p_prop1") << QLatin1String("p_prop2"), nullptr, true),
            QLatin1String("p_prop1:toto or p_prop2:toto"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("#ToTo")),
                    QStringList() << QLatin1String("p_prop1") << QLatin1String("p_prop2"), nullptr),
            QLatin1String("((i_PROPPNAME='prop1' AND REGEXP('toto',i_PROPVALUE) OR i_PROPPNAME='prop2' AND REGEXP('toto',i_PROPVALUE)))"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("t_att>10")),
                    QStringList() << QLatin1String("t_att") << QLatin1String("t_att1"), nullptr),
            QLatin1String("((lower(t_att)>'10' OR lower(t_att1)>'10'))"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("t_att.>10")),
                    QStringList() << QLatin1String("t_att") << QLatin1String("t_att1"), nullptr),
            QLatin1String("((lower(t_att)>'10'))"));

    SKGTEST(QLatin1String("STR:searchCriteriasToWhereClause"),
            SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(QLatin1String("t_att.>10")),
                    QStringList() << QLatin1String("t_att") << QLatin1String("t_att1"), nullptr, true),
            QLatin1String("t_att>10"));
}

int main(int argc, char** argv)
{
    Q_UNUSED(argc)
    Q_UNUSED(argv)

    // Init test
    SKGINITTEST(true)

    // Test class SKGError
    test_errors(nberror, nbcheck, showonlyfailures);

    // SKGServices
    {
        test_getPeriodWhereClause(nberror, nbcheck, showonlyfailures);

        test_getNeighboringPeriod(nberror, nbcheck, showonlyfailures);

        test_periodToDate(nberror, nbcheck, showonlyfailures);

        test_partialStringToDate(nberror, nbcheck, showonlyfailures);

        test_conversions(nberror, nbcheck, showonlyfailures);

        test_nbWorkingDays(nberror, nbcheck, showonlyfailures);

        test_getDateFormat(nberror, nbcheck, showonlyfailures);

        test_csv(nberror, nbcheck, showonlyfailures);

        test_stringToSearchCriterias(nberror, nbcheck, showonlyfailures);

        test_getnext(nberror, nbcheck, showonlyfailures);

        // Various test
        SKGStringListList table;
        table.push_back(QStringList() << QLatin1String("Person") << QLatin1String("Salary") << QLatin1String("Age"));
        table.push_back(QStringList() << QLatin1String("John") << QLatin1String("58000") << QLatin1String("33"));
        table.push_back(QStringList() << QLatin1String("Paul") << QLatin1String("42000") << QLatin1String("25"));
        table.push_back(QStringList() << QLatin1String("Alan") << QLatin1String("65000") << QLatin1String("41"));

        SKGServices::getBase100Table(table);
        SKGServices::getPercentTable(table, true, false);
        SKGServices::getPercentTable(table, false, true);
        SKGServices::getHistorizedTable(table);

        SKGTEST(QLatin1String("STR:encodeForUrl"), SKGServices::encodeForUrl(QLatin1String("abc")), QLatin1String("abc"))

        QLocale::setDefault(QLocale::C);

        SKGTEST(QLatin1String("STR:getMajorVersion"), SKGServices::getMajorVersion(QLatin1String("4.3.12.3")), QLatin1String("4.3"))
        SKGTEST(QLatin1String("STR:getMajorVersion"), SKGServices::getMajorVersion(QLatin1String("4.3.12")), QLatin1String("4.3"))
        SKGTEST(QLatin1String("STR:getMajorVersion"), SKGServices::getMajorVersion(QLatin1String("4.3")), QLatin1String("4.3"))
        SKGTEST(QLatin1String("STR:getMajorVersion"), SKGServices::getMajorVersion(QLatin1String("4")), QLatin1String("4"))

        SKGTEST(QLatin1String("STR:toCurrencyString"), SKGServices::toCurrencyString(5.12341234, QLatin1String("F"), 2).remove(' '), QLatin1String("5.12F"))
        SKGTEST(QLatin1String("STR:toCurrencyString"), SKGServices::toCurrencyString(-5.12341234, QLatin1String("F"), 4).remove(' '), QLatin1String("-5.1234F"))

        SKGTEST(QLatin1String("STR:toPercentageString"), SKGServices::toPercentageString(5.12341234, 2), QLatin1String("5.12 %"))
        SKGTEST(QLatin1String("STR:toPercentageString"), SKGServices::toPercentageString(5.12341234, 4), QLatin1String("5.1234 %"))

        QByteArray tmp;
        SKGTESTERROR(QLatin1String("STR:downloadToStream"), SKGServices::downloadToStream(QUrl::fromLocalFile(QLatin1String("notfound")), tmp), false)

        SKGTEST(QLatin1String("STR:getFullPathCommandLine"), SKGServices::getFullPathCommandLine(QLatin1String("skrooge-release.py --help")), QLatin1String("skrooge-release.py --help"))
    }

    // End test
    SKGENDTEST()
}
