// Header for kmgroupware the kmail groupware handler
// written by Karl-Heinz Zimmer <khz@kde.org>

#ifndef KMGROUPWARE_H
#define KMGROUPWARE_H

#include <qguardedptr.h>

#include <kfoldertree.h>

#include "kmfoldertype.h"

class QSplitter;
class QDateTime;

class DCOPClient;

class KMKernel;
class KMFolder;
class KMFolderDir;
class KMFolderTreeItem;
class KMAccount;
class KMMainWin;
class KMMessage;
class KMHeaders;
class KMReaderWin;
class KMMimePartTree;
class KMMsgBase;

namespace KParts {
  class ReadOnlyPart;
};


class KMGroupware : public QObject
{
  Q_OBJECT

public:
  KMGroupware( QObject* parent = 0, const char* name = 0 );
  virtual ~KMGroupware();

  /**
   * Returns true if groupware mode is enabled and folder is one of the
   * groupware folders.
   */
  bool isGroupwareFolder( KMFolder* folder ) const;

  /**
   * @return The account containing the (currently active) groupware folders
   */
  KMAccount* groupwareAccount() const;

  /**
   * Returns the groupware folder type. Other is returned if groupware
   * isn't enabled or it isn't a groupware folder.
   */
  KFolderTreeItem::Type folderType( KMFolder* folder ) const;

  QString folderName( KFolderTreeItem::Type type, int language = -1 ) const;
  bool folderSelected( KMFolder* folder );
  bool isContactsFolder( KMFolder* folder ) const;
  bool checkFolders() const;

  /** Initialize all folders. */
  void initFolders();

  /** Disconnect all slots and close the dirs. */
  void cleanup();

  bool setFolderPixmap(const KMFolder& folder, KMFolderTreeItem& fti) const;
  void setupKMReaderWin(KMReaderWin* reader);
  void setMimePartTree(KMMimePartTree* mimePartTree);
  void createKOrgPart(QWidget* parent);
  void reparent(QSplitter* panner);
  void moveToLast();
  void setupActions(); // Not const since it emits a signal
  void enableActions(bool on) const;
  void processVCalRequest( const QCString& receiver, QString vCalIn,
                           QString& choice );
  void processVCalReply( const QCString& sender, const QString& vCalIn,
			 const QString& choice );

  /* (Re-)Read configuration file */
  void readConfig();

  bool isEnabled() const { return mUseGroupware; }

  bool hidingMimePartTree(){ return mGroupwareIsHidingMimePartTree; }

  // Find a message matching a given UID
  static KMMessage* findMessageByUID( const QString& uid, KMFolder* folder );

  // Convenience function to delete a message
  static void deleteMsg( KMMessage* msg );

  // retrieve matching body part (either text/vCal (or vCard) or application/ms-tnef)
  // and decode it
  // returns a readable vPart in *s or in *sc or in both
  // (please make sure to set s or sc to o if you don't want ot use it)
  // note: Additionally the number of the update counter (if any was found) is
  //       returned in aUpdateCounter, this applies only to TNEF data - in the
  //       iCal standard (RfC2445,2446) there is no update counter.
  static bool vPartFoundAndDecoded( KMMessage* msg,
                                    int& aUpdateCounter,
                                    QString& s );

  enum DefaultUpdateCounterValue { NoUpdateCounter=-1 };
  // functions to be called by KMReaderWin for 'print formatting'
  static bool vPartToHTML( int aUpdateCounter, const QString& vCal, QString fname,
                           bool useGroupware, QString& prefix, QString& postfix,
			   bool isSeparateWindow );
  static bool msTNEFToVPart( const QByteArray& tnef,
                             int& aUpdateCounter, // there is no such counter in RfC2466 (khz)
                             QString& aVPart );
  static bool msTNEFToHTML( KMReaderWin* reader, QString& vPart, QString fname, bool useGroupware,
                            QString& prefix, QString& postfix, bool isSeparateWindow );

  // function to be called by KMReaderWin for analyzing of clicked URL
  static bool foundGroupwareLink( const QString aUrl,
                                  QString& gwType,
                                  QString& gwAction,
                                  QString& gwAction2,
                                  QString& gwData );

  /** KMReaderWin calls this with an URL. Return true if a groupware url was
      handled. */
  virtual bool handleLink( const KURL &aUrl, KMMessage* msg );

  /** These methods are called by KMKernel's DCOP functions. */
  virtual void requestAddresses( QString );
  virtual bool storeAddresses(QString, QStringList);
  virtual bool lockContactsFolder();
  virtual bool unlockContactsFolder();

  // automatic resource handling
  bool incomingResourceMessage( KMAccount*, KMMessage* );

  // tell KOrganizer about messages to be deleted
  void msgRemoved( KMFolder*, KMMessage* );

  void setMainWin(KMMainWin *mainWin) { mMainWin = mainWin; }
  void setHeaders(KMHeaders* headers );

  // Use this to add a vCard to the current addressbook
  void addVCard( const QString& vCard, bool checkLock=true );

public slots:
  /** View->Groupware menu */
  void slotGroupwareHide();
  /** additional groupware slots */
  void slotGroupwareShow(bool);

    /** KO informs about a new or updated event */
  void slotUpdateIncidence( const QString& type, const QString& uid, 
			    const QString& ical );
  void slotDeleteIncidence( const QString& type, const QString& uid );
  void slotInitIncidences( const QString& type );

  /** KO needs to know if we're stripping alarms from sends (another
      Outlook compatibility thing) */
  void slotGetStripAlarmsForSending( bool &isStripping );

  /** Send ical mails out. Method is one of "REQUEST", "REPLY" or "UPDATE" */
  void slotSendICalMessage( const QString& method, const QString& subject,
			    const QStringList& recipients, const QString& ical );


  /** KO informs about a new or updated note */
  void slotNewOrUpdatedNote( const QString& id, const QString& geometry,
			     const QColor& color, const QString& text,
			     bool force );
  void slotDeleteNote( const QString& id );

  // This is only used to initialize the notes view
  void slotNotesFolderChanged();

  void slotIncidenceAdded( KMFolder* folder, int idx );
  void slotIncidenceDeleted( KMFolder* folder, Q_UINT32 sernum );

  /** Delete and sync the local IMAP cache  */
  void slotInvalidateIMAPFolders();

  // Show the KOrganizer handbook
  void slotKOrgHelp();

  // Mark a folder for re-sync after a conflict has been resolved
  void slotConflictResolved( const QString& foldertype );

  // Hold back syncing for a folder
  void slotHoldSyncing( const QString&, bool );

signals:
  void signalSetGroupwareCommunicationEnabled( QObject* );

  // Make KOrganizer read everything from scratch
  void signalRefreshAll();

  /** Initialize Groupware with a list of Notes entries */
  void signalRefreshNotes( const QStringList& );

  void incidenceAdded( const QString& type, const QString& ical, bool& accepted );
  void incidenceDeleted( const QString& type, const QString& uid );

  // These two clears the calendar and task views
  void signalCalendarFolderExpunged();
  void signalTasksFolderExpunged();

  /** Make sure a given time span is visible in the Calendar */
  void signalCalendarUpdateView( const QDateTime&, const QDateTime& );

  void signalShowCalendarView();
  void signalShowContactsView();
  void signalShowNotesView();
  void signalShowTodoView();

  /** Open Groupware to consider accepting/declining an invitation */
  void signalEventRequest( const QCString& receiver, const QString&, bool&,
			   QString&, QString&, bool& );

  /** Use Groupware to create an answer to a resource request. */
  void signalResourceRequest( const QValueList<QPair<QDateTime, QDateTime> >& busy,
                              const QCString& resource,
                              const QString& vCalIn, bool& vCalInOK,
                              QString& vCalOut, bool& vCalOutOK,
                              bool& isFree, QDateTime& start, QDateTime& end );

  /** Accept an invitation without checking: Groupware will *not* show up */
  void signalAcceptedEvent( bool, const QCString&, const QString&, bool&,
			    QString&, bool& );

  /** Reject an invitation: Groupware will *not* show up */
  void signalRejectedEvent( const QCString&, const QString&, bool&, QString&,
			    bool& );

  /** Answer an invitation */
  void signalIncidenceAnswer( const QCString&, const QString&, QString& );

  /** A note was added */
  void signalNoteAdded( const QString&, bool& );

  /** A note was deleted */
  void signalNoteDeleted( const QString& );

  /** The menus were changed */
  void signalMenusChanged();

  // Hold the event handlers
  void syncRunning( const QString& type, bool running );

private slots:
  void slotSyncRunning( KMFolder* folder, bool running );

private:
  /** Helper function for initFolders. Initializes a single folder. */
  KMFolder* initFolder( KFolderTreeItem::Type itemType, const char* typeString );

  void saveActionEnable( const QString& name, bool on ) const;

  // Figure out if a vCal is a todo, event or neither
  enum VCalType { vCalEvent, vCalTodo, vCalUnknown };
  static VCalType getVCalType( const QString &vCard );

  /** This class functions as an event filter while showing groupware widgets */
  bool eventFilter( QObject *o, QEvent *e ) const;

  void internalCreateKOrgPart();
  void loadPixmaps() const;

  void connectNotes( KMFolder* folder );
  void connectCalendar( KMFolder* folder );
  void connectTasks( KMFolder* folder );

  // We use QGuardedPtr for everything, since
  // we are not the owner of any of those objects
  QGuardedPtr<KMMainWin>      mMainWin;
  QGuardedPtr<KMHeaders>      mHeaders;
  QGuardedPtr<KMReaderWin>    mReader;
  QGuardedPtr<KMMimePartTree> mMimePartTree;

  // Groupware folder icons:
  static QPixmap *pixContacts, *pixCalendar, *pixNotes, *pixTasks;

  bool mUseGroupware;
  bool mGroupwareIsHidingMimePartTree;
  int  mFolderLanguage;
  QSplitter* mPanner;
  QGuardedPtr<KParts::ReadOnlyPart> mKOrgPart;
  QGuardedPtr<QWidget> mKOrgPartParent;

  KMFolderDir* mFolderParent;
  KMFolderType mFolderType;

  bool mContactsLocked;
  KMFolder* mContacts;
  KMFolder* mCalendar;
  KMFolder* mNotes;
  KMFolder* mTasks;

  KMFolder* mOrigContacts;
  KMFolder* mOrigCalendar;
  KMFolder* mOrigNotes;
  KMFolder* mOrigTasks;
};

#endif /* KMGROUPWARE_H */
