/*
    This file is part of KOrganizer.
    Copyright (c) 2001 Cornelius Schumacher <schumacher@kde.org>
    Copyright (c) 2002 Klarlvdalens Datakonsult AB <info@klaralvdalens-datakonsult.se>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    As a special exception, permission is given to link this program
    with any edition of Qt, and distribute the resulting executable,
    without including the source code for Qt in the source distribution.
*/

#include "koeventtimewidget.h"
#include <klocale.h>
#include <kglobal.h>
#include <kdebug.h>
#include <kmessagebox.h>
#include "ktimeedit.h"
#include <libkdepim/kdateedit.h>
#include <qcheckbox.h>
#include <qgrid.h>
#include <qhbox.h>
#include <qlabel.h>

/*!  Constructs an empty KOEventTimeWidget

*/

KOEventTimeWidget::KOEventTimeWidget( QWidget* parent, const char* name ) :
    QVGroupBox( i18n( "Date && Time" ), parent, name )
{
    QGrid* grid = new QGrid( 3, this );
    grid->setSpacing( 11 );

    new QLabel(i18n("Start:"), grid);
    mStartDateEdit = new KDateEdit(grid);
    mStartTimeEdit = new KOTimeEdit(grid);
    new QLabel(i18n("End:"),grid);
    mEndDateEdit = new KDateEdit(grid);
    mEndTimeEdit = new KOTimeEdit(grid);

    QHBox* bottomLine = new QHBox( this );
    bottomLine->setSpacing( 11 );

    mRecursButton = new QCheckBox(i18n("Recurring event"),bottomLine);
    connect(mRecursButton,SIGNAL(toggled(bool)),SIGNAL(recursChanged(bool)));
#ifdef KORG_NORECURRENCE
    mRecursButton->hide();
#endif

    mNoTimeButton = new QCheckBox(i18n("No time associated"),bottomLine);
    connect(mNoTimeButton, SIGNAL(toggled(bool)),SLOT(dontAssociateTime(bool)));

    mDurationLabel = new QLabel(bottomLine);

    // time widgets are checked if they contain a valid time
    connect(mStartTimeEdit, SIGNAL(timeChanged(QTime)),
            this, SLOT(startTimeChanged(QTime)));
    connect(mEndTimeEdit, SIGNAL(timeChanged(QTime)),
            this, SLOT(endTimeChanged(QTime)));

    // date widgets are checked if they contain a valid date
    connect(mStartDateEdit, SIGNAL(dateChanged(QDate)),
            this, SLOT(startDateChanged(QDate)));
    connect(mEndDateEdit, SIGNAL(dateChanged(QDate)),
            this, SLOT(endDateChanged(QDate)));
}


void KOEventTimeWidget::startTimeChanged(QTime newtime)
{
  // kdDebug(5850) << "KOEventTimeWidget::startTimeChanged() " << newtime.toString() << endl;

  int secsep = mCurrStartDateTime.secsTo(mCurrEndDateTime);

  mCurrStartDateTime.setTime(newtime);

  // adjust end time so that the event has the same duration as before.
  mCurrEndDateTime = mCurrStartDateTime.addSecs(secsep);
  mEndTimeEdit->setTime(mCurrEndDateTime.time());

  emit dateTimesChanged(mCurrStartDateTime,mCurrEndDateTime);
}


void KOEventTimeWidget::endTimeChanged(QTime newtime)
{
  // kdDebug(5850) << "KOEventTimeWidget::endTimeChanged " << newtime.toString() << endl;

  QDateTime newdt(mCurrEndDateTime.date(), newtime);
  mCurrEndDateTime = newdt;

  emit dateTimesChanged(mCurrStartDateTime,mCurrEndDateTime);
}


void KOEventTimeWidget::startDateChanged(QDate newdate)
{
  int daysep = mCurrStartDateTime.daysTo(mCurrEndDateTime);

  mCurrStartDateTime.setDate(newdate);

  // adjust end date so that the event has the same duration as before
  mCurrEndDateTime.setDate(mCurrStartDateTime.date().addDays(daysep));
  mEndDateEdit->setDate(mCurrEndDateTime.date());

  emit dateTimesChanged(mCurrStartDateTime,mCurrEndDateTime);
}


void KOEventTimeWidget::endDateChanged(QDate newdate)
{
  QDateTime newdt(newdate, mCurrEndDateTime.time());

  if(newdt < mCurrStartDateTime) {
    // oops, we can't let that happen.
    newdt = mCurrStartDateTime;
    mEndDateEdit->setDate(newdt.date());
    mEndTimeEdit->setTime(newdt.time());
  }
  mCurrEndDateTime = newdt;

  emit dateTimesChanged(mCurrStartDateTime,mCurrEndDateTime);
}


void KOEventTimeWidget::timeStuffDisable(bool disable)
{
  if (disable) {
    mStartTimeEdit->setEnabled(false);
    mEndTimeEdit->setEnabled(false);
  } else {
    mStartTimeEdit->setEnabled(true);
    mEndTimeEdit->setEnabled(true);
  }
  setDuration();
}


void KOEventTimeWidget::setDuration()
{
  QString tmpStr, catStr;
  int hourdiff, minutediff;
  // end<date is an accepted temporary state while typing, but don't show
  // any duration if this happens
  if(mCurrEndDateTime >= mCurrStartDateTime) {

    if (mNoTimeButton->isChecked()) {
      int daydiff = mCurrStartDateTime.date().daysTo(mCurrEndDateTime.date()) + 1;
      tmpStr = i18n("Duration: ");
      tmpStr.append(i18n("1 Day","%n Days",daydiff));
    } else {
      hourdiff = mCurrStartDateTime.date().daysTo(mCurrEndDateTime.date()) * 24;
      hourdiff += mCurrEndDateTime.time().hour() -
                  mCurrStartDateTime.time().hour();
      minutediff = mCurrEndDateTime.time().minute() -
                   mCurrStartDateTime.time().minute();
      // If minutediff is negative, "borrow" 60 minutes from hourdiff
      if (minutediff < 0 && hourdiff > 0) {
        hourdiff -= 1;
        minutediff += 60;
      }
      if (hourdiff || minutediff){
        tmpStr = i18n("Duration: ");
        if (hourdiff){
          catStr = i18n("1 hour","%n hours",hourdiff);
          tmpStr.append(catStr);
        }
        if (hourdiff && minutediff){
          tmpStr += i18n(", ");
        }
        if (minutediff){
          catStr = i18n("1 minute","%n minutes",minutediff);
          tmpStr += catStr;
        }
      } else tmpStr = "";
    }
  }
  mDurationLabel->setText(tmpStr);
}


void KOEventTimeWidget::dontAssociateTime(bool noTime)
{
  timeStuffDisable(noTime);
  //if(alarmButton->isChecked()) alarmStuffDisable(noTime);
  emit allDayChanged(noTime);
}


void KOEventTimeWidget::slotDateTimesChanged(QDateTime start, QDateTime end,
                                            const QString& dtStr,
                                            bool emitSignals )
{
  // kdDebug(5850) << "+++KOEventTimeWidget::slotDateTimesChanged(): Start DateTime: " << start.toString() << endl;

  // the signal calling this slot is emitted by KOTimeEdit::setTime()

  mStartDateEdit->setDate(start.date());
  mStartTimeEdit->setTime(start.time());
  // kdDebug(5850) << "+++KOEventTimeWidget::slotDateTimesChanged(): mStartTimeEdit->getTime(): " << mStartTimeEdit->getTime().toString() << endl;
  mEndDateEdit->setDate(end.date());
  mEndTimeEdit->setTime(end.time());
  mCurrStartDateTime = start;
  mCurrEndDateTime = end;
  if( emitSignals )
    emit dateTimesChanged(mCurrStartDateTime,mCurrEndDateTime);
  setDuration();
}


void KOEventTimeWidget::setDefaults(QDateTime from,QDateTime to,bool allDay)
{
  mNoTimeButton->setChecked(allDay);
  timeStuffDisable(allDay);

  slotDateTimesChanged(from,to, "", false );

  mRecursButton->setChecked(false);
}


void KOEventTimeWidget::readEvent( KCal::Event *event, bool tmpl )
{
  if ( !tmpl ) {
    // the rest is for the events only
    mNoTimeButton->setChecked(event->doesFloat());
    timeStuffDisable(event->doesFloat());

    slotDateTimesChanged(event->dtStart(),event->dtEnd(), "", false );
  }

//   kdDebug(5850) << "KOEventTimeWidget::readEvent(). this = " << this
// 		<< ", event type = " << event->type() <<endl;
//   kdDebug(5850) << ", rec = " << event->recurrence() << endl;
  mRecursButton->setChecked(event->recurrence()->doesRecur());
}


void KOEventTimeWidget::writeEvent( KCal::Event *event )
{
  QDate tmpDate;
  QTime tmpTime;
  QDateTime tmpDT;

  // temp. until something better happens.
  QString tmpStr;

  if (mNoTimeButton->isChecked()) {
    event->setFloats(true);
    // need to change this.
    tmpDate = mStartDateEdit->date();
    tmpTime.setHMS(0,0,0);
    tmpDT.setDate(tmpDate);
    tmpDT.setTime(tmpTime);
    event->setDtStart(tmpDT);

    tmpDate = mEndDateEdit->date();
    tmpTime.setHMS(0,0,0);
    tmpDT.setDate(tmpDate);
    tmpDT.setTime(tmpTime);
    event->setDtEnd(tmpDT);
  } else {
    event->setFloats(false);

    // set date/time end
    tmpDate = mEndDateEdit->date();
    tmpTime = mEndTimeEdit->getTime();
    tmpDT.setDate(tmpDate);
    tmpDT.setTime(tmpTime);
    event->setDtEnd(tmpDT);
    // qDebug( "KOEventTimeWidget::writeEvent(): tmpDTEnd = %s", tmpDT.toString().latin1() );

    // set date/time start
    tmpDate = mStartDateEdit->date();
    tmpTime = mStartTimeEdit->getTime();
    // qDebug( "+++tmpDate = %s", tmpDate.toString().latin1() );
    // qDebug( "+++tmpTime = %s", tmpTime.toString().latin1() );
    tmpDT.setDate(tmpDate);
    tmpDT.setTime(tmpTime);
    event->setDtStart(tmpDT);
    // qDebug( "KOEventTimeWidget::writeEvent(): tmpDTStart = %s", tmpDT.toString().latin1() );
  } // check for float

}


bool KOEventTimeWidget::validateInput()
{
    if (!mNoTimeButton->isChecked()) {
        if (!mStartTimeEdit->inputIsValid()) {
            KMessageBox::sorry(0,i18n("Please specify a valid start time."));
            return false;
        }

        if (!mEndTimeEdit->inputIsValid()) {
            KMessageBox::sorry(0,i18n("Please specify a valid end time."));
            return false;
        }
    }

    if (!mStartDateEdit->inputIsValid()) {
        KMessageBox::sorry(0,i18n("Please specify a valid start date."));
        return false;
    }

    if (!mEndDateEdit->inputIsValid()) {
        KMessageBox::sorry(0,i18n("Please specify a valid end date."));
        return false;
    }

    QDateTime startDt,endDt;
    startDt.setDate(mStartDateEdit->date());
    endDt.setDate(mEndDateEdit->date());
    if (!mNoTimeButton->isChecked()) {
        startDt.setTime(mStartTimeEdit->getTime());
        endDt.setTime(mEndTimeEdit->getTime());
    }

    if (startDt > endDt) {
        KMessageBox::sorry(0,i18n("The event ends before it starts.\n"
                                  "Please correct dates and times."));
        return false;
    }

    return true;
}


void KOEventTimeWidget::setAllDay( bool flag, bool emitSignals )
{
    if( !emitSignals )
        mNoTimeButton->blockSignals( true );
    mNoTimeButton->setChecked( flag );
    mStartTimeEdit->setEnabled( !flag );
    mEndTimeEdit->setEnabled( !flag );
    if( !emitSignals )
        mNoTimeButton->blockSignals( false );
}


void KOEventTimeWidget::setRecurs( bool flag, bool emitSignals )
{
    if( !emitSignals )
        mRecursButton->blockSignals( true );
    mRecursButton->setChecked( flag );
    if( !emitSignals )
        mRecursButton->blockSignals( false );
}


/*!
  Returns a QDateTime object with the currently selected start time
*/

QDateTime KOEventTimeWidget::startDateTime() const
{
  return QDateTime( mStartDateEdit->date(), mStartTimeEdit->getTime() );
}


/*!
  Returns a QDateTime object with the currently selected end time
*/

QDateTime KOEventTimeWidget::endDateTime() const
{
    return QDateTime( mEndDateEdit->date(), mEndTimeEdit->getTime() );
}
