/* ============================================================
 *
 * This file is a part of digiKam project
 * https://www.digikam.org
 *
 * Date        : 2009-02-04
 * Description : an unit-test to check Qt PGF interface
 *
 * SPDX-FileCopyrightText: 2011-2022 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * ============================================================ */

#include "qtpgf_utest.h"

// Qt includes

#include <QTest>
#include <QFile>
#include <QBuffer>
#include <QImage>

// Local includes

#include "digikam_debug.h"
#include "dtestdatadir.h"
#include "pgfutils.h"

using namespace Digikam;

QTEST_GUILESS_MAIN(QtPGFTest)

QtPGFTest::QtPGFTest(QObject* const parent)
    : QObject(parent)
{
}

void QtPGFTest::testQtPGFIface()
{
    clock_t    start, end;
    QImage     img;
    QByteArray pgfData, jpgData, pngData;

    // QImage => PGF conversion

    qCDebug(DIGIKAM_TESTS_LOG) << "Using LibPGF version: " << PGFUtils::libPGFVersion();

    QString fname = DTestDataDir::TestData(QString::fromUtf8("core/tests/fileio"))
                       .root().path() + QLatin1String("/test.png");
    qCDebug(DIGIKAM_TESTS_LOG) << "Test Data File:" << fname;

    QVERIFY2(img.load(fname, "PNG"), "Cannot load PNG test image");

    qCDebug(DIGIKAM_TESTS_LOG) << "Generate PGF file using file stream";

    // First, write QImage as PGF file using file stream

    QVERIFY2(PGFUtils::writePGFImageFile(img, QLatin1String("test-filestream.pgf"), 0, true),
             "Cannot write PGF file");

    qCDebug(DIGIKAM_TESTS_LOG) << "Generate PGF file using data stream";

    start = clock();

    // Second, write QImage as PGF file using data stream

    QVERIFY2(PGFUtils::writePGFImageData(img, pgfData, 0, true),
             "Cannot write PGF data stream");

    end   = clock();

    qCDebug(DIGIKAM_TESTS_LOG) << "PGF Encoding time: " << double(end - start)/CLOCKS_PER_SEC << " s";

    // Write PGF file.

    QFile file(QLatin1String("test-datastream.pgf"));

    QVERIFY2(file.open(QIODevice::WriteOnly),
             "Cannot open PGF file to write");

    QDataStream stream(&file);
    stream.writeRawData(pgfData.data(), pgfData.size());
    file.close();

    // PGF => QImage conversion

    qCDebug(DIGIKAM_TESTS_LOG) << "Load PGF file generated by data stream";

    start = clock();

    QVERIFY2(PGFUtils::readPGFImageData(pgfData, img, true),
             "Cannot load PGF data stream to bytearray");

    end   = clock();

    QVERIFY2(img.save(QLatin1String("test2.png"), "PNG"),
             "Cannot generate second test PNG image");

    qCDebug(DIGIKAM_TESTS_LOG) << "PGF Decoding time: " << double(end - start)/CLOCKS_PER_SEC << " s";

    // JPEG tests for comparisons.

    QVERIFY2(img.load(fname, "PNG"), "Cannot load PNG test image");

    qCDebug(DIGIKAM_TESTS_LOG) << "Generate JPG file to compare performances";

    start = clock();

    QBuffer buffer(&jpgData);
    buffer.open(QIODevice::WriteOnly);

    // Here we will use JPEG quality = 85 to reduce artifacts.

    QVERIFY2(img.save(&buffer, "JPEG", 85),
             "Cannot save JPEG version");  // Here we will use JPEG quality = 85 to reduce artifacts.

    QVERIFY2(!jpgData.isNull(), "JPEG data stream is null");

    end   = clock();

    qCDebug(DIGIKAM_TESTS_LOG) << "JPG Encoding time: " << double(end - start)/CLOCKS_PER_SEC << " s";

    start = clock();

    buffer.open(QIODevice::ReadOnly);
    QVERIFY2(img.load(&buffer, "JPEG"),
             "Cannot load JPEG data stream to byte array");

    end   = clock();

    qCDebug(DIGIKAM_TESTS_LOG) << "JPG Decoding time: " << double(end - start)/CLOCKS_PER_SEC << " s";

    // PNG tests for comparisons.

    QVERIFY2(img.load(fname, "PNG"), "Cannot load PNG test image");

    qCDebug(DIGIKAM_TESTS_LOG) << "Generate PNG file to compare performances";

    start = clock();

    QBuffer buffer2(&pngData);
    buffer2.open(QIODevice::WriteOnly);
    QVERIFY2(img.save(&buffer2, "PNG", 100),
             "Cannot load PNG data stream to byte array");

    QVERIFY2(!pngData.isNull(), "PNG data stream is null");

    end   = clock();

    qCDebug(DIGIKAM_TESTS_LOG) << "PNG Encoding time: " << double(end - start)/CLOCKS_PER_SEC << " s";

    start = clock();

    buffer2.open(QIODevice::ReadOnly);
    QVERIFY2(img.load(&buffer2, "PNG"),
             "Cannot load PNG data stream to byte array");

    QVERIFY2(!pngData.isNull(), "PNG data stream is null");

    end   = clock();

    qCDebug(DIGIKAM_TESTS_LOG) << "PNG Decoding time: " << double(end - start)/CLOCKS_PER_SEC << " s";
}
