/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

#ifndef LIBRARY_H
#define LIBRARY_H

// Qt includes
#include <qobject.h>
#include <qvaluelist.h>

// forward declarations
namespace WorKflow
{
  class LibraryDescription;
  class CommandFactoryBase;
  class ApplicationFactoryBase;
}

namespace WorKflow
{
  /**
   * @short Base class for WorKflow command libraries.
   *
   * This class carries a set of commands, categories and datatypes, conversions
   * and application descriptions from a loadable module and makes them available
   * to the WorKflow libraries. If you want to provide you own commands, you need
   * to derive from this class to export them.
   *
   * In subclasses, you need to implement the init() function. Use this function
   * to create your command factories and application factories. Each instance of
   * these classes must be registered with the Library instance using the
   * addCommandFactory() and addApplicationFactory() methods. The registration
   * is needed to delete the instances when the library is unloaded. Example:
   *
   * @code
   *   void MyLibrary::init()
   *   {
   *     // add new commands
   *     addCommandFactory(new CommandFactory<TestCommand>("play video"));
   *   }
   * @endcode
   *
   * This class is designed to work together with KDE's plugin system, using
   * KPluginInfo in particular. To tell KDE about the existance of your command
   * library, you need to do two more things: first, you need to export your
   * Library subclass from your loadable module. The easiest way to do this is
   * to use the K_EXPORT_COMPONENT_FACTORY macro (this would go into the file
   * @c mylibrary.cpp, in this example):
   *
   * @code
   *   typedef KGenericFactory<MyLibrary> Factory;
   *   K_EXPORT_COMPONENT_FACTORY(libworkflow_mylibrary, Factory("workflow_mylibrary"))
   * @endcode
   *
   * The second thing you need to do is to provide a @c .desktop file containing
   * at least the following entries:
   *
   * - @c Type - The @c .desktop file type, needs to be @c Service
   * - @c ServiceTypes - The offered service types, needs to be at least
   *   @c WorKflow/Library.
   * - @c X-KDE-Library - Your library's name.
   * - @c X-KDE-PluginInfo-Name - An internal name, used for the library. Can be
   *   equal to @c X-KDE-Library.
   * - @c Name - The library's user visible name, possibly translated into other
   *   languages.
   *
   * The following entries are optional, but it is strongly recommended that you
   * add them to your @c .desktop file:
   *
   * - @c Comment - A short description of your library, possibly translated into
   *   other languages.
   * - @c X-KDE-PluginInfo-Author - Your name.
   * - @c X-KDE-PluginInfo-Email - Your e-mail address.
   * - @c X-KDE-PluginInfo-Version - The library's version.
   * - @c X-KDE-PluginInfo-Website - The library's website.
   * - @c X-KDE-PluginInfo-Depends - Any dependencies the command library has on
   *   other command libraries must be listed here.
   * - @c X-KDE-PluginInfo-License - The library's license.
   * - @c X-KDE-PluginInfo-EnabledByDefault - Set this to @c false if your
   *   library should not be enabled by default, e.g. if it is in an experimental
   *   stage.
   *
   * An example of a WorKflow library's @c .desktop file could look something
   * like this:
   *
   * @code
   *   [Desktop Entry]
   *   Encoding=UTF-8
   *   Type=Service
   *   Icon=kmail
   *   ServiceTypes=WorKflow/Library
   *   X-KDE-Library=workflow_kmail
   *   X-KDE-PluginInfo-Author=Thomas Kadauke
   *   X-KDE-PluginInfo-Email=tkadauke@gmx.de
   *   X-KDE-PluginInfo-Name=workflow_kmail
   *   X-KDE-PluginInfo-Version=0.2.0
   *   X-KDE-PluginInfo-Website=http://workflow.kde.org
   *   X-KDE-PluginInfo-Depends=
   *   X-KDE-PluginInfo-License=GPL
   *   X-KDE-PluginInfo-EnabledByDefault=false
   *   Name=Kmail Commands
   * @endcode
   *
   * @author Thomas Kadauke
   */
  class Library : public QObject
  {
    Q_OBJECT

  public:
    /**
     * Constructor. The three parameters are mandatory because of KDE's component
     * factory.
     * @param parent The parent object.
     * @param name The object's name.
     * @param args Some arguments given to the instance through KDE's component
     *   factory. These arguments are ignored in this class, but you're free to
     *   do something useful with them in your subclass.
     */
    Library(QObject* parent, const char* name, const QStringList& args);

    /**
     * Destructor. Deletes all registered Datatype, Category and CommandFactory
     * instances.
     */
    virtual ~Library();

    /**
     * Loads the library's XML description file and creates the described
     * objects.
     * @param libName The filename of the XML description.
     */
    void loadXML(const QString& libName);

    /**
     * Initializes the command library. In order to make your commands and
     * applications public, you need to implement this function and create the
     * appropriate instances. Then, register them with the addCommandFactory()
     * and addApplicationFactory() methods.
     */
    virtual void init() = 0;

  protected:
    /**
     * Registers a new command factory with the library. It is recommended that
     * the CommandFactory template is used here.
     * @param factory The new command factory.
     */
    void addCommandFactory(CommandFactoryBase* factory);

    /**
     * Registers a new application factory with the library. It is recommended
     * that the ApplicationFactory template is used here.
     * @param factory The new application factory.
     */
    void addApplicationFactory(ApplicationFactoryBase* factory);

  private:
    class Private;
    Private* d;
  };
}

#endif
