/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE includes
#include <kdebug.h>

// Qt includes
#include <qdom.h>

// WorKflow includes
#include "datatype.h"
#include "typemanager.h"
#include "librarydescription.h"

using namespace WorKflow;

class Datatype::Private
{
public:
  void readEnumItems(const QDomElement& e);
  void writeEnumItems(QDomDocument& doc, QDomElement& e);

  LibraryDescription* parent;

  QString pluralName;
  QString baseTypeId;

  mutable Datatype* baseType;

  bool isEnum;
  QStringList enumKeys;
  QStringList enumNames;
};

// Datatype::Private implementation
void Datatype::Private::readEnumItems(const QDomElement& e)
{
  QDomNode i = e.firstChild();
  while (!i.isNull()) {
    QDomElement item = i.toElement();
    if (!item.isNull()) {
      if (item.tagName() == "item") {
        QString key = item.attribute("key");
        QString name = item.text();
        if (key.isNull() || name.isNull()) {
          kdWarning() << "malformed enum item in datatype" << endl;
        } else {
          enumKeys.append(key);
          enumNames.append(name);
        }
      }
    }
    i = i.nextSibling();
  }
}

void Datatype::Private::writeEnumItems(QDomDocument& doc, QDomElement& e)
{
  QDomElement enums = doc.createElement("enumitems");

  QStringList::ConstIterator k = enumKeys.begin();
  QStringList::ConstIterator n = enumNames.begin();

  for ( ; k != enumKeys.end(); ++k, ++n) {
    QDomElement item = doc.createElement("item");
    item.setAttribute("key", *k);
    QDomText text = doc.createTextNode(*n);
    item.appendChild(text);
    enums.appendChild(item);
  }

  e.appendChild(enums);
}

// Datatype implementation
Datatype::Datatype(LibraryDescription* parent)
  : XMLDescription()
{
  d = new Private;
  d->parent = parent;
  d->baseType = 0;
  d->isEnum = false;
  d->parent->addDatatype(this);
}

Datatype::~Datatype()
{
  // clean up on destruction and unregister type
  unregisterType();
  d->parent->removeDatatype(this);
  delete d;
}

QString Datatype::pluralName() const
{
  return d->pluralName;
}

Datatype* Datatype::baseType() const
{
  if (!d->baseType && !d->baseTypeId.isEmpty()) {
    d->baseType = TypeManager::self()->find(d->baseTypeId);
  }
  return d->baseType;
}

void Datatype::setPluralName(const QString& plural)
{
  d->pluralName = plural;
}

QString Datatype::baseTypeId() const
{
  return d->baseTypeId;
}

void Datatype::setBaseTypeId(const QString& id)
{
  d->baseTypeId = id;
  d->baseType = 0;
}

bool Datatype::isConvertibleTo(Datatype* dest)
{
  return TypeManager::self()->findConversion(this, dest) != 0;
}

void Datatype::registerType()
{
  TypeManager::self()->registerType(this);
}

void Datatype::unregisterType()
{
  TypeManager::self()->unregisterType(this);
}

bool Datatype::isEnum() const
{
  return d->isEnum;
}

QStringList Datatype::enumKeys() const
{
  return d->enumKeys;
}

QStringList Datatype::enumNames() const
{
  return d->enumNames;
}

void Datatype::clearEnumItems()
{
  d->isEnum = false;
  d->enumKeys.clear();
  d->enumNames.clear();
}

void Datatype::addEnumItem(const QString& key, const QString& name)
{
  d->isEnum = true;
  d->enumKeys.append(key);
  d->enumNames.append(name);
}

void Datatype::readXML(const QDomElement& e)
{
  XMLDescription::readXML(e);

  QString basetype = e.attribute("basetype");
  if (!basetype.isNull()) {
    d->baseTypeId = basetype;
  }

  QDomNode n = e.firstChild();
  while (!n.isNull()) {
    QDomElement e = n.toElement();
    if (!e.isNull()) {
      if (e.tagName() == "pluralname")
        setPluralName(e.text());
      else if (e.tagName() == "enumitems") {
        d->isEnum = true;
        d->readEnumItems(e);
      }
    }
    n = n.nextSibling();
  }

  // register type automatically
  registerType();
}

void Datatype::writeXML(QDomDocument& doc, QDomElement& e)
{
  QDomElement type = doc.createElement("datatype");

  XMLDescription::writeXML(doc, type);

  if (!d->baseTypeId.isEmpty())
    type.setAttribute("basetype", d->baseTypeId);

  QDomElement plural = doc.createElement("pluralname");
  QDomText text = doc.createTextNode(d->pluralName);
  plural.appendChild(text);
  type.appendChild(plural);

  if (d->isEnum)
    d->writeEnumItems(doc, type);
  e.appendChild(type);
}

bool Datatype::inherits(Datatype* base)
{
  if (base == 0)
    return false;

  if (base == this)
    return true;
  else if (baseType())
    return baseType()->inherits(base);
  return false;
}
