#ifndef __KMTRACE_LEAKSVIEW__H__
#define __KMTRACE_LEAKSVIEW__H__

// Qt includes
#include <qwidget.h>

// KDE include
#include <klistview.h>
#include <kpopupmenu.h>

// KMtraceViewer includes
#include "KMtraceModel.hh"
#include "KMtraceLeaksViewItem.hh"
#include "KMtraceLeaksFindDialog.hh"
#include "KMtraceSuppressDialog.hh"

/**
 * The widget is an extension of a KListView class. Its list view
 * items are of type @ref KMtraceLeaksViewItem, which is the view
 * of a @ref KMtraceLeak model or a @ref KMtraceCall.
 *
 * @short The widget showing a list a memory leaks and their data
 * @author Andreas Baumann <andreas_baumann@users.sourceforge.net>
 * @see KMtraceLeaksViewItem, KMtraceLeak
 * @version $ProjectMajorVersion: 0 $
 */
class KMtraceLeaksView : public KListView
{
   Q_OBJECT
   
   public:
      /**
       * Create a new list view for all leaks of the model.
       *
       * @param parent the parent widget which should contain this widget
       * @param model the model to be shown
       */
      KMtraceLeaksView( QWidget *parent, KMtraceModel *model );
      
      /**
       * Sets (and replaces) the model of memory leaks to be shown.
       * This function creates the list items of the widget (leak
       * information and call stacks) based on the information from
       * the model.
       *
       * @param model a model containing the memory leaks and the suppression
       */
      void setModel( KMtraceModel *model );

      /**
       * Boolean flag which specifies if suppressed leaks should be shown
       * (and marked and suppressed leaks) or if they should be hidden from
       * the list view.
       *
       * @param show
       */
      void setShowSuppressed( bool show );

   signals:
       /**
        * The module with name 'module' should be displayed at line 'line'.
        * At the moment we inform a source code viewer in class
        * @ref KMtraceViewerMainWindow, could also be the kdevelop
        * main window if integrating the leak detection more into kdevelop
        * as tool plugin.
        *
        * @param module the filename of file containig the source code
        *        of the leaking code
        * @param line set the focus to this line
        */
       void moduleChanged( QString module, int line );

       /**
        * No module should be shown (as no function in the any call stack
        * is currently selected).
        * At the moment we inform a source code viewer in class
        * @ref KMtraceViewerMainWindow, could also be the kdevelop
        * main window if integrating the leak detection more into kdevelop
        * as tool plugin.
        */
       void moduleUnselect( );
   
   public slots:
      /**
       * Expand all call stacks of all memory leaks.
       */
      void slotExpandAll( );
      
      /**
       * Collapse all call stacks of all memory leaks.
       */
      void slotCollapseAll( );
      
      /**
       * Open a non-modal memory leak find dialog.
       */
      void slotFind( );
      
   private slots:
      void slotContextLeak( QListViewItem *item, const QPoint &point, int column );
      void slotSelectionChanged( QListViewItem *item );
      void slotExecuted( QListViewItem *item );
      void slotSearch( );
      void slotSearchDone( );
      void slotSuppress( KMtraceLeak *leak, QString string );
      void slotSuppressionChanged( KMtraceLeak *leak );
   
   private:
      // expand all call stacks of all memory leaks
      void doExpandAll( );
      
      // collapse all call stacks of all memory leaks
      void doCollapseAll( );
      
      // called if a memory leak gets selected by the user. Depending
      // whether the user sets the focus on an item of the call stack
      // with source code or not the signal moduleChanged respectively
      // moduleUnselect is emitted.
      void doSelectItem( QListViewItem *item );
      
      // starts or continues the search of a KMtraceLeakFindDialog and
      // select the item if a matching one is found.
      void doSearch( int where, QString what,
                     bool caseSensitive );   
       
      // contains the model of the view (list of leaks with callstacks)
      KMtraceModel *m_model;
      
      // the popup menu for leaks
      KPopupMenu *m_menu;
      
      // itentifier for the "Suppress" context menu operation
      int m_menuItemIdSuppress;
      
      // the leak find dialog (keep it non-modal)
      KMtraceLeaksFindDialog *m_findDialog;

      // the suppress dialog (simple version to add a suppression)
      KMtraceSuppressDialog *m_suppressDialog;
      
      // the current search position in the find dialog (an iterator,
      // the currently found item and a Y coordinate which indicates
      // where the found item is positioned from the origin of the
      // KListView (easier way possible?). This is needed to set the
      // focus onto the found memory leak.
      QListViewItemIterator *m_findIterator;
      QListViewItem *m_findItem;
      int m_posY;

      bool m_showSuppressed;
};

#endif
