/***************************************************************************
                          account.h  -  description
                             -------------------
    begin                : Sat Aug 10 2002
    copyright            : (C) 2002 by Richard Garand
    email                : richard@garandnet.net
    $Id: account.h,v 1.13 2002/09/09 03:09:03 richard Exp $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef ACCOUNT_H
#define ACCOUNT_H

#include <qstring.h>
#include <qvaluelist.h>
#include <qdatetime.h>
#include <qmap.h>

class KBudgetDoc;
class QTextStream;

/**Contains information and a list of transactions for an account
  *@author Richard Garand
  */

class Account {
  public:
    friend class KBudgetDoc;

    /** account types */
    enum types { AC_INCOME = 0, AC_EXPENSE = 1, AC_CASH = 2, AC_ALL };

    /** default constructor */
    Account();
    /** reads data from a stream; version is the stream's format version */
    Account(KBudgetDoc* doc, int id, types type, QTextStream &stream, int version);
    /** sets account type */
    Account(KBudgetDoc* doc, int id, types type, QString name, float initialBalance);
    /** copy constructor */
    Account(const Account& from);
    /** Account destructor */
    ~Account();

    /** writes the data to the text stream */
    void serialize(QTextStream &stream);

    /** return the account ID */
    int id() const;
    
    /** set the type of the account */
    void setType(types type);
    /** return the type of the account */
    types type() const;

    /** sets the name of the account */
    void setName(QString name);
    /** returns the name of the account */
    QString name() const;

    /** adds a transaction to the account */
    void addTransaction(int id);
    /** removes a transaction from the account */
    void removeTransaction(int id);

    /** return a list of transactions in the selected periods (0 = any) */
    QValueList<int> getTransactions(int year = 0, int month = 0) const;
    /** return list of transactions in the selected period (includes start and end dates) */
    QValueList<int> getTransactions(QDate start, QDate end) const;

    /** return the current balance */
    float getBalance() const;
    /** returns the starting balance */
    float startingBalance() const;
    /** sets the starting balance */
    void setStartingBalance(float balance);
    /** return the difference between the starting balance and the current balance */
    float change() const;

    /** returns the initial budgeted amount for the requested month (defaults to first month) */
    float startingBudget(QDate month = QDate()) const;
    /** returns the final budgeted amount for the requested month (defaults to last month) */
    float budget(QDate month = QDate()) const;
    /** returns the amount budgeted for the specified month (Default is last month) */
    float budgeted(QDate month = QDate()) const;
    /** sets the amount budgeted for the specified month. Value for day can be anything */
    void setBudgeted(QDate month, float amount);

    /** returns a copy of the account. If month is set to 0, it will be ignored.
      * If toDate is set, select a range (including fromDate and toDate) */
    Account clone(QDate fromDate, QDate toDate = QDate()) const;

    /** print debugging data */
    void dump() const;
  private:
    class Month {
      public:
        Month() : startBalance(0), endBalance(0), startBudget(0), endBudget(0), budgeted(0) {};
        float			startBalance;
        float			endBalance;
        QValueList<int> transactions;
		float           startBudget;
		float           endBudget;
		float           budgeted;
    };
    typedef QMap<int,Month> year_t;

    bool isInRange(QDate from, QDate to, QDate test) const;
    /** make sure a month is properly initialized */
    void createMonth(int year, int month, QMap<int,year_t>::iterator &yearIt, year_t::iterator &monthIt);
    /** return the last month before date or 0 */
    const Month* monthBefore(QDate date) const;
    /** add or remove a transaction */
    void transactionUpdate(int id, bool adding);
    /** add a certain amount to the start,end balance,budget for all months starting with the given iterators */
    void add(QMap<int,year_t>::iterator year, year_t::iterator month, float balance, float budget);

    /** type of account */
    types				m_type;
    /** name of account */
    QString				m_name;
    /** initial balance */
    float               initialBalance;
    /** list of transactions */
    QMap<int,year_t>	months;
	KBudgetDoc*			m_doc;
	int					m_id;
};

#endif
