/*
 * KHeiseReg
 *
 * A utility to search for articles within the Heise register.
 *
 * Copyright (C) 2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <qglobal.h>

#include <kapplication.h>
#include <klocale.h>
#include <kglobal.h>

#include "articleview.h"
#include "global.h"



ArticleViewItem::ArticleViewItem(QListView *parent, const char *name): KListViewItem(parent, name)
{
}


ArticleViewItem::~ArticleViewItem()
{
}


QString ArticleViewItem::key(int column, bool ascending) const
{
	return (column == ARTICLE_COL_EDITION) ? m_edition_key : text(column);
}


void ArticleViewItem::setEntry(const RegEntry &e)
{
	QString s;

	setText(ARTICLE_COL_TITLE, e.lineStr(REG_LINE_TITLE));
	setText(ARTICLE_COL_SUBTITLE, e.lineStr(REG_LINE_SUBTITLE));
	if (e.magazine() == REG_MAGAZINE_CT)
		setText(ARTICLE_COL_MAGAZINE, QString::fromLatin1("c't"));
	else if (e.magazine() == REG_MAGAZINE_IX)
		setText(ARTICLE_COL_MAGAZINE, QString::fromLatin1("iX"));
	s.sprintf("%d/%04d", e.edition(), e.year());
	setText(ARTICLE_COL_EDITION, s);

	m_edition_key.sprintf("%04d%02d", e.year(), e.edition());

	m_authors = e.lineStr(REG_LINE_AUTHOR);
	m_editor = e.lineStr(REG_LINE_EDITOR);
	m_page = e.page();
	m_links = e.lineStr(REG_LINE_LINKS);
}



ArticleView::ArticleView(QWidget *parent, const char *name): KListView(parent, name)
{
	config = kapp->config();

	addColumn(i18n("Title"));
	addColumn(i18n("Subtitle"));
	addColumn(i18n("Magazine"));
	addColumn(i18n("Edition"));
	setColumnAlignment(ARTICLE_COL_MAGAZINE, AlignVCenter | AlignHCenter);
	setColumnAlignment(ARTICLE_COL_EDITION, AlignVCenter | AlignRight);
	setColumnWidthMode(ARTICLE_COL_TITLE, Manual);
	setColumnWidthMode(ARTICLE_COL_SUBTITLE, Manual);
	setColumnWidthMode(ARTICLE_COL_MAGAZINE, Manual);
	setColumnWidthMode(ARTICLE_COL_EDITION, Manual);
	setMinimumSize(sizeHint());
	setSorting(ARTICLE_COL_EDITION, false);
	setAllColumnsShowFocus(true);
	setShowSortIndicator(true);
/*
	contextMenu = new QPopupMenu(0, "contextmenu");
	Q_CHECK_PTR(contextMenu);
	contextMenu->insertItem(SmallIcon("foobar"), i18n("&Foobar"), ID_FOO_BAR);
	connect(this, SIGNAL(rightButtonClicked(QListViewItem *, const QPoint&, int)), SLOT(rightButtonClickedSlot(QListViewItem *, const QPoint&, int)));
	connect(contextMenu, SIGNAL(activated(int)), SLOT(contextMenuSlot(int)));
*/
}


ArticleView::~ArticleView()
{
/*
	delete contextMenu;
*/
}


void ArticleView::readOptions()
{
	config->setGroup("ArticleWindow Options");

	setColumnWidth(0, config->readNumEntry("TitleWidth", 174));
	setColumnWidth(1, config->readNumEntry("SubTitleWidth", 350));
	setColumnWidth(2, config->readNumEntry("MagazineWidth", 64));
	setColumnWidth(3, config->readNumEntry("EditionWidth", 80));
}


void ArticleView::saveOptions()
{
	config->setGroup("ArticleWindow Options");

	config->writeEntry("TitleWidth", columnWidth(0));
	config->writeEntry("SubTitleWidth", columnWidth(1));
	config->writeEntry("MagazineWidth", columnWidth(2));
	config->writeEntry("EditionWidth", columnWidth(3));
}


void ArticleView::addEntry(const RegEntry &entry)
{
	ArticleViewItem *item;

	item = new ArticleViewItem(this);
	Q_CHECK_PTR(item);

	item->setEntry(entry);
}


/*
void ArticleView::enableCommand(int id_, bool enable)
{
	contextMenu->setItemEnabled(id_, enable);
}


void ArticleView::rightButtonClickedSlot(QListViewItem *item, const QPoint &point, int)
{
	if (item) {
		setSelected(item, true);
		contextMenu->popup(point);
	}
}


void ArticleView::contextMenuSlot(int id_)
{
	emit command(id_);
}
*/
