/*
 *
 * Copyright (C) 2002-2003 George Staikos <staikos@kde.org>
 * Copyright (C) 2002 Richard Moore <rich@kde.org>
 * Copyright (C) 2004 Dirk Ziegelmeier <dziegel@gmx.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <kapplication.h>

#include "kdetv_v4l.h"

#include <qwidget.h>
#include <qspinbox.h>
#include <qradiobutton.h>
#include <qcheckbox.h>
#include <qtimer.h>
#include <qthread.h>
#include <qevent.h>
#include <qdir.h>
#include <qsize.h>
#include <klocale.h>
#include <kdebug.h>
#include <math.h>
#include <assert.h>
#include <qapplication.h>
#include <qcolor.h>
#include <qimage.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>

#ifdef HAVE_DVBS
// not supported yet
#undef HAVE_DVBS
#endif

#ifdef HAVE_DVBS
#include <linux/dvb/frontend.h>
#include <linux/dvb/dmx.h>
#endif

#ifdef HAVE_XRANDR
#include <X11/Xlib.h>
#include <X11/extensions/Xrandr.h>
#endif

#include <kconfig.h>
#include <kscreensaver_vroot.h>

#include "v4ldev.h"
#include "v4ldevtuner.h"
#include "v4ldevcamera.h"
#include "v4lutil.h"
#include "overlaycontroller.h"

#include "kdetv_v4l.moc"


class V4LErrorEvent : public QEvent
{
public:
    V4LErrorEvent(const QString& msg)
        : QEvent(QEvent::User),
          _msg(msg)
    {
    }

    ~V4LErrorEvent()
    {
    }

    QString _msg;
};


class KDWidget : public QWidget
{
public:
	KDWidget()
        : QWidget()
    {
        create(DefaultRootWindow(qt_xdisplay()), false, true);
    }

	virtual ~KDWidget()
    {
        destroy(false, false);
    }
};


class V4LGrabber : public QObject, public QThread
{
public:
	V4LGrabber(QObject *owner, V4LDev *dev, QVideoStream* vs, int gsn);
	virtual ~V4LGrabber();
	virtual void run();
	void stop() {  _stop = true; }
	QMutex & mutex() { return _devMtx; }

private:
	QObject *_owner;
	V4LDev *_d;
	volatile bool _stop;
	QMutex _devMtx;
	V4LImage _image;
	int _errors;
	QVideoStream* _vs; 
	int _gsn;
};


struct V4LGrabberLocker
{
    V4LGrabberLocker(V4LGrabber * g)
        : _g(g) 
    {
        if (_g) _g->mutex().lock();
    }

    ~V4LGrabberLocker()
    {
        if (_g) _g->mutex().unlock();
    }

    V4LGrabber * _g;
};

V4LGrabber::V4LGrabber(QObject *owner, V4LDev *dev, QVideoStream* vs, int gsn) 
    : _owner(owner),
      _d(dev),
      _stop(false),
      _devMtx(true),
      _errors(0),
      _vs(vs),
      _gsn(gsn)
{
}

V4LGrabber::~V4LGrabber()
{
    fprintf(stderr, "V4LGrabber::~V4LGrabber(): wait().\n");
    stop();
    wait();
    fprintf(stderr, "V4LGrabber::~V4LGrabber(): deleted.\n");
}

void V4LGrabber::run()
{
	int skip = 10;   // number of frames to drop at startup (for sync purposes)
	_errors = 0;
    
	if (!_d)
		return;

	// skip the first n frames for syncing
	while ( !_stop && (skip-- > 0) ) {
		_devMtx.lock();
		_d->grab(0);
		_devMtx.unlock();
	}
	
	while (!_stop) {
		// grab image
		_devMtx.lock();
		if (_stop) {
		    _devMtx.unlock();
		    break;
		}
		int rc = _d->grab(&_image);
        _devMtx.unlock();

		if (!rc) {
			// paint image to TV widget
			qApp->lock();
			if (_stop) {
				qApp->unlock();
				break;
			}
			_vs->setInputSize(QSize(_image.width, _image.height));
            rc = _vs->displayFrame(_image.buffer);
			qApp->unlock();
		}

        if (!rc) {
			_errors = 0;
        } else if (++_errors > 50) {
            qApp->postEvent( _owner,
                             new V4LErrorEvent("Unable to grab video.\nVideo display is not possible with the current plugin configuration.\
                                                Try playing with the configuration options of the V4L plugin.") );
			fprintf(stderr, "Too many errors.  Ending V4L grabbing.\n");
			break;
		}
	}

	while (!_stop); // if we bail out above because of an error, don't delete the object
	deleteLater();
}


KdetvV4L::KdetvV4L(Kdetv *ktv, QWidget *parent, const char *name)
    : KdetvSourcePlugin(ktv, "v4l", parent, name),
      _w(parent), 
      _dtReg(new KDWidget),
      _winReg(parent),
      dev(0),
      g(0),
      _probed(false),
      _capturing(false),
      _gsn(0)
{
    _vs = new QVideoStream(_w);

    // figure out best method available
    int bestAvailable = QVIDEO_METHOD_XVSHM;
    if (!_vs->haveMethod(bestAvailable)) {
        bestAvailable = QVIDEO_METHOD_XV;
    }
    if (!_vs->haveMethod(bestAvailable)) {
        bestAvailable = QVIDEO_METHOD_XSHM;
    }
    if (!_vs->haveMethod(bestAvailable)) {
        bestAvailable = QVIDEO_METHOD_X11;
    }
    
    _cfg->setGroup("V4L Plugin");

#ifdef HAVE_DVBS
    _dvbs = _cfg->readBoolEntry("Use DVB-S", false);
#else
    _dvbs = false;
#endif
    _mirror = _cfg->readBoolEntry("Mirror Image", false);
    _frameRate = _cfg->readNumEntry("Max Framerate", 22);

    // don't annoy users that already configured plugin
    if (_cfg->hasKey("GD Method")) {
        _autoConfig = _cfg->readBoolEntry("Autoconfigure", false);
    } else {
        _autoConfig = _cfg->readBoolEntry("Autoconfigure", true);
    }

    if (_autoConfig) {
        _qvsMethod = bestAvailable;

        if ( (_qvsMethod == QVIDEO_METHOD_X11)  ||
             (_qvsMethod == QVIDEO_METHOD_XSHM)    ) {
            _useOverlay = true;
        } else {
            _useOverlay = false;
        }
        _changeRes = false;
    } else {
        _qvsMethod = _cfg->readNumEntry("GD Method", bestAvailable);

        // saved config may be invalid due to graphics card change
        if (!_vs->haveMethod(_qvsMethod)) {
            _qvsMethod = bestAvailable;
        }

        // Try to use overlay if we only have X11 display method
        if ( (_qvsMethod == QVIDEO_METHOD_X11)  ||
             (_qvsMethod == QVIDEO_METHOD_XSHM)    ) {
            _useOverlay = _cfg->readBoolEntry("Use Overlay", true);
        } else {
            _useOverlay = _cfg->readBoolEntry("Use Overlay", false);
        }

        _changeRes = _cfg->readBoolEntry("Change Screen Resolution", false);
    }

    _vs->setMethod(_qvsMethod);

    /*
     * The signal to stop capture must be delivered while qApp is in its event loop
     * because the shutdown relies on a working Qt Library Mutex handshake.
     * When qApp is destroyed (at the end of main()), it destroys all QObjects,
     * but without locking the mutex.
     * This makes the thread shutdown crash because the _stop flag is set during object deletion,
     * so the thread can paint to a widget that is being deleted using a buffer that is being
     * deleted (deletion of tv screen and V4LGrabber).
     */
    connect(qApp, SIGNAL( aboutToQuit() ),
			this, SLOT( stopVideo() ) );

    connect( parent, SIGNAL( resized(int, int) ),
             this, SLOT( viewResized() ) );
    connect( parent, SIGNAL( moved(int, int) ),
             this, SLOT( viewMoved() ) );

    kdDebug() << "Kdetv V4L plugin loaded successfully." << endl;
}


KdetvV4L::~KdetvV4L()
{
    stopVideo();
    delete dev;
    dev = 0;
    delete _vs;
    delete _dtReg;
}


bool KdetvV4L::isTuner()
{
    V4LGrabberLocker l(g);
    return dev ? dev->isTuner() : false;
}


int KdetvV4L::brightness()
{
    V4LGrabberLocker l(g);
    return dev ? dev->brightness() : -1;
}


QColor KdetvV4L::colourKey()
{
    V4LGrabberLocker l(g);
    return dev ? dev->colourKey() : QColor();
}


int KdetvV4L::colour()
{
    V4LGrabberLocker l(g);
    return dev ? dev->colour() : -1;
}


int KdetvV4L::hue()
{
    V4LGrabberLocker l(g);
    return dev ? dev->hue() : -1;
}


int KdetvV4L::contrast()
{
    V4LGrabberLocker l(g);
    return dev ? dev->contrast() : -1;
}


int KdetvV4L::whiteness()
{
    V4LGrabberLocker l(g);
    return dev ? dev->contrast() : -1;
}


int KdetvV4L::signal()
{
    V4LGrabberLocker l(g);
    return dev ? dev->signal() : -1;
}


/* ARGH, this doesn't work on my card.  It always returns 0! */
bool KdetvV4L::muted()
{
    V4LGrabberLocker l(g);
    if (dev)
        return dev->audioEnabled();
    return false;
}


int KdetvV4L::setDevice( const QString &name )
{
    if (!_probed)
        probeDevices();

    if (dev) {
        stopVideo();
        delete dev;
    }

    _device = name;
    _currentDev = _devNames[name];
    kdDebug() << "V4L: setDevice [" << name 
              << "] which maps to " << _currentDev << endl;

    dev = V4LDev::getDevice( _currentDev );
    kdDebug() << "V4L: Success? " << (dev ? "true" : "false") << endl;

    _audioModes.clear();
    if (dev) {
        _audioModes += dev->audioModes();
    }

    return dev ? 0 : -1;
}


int KdetvV4L::setSource( const QString &src )
{
    V4LGrabberLocker l(g);
    if ( !dev )
        return -1;
    
    int rc = dev->setSource( src );
    _source = dev->source();
    return rc;
}


int KdetvV4L::setEncoding( const QString &encoding )
{
    V4LGrabberLocker l(g);
    if ( !dev )
        return -1;

    int rc = dev->setEncoding( encoding );
    _encoding = dev->encoding();
    return rc;
}


int KdetvV4L::frequency()
{
    V4LGrabberLocker l(g);
    if (dev && dev->isTuner())
        return 125 * static_cast<V4LTuner*>(dev)->freq() / 2;
    return -1;
}


void KdetvV4L::setFrequency( int freq )
{
    V4LGrabberLocker l(g);
    if ( !dev || !dev->isTuner() )
        return;
    static_cast<V4LTuner*>(dev)->setFreq(freq * 2 / 125);
}

const QStringList& KdetvV4L::broadcastedAudioModes()
{
    V4LGrabberLocker l(g);
    static QStringList empty;
    
    if ( !dev )
        return empty;

    return dev->broadcastedAudioModes();
}

const QString& KdetvV4L::defaultAudioMode()
{
    const QStringList& modes = broadcastedAudioModes();
    
    // FIXME: a more intelligent and general algorithm would not hurt...
    if (modes.contains("Stereo")) {
        return modes[modes.findIndex("Stereo")];
    }
    
    if (modes.contains("Language 1")) {
        return modes[modes.findIndex("Language 1")];
    }
    
    return broadcastedAudioModes().first();
}

int KdetvV4L::setAudioMode( const QString& audioMode )
{
    V4LGrabberLocker l(g);
    if ( !dev )
        return -1;
    
    return dev->setAudioMode(audioMode);
}

void KdetvV4L::setMuted( bool muted )
{
    V4LGrabberLocker l(g);
    if ( !dev )
        return;
    
    if ( muted )
        dev->disableAudio();
    else
        dev->enableAudio();
}


void KdetvV4L::setBrightness(int val)
{
    V4LGrabberLocker l(g);
    if ( !dev )
        return;
    dev->setBrightness( val );
}


void KdetvV4L::setColour(int val)
{
    V4LGrabberLocker l(g);
    if ( !dev )
        return;
    dev->setColour( val );
}


void KdetvV4L::setHue(int val)
{
    V4LGrabberLocker l(g);
    if ( !dev )
        return;
    dev->setHue( val );
}


void KdetvV4L::setContrast(int val) {
    V4LGrabberLocker l(g);
    if ( !dev )
        return;
    dev->setContrast( val );
}


void KdetvV4L::setWhiteness(int val)
{
    V4LGrabberLocker l(g);
    if ( !dev )
        return;
    dev->setWhiteness( val );
}


void KdetvV4L::viewMoved()
{
    V4LGrabberLocker l(g);
    if (!dev)
        return;

    if (dev->overlayOn()) {
        // Center the overlay video on the widget if
        // the widget is larger than the maximum capture size
        QSize maxSize = dev->getMaxImageSize();
        QSize wSize   = _w->size();
        int dx, dy;

        if (maxSize.width() < wSize.width()) {
            dx = (wSize.width() - maxSize.width()) / 2;
            wSize.setWidth(maxSize.width());
        } else {
            dx = 0;
        }

        if (maxSize.height() < wSize.height()) {
            dy = (wSize.height() - maxSize.height()) / 2;
            wSize.setHeight(maxSize.height());
        } else {
            dy = 0;
        }

        QRect scn = QApplication::desktop()->screenGeometry(_w);
        QRect g;
        g.moveTopLeft(_w->mapToGlobal(QPoint(dx, dy)) - scn.topLeft());
        g.setSize(wSize);
        dev->setCaptureGeometry(g);
    }
}


void KdetvV4L::viewResized()
{
    V4LGrabberLocker l(g);
    if (!dev)
        return;

    if (dev->overlayOn()) {
        viewMoved();
    } else {
        dev->setImageSize(_w->width(), _w->height());
    }

    _vs->setSize(QSize(_w->width(), _w->height()));
}


int KdetvV4L::probeDevices()
{
	QString dev;
	struct stat sb;

	
	// don't probe multiple times, it's unnecessary and might yield incorrect
	// results if plugin is currently using the device
	if (_probed) return 0;

	int rc = stat("/dev/v4l", &sb);
	if (!rc && S_ISDIR(sb.st_mode) && !access("/dev/v4l", R_OK|X_OK)) {
		// DEVFS
		dev = "/dev/v4l/video%1";
	} else {
		// Normal V4L case
		dev = "/dev/video%1";
	}

	_devices.clear();
	_sources.clear();
	_tuners.clear();
	_encodings.clear();
	_devNames.clear();

	QString mainVideoDev = QString::null;
	// FIXME: when we test for devices, we must make sure that we can
	// actually use them too.  This fails for cases where something else
	// is using the device currently.
	if (!access("/dev/video", R_OK|W_OK)) {
		V4LDev *vd = V4LDev::getDevice("/dev/video");
		if (vd) {
			QString name = vd->name() + " Video4Linux(/dev/video)";
			_tuners[name] = vd->isTuner();
			_sources[name] = vd->sources();
			_encodings[name] = vd->encodings();
			_devices << name;
			_devNames[name] = "/dev/video";
			mainVideoDev = QDir("/dev/video").canonicalPath();
			delete vd;
		}
	}
    
	for (int i = 0; i <= 9; i++) {
		QString dn = dev.arg(i);
		if (dn != mainVideoDev && !access(dn.local8Bit(), R_OK|W_OK)) {
			V4LDev *vd = V4LDev::getDevice(dn);
			if (vd) {
				QString name =vd->name()+" Video4Linux("+dn+")";
				_tuners[name] = vd->isTuner();
				_sources[name] = vd->sources();
				_encodings[name] = vd->encodings();
				_devices << name;
				_devNames[name] = dn;
				delete vd;
			}
		}
	}

    _probed = true;
    return 0;
}


int KdetvV4L::startVideo()
{
    int rc = 0;
    if (!dev || g || _capturing) {
        kdDebug() << "Error starting video: " << (void *)dev << " " << (void*)g << endl;
        return -1;
    }
    
    dev->setImageSize(_w->width(), _w->height());
    if (_useOverlay && dev->canOverlay()) {
		dev->setInputFormat(VIDEO_PALETTE_YUV422);
        dev->setColourKey(0x0000ff00); // FIXME: we need to do proper colorkey for devices that support it.

		_overlayController = new OverlayController(_w);
        connect(_overlayController, SIGNAL( updateClipping() ),
                this, SLOT( updateClipping() ));
        connect(_overlayController, SIGNAL( moved() ),
                this, SLOT( viewMoved() ));
        connect(_overlayController, SIGNAL( repaintScreen() ),
                this, SLOT( repaintScreen() ));
        connect(_overlayController, SIGNAL( enableVideo(bool) ),
                this, SLOT( enableOverlay(bool) ));

        rc = enableOverlay(true);
    } else {
        _vs->setMethod(_qvsMethod);
        _vs->setSize(_w->size());
		dev->setInputFormat(getGrabFormat(_qvsMethod));
		g = new V4LGrabber(this, dev, _vs, ++_gsn);
		g->start();
    }

    if (rc == 0)
        setMuted(false);
    
    _capturing = true;
    return rc;
}


int KdetvV4L::stopVideo()
{
    V4LGrabberLocker l(g);
    
    if (!_capturing)
        return -1;
    
    setMuted(true);

    if (g) {
		g->stop();
		g = 0;
    } else {  // overlay
        delete _overlayController;
    }
    _capturing = false;
    return 0;
}


bool KdetvV4L::canVideoDesktop() const
{
    return true;
}


int KdetvV4L::setVideoDesktop(bool on)
{
    if (!dev)
        return -1;
    
    V4LGrabberLocker l(g);
    if (on) {
        _wReg = _vs->width();
        _hReg = _vs->height();
        stopVideo();
        _w = _dtReg;
        delete _vs;
        _vs = new QVideoStream(_w);
        _vs->setMethod(_qvsMethod);
        viewResized();
        _isVideoDesktop = true;
        startVideo();
        setMuted(false);
        _capturing = true;
    } else {
        if (!_isVideoDesktop)
            return -1;
        _isVideoDesktop = false;
        stopVideo();
        setMuted(true);
		_dtReg->repaint();
        _w = _winReg;
        delete _vs;
        _vs = new QVideoStream(_w);
        _vs->setMethod(_qvsMethod);
        viewResized();
        return startVideo();
    }

    return 0;
}


bool KdetvV4L::videoPlaying() const
{
    return _capturing;
}

bool KdetvV4L::canGrabStill() const
{
    // Is this dangerous?  We can't use a mutex here though...
    return dev && dev->canGrab();
}

bool KdetvV4L::grabStill(QImage *img)
{
    V4LGrabberLocker l(g);
    if (dev && dev->canGrab()) {
        V4LImage im;

        bool videoWasRunning = g || _capturing;

        stopVideo();

        dev->setInputFormat(VIDEO_PALETTE_RGB24);
        dev->setImageSize(img->width(), img->height());

        // Capture two consecutive frames without failure.
        // Necessary for good grab image quality with bt8x8 cards.
        int rc      = -1;
        int lastrc  = -1;
        int newrc   = -1;
        int timeout = 20;
        do {
            newrc = dev->grab(&im);
            if ((lastrc == 0) && (newrc == 0)) {
                rc = 0;
            }
            lastrc = newrc;
        } while ((rc != 0) && (timeout-- > 0));

        // Do the conversion before restoring the size
        // -> V4L image may contain pointer to v4ldev internal buffers,
        //    which may be resized change due to setImageSize call
        if ((rc != 0) || (im.toQImage(*img) != 0)) {
            rc = -1;
        }

        if (videoWasRunning) {
            startVideo();
        }

        if (!rc) {
            return true;
        }
    }
    return false;
}


void KdetvV4L::updateClipping()
{
    Display *dpy = qt_xdisplay();
    Window win = _w->winId();
    Window root;
    Window rroot, parent, *children;
    unsigned int nchildren = 0;
    unsigned int i = 0;
    
    int scn = QApplication::desktop()->screenNumber(_w);
    root    = QApplication::desktop()->screen(scn)->winId();
    
    // Find the window just below the root.
    for (;;) {
        if (!XQueryTree(dpy, win, &rroot, &parent, &children, &nchildren))
            return;
        XFree(children);
        
        if (root == parent)
            break;
        win = parent;
    }
    
    if (!XQueryTree(dpy, root, &rroot, &parent, &children, &nchildren)) {
        return;
    }

    // Find the root children that come after the window we just found
    for  (i = 0; i < nchildren; i++ ) {
        if ( children[i] == win ) {
            break;
        }
    }

    QRect wgeom = _w->geometry();
    QRect geom;
    QPoint ul = _w->mapToGlobal(wgeom.topLeft());
    QPoint lr = _w->mapToGlobal(wgeom.bottomRight());

    dev->clearClips();

    // Go through the rest and see if they overlap.  If so, clip.
    for (i++; i < nchildren; i++) {
        XWindowAttributes wattrs;

        XGetWindowAttributes(dpy, children[i], &wattrs);
        if (!(wattrs.map_state & IsViewable))
            continue;

        if (wattrs.x + wattrs.width < ul.x()  || wattrs.x > lr.x() ||
            wattrs.y + wattrs.height < ul.y() || wattrs.y > lr.y())
            continue;

        geom = QApplication::desktop()->screenGeometry(QRect(wattrs.x, wattrs.y, wattrs.width, wattrs.height).center());
        wattrs.x -= geom.x();
        wattrs.y -= geom.y();
        //        printf("Found a sibling of the view: %d %d %dx%d\n", wattrs.x, wattrs.y, wattrs.width, wattrs.height);

        dev->addClip(QRect(wattrs.x, wattrs.y, wattrs.width, wattrs.height));
    }

    XFree(children);

    if (XQueryTree(dpy, _w->winId(), &rroot, &parent, &children, &nchildren)) {
        // Find the root children that come after the window we just found
        for  (i = 0; i < nchildren; i++ ) {
            XWindowAttributes wattrs;

            XGetWindowAttributes(dpy, children[i], &wattrs);
            if (!(wattrs.map_state & IsViewable))
                continue;

            QPoint wpt(wattrs.x, wattrs.y);
            wpt = _w->mapToGlobal(wpt);
            wattrs.x = wpt.x();
            wattrs.y = wpt.y();
            if (wattrs.x + wattrs.width  < ul.x() || wattrs.x > lr.x() ||
                wattrs.y + wattrs.height < ul.y() || wattrs.y > lr.y())
                continue;

            geom = QApplication::desktop()->screenGeometry(QRect(wattrs.x, wattrs.y, wattrs.width, wattrs.height).center());
            wattrs.x -= geom.x();
            wattrs.y -= geom.y();
            //            printf("Found a child of the view: %d %d %dx%d\n", wattrs.x, wattrs.y, wattrs.width, wattrs.height);

            dev->addClip(QRect(wattrs.x, wattrs.y, wattrs.width, wattrs.height));
        }
        XFree(children);
    }

    dev->reClip();
}


void KdetvV4L::repaintScreen()
{
    QRect   geom = QApplication::desktop()->screenGeometry(_w);
    Display *dpy = qt_xdisplay();

    // repaint the region to clean it up
    XSetWindowAttributes xwattr;
    xwattr.override_redirect = true;
    xwattr.backing_store = NotUseful;
    xwattr.save_under = false;
    unsigned long mask = CWSaveUnder | CWBackingStore | CWOverrideRedirect;

    int scn = QApplication::desktop()->screenNumber(_w);
    Window pwin = XCreateWindow(dpy,
                                QApplication::desktop()->screen(scn)->winId(),
                                geom.x(), geom.y(),
                                geom.width(), geom.height(),
                                0,
                                CopyFromParent, InputOutput, CopyFromParent,
                                mask, &xwattr);
    XMapWindow(dpy, pwin);
    XUnmapWindow(dpy, pwin);
    XDestroyWindow(dpy, pwin);

    QApplication::flushX();
}


bool KdetvV4L::event(QEvent* e)
{
    if (e->type() == QEvent::User) {
        emit errorMessage( ((V4LErrorEvent*)e)->_msg );
        stopVideo();
        return true;
    } else {
        return KdetvSourcePlugin::event(e);
    }
}


int KdetvV4L::enableOverlay(bool enable)
{
    if (enable) {
        QPoint qp = _w->mapToGlobal(QPoint(0, 0));
        qp -= QApplication::desktop()->screenGeometry(_w).topLeft();
        int rc = dev->startCapture(qp.x(), qp.y());
        viewMoved();
        return rc;
    } else {
        return dev->stopCapture();
    }
}


QWidget *KdetvV4L::configWidget(QWidget *parent, const char *name)
{
    _cfgWidget = new V4LPluginCfg(parent, name);

    _cfgWidget->_xvideo->setEnabled(_vs->haveMethod(QVIDEO_METHOD_XV));
    _cfgWidget->_xvshm->setEnabled(_vs->haveMethod(QVIDEO_METHOD_XVSHM));
    _cfgWidget->_x11->setEnabled(_vs->haveMethod(QVIDEO_METHOD_X11));
    _cfgWidget->_x11shm->setEnabled(_vs->haveMethod(QVIDEO_METHOD_XSHM));

    switch(_qvsMethod) {
    case QVIDEO_METHOD_XV:
        _cfgWidget->_xvideo->setChecked(true);
        break;
    case QVIDEO_METHOD_XVSHM:
        _cfgWidget->_xvshm->setChecked(true);
        break;
    case QVIDEO_METHOD_X11:
        _cfgWidget->_x11->setChecked(true);
        break;
    case QVIDEO_METHOD_XSHM:
        _cfgWidget->_x11shm->setChecked(true);
        break;
    default:
        assert(0);
        break;
    }
    
    _cfgWidget->_autoConfig->setChecked(_autoConfig);
    _cfgWidget->_overlay->setChecked(_useOverlay);
    _cfgWidget->_changeRes->setChecked(_changeRes);
    //    _cfgWidget->_mirror->setChecked(_mirror);
    //    _cfgWidget->_frameRate->setValue(_frameRate);
    //#ifdef HAVE_DVBS
    //    _cfgWidget->_dvbs->setChecked(_dvbs);
    //#else
    //    _cfgWidget->_dvbs->setChecked(false);
    //    _cfgWidget->_dvbs->setEnabled(false);
    //#endif

    return _cfgWidget;
}


void KdetvV4L::saveConfig()
{
    _changeRes  = _cfgWidget->_changeRes->isChecked();
    _useOverlay = _cfgWidget->_overlay->isChecked();
    _autoConfig = _cfgWidget->_autoConfig->isChecked();
    //    _dvbs = _cfgWidget->_dvbs->isChecked();
    //    _mirror = _cfgWidget->_mirror->isChecked();
    //    _frameRate = _cfgWidget->_frameRate->value();

    _qvsMethod = QVIDEO_METHOD_NONE;
    if (_cfgWidget->_xvideo->isChecked()) {
        _qvsMethod = QVIDEO_METHOD_XV;
    } else if (_cfgWidget->_xvshm->isChecked()) {
        _qvsMethod = QVIDEO_METHOD_XVSHM;
    } else if (_cfgWidget->_x11shm->isChecked()) {
        _qvsMethod = QVIDEO_METHOD_XSHM;
    } else if (_cfgWidget->_x11->isChecked()) {
        _qvsMethod = QVIDEO_METHOD_X11;
    }

    _cfg->writeEntry("Change Screen Resolution", _changeRes);
    _cfg->writeEntry("Use Overlay", _useOverlay);
    _cfg->writeEntry("GD Method", _qvsMethod);
    _cfg->writeEntry("Autoconfigure", _autoConfig);
    _cfg->writeEntry("Use DVB-S", _dvbs);
    _cfg->writeEntry("Mirror Image", _mirror);
    _cfg->writeEntry("Max Framerate", _frameRate);

    if (_capturing) {
        stopVideo();
        _vs->setMethod(_qvsMethod);
        startVideo();
    } else {
        _vs->setMethod(_qvsMethod);
    }

    _cfg->sync();
}

int KdetvV4L::getGrabFormat(int qvsMethod)
{
    int fmt;
    int dummy;
	
    switch (qvsMethod) {
    case QVIDEO_METHOD_X11:
    case QVIDEO_METHOD_XSHM:
        if (!V4LUtil::findDisplayProperties(fmt, dummy, dummy, dummy)) {
            kdDebug() << "Unable to figure out display properties. Grab format may be wrong." << endl;
			fmt = VIDEO_PALETTE_RGB32;
        }
        break;
    case QVIDEO_METHOD_XV:
    case QVIDEO_METHOD_XVSHM:
    default:
		fmt = VIDEO_PALETTE_YUV422;
		break;
    }
	
    return fmt;
}

void KdetvV4L::setFullscreen(bool fs)
{
    if ( !dev || !dev->overlayOn() || !_changeRes)
        return;

    if(fs) {
        _previousSize = setScreenResolution(dev->getMaxImageSize());
    } else {
        setScreenResolution(_previousSize);
    }
}

#ifdef HAVE_XRANDR
QSize KdetvV4L::setScreenResolution(const QSize& size)
{
    Display* dpy  = qt_xdisplay();
    int      scn  = QApplication::desktop()->screenNumber(_w);
    Window   root = QApplication::desktop()->screen(scn)->winId();

    // First check for XRANDR extension and refresh sizes array
    int foo, bar;
    int nrandr = 0;
    XRRScreenSize* randr = 0;

    if (XRRQueryExtension(dpy, &foo, &bar)) {
        randr = XRRSizes(dpy, scn, &nrandr);
	}
    if (nrandr == 0) {
        kdDebug() << "KdetvV4L: No XRANDR available. Cannot change resolution." << endl;
        return QSize();
    }

    XRRScreenConfiguration* sc = XRRGetScreenInfo(dpy, root);
    Rotation rotation;
    SizeID current = XRRConfigCurrentConfiguration(sc, &rotation);
    SizeID newRes = current;

    // Find closest match for the desired size
    int mindist = 1000000;
    for (SizeID i=0; i<nrandr; i++) {
        int d1   = randr[i].width  - size.width();
        int d2   = randr[i].height - size.height();
        int dist = d1 + d2;

        if( (d1>=0) && (d2>=0) && (dist<mindist) ) {
            mindist = dist;
            newRes  = i;
        }
    }
    if (newRes != current) {
        kdDebug() << "KdetvV4L: XRANDR: switch to " << randr[newRes].width << "x" << randr[newRes].height << endl;
        XRRSetScreenConfig(dpy, sc, root, newRes, rotation, CurrentTime);
    }

    XRRFreeScreenConfigInfo(sc);

    return QSize(randr[current].width, randr[current].height);
}
#else // HAVE_XRANDR
QSize KdetvV4L::setScreenResolution(const QSize&)
{
    return QSize();
}
#endif // HAVE_XRANDR

extern "C" {
    KdetvV4L* create_v4l(Kdetv *ktv, QWidget *w) {
		return new KdetvV4L(ktv, w, "v4l plugin");
    }
}
