/*
 * KMLOFax
 *
 * A utility to process facsimiles received with a modem of the
 * ELSA MicroLink(tm) Office family.
 *
 * Copyright (C) 1999,2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen, Germany.
 */

#include <qglobal.h>
#include <qframe.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qsize.h>
#include <qtooltip.h>
#include <qfile.h>

#include <kapplication.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <ksimpleconfig.h>
#include <kseparator.h>

#include "senderaliases.h"
#include "global.h"



SenderAliases *senderaliases = 0;



EditAlias::EditAlias(QWidget *parent, const char *name) : KDialog (parent, name, true)
{
	QVBoxLayout *vbox;
	QHBoxLayout *hbox;
	QGridLayout *grid;
	QLabel *label;
	KSeparator *separator;
	QPushButton *ok, *cancel;
	QSize size;

	vbox = new QVBoxLayout(this, marginHint());
	Q_CHECK_PTR(vbox);

	vbox->addStretch();

	grid = new QGridLayout(2, 2);
	Q_CHECK_PTR(grid);
	vbox->addLayout(grid);

	label = new QLabel(i18n("&Sender ID:"), this);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 0, 0, AlignVCenter | AlignLeft);

	m_sender = new QLineEdit(this);
	label->setBuddy(m_sender);
	m_sender->setMinimumWidth(200);
	m_sender->setFixedHeight(m_sender->sizeHint().height());
	grid->addWidget(m_sender, 0, 1);

	label = new QLabel(i18n("&Alias:"), this);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 1, 0, AlignVCenter | AlignLeft);

	m_alias = new QLineEdit(this);
	label->setBuddy(m_alias);
	m_alias->setMinimumWidth(200);
	m_alias->setFixedHeight(m_alias->sizeHint().height());
	grid->addWidget(m_alias, 1, 1);

	vbox->addStretch();
	
	vbox->addSpacing(marginHint());

	separator = new KSeparator(KSeparator::HLine, this);
	vbox->addWidget(separator);
	
	hbox = new QHBoxLayout();
	vbox->addLayout(hbox);

	ok = new QPushButton(i18n("&OK"), this);
	ok->setDefault(true);
	cancel = new QPushButton(i18n("&Cancel"), this);

	size = ok->sizeHint().expandedTo(cancel->sizeHint());

	ok->setFixedSize(size);
	cancel->setFixedSize(size);

	hbox->addStretch();
	hbox->addWidget(ok);
//	hbox->addSpacing(spacingHint());
	hbox->addWidget(cancel);

	vbox->activate();

	connect(ok, SIGNAL(clicked()), SLOT(accept()));
	connect(cancel, SIGNAL(clicked()), SLOT(reject()));

	setCaption(i18n("Sender Alias"));
}


EditAlias::~EditAlias()
{
}



SenderAliases::SenderAliases(QWidget *parent, const char *name) : KDialog (parent, name)
{
	QVBoxLayout *vbox, *gvbox;
	QHBoxLayout *hbox, *ghbox;
	QGroupBox *gb;
	QSize size;
	QPushButton *help, *bclose;

	gotToolTips = false;

	vbox = new QVBoxLayout(this, marginHint());

	gb = new QGroupBox(this);

	gvbox = new QVBoxLayout(gb, marginHint());
	
	list = new QListView(gb, "aliaslist");
	Q_CHECK_PTR(list);
	list->addColumn(i18n("Sender ID"));
	list->addColumn(i18n("Alias"));
	list->setColumnWidth(0, 200);
	list->setColumnWidth(1, 280);
	list->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	list->setAllColumnsShowFocus(true);
	list->setShowSortIndicator(true);

	gvbox->addWidget(list, 1);

	ghbox = new QHBoxLayout();
	Q_CHECK_PTR(ghbox);
	gvbox->addLayout(ghbox);

	add_item = new QPushButton(i18n("&Add"), gb);
	edit_item = new QPushButton(i18n("&Edit"), gb);
	edit_item->setEnabled(false);
	del_item = new QPushButton(i18n("&Delete"), gb);
	del_item->setEnabled(false);

	size = add_item->sizeHint().expandedTo(edit_item->sizeHint()).expandedTo(del_item->sizeHint());

	add_item->setFixedSize(size);
	edit_item->setFixedSize(size);
	del_item->setFixedSize(size);

	ghbox->addWidget(add_item);
	ghbox->addStretch();
	ghbox->addWidget(edit_item);
	ghbox->addStretch();
	ghbox->addWidget(del_item);

	gvbox->activate();

	vbox->addWidget(gb, 1);
	
	hbox = new QHBoxLayout(8);
	vbox->addLayout(hbox);

	help = new QPushButton(i18n("&Help"), this);
	bclose = new QPushButton(i18n("&Close"), this);
	bclose->setDefault(true);
	size = help->sizeHint().expandedTo(bclose->sizeHint());
	help->setFixedSize(size);
	bclose->setFixedSize(size);

	hbox->addWidget(help);
	hbox->addStretch();
	hbox->addWidget(bclose);

	vbox->activate();

	connect(list, SIGNAL(selectionChanged(QListViewItem *)), SLOT(selectionChangedSlot(QListViewItem *)));
	connect(list, SIGNAL(doubleClicked(QListViewItem *)), SLOT(executedSlot(QListViewItem *)));
	connect(list, SIGNAL(returnPressed(QListViewItem *)), SLOT(executedSlot(QListViewItem *)));
	connect(add_item, SIGNAL(clicked()), SLOT(addItem()));
	connect(edit_item, SIGNAL(clicked()), SLOT(editItem()));
	connect(del_item, SIGNAL(clicked()), SLOT(deleteItem()));
	connect(help, SIGNAL(clicked()), SLOT(slotHelp()));
	connect(bclose, SIGNAL(clicked()), SLOT(close()));

	setCaption(i18n("Sender Aliases"));

	loadDir();
}


SenderAliases::~SenderAliases()
{
}


void SenderAliases::loadDir()
{
	KSimpleConfig *sconfig;
	int number, i;
	QListViewItem *item;
	
	list->clear();

	sconfig = new KSimpleConfig(expandPath(KMLOFAX_ALIASLIST), true);
	Q_CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	number = sconfig->readNumEntry("Entries");
	for (i = 0; i < number; i++) {
		sconfig->setGroup(QString("Entry %1").arg(i));
		item = new QListViewItem(list);
		Q_CHECK_PTR(item);
		item->setText(0, sconfig->readEntry("Sender"));
		item->setText(1, sconfig->readEntry("Alias"));
	}
	delete sconfig;
}


void SenderAliases::saveDir()
{
	KSimpleConfig *sconfig;
	QListViewItem *item;
	int i;

	QFile::remove(expandPath(KMLOFAX_ALIASLIST));
	sconfig = new KSimpleConfig(expandPath(KMLOFAX_ALIASLIST), false);
	Q_CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	sconfig->writeEntry("Entries", list->childCount());
	item = list->firstChild();
	for (i = 0; (item); i++) {
		sconfig->setGroup(QString("Entry %1").arg(i));
		sconfig->writeEntry("Sender", item->text(0));
		sconfig->writeEntry("Alias", item->text(1));
		item = item->nextSibling();
	}
	sconfig->sync();
	delete sconfig;
}


void SenderAliases::addToolTips(bool on)
{
	if (on == gotToolTips)
		return;

	gotToolTips = on;

	if (gotToolTips) {
		QToolTip::add(add_item, i18n("Add an entry to the list"));
		QToolTip::add(edit_item, i18n("Edit selected entry"));
		QToolTip::add(del_item, i18n("Delete selected entry"));
	}
	else {
		QToolTip::remove(add_item);
		QToolTip::remove(edit_item);
		QToolTip::remove(del_item);
	}
}	


QString SenderAliases::alias(const QString& id_)
{
	QListViewItem *item;
	QString s = id_.stripWhiteSpace();

	item = list->firstChild();
	while (item) {
		if (item->text(0) == s)
			return item->text(1);
		item = item->nextSibling();
	}

	return s;
}


void SenderAliases::setAlias(const QString& id_)
{
	QListViewItem *item;
	QString s = id_.stripWhiteSpace();

	item = list->firstChild();
	while (item) {
		if (item->text(0) == s)
			break;
		item = item->nextSibling();
	}

	if (!item) {
		item = new QListViewItem(list);
		item->setText(0, s);
	}

	list->setCurrentItem(item);
	list->ensureItemVisible(item);

	if (isVisible())
		raise();
	else
		show();
}


void SenderAliases::selectionChangedSlot(QListViewItem *item)
{
	if (item) {
		edit_item->setEnabled(true);
		del_item->setEnabled(true);
	}
}


void SenderAliases::executedSlot(QListViewItem *)
{
	editItem();
}


void SenderAliases::addItem()
{
	EditAlias edit;
	QListViewItem *item;

	if (edit.exec()) {
		item = new QListViewItem(list);
		item->setText(0, edit.sender());
		item->setText(1, edit.alias());
		list->setSelected(item, true);
		list->ensureItemVisible(item);
		saveDir();
		emit applyButtonPressed();
	}
}


void SenderAliases::editItem()
{
	EditAlias edit;
	QListViewItem *item;

	if ((item = list->currentItem())) {
		edit.setSender(item->text(0));
		edit.setAlias(item->text(1));
		if (edit.exec()) {
			item->setText(0, edit.sender());
			item->setText(1, edit.alias());
			saveDir();
			emit applyButtonPressed();
		}
	}
}


void SenderAliases::deleteItem()
{
	QListViewItem *item;

	if ((item = list->currentItem())) {
		if (KMessageBox::warningYesNo(this, i18n("Really delete entry\n\"%1\"?").arg(item->text(0)), i18n("Question"), i18n("&Delete"), i18n("&Cancel")) == KMessageBox::Yes) {
			delete item;
			edit_item->setEnabled(false);
			del_item->setEnabled(false);
			saveDir();
			emit applyButtonPressed();
		}
	}
}


void SenderAliases::slotHelp()
{
	kapp->invokeHelp("senderaliases-help");
}
