// -*- C++ -*-
/*****************************************************************************
 * KWATCH --- KDE Log File Viewer
 *
 * $Id: klogwidget.cpp,v 0.11 2000/12/14 20:37:23 mj Exp $
 *
 * QMultiLineEdit based widget for viewing log files
 *
 *****************************************************************************
 * Copyright (C) 2000
 *  _____ _____
 * |     |___  |   Martin Junius             Internet:  mailto:mj@m-j-s.net
 * | | | |   | |   Radiumstr. 18                        http://www.m-j-s.net/
 * |_|_|_|@home|   D-51069 Koeln, Germany
 *
 * Based on kwatch 0.3, with the following copyright notice:
 *
 * Kwatch is Copyright (C) 1997,1998 Ralph C. Weichert and can be
 * distributed freely under the terms of the GNU General Public License
 * (see file COPYING)
 *****************************************************************************
 * This file is part of KWATCH.
 *
 * KWATCH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * KWATCH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KWATCH; see the file COPYING.  If not, write to the Free
 * Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *****************************************************************************/


#include <string.h>

#include <qdragobject.h>

#include <kurl.h>
#include <kapp.h>

#include "klogwidget.h"
#include "prefs.h"



/*
 * KLogFile
 */

KLogFile::KLogFile(const QString &name, QObject *parent, kwatchConf &cf)
    : QObject(parent), m_cf(cf)
{
    m_buf  = new char[m_cf.bufsize];
    m_file = new QFile(name);
    activate();
}


KLogFile::~KLogFile()
{
    delete m_buf;
    delete m_file;
}



void KLogFile::activate()
{
    if(m_file->isOpen())
	return;
    m_file->open( IO_ReadOnly | IO_Raw);
    if(m_file->isOpen())
        // Go to EOF
        m_file->at(m_file->size());
    else {
        // Try again in a minute
        QTimer::singleShot( 60000, this, SLOT(activate()) );
    }
}



bool KLogFile::readMore()
{
    if( !m_file->isOpen() )
	return FALSE;
    
    QString text;
    int len;

    do {
        len = m_file->readBlock( m_buf, m_cf.bufsize-1 );
        if (len < 0) {
            text += "\nRead Error!\n";
            m_file->close();
            QTimer::singleShot( 60000, this, SLOT(activate()) );
            break;
        }
        m_buf[len] = 0;
        text += m_buf;
    } while (len == m_cf.bufsize-1);

    if ( !text.isEmpty() ) {
        emit moreLines( this, text );
        return TRUE;
    }

    return FALSE;
}




/*
 * KLogWidget
 */

KLogWidget::KLogWidget(QWidget *parent, kwatchConf &cf)
    : QMultiLineEdit(parent), m_cf(cf),
      m_timer(0), m_files(0), m_current(0),
      m_r(0), m_c(0)
{
    // set QMultiLineEdit parameters
    setReadOnly( TRUE );

    // set configuration parameters
    updateConf();

    // accept drops
    setAcceptDrops(true);

    // start timer
    m_timer = new QTimer( this );
    m_timer->start(0);
    connect( m_timer, SIGNAL(timeout()), SLOT(timeout()) );
}



KLogWidget::~KLogWidget()
{
    if(m_timer)
	delete m_timer;
    if(m_files)
	delete m_files;

}



void KLogWidget::dropEvent(QDropEvent *ev)
{
    fprintf(stderr, "KLogWidget::dropEvent()\n");

    if(QUriDrag::canDecode(ev))
    {
	QStringList files;
 
	if(QUriDrag::decodeLocalFiles(ev, files))
	{
	    QStringList::Iterator it;
	    for (it=files.begin(); it!=files.end(); it++)
	    {
		QString f = (*it);
		fprintf(stderr, "KLogWidget::dropEvent() - file = %s\n",
			(const char *)f);

		// add to list and open
		if (m_cf.logfiles.find(f) == m_cf.logfiles.end()) {
		    m_cf.logfiles.append(f);
		    addFile(f);
		}
	    }
	}
    }
}

void KLogWidget::dragEnterEvent(QDragEnterEvent *ev)
{
    fprintf(stderr, "KLogWidget::dragEnterEvent()\n");

    ev->accept(QUriDrag::canDecode(ev));
}
    


void KLogWidget::timeout()
{
    bool text_added = FALSE;
    QListIterator<KLogFile> it(*m_files);

    for(; it.current(); ++it)
        if(it.current()->readMore())
            text_added = TRUE;
    if(text_added)
        m_timer->changeInterval(0);
    else
        m_timer->changeInterval(1000);
}



void KLogWidget::moreText(KLogFile *f, QString s)
{
    if (s.isEmpty()) return;

    QString more = s;

    if(f && f != m_current )
    {
        m_current = f;
        more.prepend(formatName(f->m_file->name()));
    }

    // disable autoUpdate to avoid flicker
    setAutoUpdate(false);

    // insert new text
    insertAt(more, m_r, m_c);

    // remove lines at beginning if necessary
    for(int l = numLines() - m_cf.maxlines; l > 0; l--)
        removeLine(0);
    m_r = numLines() - 1;
    while(0 == (m_c = lineLength(m_r)) && m_r > 0)
	m_r--;
    if(more.right(1) == "\n")
	m_r++;

    // enable autoUpdate and repaint
    setAutoUpdate(true);
    repaint();
    
    setCursorPosition(numLines()-1, 0);
}



QString KLogWidget::formatName(const QString &name)
{
    QString std = "---------------[" + name + "]---------------\n";

    QString s = textLine(0);
    if(s.isEmpty() || s.isNull())
	return std;
    
    int l    = textWidth(0);
    int cols = width() / (l / s.length());
    int n    = cols - name.length() - 15 - 1 - 1 - 4;
    if(n < 1)
	return std;
    
    QString f;
    f.fill('-', n);
    
    return "---------------[" + name + "]" + f + "\n";
}



void KLogWidget::updateConf()
{
    fprintf(stderr, "KLogWidget::updateConf()\n");

    // list of files
    if(m_files)
	delete m_files;
    m_files = new QList<KLogFile>();
    m_files->setAutoDelete(TRUE);
    m_current = NULL;

    QStringList::Iterator it;
    for (it=m_cf.logfiles.begin(); it!=m_cf.logfiles.end(); it++)
        addFile((*it));

    // set font, fg, bg colors
    kwatchPrefs::setFontFgBg(this, m_cf.font, m_cf.fg, m_cf.bg);
}



void KLogWidget::reopen()
{
    QListIterator<KLogFile> it(*m_files);
    KLogFile *f;

    for(; (f = it.current()); ++it)
    {
        if(f->m_file->isOpen())
	    f->m_file->close();
        f->activate();
    }
}



void KLogWidget::addFile(const QString &name)
{
    fprintf(stderr, "KLogWidget::addFile(%s)\n", (const char *)name);

    KLogFile *f = new KLogFile(name, this, m_cf);
    if(! f->m_file->isOpen())
	moreText(f, "ERROR: cannot open file\n");
    connect(f, 
	    SIGNAL(moreLines(KLogFile *, QString)),
	    SLOT(moreText(KLogFile *, QString))    );
    m_files->append(f);
}



#include "klogwidget.moc"
