/*
    KMLOVoice

    A utility to process voice messages received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 2000 Oliver Gantz <Oliver.Gantz@epost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include <stdlib.h>
#include <string.h>

#include <qglobal.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qdir.h>

#include <kapp.h>
#include <klocale.h>
#include <kiconloader.h>
#include <ksimpleconfig.h>

#include "voiceview.h"
#include "global.h"
#include "senderaliases.h"



VoiceViewItem::VoiceViewItem(QListView *parent, const char *name) : QListViewItem(parent, name)
{
	m_new = false;
	m_type = VOI_UNKNOWN;
	m_size = 0;
	m_file = false;
}


VoiceViewItem::~VoiceViewItem()
{
}


void VoiceViewItem::setName(const QString& name)
{
	setText(0, name);
	setText(2, mloDateStr(name));
}


void VoiceViewItem::setNew(bool isnew)
{
	m_new = isnew;
	setPixmap(1, m_new ? UserIcon("yes") : UserIcon("no"));
}


void VoiceViewItem::setCaller(const QString& caller_)
{
	m_caller = caller_;
	
	setAlias();
}


void VoiceViewItem::setType(uchar type_)
{
	m_type = type_;
	setDuration();
}


void VoiceViewItem::setSize(uint size_)
{
	m_size = size_;
	setText(5, KGlobal::locale()->formatNumber((double)size_, 0));
	setDuration();
}


void VoiceViewItem::setFile(bool hasfile)
{
	m_file = hasfile;
	setPixmap(6, m_file ? UserIcon("yes") : UserIcon("no"));
}


void VoiceViewItem::setAlias()
{
	setText(3, CALLER_ALIAS(m_caller));
}


void VoiceViewItem::setDuration()
{
	static const int ratios[7] = { 1, 1800, 2000, 3600, 4000, 7200, 8000 };

	setText(4, KGlobal::locale()->formatNumber((double)m_size / (double)ratios[m_type], 1));
}


void VoiceViewItem::setParams(const QString& name_, bool isnew_, const QString& caller_, uchar type_, uint size_)
{
	setName(name_);
	setNew(isnew_);
	setCaller(caller_);
	setType(type_);
	setSize(size_);
	setFile(QFile::exists(expandPath(name_)));
}



VoiceView::VoiceView(QWidget *parent, const char *name) : KListView(parent, name)
{
	config = kapp->config();

	selectedItem = 0;

	addColumn(i18n("Name"));
	addColumn(i18n("New"));
	addColumn(i18n("Date (Time)"));
	addColumn(i18n("Caller"));
	addColumn(i18n("Duration"));
	addColumn(i18n("Size"));
	addColumn(i18n("File"));
	setColumnWidth(0, 140);
	setColumnWidth(1, 40);
	setColumnWidth(2, 140);
	setColumnWidth(3, 160);
	setColumnWidth(4, 60);
	setColumnWidth(5, 60);
	setColumnWidth(6, 40);
	setColumnAlignment(1, AlignVCenter | AlignRight);
	setColumnAlignment(4, AlignVCenter | AlignHCenter);
	setColumnAlignment(5, AlignVCenter | AlignRight);
	setMinimumSize(sizeHint());
	setSorting(0, false);
	setAllColumnsShowFocus(true);
	setShowSortIndicator(true);

	contextMenu = new QPopupMenu(0, "contextmenu");
	CHECK_PTR(contextMenu);
	contextMenu->insertItem(SmallIcon("play"), i18n("&Playback..."), ID_FILE_PLAYBACK);
	contextMenu->insertItem(SmallIcon("filesave"), i18n("&Export..."), ID_FILE_EXPORT);
	contextMenu->insertItem(SmallIcon("mail_send"), i18n("&Mail..."), ID_FILE_MAIL);
	contextMenu->insertItem(SmallIcon("remove"), i18n("De&lete"), ID_FILE_DELETE);
	contextMenu->insertItem(i18n("&Caller Alias..."), ID_FILE_CALLER_ALIAS);

	connect(this, SIGNAL(selectionChanged(QListViewItem *)), SLOT(selectionChangedSlot(QListViewItem *)));
	connect(this, SIGNAL(doubleClicked(QListViewItem *)), SLOT(doubleClickedSlot(QListViewItem *)));
	connect(this, SIGNAL(returnPressed(QListViewItem *)), SLOT(returnPressedSlot(QListViewItem *)));
	connect(this, SIGNAL(rightButtonClicked(QListViewItem *, const QPoint&, int)), SLOT(rightButtonClickedSlot(QListViewItem *, const QPoint&, int)));
	connect(calleraliases, SIGNAL(applyButtonPressed()), SLOT(refreshAliases()));
	connect(contextMenu, SIGNAL(activated(int)), SLOT(contextMenuSlot(int)));

	setMinimumSize(40, 30);
}


VoiceView::~VoiceView()
{
	delete contextMenu;
}


void VoiceView::readOptions()
{
	config->setGroup("VoiceWindow Options");

	setColumnWidth(0, config->readNumEntry("NameWidth", 140));
	setColumnWidth(1, config->readNumEntry("NewWidth", 40));
	setColumnWidth(2, config->readNumEntry("DateWidth", 140));
	setColumnWidth(3, config->readNumEntry("CallerWidth", 160));
	setColumnWidth(4, config->readNumEntry("DurationWidth", 60));
	setColumnWidth(5, config->readNumEntry("SizeWidth", 60));
	setColumnWidth(6, config->readNumEntry("FileWidth", 40));
}


void VoiceView::saveOptions()
{
	config->setGroup("VoiceWindow Options");
	
	config->writeEntry("NameWidth", columnWidth(0));
	config->writeEntry("NewWidth", columnWidth(1));
	config->writeEntry("DateWidth", columnWidth(2));
	config->writeEntry("CallerWidth", columnWidth(3));
	config->writeEntry("DurationWidth", columnWidth(4));
	config->writeEntry("SizeWidth", columnWidth(5));
	config->writeEntry("FileWidth", columnWidth(6));
}


int VoiceView::loadDir()
{
	KSimpleConfig *sconfig;
	int i, number = 0;
	QString s;
	VoiceViewItem *item;

	clear();
	
	sconfig = new KSimpleConfig(expandPath(KMLOVOICE_MESSAGEDIR), true);
	CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	number = sconfig->readNumEntry("Entries");
	for (i = 0; i < number; i++) {
		s.sprintf("Entry %d", i);
		sconfig->setGroup(s);
		item = new VoiceViewItem(this);
		CHECK_PTR(item);
		item->setParams(sconfig->readEntry("FileName"), sconfig->readBoolEntry("New"), sconfig->readEntry("Caller"), sconfig->readNumEntry("Type"), sconfig->readNumEntry("Size"));
	}
	delete sconfig;
	
	return number;
}


void VoiceView::saveDir()
{
	KSimpleConfig *sconfig;
	QString s;
	VoiceViewItem *item;
	int i;

	QFile::remove(expandPath(KMLOVOICE_MESSAGEDIR));
	sconfig = new KSimpleConfig(expandPath(KMLOVOICE_MESSAGEDIR), false);
	CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	sconfig->writeEntry("Entries", childCount());
	item = (VoiceViewItem *)firstChild();
	for (i = 0; (item); i++) {
		s.sprintf("Entry %d", i);
		sconfig->setGroup(s);
		sconfig->writeEntry("FileName", item->name());
		sconfig->writeEntry("New", item->isNew());
		sconfig->writeEntry("Type", (int)item->type());
		sconfig->writeEntry("Caller", item->caller());
		sconfig->writeEntry("Size", item->size());
		item = (VoiceViewItem *)item->nextSibling();
	}
	sconfig->sync();
	delete sconfig;
}


void VoiceView::purgeDir()
{
	VoiceViewItem *item, *item_buff;

	item = (VoiceViewItem *)firstChild();
	while (item) {
		if (!(item->hasFile())) {
			item_buff = item;
			item = (VoiceViewItem *)item->nextSibling();
			delete item_buff;
		}
		else
			item = (VoiceViewItem *)item->nextSibling();
	}
}

/*
int VoiceView::rebuildDir()
{
	VoiceViewItem *item;
	int number = 0;

	QDir d(expandPath(""));
	const QFileInfoList *files;
	
	d.setNameFilter("[1-9][0-9][0-9][0-9][01][0-9][0-3][0-9][012][0-9][0-5][0-9][0-5][0-9].VOI");
	files = d.entryInfoList();

	if (files) {
		clear();
		kapp->setOverrideCursor(waitCursor);
		QFileInfoListIterator it(*files);
		QFileInfo *f;
		while ((f=it.current()) != 0) {
			++it;
			mlofile.setName(f->filePath());
			if (!(mlofile.open()))
				continue;
			item = new VoiceViewItem(this);
			CHECK_PTR(item);
			item->setParams(f->fileName(), false, mlofile.caller(), mlofile.pages(), mlofile.size());
			mlofile.close();
			number++;
		}
		kapp->restoreOverrideCursor();
	}

	return number;
}
*/

void VoiceView::addVoiceItem(const QString& name, uchar type, const QString& caller)
{
	VoiceViewItem *item;
	QFile file;

	for (item = (VoiceViewItem *)firstChild(); (item); item = (VoiceViewItem *)item->nextSibling())
		if (item->name() == name)
			break;
	
	if (!item)
		item = new VoiceViewItem(this);

	item->setName(name);
	item->setType(type);
	item->setCaller(caller);
	
	file.setName(expandPath(name));
	if (file.open(IO_ReadOnly)) {
		item->setNew(true);
		item->setSize(file.size());
		item->setFile(true);
		file.close();
	}
	else {
		item->setNew(false);
		item->setSize(0);
		item->setFile(false);
	}
}


void VoiceView::setSelectedMsgNew(bool isnew)
{
	if (selectedItem)
		selectedItem->setNew(isnew);
}


bool VoiceView::selectedMsgNew()
{
	if (selectedItem)
		return selectedItem->isNew();

	return false;
}


QString VoiceView::selectedMsgName()
{
	if (selectedItem)
		return selectedItem->name();

	return QString::null;
}


uchar VoiceView::selectedMsgType()
{
	if (selectedItem)
		return selectedItem->type();

	return VOI_UNKNOWN;
}


QString VoiceView::selectedMsgCaller()
{
	if (selectedItem)
		return selectedItem->caller();

	return QString::null;
}


QString VoiceView::selectedMsgAlias()
{
	if (selectedItem)
		return selectedItem->alias();

	return QString::null;
}


void VoiceView::setSelectedMsgFile(bool fexists)
{
	if (selectedItem)
		selectedItem->setFile(fexists);
}


bool VoiceView::selectedMsgFile()
{
	if (selectedItem)
		return selectedItem->hasFile();

	return false;
}


void VoiceView::enableCommand(int id_, bool enable)
{
	contextMenu->setItemEnabled(id_, enable);
}


void VoiceView::gotSelection(bool on)
{
	enableCommand(ID_FILE_CALLER_ALIAS, on);
	
	if (on)
		on = selectedMsgFile();

	enableCommand(ID_FILE_PLAYBACK, on);
	enableCommand(ID_FILE_EXPORT, on);
	enableCommand(ID_FILE_MAIL, on);
	enableCommand(ID_FILE_DELETE, on);
}


void VoiceView::selectionChangedSlot(QListViewItem *item)
{
	selectedItem = (VoiceViewItem *)item;
	
	gotSelection((selectedItem));
	
	emit hasSelection((selectedItem));
}


void VoiceView::doubleClickedSlot(QListViewItem *)
{
	if (selectedItem)
		emit command(ID_FILE_PLAYBACK);
}


void VoiceView::returnPressedSlot(QListViewItem *item)
{
	if (item) {
		setSelected(item, true);
		emit command(ID_FILE_PLAYBACK);
	}
}


void VoiceView::rightButtonClickedSlot(QListViewItem *item, const QPoint& point, int)
{
	if (item) {
		setSelected(item, true);
		contextMenu->popup(point);
	}
}


void VoiceView::refreshAliases()
{
	VoiceViewItem *item;

	for (item = (VoiceViewItem *)firstChild(); (item); item = (VoiceViewItem *)item->nextSibling())
		item->setAlias();
}


void VoiceView::contextMenuSlot(int id_)
{
	emit command(id_);
}
