/*
    KMLOVoice

    A utility to process voice messages received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 2000 Oliver Gantz <Oliver.Gantz@epost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include <unistd.h>
#include <stdlib.h>

#include <qglobal.h>

#include <kapp.h>
#include <klocale.h>
#include <kmessagebox.h>

#include "downloaddlg.h"
#include "global.h"



DownloadDlg::DownloadDlg(QWidget *parent, const char *name) : QProgressDialog(parent, name, false)
{
	config = kapp->config();
	
	modem = new Modem(this, "modem");
	CHECK_PTR(modem);
	init();

	setMinimumDuration(0);
	setAutoReset(false);
	setAutoClose(false);

	connect(modem, SIGNAL(gotLine(const char *)), SLOT(fetchModemLine(const char *)));
	connect(modem, SIGNAL(gotXBlock(const uchar *, int)), SLOT(fetchXModemBlock(const uchar *, int)));
	connect(modem, SIGNAL(xmodemDone(bool)), SLOT(fetchXModemDone(bool)));
	connect(modem, SIGNAL(timeout()), SLOT(fetchTimeout()));
	connect(this, SIGNAL(cancelled()), SLOT(cancel()));

	setCaption(i18n("Download"));
}


DownloadDlg::~DownloadDlg()
{
	reset();
}


bool DownloadDlg::startDownload()
{
	static int speeds[] = { 19200, 38400, 57600, 115200, 230400 };
	QString dev;
	int speed;

	if (status)
		return false;

	config->setGroup(ID_PREF_GROUP_GENERAL);
	load_new = config->readBoolEntry(ID_PREF_GEN_LOADNEW, PREF_GEN_LOADNEW);
	delete_message = config->readBoolEntry(ID_PREF_GEN_DELETEMSG, PREF_GEN_DELETEMSG);

	config->setGroup(ID_PREF_GROUP_MODEM);
	dev = config->readEntry(ID_PREF_MOD_DEVICE, PREF_MOD_DEVICE);
	speed = config->readNumEntry(ID_PREF_MOD_SPEED, PREF_MOD_SPEED);
	if ((speed < 0) || (speed > 4))
		speed = PREF_MOD_SPEED;
	
	modem->setDevice(dev);
	modem->setSpeed(speeds[speed]);
	modem->setData(8);
	modem->setParity('N');
	modem->setStop(1);

	if (!modem->open()) {
		KMessageBox::sorry(this, i18n("Cannot open modem device."), i18n("Modem Error"));
		return false;
	}
	if (!modem->dsrOn()) {
		KMessageBox::sorry(this, i18n("Modem is off."), i18n("Modem Error"));
		modem->close();
		return false;
	}
	if (!modem->ctsOn()) {
		KMessageBox::sorry(this, i18n("Modem is busy."), i18n("Modem Error"));
		modem->close();
		return false;
	}

	modem->writeLine("");
	usleep(250000);
	modem->flush();
	modem->writeLine("ATZ");

	status = 1;

	return true;
}


void DownloadDlg::cancel()
{
	QProgressDialog::cancel();

	if (status == 3)
		modem->abortXModem();
	else {
		reset();
		emit done();
	}
}


void DownloadDlg::fetchModemLine(const char *line)
{
	static const char *file_type_ids[6] = { "ADPCM2-7", "ADPCM2-8", "ADPCM4-7", "ADPCM4-8", "PCM8L-7", "PCM8L-8" };
	static const uchar file_types[6] = { VOI_ADPCM2_7, VOI_ADPCM2_8, VOI_ADPCM4_7, VOI_ADPCM4_8, VOI_PCM8L_7, VOI_PCM8L_8 };
	char buff[81], *p;
	fitem_t *fitem;
	int year, month, day, hour, minute, i;
  uint size;

	switch (status) {
		case 1:					/* "ATZ" sent	*/
			if (!strcmp(line, "OK")) {
				usleep(250000);
				if (load_new)
					modem->writeLine("AT$JDIR=,VOICE,N");
				else
					modem->writeLine("AT$JDIR=,VOICE");
				status++;
			}
			break;
		case 2:					/* "AT$JDIR=,VOICE[,N]" sent	*/
			if (strlen(line) > 59) {
				usleep(250000);
				strncpy(buff, line, 80);
				buff[80] = 0;
				if ((buff[32] != 'G') && (buff[33] != 'G') && (buff[34] != 'G') && (buff[35] != 'G') && (sscanf(&buff[41], "%d.%d.%d, %d:%d, %u", &day, &month, &year, &hour, &minute, &size) == 6)) {
					fitem = (fitem_t *)malloc(sizeof(fitem_t));
					CHECK_PTR(fitem);
					fitem->next = fitems;
					fitems = fitem;
					if ((p = strchr(buff, ',')))
						*p = 0;
					strncpy(fitems->name, buff, 18);
					fitems->name[18] = 0;
					sprintf(fitems->file, "%04d%02d%02d%02d%02d00.VOI", year, month, day, hour, minute);
					if (!strncmp(fitems->name, fitems->file, 12)) {
						if ((fitems->name[12] >= '0') && (fitems->name[12] <= '9'))
							fitems->file[12] = fitems->name[12];
						if ((fitems->name[13] >= '0') && (fitems->name[13] <= '9'))
							fitems->file[13] = fitems->name[13];
					}
					fitems->type = VOI_UNKNOWN;
					for (i=0; i < 6; i++)
						if (!strncmp(file_type_ids[i], &buff[22], strlen(file_type_ids[i]))) {
							fitems->type = file_types[i];
							break;
						}
					fitems->caller[0] = 0;
					fitems->size = size;
					nfiles++;
				}
			}
			else if (!strcmp(line, "OK")) {
				usleep(250000);
				count = 0;
				status++;
				downloadFile();
			}
			break;
		case 3:					/* XModem confirmed	*/
			if (!strcmp(line, "OK")) {
				usleep(250000);

				if (delete_message) {
					modem->writeLine(QString("AT$JDEL=\"%1\"").arg(fitems->name));
					status++;
				}
				else {
					setProgress(fitems->size + 1);
					fitem = fitems;
					fitems = fitems->next;
					free(fitem);
					downloadFile();
				}
			}
			break;
		case 4:					/* Delete confirmed	*/
			if (!strcmp(line, "OK")) {
				usleep(250000);
				setProgress(fitems->size + 1);
				fitem = fitems;
				fitems = fitems->next;
				free(fitem);
				status = 3;
				downloadFile();
			}
	}
}


void DownloadDlg::fetchXModemBlock(const uchar *block, int size)
{
	int diff;

	if (file.isOpen() && ((diff = fitems->size - progress()) > 0)) {
		if (size > diff)
			size = diff;
		file.writeBlock((const char *)block, size);
		setProgress(progress() + size);
	}
}


void DownloadDlg::fetchXModemDone(bool success)
{
	QString s;

	if (file.isOpen()) {
		file.close();
		if (success)
			emit gotFile(fitems->file, fitems->type, fitems->caller);
		else {
			emit message(QString("%1: %2").arg(i18n("Error")).arg(i18n("Download failed.")));

			reset();
			emit done();
		}
		setLabelText(i18n("Waiting for confirmation..."));
	}
}


void DownloadDlg::fetchTimeout()
{
	QString s;
	
	KMessageBox::sorry(this, i18n("Modem response timeout."), i18n("Modem Error"));

	emit message(QString("%1: %2").arg(i18n("Error")).arg(i18n("Modem response timeout.")));

	reset();
	emit done();
}


void DownloadDlg::init()
{
	load_new = false;
	delete_message = false;
	
	status = 0;

	fitems = 0;
	nfiles = 0;
	count = 0;
}


void DownloadDlg::reset()
{
	fitem_t *p;

	QProgressDialog::reset();

	modem->close();

	while (fitems) {
		p = fitems;
		fitems = fitems->next;
		free(p);
	}

	if (file.isOpen())
		file.close();

	init();

	hide();
}


void DownloadDlg::downloadFile()
{
	QString s;

	if (fitems) {
		count++;
		file.setName(expandPath(fitems->file));
		if (!file.open(IO_WriteOnly)) {
			KMessageBox::sorry(this, i18n("Cannot create file."), i18n("File Error"));
			emit message(QString("%1: %2!").arg(i18n("Error")).arg(i18n("Download failed")));
			reset();
			emit done();
			return;
		}
		modem->writeLine(QString("AT$JDNL=\"%1\"").arg(fitems->name));
		QProgressDialog::reset();
		setLabelText(i18n("Receiving file %1 of %2...").arg(count).arg(nfiles));
		setProgress(0);
		setTotalSteps(fitems->size);
		modem->receiveXModem(true);
	}
	else {
		if (nfiles)
			emit message(i18n("Received %1 file(s).").arg(nfiles));
		else
			emit message(i18n("No message in modem."));

		reset();
		emit done();
	}
}
