/**
 * KPS2DParametric
 * David Sweet
 *
 * A KPSprite which plots (x,y) points, optionally with lines connecting
 *  them in order.
 **/


#ifndef __KPS2DPARAMETRIC_H__
#define __KPS2DPARAMETRIC_H__

#include <stdlib.h>
#include <qcombobox.h>
#include <qcheckbox.h>

#include <klined.h>
#include <kcolorbtn.h>

#include "kps2d.h"
#include "kpmatrix.h"
#include "kpcoords.h"

#include "kps2doptions.h"
#include "kpswhatami.h"


class KPS2DParametricSeriesProp;

class KPS2DParametricOptions
{
 public:
  KPS2DParametricOptions () { matrix=0; pname="Name"; condots=false; pointsize=.02;
  pointsizeo2=pointsize/2; pointstyle=FilledCircles; color=Qt::black;}

  KPMatrix *matrix;
  bool condots;
  double pointsize, pointsizeo2;
  unsigned int pointstyle;
  QColor color;
  QString pname;
};


class KPS2DParametric : public KPS2D
{
 Q_OBJECT


 public:
  /**
   * This sprite creates 2D Parametric plots.  The first column
   *  of the matrix specifies the x values. 
   */
  KPS2DParametric (KPView *view, const KPCoords &coords, KPMatrix *matrix, 
		double x, double y, double w, double h, int z=0);

  KPS2DParametric (KPView *view, const KPCoords &coords,
		   double x, double y, double w, double h, int z=0);

  /**
   * Add a matrix to the plot.  Returns the number of the added matrix.
   **/
  int addMatrix (KPMatrix *_matrix);


  /**
   * Get a matrix from the plot.
   **/
  KPMatrix *matrix (unsigned int series=0)
    { return plots.count()==0 ? 0L : plots.at(series)->matrix; }
 
  /**
   * Number of matrices in plot.
   **/
  int nMatrices (void) { return plots.count(); }

  /**
   * Sets the point size (edge length, diameter, or whatever,
   *  depending on the point type).  Returns TRUE if the size was
   *  set, FALSE if an invalid size was requested.
   *
   * The point size is requested as a fraction of the plot size,
   *  so the maximum point size would be 1, the size of the plot.
   **/
  bool setPointSize (double size, unsigned int series=0);
 
  /**
   * Get the point size.
   **/
  double pointSize (unsigned int series=0)
    {return plots.at(series)->pointsize;}
 
  /**
   * Set the point style.  See kps2doptions.h for a list of
   * plot styles.
   *
   **/
  void setPointStyle (unsigned int _pointstyle, unsigned int series=0)
    { bneedredraw = TRUE; plots.at(series)->pointstyle = _pointstyle;}

  /**
   * Get the point style.
   **/
  unsigned int pointStyle (unsigned int series=0)
    { return plots.at(series)->pointstyle; }
 
  /**
   * Should we connect the dots?
   **/
  void setConnected (bool c=TRUE, unsigned int series=0)
    { bneedredraw = TRUE; plots.at(series)->condots = c;}
 
  /**
   * Are the points connected with line segments?
   **/
  bool connected (unsigned int series=0) {return plots.at(series)->condots;}

  /**
   * Set the marker color.
   **/
  void setColor (QColor col, unsigned int series=0)
    {bneedredraw=true; plots.at(series)->color = col; }

  /**
   * Get the marker color.
   **/
  QColor color (unsigned int series=0)
    { return plots.at(series)->color; }

  /**
   * Set the series name (used on properites tab and, eventually, in
   *  KPS2DParametricLengend.
   **/
  void setSeriesName (QString n, unsigned int series=0);

  /**
   * Get the series name.
   **/
  QString seriesName (unsigned int series=0)
    { return plots.at(series)->pname; }

  /**
   * The number of series (matrices) in this plot.
   **/
  unsigned int nSeries (void) { return plots.count(); }

  /**
   * Set the whole options class for a series.
   **/
  void setOptions (const KPS2DParametricOptions &op, unsigned int series=0);

  /**
   * Get the whole options class for a series.
   **/
  KPS2DParametricOptions options (unsigned int series=0)
    { return *plots.at(series); }


  /**
   * Find coordinates which fit a KPMatrix.
   * If you pass in a KPCoords, then this will find coordinates which
   *  fit these KPCoords as well as the KPMatrix.
   **/
  static KPCoords autoCoords (const KPMatrix *,
			      const KPCoords * = 0);

  /**
   * Find coordinates which fit the current plot.
   * If you pass in a KPCoords, then this will find coordinates which
   *  fit these KPCoords as well as the KPMatrix.
   **/
  virtual KPCoords autoCoords (const KPCoords *c = 0)
    {return autoCoords (matrix(), c);}


 
  /**
   * Installs the default RMB menu.
   **/
  void installDefaultRMBMenu (void);
 
 
  /**
   * This KPSprite offers a properties widget.
   **/
  QWidget *propWidget (unsigned int series, QWidget *parent, const char *name=0);
 
                       
  /**
   * I am a KPS2DParametric.
   **/
   virtual int whatami() const;

  ~KPS2DParametric();

  public slots:
   void slotProperties();

 protected:

  virtual void reDraw(QPainter *);
  //virtual void setRect();

  //virtual void resizeEvent (QResizeEvent *);

 private:

  bool rdflag;
  int rdcol, rdpoints, rdi, rdj;
  double rdlastx, rdlasty;

  QList<KPS2DParametricOptions> plots;


 protected:
  KPS2DParametricSeriesProp *propwid;
  
};






class KPS2DParametricSeriesProp : public QWidget
{
  Q_OBJECT

 public:
  KPS2DParametricSeriesProp (KPS2DParametric *, unsigned int series, QWidget *parent, const char *name, WFlags f=0);
  
  /**
   * Should the points be connected with line segments?
   **/
  bool connected (void) {return cbcon->isChecked();}

  /**
   * Get the point style.
   **/
  unsigned int pointStyle (void)
    { return cbstyle->currentItem(); }

  /**
   * Get the point size.
   **/
  double pointSize (void)
  {return atof (lesize->text())/100.;}
  
  /**
   * Get the marker color.
   **/
  QColor color (void) { return colb->color(); }
  
 protected:
  KLineEdit *lesize;
  QComboBox *cbstyle;
  QCheckBox *cbcon;
  KColorButton *colb;

  protected slots:
   void slotCombo (int);
};

#endif



