#ifndef __KPMATRIX_H__
#define __KPMATRIX_H__

#include <qglobal.h>
#include <qstring.h>

class QWidget;

/**
 * KPMatrix
 *
 * David Sweet
 *
 * A simple matrix storage class.
 * Check out the @ref load() method.  It's loads a matrix from a file.
 *
 **/
class KPMatrix
{
 public:
  /**
   * Pass a pointer to a matrix along with its size.
   * <i>nc</i> - number of columns
   * <i>nr</i> - number of rows
   * If <i>deep</i>==TRUE, then KPMatrix makes an internal copy
   *  of data which is deleted upon destruction.  You can
   *  then set () beyond the original <i>nr</i> and the
   *  matrix will grow.  You cannot add beyond <i>nc</i>.  Passing
   *  0 for <i>_matrix</i>, is fine when <i>deep</i>==TRUE.
   * If <i>deep</i>==FALSE, the matrix size is fixed and you
   *  should not pass 0 for <i>_matrix</i>
   **/
  KPMatrix (double *_matrix, unsigned int nc, unsigned int nr,
	    bool deep=TRUE);

  /**
   * This constructor is equivalent to calling the above constructor as
   *  KPMatrix (0, nc, 1, TRUE);
   **/
  KPMatrix (unsigned int nc);

  /**
   * Use this directly ( along with nRows(), nCols() ) for
   *  fast scanning through the matrix.
   **/
  double *matrix () {return data;}

  /**
   * Get the value at <i>row</i>, <i>col</i>.
   **/
  double at (unsigned int row, unsigned int col) 
    { return data [row*nc + col]; }

  /**
   * Number of rows in matrix
   **/
  unsigned int nRows (void) {return nr;}

  /**
   * Number of columns in matrix
   **/
  unsigned int nCols (void) {return nc;}

  /**
   * Set an element of the matrix.
   * It will generally be quicker to fill the matrix and then pass
   *  it to the constructor rather than the other way around.  But
   *  you may only notice that for large matrices.
   * You may call set() for <i>nc</i> and <i>nr</i> values
   *  beyond the inital limits.  It will extend the size of the
   *  matrix.
   **/
  void set (unsigned int nr, unsigned int nc, double value);

  /**
   * Returns the minimum value in column <i>col</i> (efficiently).
   **/
  double min (unsigned int col) const { return amin [col]; }

  /**
   * Returns the maximum value in column <i>col</i> (efficiently).
   **/
  double max (unsigned int col) const { return amax [col]; }

  /**
   * Load a matrix from an ASCII file.
   *
   * This will process a standard space- (or comma-, or semicolon-)
   *  separated-column file.
   **/
  static KPMatrix * load (const QString &filename, QWidget *parent=0);

  ~KPMatrix();
  
 private:
    unsigned int nr, nc, memnr, growby;
    bool bdeep;
    double *data;
    double *amin, *amax;

    void init (double *_matrix, unsigned int nc, unsigned int nr,
	    bool deep=TRUE);

    void minmax (unsigned int col, int index);
};

inline void
KPMatrix::minmax (unsigned int col, int index)
{
  if (data [index]>amax[col])
    amax[col]=data [index];
  if (data [index]<amin[col])
    amin[col]=data [index];
}
#endif
