/*************************************************/
/* methods for class Device                      */
/*                                               */
/* logical simulation for wired gates            */
/*                                               */
/* Andreas Rostin                                */
/* 15.03.99                                      */
/*************************************************/
#include <stdio.h>
#include <string.h>
#include <klogic.h>
#include <device.h>

/***************************************************/
/* static methods of Device                        */
/***************************************************/
int Device::STATcnt[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
int Device::STATdef_delay = 1;
int Device::STATdef_undefined = 0;
int Device::STATdef_clock = CLK_FALLING_EDGE;
int Device::STATreverse_trigger = 0;

// static method
void Device::resetCounter()
{	int i;

	for (i = 0; i < 20; i++)
		STATcnt[i] = 0;
}

// static method
int Device::defDelay()
{
	return STATdef_delay;
}

// static method
void Device::setDefDelay(int newi)
{
	if (newi > MAXDELAY) STATdef_delay = MAXDELAY;
	else if (newi < MINDELAY) STATdef_delay = MINDELAY;
	else STATdef_delay = newi;
}

// static method
int Device::defUndefined()
{
	return STATdef_undefined;
}

// static methods
void Device::setDefUndefined(int newi)
{
	if (newi >= 1) STATdef_undefined = 1;
	else STATdef_undefined = 0;
}

int Device::defClock()
{
	return STATdef_clock;
}

void Device::setDefClock(int newi)
{
	if ((newi >= 0) && (newi <4))
		STATdef_clock = newi;
	else STATdef_clock = CLK_FALLING_EDGE;
}

void Device::invertTrigger(int newi)
{
	STATreverse_trigger = newi;
}

int Device::triggerInverted()
{
	return STATreverse_trigger;
}

/***************************************************/
/* methods of Device                               */
/***************************************************/
// Constructor
Device::Device(int func, int del=-1, int undef=-1, int clock=-1)
{	int undef_value;
	int delay;
	char buf[101];

	id = uniqueID::getID();
	mq_id = 0;
	c1_1_id = 0;

	// these counters are only used to get a new name!
	sprintf(buf, "%d", ++STATcnt[func - fBASE]);

	text = (char *)malloc(sizeof(char) * 111);
	switch (func) {
		case Device::fTXT:
			strcpy(text, "text");
			break;
		case Device::fIN:
			strcpy(text, "in");
			break;
		case Device::fOUT:
			strcpy(text, "out");
			break;
		case Device::fAND:
			strcpy(text, "and");
			break;
		case Device::fOR:
			strcpy(text, "or");
			break;
		case Device::fXOR:
			strcpy(text, "xor");
			break;
		case Device::fSWI:
			strcpy(text, "swi");
			break;
		case Device::fRS:
			strcpy(text, "rs");
			break;
		case Device::fJK:
			strcpy(text, "jk");
			break;
		case Device::fDFF:
			strcpy(text, "d");
			break;
		case Device::fNET:
			strcpy(text, "sub");
			break;
		case Device::fEQU:
			strcpy(text, "sub");
			break;
		case Device::fSS:
			strcpy(text, "7s");
			break;
		case Device::fLEDred:
		case Device::fLEDblue:
		case Device::fLEDgreen:
		case Device::fLEDyellow:
			strcpy(text, "led");
			break;
		case Device::fOSZ:
			strcpy(text, "osc");
			break;
		case Device::fONE:
			strcpy(text, "inv");
			break;
		default:
			strcpy(text, "dev");
			break;
	}
	strcat(text, buf);

	function = func;

	if (undef < 0) undef_value = Device::STATdef_undefined;
	else undef_value = undef;

	// all outputs have the same delay
	if (del <= 0) delay = Device::STATdef_delay;
	else delay = del;
	if (function == fNET ||
	    function == fLEDred ||
	    function == fLEDblue ||
	    function == fLEDgreen ||
	    function == fLEDyellow ||
	    function == fSS ||
            function == fSWI) delay = 0;

	osz_off_cnt = 10;
	osz_on_cnt = 10;
	osz_curr_step = 0;
	if (clock < 0)
		if (function == fRS) clock_type = CLK_NONE;
		else if (function == fOSZ) clock_type = CLK_MULTI;
		else clock_type = Device::STATdef_clock;
	else
		clock_type = clock;
	last_clk = undef_value;

	// master-slave flipflop: master enabled
	if (function == fRS) master = 0;
	else master = 1;

	// detect changes of the output-value
	old_result = 0;
	new_result = 1;

	// the first output in the list of outputs is an unnamed output, each device can use
	// it can be reached by "named_output.Get()->"
	opStack *op = new opStack(this, undef_value, delay);
	op->setPosition(-1);
	named_output.Append(op);

	static_input_value = 0;

}

// Destructor
Device::~Device()
{	list<value> *l;
	list<opStack> *o;

	// destroy lists
	connection_list.Destroy();
	l = named_input.First();
	while(l) {
		delete l->Get();
		l = l->Next();
	}
	named_input.Destroy();

	o = named_output.First();
	while(o) {
		delete o->Get();
		o = o->Next();
	}
	named_output.Destroy();

	// free storage
	if (text) free(text);
}

int Device::oszOn()
{
	return osz_on_cnt;
}

void Device::setOszOn(int value)
{
	osz_on_cnt = value;
}

int Device::oszOff()
{
	return osz_off_cnt;
}
void Device::setOszOff(int value)
{
	osz_off_cnt = value;
}

int Device::clock()
{
	return clock_type;
}

void Device::setClock(int newc)
{
	if ((newc >= 0) && (newc <= 6)) {
		if (function == fOSZ && newc != CLK_MULTI && newc != CLK_MONO) newc = CLK_MULTI; 
		clock_type = newc;
	} else {
		if (function == fOSZ) clock_type = CLK_MULTI; 
		clock_type = CLK_RISING_EDGE;
	}
	setEquation();
	parseEquation();
}

int Device::hasMaster()
{
	return master;
}

// enable/disable master of a master-slave flipflop
void Device::setMaster(int newi)
{
	if (newi >= 1)
		master = 1;
	else
		master = 0;
	setEquation();
	parseEquation();
}

char * Device::getText(int nospace = 0)
{	static char buf[10001];

	if (!nospace) return text;

	// remove all spaces from the text
	if (text) {
		strncpy(buf, text, 10000);
		buf[10000] = 0;
		char *s;
		while (NULL != (s = strchr(buf, ' '))) *s = '_';
		return buf;
	}

	return (char *)NULL;
}

void Device::setText(char *new_text)
{	char *newp;

	if (new_text && strlen(new_text) > 0) {
		if (!text || (text && strcmp(text, new_text))) {
			newp = (char *)malloc(sizeof(char) * (strlen(new_text) + 1));
			strcpy(newp, new_text);
			if (text) free(text);
			text = newp;
		}

	}
}

// connect two devices
//	out_dev:   connected device
//	input_id:  id of this named input
//	output_id: id of out_dev's named output
int Device::connectDevices(Device *out_dev, int input_id, int output_id)
{
	if (!connection_list.Append(out_dev, input_id, output_id)) return 0;
	return 1;
}

// disconnect two devices
int Device::disconnectDevices(Device *out_dev, int input_id, int output_id)
{	
	if (!connection_list.Destroy(out_dev, input_id, output_id)) return 0;
	return 1;
}

// return type of device
int Device::type()
{	
	return function;
}

// change LED color by changing the type
void Device::changeLEDType(int new_type) {
	if (new_type != fLEDred &&
	    new_type != fLEDblue &&
	    new_type != fLEDgreen &&
	    new_type != fLEDyellow) return;
	function = new_type;
}

// static: devices with user interaction
int Device::isInteractive()
{	
	if (function == fSWI)
		return 1;
	return 0;
}

int Device::sizeChangeable()
{	
	if (function == fSS ||
	    function == fRS ||
	    function == fJK ||
	    function == fLEDred ||
	    function == fLEDblue ||
	    function == fLEDgreen ||
	    function == fLEDyellow ||
	    function == fDFF ||
	    function == fOSZ ||
	    function == fSWI)
		return 0;
	return 1;
}

int Device::hasNamedInput()
{
	if (function == fSS ||
	    function == fRS ||
	    function == fJK ||
	    function == fDFF ||
	    function == fNET ||
	    function == fEQU)
		return 1;
	return 0;
}

// check if name is unused
int Device::inputNameIsUnique(const char *input_name)
{
	if (named_input.With(input_name)) return 0;
	return 1;
}

// add a new named input for this device
// the dev_id argument is used for net interfaces
// it contains the id of the position of the named input device
int Device::addInputName(const char *name, int pos, int input_id)
{
	value *v = new value;
	v->setPosition(pos);
	if (!input_id) input_id = uniqueID::getID();
	named_input.Append(v, input_id, pos)->setText(name);
	return input_id;
}

// change the name of an input
void Device::changeInputName(int input_id, const char *new_name)
{	list<value> *v = named_input.With(input_id);
	if (v) v->setText(new_name);
}

// remove an existing named input from this device
void Device::removeInputName(int input_id)
{	list<value> *v = named_input.With(input_id);
	if (v) {
		delete v->Get();
		named_input.Destroy(input_id);
	}
}

// return the id of an output name (equation devices: unique names!)
int Device::getInputID(const char *input_name)
{	list <value> *lv = named_input.With(input_name);

	if (lv) return lv->getID1();
	fprintf(stderr, "warning: cannot find input <%s>\n", input_name);
	return 0;
}

int Device::hasNamedOutput()
{
	if (function == fNET || function == fEQU) return 1;
	return 0;
}

// add a new named output for this device
// the dev_id argument is used for net interfaces
// it contains the id of the position of the named output device
int Device::addOutputName(const char *name, int pos, int output_id = 0)
{
	opStack *op = new opStack(this, undef(), delay());
	op->setPosition(pos);
	if (!output_id) output_id = uniqueID::getID();
	named_output.Append(op, output_id, pos)->setText(name);
	return output_id;
}

// change the name of an input
void Device::changeOutputName(int output_id, const char *new_name)
{	
	list<opStack> *op = named_output.With(output_id);
	if (op) {
		op->setText(new_name);
	}
}

// remove a named output from this device
void Device::removeOutputName(int output_id)
{
	list<opStack> *op = named_output.With(output_id);
	if (op) {
		delete op->Get();
		named_output.Destroy(output_id);
	}
}

// return the name of an output id (equation devices: unique names!)
char * Device::getOutputName(int output_id)
{	list <opStack> *lop = named_output.With(output_id);

	if (lop) return lop->getText();
	fprintf(stderr, "warning: cannot find output with id <%d>\n", output_id);
	return (char *)NULL;
}

// return the name of an output id (equation devices: unique names!)
int Device::getOutputID(const char *output_name)
{	list <opStack> *lop = named_output.With(output_name);

	if (lop) return lop->getID1();
	fprintf(stderr, "warning: cannot find output with name <%s>\n", output_name);
	return 0;
}

// change type of output
void Device::setOutputNameType(int output_id, int new_type)
{	list<opStack> *op = named_output.With(output_id);

	if (op) {
		if (new_type == INTERNAL_OUTPUT)
			op->Get()->setInternal(true);
		else
			op->Get()->setInternal(false);
	}
}

// add a new named internal output for this device
int Device::addInternalName(const char *name)
{	int int_id = uniqueID::getID();

	opStack *op = new opStack(this, undef(), delay());
	op->setInternal(true);
	op->setPosition(0);
	named_output.Append(op, int_id)->setText(name);
	return int_id;
}

// returns if id is an internal output name
int Device::isInternalName(int internal_id)
{	list<opStack> *op = named_output.With(internal_id);

	if (op && op->Get()->isInternal()) return INTERNAL_OUTPUT;
	if (type() == fLEDred ||
	    type() == fLEDblue ||
	    type() == fLEDgreen ||
	    type() == fLEDyellow ||
	    type() == fSS) return FINAL_OUTPUT;

	if (type() == fNET || type() == fEQU) {
		if (connection_list.With(internal_id)) return FINAL_OUTPUT;
	}

	return TMP_OUTPUT;
}

// remove a named internal output from this device
void Device::removeInternalName(int internal_id)
{
	removeOutputName(internal_id);
}

list<value> * Device::getNamedIRef()
{
	return &named_input;
}

list<opStack> * Device::getNamedORef()
{
	return &named_output;
}

int Device::hasStaticInput()
{
	if (function == fIN || function == fSWI || function == fPWR) return 1;
	return 0;
}

// set static output-value
void Device::toggleStaticInput()
{
	old_result = static_input_value;
	if (static_input_value) static_input_value = 0;
	else static_input_value = 1;
	named_output.Get()->flush(static_input_value);
	new_result = static_input_value;
}

// set static input-value
void Device::setStaticInput(int val)
{
	old_result = static_input_value;
	static_input_value = val;
	named_output.Get()->flush(static_input_value);
	new_result = static_input_value;
}

// set static output-value
void Device::setStaticOutput(int output_id, int val)
{	list<opStack> *lop = named_output.With(output_id);

	if (lop && lop->Get() && lop->Get()->getResult() != val) {
		old_result = new_result;
		new_result -= lop->Get()->getResult() * lop->current();
		lop->Get()->flush(val);
		new_result += lop->Get()->getResult() * lop->current();
	}
}

int Device::hasClock()
{
	if (function == fJK || function == fDFF || function == fRS)
		return 1;
	return 0;
}

void Device::setID(int newid)
{
	id = newid;
}

// name of device
int Device::getID()
{
	return id;
}

// set value for undefined outputs
void Device::setUndef(int undef)
{
	list<opStack> *lo = named_output.First();
	while(lo) {
		lo->Get()->setUndefined(undef);
		lo = lo->Next();
	}
}

int Device::undef()
{
	return named_output.Get()->getUndefined();
}

// set signal delay
void Device::setDelay(int d)
{
	list<opStack> *lop = named_output.First();
	while(lop) {
		lop->Get()->setDelay(d);
		lop = lop->Next();
	}
}

int Device::delay()
{
	return named_output.Get()->getDelay();
}

// return input value
int Device::input(int input_id)
{	list<Device> *l = connection_list.With(input_id);

	if (l) return l->Get()->output(l->getID2());
	return 0;
}

// return output value
int Device::output(int output_id = 0)
{
	// internal device
	if (function == fINV_INTERNAL) {
		if (connection_list.First()) {
			// internal inverter (recursive method call)
			if (connection_list.First()->Get()->output(connection_list.getID2()) >= 1)
				return 0;
			else
				return 1;
		}
		else {
			if (undef() >= 1) return 0;
			else return 1;
		}
	}

	// named outputs
	if (output_id != 0) {
		list<opStack> *lo = named_output.With(output_id);
		if (lo) return lo->Get()->getResult();
		else {
			fprintf(stderr, "Device::output() no such named output <%d> for %d\n", output_id, getID());
			exit(-1);
		}
	}

	// all other output requests: stored within the first named output without name (unnamed output)
	int ret = named_output.Get()->getResult();
	if (ret >= 0 && ret <= 15) return ret;
	return 0;
	//return named_output.Get()->getResult();
}

void Device::flush(int value)
{
	list<opStack> *lop = named_output.First();
	while(lop) {
		lop->Get()->flush(value);
		lop = lop->Next();
	}
}

// return 1 if output value has changed since last calculation
int Device::outputChanged()
{
	if (new_result != old_result) return 1;
	return 0;
}

// two step calculation
// 1. calculate new output-values (old value at output visible)
// 2. switch new output values (a new (old) value at output visible)
void Device::Calculate(int step, int substep=0)
{
	if (step) {
		// switch all outputs
		list<opStack> *lop = named_output.First();
		int result_weight = 0;
		if (substep) {
			new_result = 0;
			while(lop) {
				result_weight++;
				lop->Get()->shift();
				new_result += lop->Get()->getResult() * result_weight;
				lop = lop->Next();
			}
		} else {
			new_result = 0;
			old_result = 0;
			while(lop) {
				result_weight++;
				old_result += lop->Get()->shift() * result_weight;
				new_result += lop->Get()->getResult() * result_weight;
				lop = lop->Next();
			}
		}
	} else calculate(substep);
}

// calculates output-value
inline void Device::calculate(int substep)
{	int ip1, ip2, ip3, ip4;
	list<opStack> *lop;
	opStack *op;
	list<Device> *l;

	// determine new output value(s)
	switch (function) {
	// special device
	case fSS:
		static_output_value = undef();
		l = connection_list.With(s7_1_id);
		if (l) ip1 = l->Get()->output(l->getID2());
		else ip1 = 0;
		l = connection_list.With(s7_2_id);
		if (l) ip2 = l->Get()->output(l->getID2());
		else ip2 = 0;
		l = connection_list.With(s7_4_id);
		if (l) ip3 = l->Get()->output(l->getID2());
		else ip3 = 0;
		l = connection_list.With(s7_8_id);
		if (l) ip4 = l->Get()->output(l->getID2());
		else ip4 = 0;

		static_output_value = ip1 + (ip2 * 2) + (ip3 * 4) + (ip4 * 8);
		named_output.Get()->push(static_output_value);
		named_output.Get()->calculate();
		break;
	// special device
	case fOSZ:
		if (!substep) {
			if (clock() == CLK_MULTI) {
				osz_curr_step++;
				if (static_output_value && osz_curr_step >= osz_on_cnt) {
					static_output_value = 0;
					osz_curr_step = 0;
				}
				if (!static_output_value && osz_curr_step >= osz_off_cnt) {
					static_output_value = 1;
					osz_curr_step = 0;
				}
			} else {
				l = connection_list.First();
				if (l) {
					if (static_output_value) {
						if (osz_curr_step > osz_on_cnt) static_output_value = 0;
						else osz_curr_step++;
					} else {
						if (l->Get()->output()) {
							if (!osz_curr_step) static_output_value = 1;
						} else osz_curr_step = 0;
					}
				}
			}
			named_output.Get()->push(static_output_value);
			named_output.Get()->calculate();
		}
		break;
	default:
		// calculate all stacks
		lop = named_output.First();
		while(lop) {
			op = lop->Get();
			// static operation devices: push all input values to the calculation stack
			if (op->hasStaticOP()) {
				if (hasStaticInput()) {
					// e.g the switch: device without inputs
					op->push(static_input_value);
				} else {
					// e.g. the AND gate
					l = connection_list.First();
					while(l) {
						op->push(l->Get()->output(l->getID2()));	// id2: id of the named output of the connected device
						l = l->Next();
					}
				}
			}
			op->calculate();
			lop = lop->Next();
		}
		break;
	}
	return;
}

// return equation string for an output
char * Device::getEquation(int output_id = 0)
{
	if (!output_id) return named_output.Get()->getEquation();

	list<opStack> *lo = named_output.With(output_id);

	if (lo) return lo->Get()->getEquation();
	else return (char *)NULL;
}

// set equation operation for predefined devices
void Device::setEquation()
{       int clk = clock();

	if (STATreverse_trigger) {
		switch(clk) {
			case CLK_RISING_EDGE:
				clk = CLK_FALLING_EDGE;
				break;
			case CLK_FALLING_EDGE:
				clk = CLK_RISING_EDGE;
				break;
			case CLK_HIGH_VALUE:
				clk = CLK_LOW_VALUE;
				break;
			case CLK_LOW_VALUE:
				clk = CLK_HIGH_VALUE;
				break;
		}
	}

	switch(function) {
	case fONE:
		setEquation(OP_NONE);
		break;
	case fPWR:
		setEquation(OP_NONE);
		break;
	case fSWI:
		setEquation(OP_NONE);
		break;
	case fTXT:
		setEquation(OP_NONE);
		break;
	case fIN:
		setEquation(OP_NONE);
		break;
	case fOUT:
		setEquation(OP_OR);
		break;
	case fXOR:
		setEquation(OP_XOR);
		break;
	case fAND:
		setEquation(OP_AND);
		break;
	case fOR:
		setEquation(OP_OR);
		break;
	case fLEDred:
	case fLEDblue:
	case fLEDgreen:
	case fLEDyellow:
		setEquation(OP_OR);
		break;
	case fSS:
		setEquation(OP_NONE);
		break;
	case fOSZ:
		setEquation(OP_NONE);
		break;
	case fDFF:
		if (master) {
			switch(clk) {
			case CLK_RISING_EDGE:
				// equations for internal outputs
				setEquation("/(C1 & /C1-1) & MQ + C1 & /C1-1 & 1D", mq_id);
				setEquation("C1", c1_1_id);
				// equation for unnamed output
				setEquation("/(/C1 & C1-1) & _this_ + /C1 & C1-1 & MQ");
				break;
			case CLK_FALLING_EDGE:
				setEquation("/(/C1 & C1-1) & MQ + /C1 & C1-1 & 1D", mq_id);
				setEquation("C1", c1_1_id);
				setEquation("/(C1 & /C1-1) & _this_ + C1 & /C1-1 & MQ");
				break;
			case CLK_HIGH_VALUE:
				setEquation("/C1 & MQ + C1 & 1D", mq_id);
				setEquation((char *)NULL, c1_1_id);
				setEquation("C1 & _this_ + /C1 & MQ");
				break;
			case CLK_LOW_VALUE:
				setEquation("C1 & MQ + /C1 & 1D", mq_id);
				setEquation((char *)NULL, c1_1_id);
				setEquation("/C1 & _this_ + C1 & MQ");
				break;
			}
		} else {
			switch(clk) {
			case CLK_HIGH_VALUE:
				setEquation((char *)NULL, mq_id);
				setEquation((char *)NULL, c1_1_id);
				setEquation("/C1 & _this_ + C1 & 1D");
				break;
			case CLK_LOW_VALUE:
				setEquation((char *)NULL, mq_id);
				setEquation((char *)NULL, c1_1_id);
				setEquation("C1 & _this_ + /C1 & 1D");
				break;
			}
		}
		break;
	case fJK:
		if (master) {
			switch(clk) {
			case CLK_RISING_EDGE:
				setEquation("/(C1 & /C1-1) & MQ + /1J & /1K & MQ + C1 & /C1-1 & 1J & /1K + C1 & /C1-1 & 1J & 1K & /MQ", mq_id);
				setEquation("C1", c1_1_id);
				setEquation("/(/C1 & C1-1) & _this_ + /C1 & C1-1 & MQ");
				break;
			case CLK_FALLING_EDGE:
				setEquation("/(/C1 & C1-1) & MQ + /1J & /1K & MQ + /C1 & C1-1 & 1J & /1K + /C1 & C1-1 & 1J & 1K & /MQ", mq_id);
				setEquation("C1", c1_1_id);
				setEquation("/(C1 & /C1-1) & _this_ + C1 & /C1-1 & MQ");
				break;
			case CLK_HIGH_VALUE:
				setEquation("/C1 & MQ + /1J & /1K & MQ + C1 & 1J & /1K + C1 & 1J & 1K & /MQ", mq_id);
				setEquation((char *)NULL, c1_1_id);
				setEquation("C1 & _this_ + /C1 & MQ");
				break;
			case CLK_LOW_VALUE:
				setEquation("C1 & MQ + /1J & /1K & MQ + /C1 & 1J & /1K + /C1 & 1J & 1K & /MQ", mq_id);
				setEquation((char *)NULL, c1_1_id);
				setEquation("/C1 & _this_ + C1 & MQ");
				break;
			}
		} else {
			fprintf(stderr, "JK without master??\n");
			exit(-1);
		}
		break;
	case fRS:
		if (master) {
			switch(clk) {
			case CLK_RISING_EDGE:
				setEquation("/(C1 & /C1-1) & MQ + /1S & /1R & MQ + C1 & /C1-1 & 1S", mq_id);
				setEquation("C1", c1_1_id);
				setEquation("/(/C1 & C1-1) & _this_ + /C1 & C1-1 & MQ");
				break;
			case CLK_FALLING_EDGE:
				setEquation("/(/C1 & C1-1) & MQ + /1S & /1R & MQ + /C1 & C1-1 & 1S", mq_id);
				setEquation("C1", c1_1_id);
				setEquation("/(C1 & /C1-1) & _this_ + C1 & /C1-1 & MQ");
				break;
			case CLK_HIGH_VALUE:
				setEquation("/C1 & MQ + /1S & /1R & MQ + C1 & 1S", mq_id);
				setEquation((char *)NULL, c1_1_id);
				setEquation("C1 & _this_ + /C1 & MQ");
				break;
			case CLK_LOW_VALUE:
				setEquation("C1 & MQ + /1S & /1R & MQ + /C1 & 1S", mq_id);
				setEquation((char *)NULL, c1_1_id);
				setEquation("/C1 & _this_ + C1 & MQ");
				break;
			}
		} else {
			switch(clk) {
			case CLK_NONE:
				setEquation((char *)NULL, mq_id);
				setEquation((char *)NULL, c1_1_id);
				setEquation("/S & /R & _this_ + S");
				break;
			case CLK_HIGH_VALUE:
				setEquation((char *)NULL, mq_id);
				setEquation((char *)NULL, c1_1_id);
				setEquation("/C1 & _this_ + /1S & /1R & _this_ + C1 & 1S");
				break;
			case CLK_LOW_VALUE:
				setEquation((char *)NULL, mq_id);
				setEquation((char *)NULL, c1_1_id);
				setEquation("C1 & _this_ + /1S & /1R & _this_ + /C1 & 1S");
				break;
			}
		}
		break;
	default:
		setEquation(OP_NULL);
		break;
	}
}

// set equation operation for the (unnamed) output
void Device::setEquation(char _new_equation)
{
	named_output.Get()->setEquation(_new_equation);
}

void Device::setEquation(const char *_new_equation, int output_id = 0)
{
	setEquation((char *)_new_equation, output_id);
}

// set equation string for an output
void Device::setEquation(char *_new_equation, int output_id = 0)
{	list<opStack> *lo;

	if (!output_id) lo = named_output.First();
	else lo = named_output.With(output_id);

	if (lo) lo->Get()->setEquation(_new_equation);
}

// parse all equations, build the calculation stacks for this device
StackInfo Device::parseEquation()
{	list<opStack> *lo = named_output.First();
	StackInfo ret;

	while(lo) {
		ret = lo->Get()->parse(&named_input, &named_output);
		if (ret.no) {
			// error <no> occured at <position> and <length>
			ret.output_name = lo->getText();
			ret.output_id = lo->getID1();
			return ret;
		}
		lo = lo->Next();
	}
	return ret;
}

// clear all calculation stacks for this device
void Device::resetParsing()
{	list<opStack> *lo = named_output.First();

	while (lo) {
		lo->Get()->clear();
		lo = lo->Next();
	}
}

// deliveres a string like [/][qualifier_prefix.]devicename[.outputname]
// depends on the given device and output name
char * Device::qualifiedOutputName(Device *d, char *out_name, const char *qual_pref=(const char *)NULL)
{	static char buf[1000];
	char *new_out_name;
	list<Device> *li;
	int dbl_inv = 0;

	char *prefix=(char *)qual_pref;
	if (!prefix) prefix = "";
	buf[0] = 0;

	if (d->type() == fINV_INTERNAL) {
		li = d->connection_list.First();
		if (li) {
			if (li->Get()->type() == fINV_INTERNAL) {
				li = li->Get()->connection_list.First();
				dbl_inv = 1;
			}
		}
		if (li) {
			new_out_name = li->getText();
			if (new_out_name && strlen(new_out_name)) {
				if (dbl_inv) sprintf(buf, "%s%s.%s", prefix, li->Get()->getText(NOSPACE), new_out_name);
				else sprintf(buf, "/%s%s.%s", prefix, li->Get()->getText(NOSPACE), new_out_name);
			} else {
				if (dbl_inv) sprintf(buf, "%s%s", prefix, li->Get()->getText(NOSPACE));
				else sprintf(buf, "/%s%s", prefix, li->Get()->getText(NOSPACE));
			}
		} else buf[0] = 0;
	} else {
		if (out_name && strlen(out_name))
			sprintf(buf, "%s%s.%s", prefix, d->getText(NOSPACE), out_name);
		else
			sprintf(buf, "%s%s", prefix, d->getText(NOSPACE));
	}

	// remove all spaces from the used names!
	char *s;
	while (NULL != (s = strchr(buf, ' '))) *s = '_';

	return buf;
}

// append all equations of this device to the given list
// the qualifier will be appended to each name and equation
// isolate: just look at this device, not outside
void Device::getAllEquations(list<OutputInfo> *ls, char *qualifier, int isolate = 0)
{	list<Device> *lconn;
	list<value> *li;
	list<opStack> *lo;
	QString qual_pref(qualifier);
	QString final;

	if (!ls) fatal("no list??\n");

	// remember the interface circuits
	if (!qual_pref.length() && function == fOUT ||
	    function == fSS ||
	    function == fLEDred ||
	    function == fLEDblue ||
	    function == fLEDgreen ||
	    function == fLEDyellow) final = "(final)";
	else final = "";

	if (hasNamedInput()) {
		// device has named inputs

		if (!isolate) {
			// build artificial output for each input
			li = named_input.First();
			while(li) {
				// look for a connection (could be more than one)
				int connected = 0;
				QString eq;
				lconn = connection_list.First();
				while(lconn) {
					if (li->getID1() == lconn->getID1()) {
						if (!connected) {
							eq = qualifiedOutputName(lconn->Get(), lconn->Get()->getOutputName(lconn->getID2()), qual_pref);
							connected = 1;
						} else if (connected == 1) {
							eq = "(" + eq + ") + (";
							eq += qualifiedOutputName(lconn->Get(), lconn->Get()->getOutputName(lconn->getID2()), qual_pref);
							eq += ")";
							connected = 2;
						} else {
							eq += " + (";
							eq += qualifiedOutputName(lconn->Get(), lconn->Get()->getOutputName(lconn->getID2()), qual_pref);
							eq += ")";
						}

					}
					lconn = lconn->Next();
				}
				if (connected) {
					OutputInfo *oi = new OutputInfo();
					oi->device_id = this->getID();
					oi->output_id = 0;
					oi->output_name = qualifiedOutputName(this, li->getText(), qual_pref);
					oi->original_output_name = qualifiedOutputName(this, li->getText(), qual_pref);
					oi->output_type = INPUT;
					oi->equation = eq;
					oi->prefix = qual_pref;
					oi->suffix = final;
					ls->Append(oi)->setText(oi->output_name);
				}

				li = li->Next();
			}
		}
	}

	// named output equation handling
	lo = named_output.First();
	while(lo) {
		if (lo->Get()->hasEquation() && !(!lo->Get()->hasStaticOP() && lo == named_output.First())) {
			QString qual;

			if (!isolate) qual = qual_pref + this->getText(NOSPACE) + ".";
			else qual = qual_pref;

			OutputInfo *oi = new OutputInfo();
			oi->device_id = getID();
			oi->output_id = lo->getID1();
			if (lo->Get()->hasStaticOP()) {
				// concatenate all inputs with the static operation
				lconn = connection_list.First();
				while(lconn) {
					oi->equation += qualifiedOutputName(lconn->Get(), lconn->Get()->getOutputName(lconn->getID2()), qual_pref);
					lconn = lconn->Next();
					if (lconn) {
						oi->equation += " ";
						oi->equation += lo->Get()->getEquation();	// the static operation
						oi->equation += " ";
					}
				}
			} else {
				// the opStack instance has the equation
				oi->equation = lo->Get()->getEquation(qual);
			}
			if (!isolate) {
				oi->output_name = qualifiedOutputName(this, lo->getText(), qual_pref);
				oi->original_output_name = qualifiedOutputName(this, lo->getText(), qual_pref);
			} else {
				oi->output_name = lo->getText();
				oi->original_output_name = lo->getText();
			}
			if (!isolate) {
				oi->output_type = isInternalName(oi->output_id);
			} else {
				if (isInternalName(oi->output_id) != INTERNAL_OUTPUT) oi->output_type = FINAL_OUTPUT;
				else oi->output_type = INTERNAL_OUTPUT;
			}
			oi->prefix = qual_pref;
			oi->suffix = final;
			ls->Append(oi)->setText(oi->output_name);
		}
		lo = lo->Next();
	}
}

// device import/export
// returns 0 on error
QString Device::device2string()
{	QString ret;

	ret.truncate(0);

	// all other devices
	ret.sprintf("%d%c%d%c%d%c%d%c%d%c%d%c%d%c%d%c%d%c%d%c",
		id, DLM,
		function, DLM,
		delay(), DLM,
		undef(), DLM,
		clock_type, DLM,
		master, DLM,
		osz_on_cnt, DLM,
		osz_off_cnt, DLM,
		named_output.Get()->getResult(), DLM,
		osz_curr_step, DLM);
	if (text) ret += text;

	return ret;
}

// returns 0 on error
int Device::string2device(QString s)
{	int apos, epos;
	QString original_text;
	QString alternate_text;

	// name
	apos = 1;
	epos = s.find(DLM, 0);
	if (epos == -1) return -1;
	alternate_text = s.mid(apos, epos - apos);

	// function
	apos = epos + 1;
	epos = s.find(DLM, apos);
	if (epos == -1) return -1;

	// delay
	apos = epos + 1;
	epos = s.find(DLM, apos);
	if (epos == -1) return -1;
	setDelay(s.mid(apos, epos - apos).toInt());

	// undefined value
	apos = epos + 1;
	epos = s.find(DLM, apos);
	if (epos == -1) return -1;
	setUndef(s.mid(apos, epos - apos).toInt());

	// clock type
	apos = epos + 1;
	epos = s.find(DLM, apos);
	if (epos == -1) return -1;
	setClock(s.mid(apos, epos - apos).toInt());

	// master ff flag
	apos = epos + 1;
	epos = s.find(DLM, apos);
	if (epos == -1) return -1;
	setMaster(s.mid(apos, epos - apos).toInt());

	// oscillator on time
	apos = epos + 1;
	epos = s.find(DLM, apos);
	if (epos == -1) return -1;
	setOszOn(s.mid(apos, epos - apos).toInt());

	// oscillator off time
	apos = epos + 1;
	epos = s.find(DLM, apos);
	if (epos == -1) return -1;
	setOszOff(s.mid(apos, epos - apos).toInt());

	// output value
	apos = epos + 1;
	epos = s.find(DLM, apos);
	if (epos == -1) return -1;
	flush(s.mid(apos, epos - apos).toInt());
	setStaticInput(s.mid(apos, epos - apos).toInt());

	// osz_curr_step
	apos = epos + 1;
	epos = s.find(DLM, apos);
	if (epos == -1) return -1;
	osz_curr_step = s.mid(apos, epos - apos).toInt();

	// text label
	apos = epos + 1;
	epos = s.find(DLM, apos);
	if (epos == -1) return -1;
	original_text = s.mid(apos, epos - apos);
	if (original_text.length() > 0) setText((char *)(const char *)original_text);
	else setText((char *)(const char *)alternate_text);

	return 0;
}

