/*************************************************/
/* member methods for class opStack              */
/*                                               */
/* equation analyzer and calculator              */
/*                                               */
/* Andreas Rostin                                */
/* 15.10.99                                      */
/*************************************************/
#include <opStack.h>
#include <qstring.h>
#include <device.h>
#include <math.h>
#include <array.h>

opStack::opStack(Device *_dev, int _undef, int _delay)
	:value()
{
	init();

	dev = _dev;
	undefined_result = _undef;
	result = _undef;
	delayed_result = _undef;
	delay = _delay;
	result_fifo = new Fifo(delay);
}

opStack::opStack()
	:value()
{
	init();
}

void opStack::init()
{
	// no of variables used for the calculation of the queue
	total = 0;
	// max number of items for queue, stack and stack_info
	maxitem = 0;
	// number of terminal on the stack
	plane = 0;

	dev = (Device *)NULL;

	equation = (char *)NULL;
	queue = (char **)NULL;
	queueid = (int *)NULL;
	queue_info = (StackInfo **)NULL;

	result_fifo = (Fifo *)NULL;
	stack = (int *)NULL;

	undefined_result = 0;
	result = 0;
	delayed_result = 0;
	delay = 0;
	staticOP = OP_NULL;

	tmp_equ = (char *)NULL;

	input_list = (char **)NULL;
	input_cnt = 0;

	pattern_idx = (int *)NULL;
	patterns = (int **)NULL;
	daa_patterns = (char *)NULL;
	pattern = (int *)NULL;
	pattern_cnt = 0;
	pattern_mode = 0;

	daa_group = (char *)NULL;
	group_size = (int *)NULL;
}

opStack::~opStack()
{
	clear();
	if (equation) free(equation);
	if (queue) free(queue);
	if (queueid) free(queueid);
	if (queue_info) {
		for(int i = 0; i < total; i++)
			delete queue_info[i];
		free(queue_info);
	}
	if (stack) free(stack);
	if (tmp_equ) free(tmp_equ);
	if (pattern_idx) free(pattern_idx);
	if (daa_patterns) free(daa_patterns);
	if (input_list) {
		int i;
		for (i = 0; i < input_cnt; i++)
			free(input_list[i]);
		free (input_list);
	}
	if (result_fifo) delete result_fifo;
	if (daa_group) free(daa_group);
	if (group_size) free(group_size);
}

// set a new equation
void opStack::setEquation(char _new_equation)
{
	clear();
	if (equation) free(equation);
	equation = (char *)malloc(sizeof(char) * 2);
	equation[0] = _new_equation;
	equation[1] = '\0';
}

// set a new equation
void opStack::setEquation(const char * _new_equation)
{
	setEquation((char *)_new_equation);
}

// set a new equation
void opStack::setEquation(char * _new_equation)
{
	clear();
	if (equation) free(equation);
	if (_new_equation && strlen(_new_equation))
		equation = strdup(_new_equation);
	else
		equation = (char *)NULL;

	// replace double-invertations by spaces
	if (equation) {
		char *buf;
		while (NULL != (buf = strstr(equation, "//"))) {
			*buf = ' ';
			*(buf + 1) = ' ';
		}
	}
}

// an empty equation has a content
int opStack::hasEquation()
{
	if (equation && equation[0] != OP_NONE && equation[0] != OP_NULL) return 1;
	return 0;
}

char * opStack::getEquation()
{
	return equation;
}

// build equation with all non-terminal qualified by a string
// attention: allocates uncontrolled memory
char * opStack::getEquation(const char *qualifier)
{	char *new_equation = (char *)malloc(sizeof(char) * 501);
	char *result;
	unsigned int current_size = 500;
	unsigned int cnt;
	unsigned int ncnt = 0;

	if (!qualifier || !strlen(qualifier)) {
		free(new_equation);
		new_equation = strdup(equation);
		return new_equation;
	}

	//dirty loops!
	for(cnt = 0; cnt < strlen(equation); cnt++) {
		// enough memory?
		while (ncnt + strlen(qualifier) + 100 > current_size) {
			result = (char *)malloc(sizeof(char) * current_size * 2 + 1);
			current_size = current_size * 2;
			new_equation[ncnt] = 0;
			strcpy(result, new_equation);
			free(new_equation);
			new_equation = result;
		}
		// copy string byte by byte
		if (equation[cnt] != ' ' && equation[cnt] != '(' && equation[cnt] != ')' && equation[cnt] != OP_AND &&
		    equation[cnt] != OP_OR && equation[cnt] != OP_XOR && equation[cnt] != OP_NOT) {
			new_equation[ncnt] = 0;
			strcat(new_equation, qualifier);
			// do not copy "_this_"
			result = strstr(equation, "_this_");
			if (result && ((result == equation + cnt) || (result == equation + cnt + 1))) {
				cnt += (result - equation) + 6;
				ncnt += strlen(qualifier);
			} else {
				//strcat(new_equation, ".");
				ncnt += strlen(qualifier);// + 1;
			}
			while (cnt < strlen(equation) && equation[cnt] != ' ' && equation[cnt] != '(' && equation[cnt] != ')' &&
			       equation[cnt] != OP_AND && equation[cnt] != OP_OR && equation[cnt] != OP_XOR && equation[cnt] != OP_NOT) {
				new_equation[ncnt++] = equation[cnt++];
			}
			cnt--;
		} else new_equation[ncnt++] = equation[cnt];
	}
	new_equation[ncnt] = 0;
	if (tmp_equ) free(tmp_equ);
	tmp_equ = new_equation;
	return new_equation;
}

void opStack::setUndefined(int _undef)
{
	undefined_result = _undef;
}

int opStack::getUndefined()
{
	return undefined_result;
}

int opStack::getDelay()
{
	return delay;
}

void opStack::setDelay(int _delay)
{
	delay = _delay;
	result_fifo->setDepth(_delay);
}

void opStack::flush(int value)
{
	old_result = result;
	result_fifo->flush(value);
	result = value;
	delayed_result = value;
}

int opStack::getResult()
{
	if (!delay) return result;
	return delayed_result;
}

int opStack::shift()
{	int ret;

	if (!delay) return old_result;
	ret = result_fifo->put(result);
	delayed_result = result_fifo->get();
	return ret;
}

char opStack::hasStaticOP()
{
	if (staticOP != OP_NULL) return staticOP;
	return 0;
}

// debugging method!
void opStack::stackDump()
{
	printf("\n**********\n");
	printf("** Equation: %s\n", equation);
	printf("** Maxitem: %d\n", maxitem);
	for (int i = 0; i < total;i++)
		printf("<%s> at offset %d, length %d\n", queue[i], queue_info[i]->position, queue_info[i]->length);
	printf("**********\n");
}

// reset opStack to init state
void opStack::clear()
{
	if (queue) {
		for (int i = 0; i < maxitem; i++) free(queue[i]);
		free(queue);
		queue = (char **)NULL;
	}

	if (queueid) free(queueid);
	queueid = (int *)NULL;

	// modified 12/08/2000 start
	if (queue_info) {
		for(int i = 0; i < total; i++)
			delete queue_info[i];
		free(queue_info);
	}
	// modified 12/08/2000 end
	queue_info = (StackInfo **)NULL;

	if (stack) free(stack);
	stack = (int *)NULL;

	maxitem = 0;
	total = 0;
	staticOP = 0;
}

// check stack if it is computational
// returns an offset to the suspicious nonterminal/operation contained in equation
// returns 0 on success (!)
// allocate memory for stack and queue, copy values
StackInfo opStack::check(list<value> *input, list<opStack> *output)
{	int cnt = 0;
	int i;
	StackInfo ret;
	list<value> *li;
	list<opStack> *lo;

	if (total == 0) return ret;

	// the number of nonterminal and operations must agree (make a test computation!)
	for (i = 0; i < total; i++) {
		switch (queue[i][0]) {
			case OP_AND:
			case OP_OR:
			case OP_XOR:
				if (cnt < 2) {
					queue_info[i]->no = STACK_NEG_ERROR;
					return *queue_info[i];
				}
				cnt--;
				break;
			case OP_NOT:
				if (cnt < 1) {
					queue_info[i]->no = STACK_NEG_ERROR;
					return *queue_info[i];
				}
				break;
			default:
				// check if nonterminal exists as an named input or output
				if (!strncmp(queue[i], "_this_", 6)) {
					if (queue_info[i]->output_name) free(queue_info[i]->output_name);
					queue_info[i]->output_name = strdup(dev->getText());
					queue_info[i]->output_id = 0;
					queue_info[i]->type = THIS;	// special operand: current output
					queueid[i] = dev->getID();
				} else if (NULL != (li = input->With(queue[i]))) {
					queue_info[i]->output_name = input->With(queue[i])->getText();
					queue_info[i]->output_id = li->getID1();
					queue_info[i]->type = Device::INPUT;
					queueid[i] = li->getID1();	// the id of the named input of the device
				} else if (NULL != (lo = output->With(queue[i]))) {
					queue_info[i]->output_name = output->With(queue[i])->getText();
					queue_info[i]->output_id = lo->getID1();
					queue_info[i]->type = Device::FINAL_OUTPUT;
					queueid[i] = lo->getID1();
				} else if (!strncmp(queue[i], "0", 1)) {
					queue_info[i]->type = CONST_0;
					queueid[i] = 0;
				} else if (!strncmp(queue[i], "1", 1)) {
					queue_info[i]->type = CONST_1;
					queueid[i] = 0;
				} else {
					queue_info[i]->no = NTERM_ERROR;
					return *queue_info[i];
				}
				cnt++;
				break;
		}
	}
	if (cnt != 1) {
		ret.no = STACK_POS_ERROR;
	}
	return ret; 
}

StackInfo opStack::parse()
{
	return parse((list<value> *)NULL, (list<opStack> *)NULL);
}

// public parse method
// transform equation into reverse polish notation and create an operation stack
StackInfo opStack::parse(list<value> *input, list<opStack> *output)
{	StackInfo rets;
	int ret;
	int i;

	if (!equation) return rets;
	if (strlen(equation) == 0) return rets;

	// set unique operation on all input values?
	if (strlen(equation) == 1 &&
	    (equation[0] == OP_AND ||
             equation[0] == OP_OR || 
	     equation[0] == OP_XOR ||
             equation[0] == OP_NONE ||
             equation[0] == OP_NOT ||
	     equation[0] == OP_NULL)) {
		if (stack) free(stack);
		stack = (int *)malloc(sizeof(int) * OP_MAXINP);
		staticOP = equation[0];
		total = 0;
		plane = 0;
		return rets;
	}

	// *** count items, allocate needed memory ***
	clear();

	parse(equation, 0, 1);
	stack = (int *)malloc(sizeof(int) * total);
	queue = (char **)malloc(sizeof(char *) * total);
	queueid = (int *)malloc(sizeof(int) * total);
	queue_info = (StackInfo **)malloc(sizeof(StackInfo *) * total);
	for (i = 0; i < total; i++)
		queue_info[i] = new StackInfo;
	maxitem = total;
	total = 0;

	// *** parse the equation, fillup queue ***
	ret = parse(equation, 0);

	// *** check the stack ***
	if (ret == 0) {
		// iput == output == NULL: do not check equation
		if (input || output) {
			rets = check(input, output);
		}
	} else {
		rets.no = PAR_ERROR;
		rets.position = ret;
		rets.length = 1;
	}

	return rets;
}

// parse the equation and resolve it
// push it to the computation stack in RPN
// if docnt is set, just count the items
int opStack::parse(char *_equation, int current_offset, int docnt = 0)
{	char *bracket_start;	// parenthesis
	int bracket_length;	// len of parenthesis
	char *op;		// operation
	char *left;		// left operand
	int left_length;	// length of left operand
	char *right;		// right operand
	int right_length;	// length of right operand
	int new_offset;		// new current offset relative to the original equation
	int optype;		// type of operator
	int res;

	char *eq = strdup(_equation);

	// find the next operator/bracket
	while (NULL != (op = getOP(eq))) {
		optype = opType(*op);

		// parenthesis found
		if (optype == OP_PARENTHESIS) {
			bracket_start = op;
			bracket_length = getBracket(op);
			if (bracket_length < 0) return bracket_start - eq + 1 + current_offset;
			*(bracket_start + bracket_length) = '\0';

			// if parenthesis contains no operands, drop it
			if (!strchr(bracket_start + 1, OP_AND) &&
			    !strchr(bracket_start + 1, OP_OR) &&
			    !strchr(bracket_start + 1, OP_XOR) &&
			    !strchr(bracket_start + 1, OP_NOT)) {
				*bracket_start = ' ';
				*(bracket_start + bracket_length) = ' ';
				continue;
			}

			new_offset = current_offset + (bracket_start  + 1 - eq);
			res = parse(bracket_start + 1, new_offset, docnt);      // recursive call
			if (res != 0) return res;
			// replace the complete bracket with spaces
			memset(bracket_start, ' ', bracket_length + 1);
			continue;
		}

		// determine operands and their length
		if (optype == OP_LR) {
			left = op - 1;
			left_length = getLeft(eq, &left);
		} else left_length = 0;
		right = op + 1;
		right_length = getRight(eq + strlen(eq), &right);

		// ****************************
		// docnt: just count the items!
		// ****************************

		// push operands to the calculation stack
		if (right_length) {
			if (!docnt) push(right, right_length, current_offset + (right - eq));
			else total++;
		}
		if (left_length) {
			if (!docnt) push(left, left_length, current_offset + (left - eq));
			else total++;
		}

		// push operation to the calculation stack
		if (!docnt) push(op, 1,  current_offset + (op - eq));
		else total++;

		// replace the operation and its operands with spaces
		*op = ' ';
		if (right_length)
			memset(right, ' ', right_length);
		if (left_length)
			memset(left, ' ', left_length);
	}
	// find operand without operator
	right = eq;
	right_length = getRight(eq + strlen(eq), &right);
	if (right_length) {
		if (!docnt) push(right, right_length, current_offset + (right - eq));
		else total++;
	}
	free(eq);
	return 0;	// OK
}

// returns length of the bracket
inline
int opStack::getBracket(char *__equation)
{	char *start = __equation;
	char *end = start;

	while (NULL != (end = strchr(end + 1, ')'))) {
		start = strchr(start + 1, '(');
		if (!start || start > end) {
			return end - __equation;
		}
	}
	return -1;
}

// 0: string contains forbidden characters/strings
// 1: string is OK
int opStack::checkName(char *name)
{
	if (getOP(name)) return 0;
	return 1;
}

// return a pointer to the next operation
char * opStack::getOP(char *_equation)
{	char *pt0, *pt1, *pt2, *pt3, *pt4;
	char *bet;

	pt0 = strchr(_equation, OP_BRACKET);
	pt1 = strchr(_equation, OP_NOT);
	pt2 = strchr(_equation, OP_XOR);
	pt3 = strchr(_equation, OP_AND);
	pt4 = strchr(_equation, OP_OR);

	// find two operators to the left for which is valid: op1 left of op2 left of op0, then return op1
	// (numbers: priority!)
	if (pt0) {
		if (pt1 && pt1 < pt0) {
			bet = strchr(pt1, OP_XOR);
			if (pt1 < bet && bet < pt0) return pt1;
			bet = strchr(pt1, OP_AND);
			if (pt1 < bet && bet < pt0) return pt1;
			bet = strchr(pt1, OP_OR);
			if (pt1 < bet && bet < pt0) return pt1;
		}
		if (pt2 && pt2 < pt0) {
			bet = strchr(pt2, OP_AND);
			if (pt2 < bet && bet < pt0) return pt2;
			bet = strchr(pt2, OP_OR);
			if (pt2 < bet && bet < pt0) return pt2;
		}
		if (pt3 && pt3 < pt0) {
			bet = strchr(pt3, OP_OR);
			if (pt3 < bet && bet < pt0) return pt3;
		}
		return pt0;
	}

	if (pt1) {
		if (pt2 && pt2 < pt1) {
			bet = strchr(pt2, OP_AND);
			if (pt2 < bet && bet < pt1) return pt2;
			bet = strchr(pt2, OP_OR);
			if (pt2 < bet && bet < pt1) return pt2;
		}
		if (pt3 && pt3 < pt1) {
			bet = strchr(pt3, OP_OR);
			if (pt3 < bet && bet < pt1) return pt3;
		}
		return pt1;
	}

	if (pt2) {
		if (pt3 && pt3 < pt2) {
			bet = strchr(pt3, OP_OR);
			if (pt3 < bet && bet < pt1) return pt3;
		}
		return pt2;
	}
	if (pt3) return pt3;
	return pt4;
}

// return type of operation:
//   OP_LR operator needs left and right operand
//   OP_R operator needs right operand
int opStack::opType(char op)
{
	switch(op) {
		case OP_AND:
		case OP_OR:
		case OP_XOR:
			return OP_LR;
			break;
		case OP_NOT:
			return OP_R;
			break;
	}
	return 0;
}

// sets pointer to the start of the left operand nonterminal
// returns length of the nonterminal
inline
int opStack::getLeft(char *start, char **left)
{	int length = 0;

	// jump over spaces
	while (**left == ' ' && *left > start)
		(*left)--;

	// reverse find next space, operand, ( or )
	while (	*left >= start &&
	  **left != '(' &&
	  **left != ')' &&
	  **left != ' ' &&
	  **left != OP_AND &&
	  **left != OP_OR &&
	  **left != OP_XOR &&
	  **left != OP_NOT &&
	  **left != 0) {
		(*left)--;
		length++;
	}
	if (length) (*left)++;

	return length;
}

// sets pointer to the start of the right operand nonterminal
// returns length of the nonterminal
inline
int opStack::getRight(char *end, char **right)
{	int length = 0;
	char *pt;

	// jump over spaces
	while (**right == ' ' && *right < end)
		(*right)++;

	pt = *right;
	// find next space, operand, ( or )
	while (	pt <= end &&
	  *pt != '(' &&
	  *pt != ')' &&
	  *pt != ' ' &&
	  *pt != OP_AND &&
	  *pt != OP_OR &&
	  *pt != OP_XOR &&
	  *pt != OP_NOT &&
	  *pt != 0) {
		pt++;
		length++;
	}

	return length;
}

// push something into the queue
inline
void opStack::push(char *str, int length, int current_offset)
{
	// push string to the queue
	if (length) {
		if (total > maxitem) {
			fprintf(stderr, "too many items on queue??\n");
			exit(-1);
		}

		queue[total] = (char *)malloc(length * sizeof(char) + 1);
		strncpy(queue[total], str, length);
		*(queue[total] + length) = '\0';

		queue_info[total]->position = current_offset;
		queue_info[total]->length = length;

		total++;
	}
}

// push an input value to the stack
void opStack::push(int val)
{
	if (staticOP == OP_NULL) return;

	if (!staticOP) {
		fprintf(stderr, "pushing values to the stack??\n");
		exit(-1);
	}
	if (plane >= OP_MAXINP) {
		fprintf(stderr, "too many inputs (100)\n");
		exit(-1);
	}
	stack[plane++] = val;
	total++;
}

// stack calculation
// returns the result
int opStack::calculate()
{	int i;
	char currOP = 0;

	old_result = result;
	result = undefined_result;
	if (!total) return undefined_result;		// nothing to do

	if (!staticOP) plane = -1;
	else {
		if (staticOP == OP_NULL) return undefined_result;
		currOP = staticOP;
		if (currOP == OP_AND || currOP == OP_OR || currOP == OP_XOR) {
			total--;
		}
		plane--;
	}

	for (i = 0; i < total; i++) {
		if (!staticOP) currOP = queue[i][0];

		switch (currOP) {
			case OP_AND:
				stack[plane - 1] = stack[plane - 1] & stack[plane];
				plane--;
				break;
			case OP_OR:
				stack[plane - 1] = stack[plane - 1] | stack[plane];
				plane--;
				break;
			case OP_XOR:
				if (stack[plane - 1] + stack[plane] == 1) stack[plane - 1] = 1;
				else stack[plane - 1] = 0;
				plane--;
				break;
			case OP_NOT:
				if (stack[plane]) stack[plane] = 0;
				else stack[plane] = 1;
				break;
			case OP_NONE:
				break;
			case OP_NULL:
				break;
			default:
				plane++;
				// external method access: always !pattern_mode
				if (pattern_mode) {
					stack[plane] = pattern[pattern_idx[i]];
				} else {
					if (plane > maxitem) {
						fprintf(stderr, "opStack: stack overflow!\n");
						exit(-1);
					}
					if (queue_info[i]->type == THIS)
						stack[plane] = dev->output();
					else if (queue_info[i]->type == Device::FINAL_OUTPUT)
						stack[plane] = dev->output(queueid[i]);
					else if (queue_info[i]->type == Device::INPUT)
						stack[plane] = dev->input(queueid[i]);
					else if (queue_info[i]->type == CONST_0)
						stack[plane] = 0;
					else if (queue_info[i]->type == CONST_1)
						stack[plane] = 1;
				}
				break;
		}
	}
	if (plane != 0) {
		plane = 0;
		if (currOP) total = 0;
		return undefined_result;
	}
	plane = 0;
	if (staticOP) total = 0;
	result = stack[0];
	return stack[0];
}

// symbolic stack calculation
// returns the equation
char * opStack::calculateSymbolic()
{	int i;
	char **cstack = (char **)malloc(sizeof(char *) * maxitem);
	char *item;
	int lpar, rpar;

	if (!total) {
		free(cstack);
		return (char *)NULL;	// nothing to do
	}

	if (staticOP) {
		fprintf(stderr, "unable to calculate static operations symbolic\n");
		free(cstack);
		return (char *)NULL;	// nothing to do
	}

	plane = -1;
	for (i = 0; i < total; i++) {
		switch (queue[i][0]) {
			case OP_AND:
				lpar = needsParenthesis(cstack[plane - 1], OP_AND);
				rpar = needsParenthesis(cstack[plane], OP_AND);
				item = (char *)malloc(sizeof(char) * (strlen(cstack[plane - 1]) + strlen(cstack[plane]) + lpar + rpar + 4));
				if (!lpar && !rpar) sprintf(item, "%s %c %s", cstack[plane - 1], OP_AND, cstack[plane]);
				if (lpar && !rpar) sprintf(item, "(%s) %c %s", cstack[plane - 1], OP_AND, cstack[plane]);
				if (!lpar && rpar) sprintf(item, "%s %c (%s)", cstack[plane - 1], OP_AND, cstack[plane]);
				if (lpar && rpar) sprintf(item, "(%s) %c (%s)", cstack[plane - 1], OP_AND, cstack[plane]);
				free(cstack[plane - 1]);
				free(cstack[plane]);
				cstack[plane - 1] = item;
				plane--;
				break;
			case OP_OR:
				lpar = needsParenthesis(cstack[plane - 1], OP_OR);
				rpar = needsParenthesis(cstack[plane], OP_OR);
				item = (char *)malloc(sizeof(char) * (strlen(cstack[plane - 1]) + strlen(cstack[plane]) + lpar + rpar + 4));
				if (!lpar && !rpar) sprintf(item, "%s %c %s", cstack[plane - 1], OP_OR, cstack[plane]);
				if (lpar && !rpar) sprintf(item, "(%s) %c %s", cstack[plane - 1], OP_OR, cstack[plane]);
				if (!lpar && rpar) sprintf(item, "%s %c (%s)", cstack[plane - 1], OP_OR, cstack[plane]);
				if (lpar && rpar) sprintf(item, "(%s) %c (%s)", cstack[plane - 1], OP_OR, cstack[plane]);
				free(cstack[plane - 1]);
				free(cstack[plane]);
				cstack[plane - 1] = item;
				plane--;
				break;
			case OP_XOR:
				lpar = needsParenthesis(cstack[plane - 1], OP_XOR);
				rpar = needsParenthesis(cstack[plane], OP_XOR);
				item = (char *)malloc(sizeof(char) * (strlen(cstack[plane - 1]) + strlen(cstack[plane]) + lpar + rpar + 4));
				if (!lpar && !rpar) sprintf(item, "%s %c %s", cstack[plane - 1], OP_XOR, cstack[plane]);
				if (lpar && !rpar) sprintf(item, "(%s) %c %s", cstack[plane - 1], OP_XOR, cstack[plane]);
				if (!lpar && rpar) sprintf(item, "%s %c (%s)", cstack[plane - 1], OP_XOR, cstack[plane]);
				if (lpar && rpar) sprintf(item, "(%s) %c (%s)", cstack[plane - 1], OP_XOR, cstack[plane]);
				free(cstack[plane - 1]);
				free(cstack[plane]);
				cstack[plane - 1] = item;
				plane--;
				break;
			case OP_NOT:
				rpar = needsParenthesis(cstack[plane], OP_NOT);
				item = (char *)malloc(sizeof(char) * (strlen(cstack[plane]) + rpar + 2));
				if (!rpar) sprintf(item, "%c%s", OP_NOT, cstack[plane]);
				if (rpar) sprintf(item, "%c(%s)", OP_NOT, cstack[plane]);
				free(cstack[plane]);
				cstack[plane] = item;
				break;
			case OP_NONE:
				break;
			case OP_NULL:
				break;
			default:
				plane++;
				if (plane > maxitem) {
					fprintf(stderr, "opstack: cstack overflow??\n");
					exit(-1);
				}
				cstack[plane] = strdup(queue[i]);
				break;
		}
	}
	if (plane != 0) {
		fprintf(stderr, "cstack error during symbolic calculation\n");
		exit(-1);
	}
	plane = -1;
	if (staticOP) total = 0;
	if (tmp_equ) free(tmp_equ);
	tmp_equ = cstack[0];
	free(cstack);
	return tmp_equ;
}

// return a minimized and normalized equation
// derived from the "karnough" graphical analyzing method, but up to 15 input variables
// algorithm:
// (1) find as big as possible rectangular pattern groups in the equation, where each neighbor pattern toggles only one bit
// (2) remove patterns contained in other patterns
// (3) create a minimized subjunctive/disjunctive normalized equation from the pattern-groups found
//  - advantage: againgst graphical method: it is not restricted to a maximum of 6 input variables
//  - disadvantages: needs some MB of memory resources and calculation power for really big equations!
//                 : the resulting minimized and normalized equation must not be shorter than the original one! so check the result!
//
// Arguments:
// (1) 1 : returns a subjunctive normalized equation
//     0 : returns a disjunctive normalized equation
// (2) 1 : remove not as many pattern groups as 0
char * opStack::calculateSymbolicNormalized(int subjunctive, int secure)
{
	int **table;				// an array of patterns with some exchanged columns
	char *daa_table;			// the pointer to use with free()
	unsigned daa_table_dimensions[2];	// the dimensions of the table array
	unsigned daa_ret;			// not really used .. the return value of the array allocation function
	int start_arry[2] = {0,0};		// initialisation values of the arrays

	int i, j, k, found, all_found, single_found, bit_found;
	int ii, ij, ik, jj, kk;
	int bit;

	if (!total) {
		return (char *)NULL;	// nothing to do
	}

	if (staticOP) {
		fprintf(stderr, "unable to analyze static operations\n");
		return (char *)NULL;	// nothing to do
	}

	// create all result patterns possible
	// daa_patterns		the patterns
	// pattern_cnt		the number of patterns stored
	if (!patterns) calculatePattern();

	// ****** find groups greater than 1 pattern (remember the patterns of the group as integer value) ******
	// group: a set of patterns with an equal output and not toggling more than one bit to the neighbors patterns
	// group_size: the number of patterns of a specific groups
	// group_cnt: the total number of groups
	//
	// further expanations (for a group-size of 2):
	//     Two patterns where the first pattern has the same result as the following pattern builds a group of two patterns
	//     if the group does not lies inside a binary jump of bits greater than or equals the group-size.
	//     So a valid group for a group size=2 and 3 input bits is (000 == 1 --> 001 == 1) or (100 == 1 --> 110 == 1) or ...,
	//     and an impossibe group is (011 == 1 --> 100 == 1) or (000 == 0 --> 001 = 1) ...

	int max_groups = pattern_cnt * input_cnt;
	int max_patterns = pattern_cnt;

	// allocate arrays on the heap (up to xxxx MBytes)
	daa_group_dimensions[0] = max_groups;
	daa_group_dimensions[1] = max_patterns;
	if (daa_group) free(daa_group);
	group = (int **)daa(sizeof(int), 2, daa_group_dimensions, start_arry, &daa_ret, &daa_group, (char *)NULL);	// groups to be found in patterns

	if (group_size) free(group_size);
	group_size = (int *)malloc(sizeof(int) * max_groups);	// no of patterns in the groups
	group_cnt = 0;						// current group_cnt / total number of groups

	daa_table_dimensions[0] = max_patterns;
	daa_table_dimensions[1] = 2;
	table = (int **)daa(sizeof(int), 2, daa_table_dimensions, start_arry, &daa_ret, &daa_table, (char *)NULL);	// pattern with exchanged columns [][value,result]

	int current_group_size;			// group field size (1, 2, 4, ..)
	int group_start;			// group start at table position
	int curr_pos;				// current group position while scanning

	int bin_a, bin_na, bin_b, bin_nb;	// column exchange
	int oldval, newval;			// pattern value before and after exchange

	for(i=0; i < input_cnt; i++) {
		for (j = i; j < input_cnt; j++) {
			// create tables with exchanged columns: exchange column i with column j
			bin_a = pow2(i);
			bin_na = pow2complement(input_cnt - 1, i);
			bin_b = pow2(j);
			bin_nb = pow2complement(input_cnt - 1, j);
			for (oldval = 0; oldval < pattern_cnt; oldval++) {
				newval = oldval;
				if (i != j) {
					if (oldval & bin_a) newval |= bin_b;	// set bit
					else newval &= bin_nb;			// clear bit
					if (oldval & bin_b) newval |= bin_a;
					else newval &= bin_na;
				}

				table[newval][0] = oldval;			// remember the original position in patterns
				table[newval][1] = patterns[oldval][input_cnt];	// the result value
			}

			// try to find groups in the new table (leaving single groups)
			for (k = input_cnt - 1; k > 0; k--) {
				current_group_size = pow2(k);				// beginning with the biggest possible current_group_size
				group_start = 0;
				while (group_start < pattern_cnt) {
					curr_pos = group_start;
					group_size[group_cnt] = 0;
					group[group_cnt][0] = 0;
					found = 1;

					// try find a new group
					while (found && curr_pos < pattern_cnt && group_size[group_cnt] < current_group_size) {
						if (table[curr_pos][1] == subjunctive) {
							group[group_cnt][group_size[group_cnt]++] = table[curr_pos][0];
							curr_pos++;
						}
						else found = 0;
					}

					// validate new group
					if (group_size[group_cnt] < current_group_size) {
						group_size[group_cnt] = 0;		// not enough patterns
					} else {
						// look if group already exists completely in another group
						// the currently found group has a smaller or equal size than all others!
						found = 0;
						ii = 0;
						while (!found && ii < group_cnt) {				// for all old groups..
							all_found = 1;
							ij = 0;
							while (all_found && ij < group_size[group_cnt]) {		// the new group
								single_found = 0;
								ik = 0;
								while (!single_found && ik < group_size[ii]) {	// the old group
									if (group[group_cnt][ij] == group[ii][ik])
										single_found = 1;
									ik++;
								}
								if (!single_found) all_found = 0;
								ij++;
							}
							if (all_found) found = 1;
							ii++;
						}
						if (found) group_size[group_cnt] = 0;
						else group_cnt++;
					}

					// continue search on the next possible starting point
					group_start += current_group_size;

				} // increment group_start
			} // decrement group size
		} // exchange table column
	} // exchange table column

	// ****** remove groups contained in other groups ******
    	if (secure) {
    		// secure method: remove only groups contained in one other group (gliches are surely excluded)
    		i = 0;
    		while (i < group_cnt) {
    			// look if group already exists completely in another group
    			// the currently found group has a smaller or equal size than all others!
    			found = 0;
    			j = 0;
    			while (!found && j < group_cnt) {				// for all other groups..
    				if (i == j || group_size[i] > group_size[j]) all_found = 0;
    				else all_found = 1;
    				ii = 0;
    				while (all_found && ii < group_size[i]) {		// the current group
    					single_found = 0;
    					jj = 0;
    					while (!single_found && jj < group_size[j]) {	// the other group
    						if (group[i][ii] == group[j][jj])
    							single_found = 1;
    						jj++;
    					}
    					if (!single_found) all_found = 0;
    					ii++;
    				}
    				if (all_found) found = 1;
    				j++;
    			}
    			if (found) {
    				for (j = i + 1; j < group_cnt; j++) {
					for (jj=0; jj < group_size[j]; jj++)
						group[j - 1][jj] = group[j][jj];
    					group_size[j - 1] = group_size[j];
    				}
    				group_cnt--;
    			} else i++;
    		}
    	} else {
		// insecure method: could cause glitches in cause of different delays of the gates
		// remove all groups contained in n other groups
		// beginning with the smallest groups
		int all_grp[pattern_cnt * group_cnt];
		int all_grp_cnt = 0;
		for (i = 1; i < input_cnt; i++) {
			current_group_size = pow2(i);
			j = 0;
			while (j < group_cnt) {
				if (group_size[j] == current_group_size) {
					// create list with all available values
					all_grp_cnt = 0;
					for (k = 0; k < group_cnt; k++) {
						if (k != j) {
							for (kk=0; kk < group_size[k]; kk ++) {
								all_grp[all_grp_cnt++] = group[k][kk];
							}
						}
					}
					// look if all values of the current group are contained in all_grp
					all_found = 1;
					jj = 0;
					while (all_found && jj < current_group_size) {
						single_found = 0;
						k = 0;
						while (!single_found && k < all_grp_cnt) {
							if (all_grp[k] == group[j][jj]) single_found = 1;
							k++;
						}
						if (!single_found) all_found = 0;
						jj++;
					}
					// remove group if all pattern were found
					if (all_found) {
						for (k = j + 1; k < group_cnt; k++) {
							for (kk=0; kk < group_size[k]; kk++)
								group[k - 1][kk] = group[k][kk];
							group_size[k - 1] = group_size[k];
						}
						group_cnt--;
					} else j++;
				} else j++;
			}
		}
	}

	// ****** at least add single groups, hopefully some lost bits ******
	for (i = 0; i < pattern_cnt; i++) {
		if (patterns[i][input_cnt] == subjunctive) {
			found = 0;
			j = 0;
			while (!found && j < group_cnt) {
				jj = 0;
				while (!found && jj < group_size[j]) {
					if (i == group[j][jj]) found = 1;
					jj++;
				}
				j++;
			}
		} else found = 1;
		if (!found) {
			group[group_cnt][0] = i;
			group_size[group_cnt] = 1;
			group_cnt++;
		}
	}

	// ****** create an equation from the groups found ******
	char newpart[1024];
	char *result;
	unsigned int current_size = 16384;
	if (tmp_equ) free(tmp_equ);
	tmp_equ = (char *)malloc(sizeof(char) * current_size + 1);
	tmp_equ[0] = 0;

	int bit_value;
	int first_part = 1;
	for (i = 0; i < group_cnt; i++) {
		// first group: open parenthesis
		if (i == 0 && !subjunctive) strcat(tmp_equ, "(");

		// in between groups
		if (i != 0) {
			if (subjunctive) strcat(tmp_equ, " + ");
			else strcat(tmp_equ, ") & (");
		}

		first_part = 1;
		// figure out the bits needed for the equation
		for (bit = 0; bit < input_cnt; bit++) {
			// scan the whole group if the bit toggles
			// if it does not toggle, the bit is needed, otherwise not
			bit_found = 1;
			bit_value = patterns[group[i][0]][bit];		// the value of the current bit of the first pattern
			ii = 0;
			while (bit_found && group_size[i] > 1 && ii < group_size[i]) {
				if (bit_value != patterns[group[i][ii]][bit]) bit_found = 0;
				ii++;
			}
			if (bit_found) {
				// create the equation part related to the current bit
				if (first_part) {
					if (subjunctive) {
						if (bit_value) sprintf(newpart, "%s", input_list[bit]);
						else sprintf(newpart, "/%s", input_list[bit]);
					} else {
						if (bit_value) sprintf(newpart, "/%s", input_list[bit]);
						else sprintf(newpart, "%s", input_list[bit]);
					}
					first_part = 0;
				} else {
					if (subjunctive) {
						if (bit_value) sprintf(newpart, " & %s", input_list[bit]);
						else sprintf(newpart, " & /%s", input_list[bit]);
					} else {
						if (bit_value) sprintf(newpart, " + /%s", input_list[bit]);
						else sprintf(newpart, " + %s", input_list[bit]);
					}
					
				}

				// get enough memory
				if (current_size < (strlen(newpart) + strlen(tmp_equ) + 1024)) {
					current_size += 16384;
					result = (char *)malloc(sizeof(char) * current_size + 1);
					strcpy(result, tmp_equ);
					free(tmp_equ);
					tmp_equ = result;
				}

				// add new part
				strcat(tmp_equ, newpart);
			} // bit found
		} // next bit

		// last group: close parenthesis
		if (i == group_cnt - 1 && !subjunctive) strcat(tmp_equ, ")");

	} // next group

	free(daa_table);
	return tmp_equ;

	// ******** debugging part ************

	// print original table
	printf("\n******************************\n");
	printf("\noriginal: %s\n", equation);
	printf("******************************\n");
	for (i=input_cnt - 1; i >=0;i--) {
		if (i < input_cnt - 1) printf(" ");
		printf("%s", input_list[i]);
		if (i == 0) printf("\n");
	}
	for (i=0; i < pattern_cnt;i++) {
		for (j=input_cnt - 1; j >= 0;j--)
			printf("%d", patterns[i][j]);
		printf (" %d\n", patterns[i][input_cnt]);
	}

	// print found groups
	printf("%d groups found:\n", group_cnt);
	for (i = 0; i < group_cnt; i++) {
		printf("group contains %d pattern\n", group_size[i]);
		for (j=0; j < group_size[i]; j++) {
			for (bit = input_cnt - 1; bit >= 0; bit--) {
				if (group[i][j] & pow2(bit)) printf("1");
				else printf("0");
			}
			printf("\n");
		}
		printf("\n");
	}

	// print new table
	printf("******************************\n");
	printf("new: %s\n", tmp_equ);
	printf("******************************\n");
	setEquation(tmp_equ);
	parse();
	calculatePattern();

	for (i=input_cnt - 1; i >=0;i--) {
		if (i < input_cnt - 1) printf(" ");
		printf("%s", input_list[i]);
		if (i == 0) printf("\n");
	}
	for (i=0; i < pattern_cnt;i++) {
		for (j=input_cnt - 1; j >= 0;j--)
			printf("%d", patterns[i][j]);
		printf (" %d\n", patterns[i][input_cnt]);
	}

	free(daa_table);
	free(daa_group);
	free(group_size);
	return tmp_equ;
}

// return the first/next symbol of the queue
const char * opStack::getSymbol(int no)
{
	if (!no) scanInputList();

	if (!input_list || !input_cnt) return (const char *)NULL;
	if (no >= 0 && no < input_cnt) return input_list[no];

	return (const char *)NULL;
}

// allocate a list with all input strings
inline void opStack::scanInputList()
{	int i, j, found, plane;

	if (input_list) {
		for (i = 0; i < input_cnt; i++)
			free(input_list[i]);
		free (input_list);
	}

	// create unique list with all inputs
	input_list = (char **)malloc(sizeof(char *) * maxitem);
	pattern_idx = (int *)malloc(sizeof(int) * total);
	plane = -1;
	input_cnt = 0;
	for (i = 0; i < total; i++) {
		switch (queue[i][0]) {
			case OP_AND:
			case OP_OR:
			case OP_XOR:
			case OP_NOT:
			case OP_NONE:
			case OP_NULL:
				break;
			default:
				j = 0;
				found = 0;
				plane++;
				while (j < input_cnt && !found) {
					if (!strcmp(input_list[j], queue[i])) {
						found = 1;
						pattern_idx[i] = j;
					}
					j++;
				}
				if (!found) {
					if (input_cnt > maxitem) {
						fprintf(stderr, "opstack: analyze: input overflow??\n");
						exit(-1);
					}
					input_list[input_cnt] = strdup(queue[i]);
					pattern_idx[i] = input_cnt;
					input_cnt++;
				}
				break;
		}
	}
}

// stack calculation for the whole number of possible patterns
inline void opStack::calculatePattern()
{	int i, j;
	unsigned daa_ret;
	int start_arry[2] = {0,0};

	if (daa_patterns) free(daa_patterns);
	daa_patterns = (char *)NULL;
	if (pattern_idx) free(pattern_idx);
	pattern_idx = (int *)NULL;
	pattern = (int *)NULL;

	scanInputList();
	if (!input_list || !input_cnt) return;

	// ****** number of possible pattern ******
	pattern_cnt = pow2(input_cnt);
	if (!pattern_cnt) return;

	daa_patterns_dimensions[0] = pattern_cnt;
	daa_patterns_dimensions[1] = input_cnt + 1;
	patterns = (int **)daa(sizeof(int), 2, daa_patterns_dimensions, start_arry, &daa_ret, &daa_patterns, (char *)NULL);

	// ****** calculate patterns ******
	pattern_mode = 1;
	for (i = 0; i < pattern_cnt; i++) {
		// set input
		pattern = patterns[i];
		for (j = 0; j < input_cnt; j++) {
			if (i & pow2(j)) pattern[j] = 1;
			else pattern[j] = 0;
		}
		// calculate output
		pattern[input_cnt] = calculate();
	}
	pattern_mode = 0;
}

// returns a set of all karnaugh groups found in the patterns of the current equation
//    the list of groups in the "_group_list" argument
//    the size of the groups in the "_group_size" argument
//    the number of groups in the "_group_cnt" argument
// the pointers may become invalid on further operations with the same opStack instance!!!
// the pointers will become invalid on destroying this instance!
void opStack::getGroups(int ***_group_list, int **_group_size, int *_group_cnt)
{
	if (!daa_group) calculateSymbolicNormalized(1,0);
	*_group_list = group;
	*_group_size = group_size;
	*_group_cnt = group_cnt;
}

// returns a set of all inputs, and all patterns which are possible,
//    the list of patterns in the "_pattern_list" argument
//    the number of patterns in the "_pattern_cnt" argument
//    the list of inputs in the "_input_list" argument
//    the number of inputs in the "_input_cnt" argument
// the pointers may become invalid on further operations with the same opStack instance!!!
// the pointers will become invalid on destroying this instance!
void opStack::getPatterns(char ***_input_list, int *_input_cnt, int ***_pattern_list, int *_pattern_cnt)
{
	if (!daa_patterns) calculatePattern();
	*_pattern_cnt = pattern_cnt;
	*_pattern_list = patterns;
	*_input_list = input_list;
	*_input_cnt = input_cnt;
}

int opStack::pow2(int value)
{
	if (value < 0) return 0;
	if (value == 0) return 1;
	if (value == 1) return 2;
	if (value == 2) return 4;
	if (value == 3) return 8;
	if (value == 4) return 16;
	if (value == 5) return 32;
	if (value == 6) return 64;
	if (value == 7) return 128;
	if (value == 8) return 256;
	if (value == 9) return 512;
	if (value == 10) return 1024;
	if (value == 11) return 2048;
	if (value == 12) return 4096;
	if (value == 13) return 8192;
	if (value == 14) return 16384;
	if (value == 15) return 32768;
	fprintf(stderr, "opStack::pow2(%d) overflow\n", value);
	return 0;
}

inline int opStack::pow2complement(int base, int clearbit)
{
	int value = pow2(base);
	value += value - 1;		//all bits on
	value -= pow2(clearbit);	//clear bit
	return value;
}

// figure out if string needs an parenthesis
// asumes that there are no parenthesis errors!
inline int opStack::needsParenthesis(char * eq, char op)
{
	char *par = strchr(eq, '(');
	if (!par) {
		switch (op) {
			case OP_NOT:
				if (strchr(eq, OP_XOR)) return 2;
				if (strchr(eq, OP_AND)) return 2;
				if (strchr(eq, OP_OR)) return 2;
				return 0;
				break;
			case OP_XOR:
				if (strchr(eq, OP_AND)) return 2;
				if (strchr(eq, OP_OR)) return 2;
				return 0;
				break;
			case OP_AND:
				if (strchr(eq, OP_OR)) return 2;
				return 0;
				break;
			default:
				return 0;
				break;
		}
	}
	char * deq = strrchr(eq, ')');
	if (!deq) fatal("fatal parenthesis error in class opStack!\n");
	*par = 0;
	switch (op) {
		case OP_NOT:
			if (strchr(deq, OP_XOR) || strchr(eq, OP_XOR)) {
				*par = '(';
				return 2;
			}
			if (strchr(deq, OP_AND) || strchr(eq, OP_AND)) {
				*par = '(';
				return 2;
			}
			if (strchr(deq, OP_OR) || strchr(eq, OP_OR)) {
				*par = '(';
				return 2;
			}
			break;
		case OP_XOR:
			if (strchr(deq, OP_AND) || strchr(eq, OP_AND)) {
				*par = '(';
				return 2;
			}
			if (strchr(deq, OP_OR) || strchr(eq, OP_OR)) {
				*par = '(';
				return 2;
			}
			break;
		case OP_AND:
			if (strchr(deq, OP_OR) || strchr(eq, OP_OR)) {
				*par = '(';
				return 2;
			}
			break;
	}
	*par = '(';
	return 0;
}

char ** opStack::getQueue()
{
	return queue;
}

int opStack::getCounter()
{
	return total;
}

StackInfo ** opStack::getInfo()
{
	return queue_info;
}


