/*************************************************/
/* methods for class XDeviceNet                  */
/*                                               */
/* widget containing a circuit                   */
/* actions with/at objects in the net-widget     */
/*                                               */
/* Andreas Rostin                                */
/* 15.03.99                                      */
/*************************************************/
#include <qfile.h>
#include <qtextstream.h>
#include <qregexp.h>

#include <netw.h>
#include <xnet.h>
#include <xwire.h>
#include <klogic.h>

/***************************************************/
/* methods of XDeviceNet                           */
/***************************************************/
XDeviceNet::XDeviceNet()
	: XDevice(fNET, 0, 0)
{
	_parent = (XDeviceNet *)NULL;
	delay = 0;
}

XDeviceNet::XDeviceNet(XDeviceNet *parent, int x, int y)
	: XDevice(fNET, x, y)
{
	_parent = parent;
	delay = 0;
}

XDeviceNet::~XDeviceNet()
{	list<XDevice> *d = netdev.First();
	list<XWire> *w = netwire.First();

	// destroy all devices and wires
	while(d) {
		delete d->Get();
		d = d->Next();
	}
	while(w) {
		delete w->Get();
		w = w->Next();
	}
	// destroy lists
	netnet.Destroy();
	netdev.Destroy();
	netwire.Destroy();
	netin.Destroy();
	netout.Destroy();

	initImport();
}

// returns 1 if all names are unique, otherwise 0
int XDeviceNet::checkCircuitNames()
{	list<XDevice> *ld1 = netdev.First();
	list<XDevice> *ld2;

	while(ld1) {
		// sub circuits ...
		if (ld1->Get()->type() == XDevice::fNET)
			if (!devIsNet(ld1->Get())->checkCircuitNames()) return 0;

		// devices ...
		ld2 = netdev.First();
		while(ld2) {
			if (ld2->Get() != ld1->Get()) {
				if (!strcmp(ld1->Get()->getText(), ld2->Get()->getText())) return 0;
			}
			ld2 = ld2->Next();
		}

		ld1 = ld1->Next();
	}
	return 1;
}

// change all duplicate names to be unique
int XDeviceNet::unifyCircuitNames()
{	list<XDevice> *ld1 = netdev.First();
	list<XDevice> *ld2;

	while(ld1) {
		// sub circuits ...
		if (ld1->Get()->type() == XDevice::fNET)
			if (!devIsNet(ld1->Get())->checkCircuitNames()) return 0;

		// devices ...
		ld2 = netdev.First();
		while(ld2) {
			if (ld2->Get() != ld1->Get()) {
				if (!strcmp(ld1->Get()->getText(), ld2->Get()->getText())) {
					QString buffer;
					buffer.sprintf("uni%d", ld2->Get()->getID());
					setText(ld2->Get()->getID(), buffer);
				}
			}
			ld2 = ld2->Next();
		}

		ld1 = ld1->Next();
	}
	return 1;
}

list<OutputInfo> * XDeviceNet::parseCircuit(char *qualifier=(char *)NULL, list<OutputInfo> *ls = (list<OutputInfo> *)NULL)
{	list<XDevice> *ld = netdev.First();
	char buf[10000];
	XDevice *dev;

	// init level
	if (!ls) {
		ls = new list<OutputInfo>;
	}

	while(ld) {
		dev = ld->Get();
		if (dev->type() == XDevice::fNET) {
			if (qualifier) sprintf(buf, "%s%s.", qualifier, dev->getText());
			else sprintf(buf, "%s.", dev->getText());

			// remove all spaces from the used names!
			char *s;
			while (NULL != (s = strchr(buf, ' '))) *s = '_';

			devIsNet(dev)->parseCircuit(buf, ls);	// recursive call
		}
		dev->getAllEquations(ls, qualifier);		// device equations
		ld = ld->Next();
	}

	return ls;
}

void XDeviceNet::deleteNet(QPainter *p)
{	list<XDevice> *ld = netdev.First();
	list<XWire> *lw = netwire.First();

	// destroy all devices and wires
	while(ld) {
		deleteDevice(p, ld->Get());
		ld = netdev.First();
	}
	while(lw) {
		deleteWire(lw->Get());
		lw = netwire.First();
	}
	// destroy lists
	netnet.Destroy();
	netdev.Destroy();
	netwire.Destroy();
	netin.Destroy();
	netout.Destroy();
	delay = 0;
	Device::resetCounter();
	uniqueID::reset();
}

XDeviceNet * XDeviceNet::rootParent()
{	XDeviceNet *top;

	top = this;
	while(top->parent()) top = top->parent();
	return top;
}

XDeviceNet * XDeviceNet::parent()
{
	return _parent;
}

// call only from top level circuit!
// calculate sum of delays for the whole circuit
int XDeviceNet::calcDelay()
{	list<XDevice> *d = netdev.First();
	XDeviceNet *n;

	delay = 0;
	while(d) {
		if (NULL != (n = devIsNet(d->Get())))
			delay += n->calcDelay();
		else
			delay += d->Get()->delay();
		d = d->Next();
	}
	delay += XDevice::delay();
	return delay;
}

int XDeviceNet::netDelay()
{
	return delay;
}

list<XDeviceNet> *XDeviceNet::subnetPointer()
{	return netnet.First();
}

list<XDevice> *XDeviceNet::devPointer()
{	return &netdev;
}

int XDeviceNet::empty()
{
	if (!netnet.First() && !netdev.First() &&
	    !netwire.First() && !netin.First() && !netout.First())
		return 1;
	return 0;
}

void XDeviceNet::applyDefaults()
{	list<XDevice> *ld = netdev.First();

	setDelay(Device::defDelay());
	setInverted(XDevice::isDefInverted());
	setUndef(Device::defUndefined());
	invertTrigger(Device::triggerInverted());
	displayText(XDevice::textDisplayedGlobal());
	while(ld) {
		if (ld->Get()->type() == fNET) devIsNet(ld->Get())->applyDefaults();
		else {
			ld->Get()->setDelay(Device::defDelay());
			ld->Get()->setInverted(XDevice::isDefInverted());
			ld->Get()->setUndef(Device::defUndefined());
			ld->Get()->invertTrigger(Device::triggerInverted());
			ld->Get()->displayText(XDevice::textDisplayedGlobal());
			ld->Get()->setEquation();
			ld->Get()->parseEquation();
		}
		ld = ld->Next();
	}
}

//*************************************************************************
//* drawing methods
//*************************************************************************

// erase device, then draw all
void XDeviceNet::drawDev(QPaintDevice *paintd, QPainter *p, XDevice *dev)
{	// erase device and its connected wires
	dev->erase(p);
	// redraw all
	drawAll(paintd, p);
}

// draw all devices and wires in the net
// if paintd is NULL then drawing device is a printer
void XDeviceNet::drawAll(QPaintDevice *paintd, QPainter *p)
{	list<XDevice> *ld = netdev.First();
	list<XWire> *lw = netwire.First();

	while(ld) {
		if (paintd) ld->Get()->drawImage(paintd, p);
		else ld->Get()->drawImage(p);
		ld = ld->Next();
	}
	while(lw) {
		if (paintd) {
			lw->Get()->setColor(1);
			lw->Get()->drawImage(p);
		}
		else {
			// printer output
			lw->Get()->Wire::setColor(Qt::black);
			lw->Get()->drawImage(p);
		}
		lw = lw->Next();
	}
}

// draw all wires with its actual potential
// draw all devices with a changed image
void XDeviceNet::drawStatus(QPaintDevice *paintd, QPainter *p)
{	list<XWire> *lw = netwire.First();
	list<XDevice> *ld = netdev.First();

	while(ld) {
		// if image changes..
		if (ld->Get()->setColor()) {
			// ..draw device
			ld->Get()->drawImage(paintd, p);
		}
		ld = ld->Next();
	}
	// draw all wires in color of connected device-status
	while(lw) {
		// if color changes..
		if (lw->Get()->setColor(0)) {
			// ..draw wire
			lw->Get()->drawImage(p);
		}
		lw = lw->Next();
	}
}

//*************************************************************************
//* device methods
//*************************************************************************

// create new device
XDevice * XDeviceNet::newDevice(int func, int x, int y, int import = 0)
{
	// new device
	if (func != Device::fNET) {
		XDevice *dev = new XDevice(func, x, y);
		if (!dev) fatal(LOMEM);

		// append named input/output to this device
		if (func == XDevice::fIN) {
			if (!import) {
				if (ADDNAMED_OK > addInputName(dev->getText(), -1, dev->getID())) {
					delete dev;
					return (XDevice *)NULL;
				}
			}
			netin.Append(dev, dev->getID());
		}
		if (func == XDevice::fOUT) {
			if (!import) {
				if (ADDNAMED_OK > addOutputName(dev->getText(), -1, dev->getID())) {
					delete dev;
					return (XDevice *)NULL;
				}
			}
			netout.Append(dev, dev->getID());
		}
		netdev.Append(dev, dev->getID());
		rootParent()->calcDelay();
		return dev;
	}

	// new net device
	XDeviceNet *net = new XDeviceNet(this, x, y);
	if (!net) fatal(LOMEM);

	netnet.Append(net, net->getID());
	netdev.Append(net, net->getID());
	rootParent()->calcDelay();

	return net;
}

// returns pointer to the net-device if device is a net-device
XDeviceNet * XDeviceNet::devIsNet(XDevice *dev)
{	list<XDeviceNet> *ln;

	if (dev->type() != XDevice::fNET) return (XDeviceNet *)NULL;
	ln = netnet.With(dev->getID());
	if (ln) return ln->Get();
	return (XDeviceNet *)NULL;
}

// get subnet-pointer by name (used by import methods)
XDeviceNet * XDeviceNet::subNet(QString s)
{	list<XDeviceNet> *ln;
	int epos;
	QString sid;

	epos = s.find(DLM, 0);
	if (epos == -1) epos = s.length();
	sid = s.mid(1, epos - 1);

	ln = netnet.With(devmap.mapID(sid));
	if (ln) return ln->Get();
	return (XDeviceNet *)NULL;
}

// look for a device containing point
XDevice * XDeviceNet::containsDev(QPoint p)
{	list <XDevice> *l = netdev.First();

	while(l) {
		if (l->Get()->contains(p))
			return l->Get();
		l  = l->Next();
	}
	return (XDevice *)NULL;
}

// look for a device containing device
XDevice * XDeviceNet::containsDev(XDevice *dev)
{	list <XDevice> *l = netdev.First();

	while(l) {
		if (l->Get() != dev) {
			if (l->Get()->contains(dev->getPos())) {
				return l->Get();
			}
		}
		l  = l->Next();
	}
	return (XDevice *)NULL;
}

// mark all devices in rect as selected
void XDeviceNet::selectDev(QRect r, list<XDevice> *d)
{	list <XDevice> *l = netdev.First();

	while(l) {
		if (l->Get()->contains(r)) {
			if (!d->With(l->Get())) d->Append(l->Get());
			l->Get()->select(1);
		}
		else {
			d->Destroy(l->Get());
			l->Get()->select(0);
		}
		l = l->Next();
	}
}

// called from class Selection
// add all selected devices to the selection (paste operation)
void XDeviceNet::grabSelection(Selection *sel)
{	list <XDevice> *ld = netdev.First();
	list <XWire> *lw = netwire.First();

	while(ld) {
		if (ld->Get()->isSelected()) sel->add(ld->Get());
		ld = ld->Next();
	}
	while(lw) {
		if (lw->Get()->isSelected()) sel->add(lw->Get());
		lw = lw->Next();
	}
}

// delete a device
void XDeviceNet::deleteDevice(QPainter *p, XDevice *dev)
{
	if (dev->type() == XDevice::fIN) {
		netin.Destroy(dev);
		removeInputName(dev->getID());
	}
	if (dev->type() == XDevice::fOUT) {
		netout.Destroy(dev);
		removeOutputName(dev->getID());
	}

	// erase bitmap from screen
	dev->erase(p);

	// delete device reference in list
	if (dev->type() == XDevice::fNET)
		netnet.Destroy(devIsNet(dev));
	netdev.Destroy(dev);

	// delete device, in destructor: delete connections
	delete dev;

	rootParent()->calcDelay();
}

//*************************************************************************
//* wire methods
//*************************************************************************

// look for a wire containing point
XWire * XDeviceNet::containsWire(QPoint p)
{	list<XWire> *l = netwire.First();

	while(l) {
		// if wire contains point, the node will be locked automaticly
		// if not, a new node is created
		if (l->Get()->createNode(p)) return l->Get();
		l = l->Next();
	}
	return (XWire *)NULL;
}

// look for a wire containing (parts of) rect and select them
void XDeviceNet::selectWire(QRect r, list<XWire> *w)
{	list<XWire> *l = netwire.First();

	while(l) {
		// catch nodes, mark them as selected
		if (l->Get()->select(r)) {
			if (!w->With(l->Get())) w->Append(l->Get());
		}
		else {
			w->Destroy(l->Get());
		}
		l = l->Next();
	}
}

int XDeviceNet::removeNode(QPainter *p, QPoint pt)
{	list<XWire> *l = netwire.First();
	XWire *w1, *w2;
	int ret = 0;
	int nextl = 1;

	// can remove more than one node if required
	while(l) {
		w1 = l->Get();
		if (w1->contains(pt)) {
			w1->erase(p);
			w2 = w1->removeNode();
			if (!wireOK(p, w1)) {
				nextl = 0;
				l = netwire.First();
			}
			if (w2 && !wireOK(p, w2)) {
				nextl = 0;
				l = netwire.First();
			}
			ret++;
		}
		if (nextl) l = l->Next();
		else nextl = 1;
	}
	return ret;
}

// look for a device or wire, connected to given wire
// connections will be made automaticly on active node of wire
int XDeviceNet::checkConnection(XWire *w) 
{	list<XDevice> *ld = netdev.First();
	list<XWire> *lw = netwire.First();
	int ret;

	// look for device connection/disconnection
	while(ld) {
		if (DOK != (ret = ld->Get()->checkConnection(w))) {
			w->setColor(1);
			return ret;
		}
		ld = ld->Next();
	}
	// look for wire connection/disconnection
	while(lw) {
		if (w != lw->Get()) {
			if (WOK != (ret = lw->Get()->checkConnection(w))) {
				w->setColor(1);
				return ret;
		}
	        }
		lw = lw->Next();
	}
	return NOK;
}

// look for a device, connected to given wire
// (used as import-method only)
int XDeviceNet::checkConnection(int devid, int inverted, XWire *w)
{	XDevice *d = netdev.Get(devid);
	int ret;

	// look for new device connection
	if (d) {
		if (DOK != (ret = d->checkConnection(w, inverted))) return ret;
	}
	return NFAIL;
}

// create a new wire
XWire *XDeviceNet::newWire(QPainter *p, QPoint pt)
{	XWire *w;
	int ret;

	// create new wire and its first node
	w = new XWire(pt);

	// look for connection in the first point
	ret = checkConnection(w);
	if ((WCONN == ret) || (DCONN == ret))
		pt = w->getActive();

	// wire possibly has changed
	w->erase(p);
	w->draw(p);

	// lock new wire, create second node
	if (!w->createNode(pt)) {
		delete w;
		warning(NEWWIRE);
		return (XWire *)NULL;
	}
	// append new wire to wirelist
	netwire.Append(w, w->getID());
	return w;
}	
  
// no wires with single nodes allowed
int XDeviceNet::wireOK(QPainter *p, XWire *w)
{	XWire *w2;

	if (w->countNodes() < 2) {
		w->lockLastNode();
		// erase last locked node
		w->erase(p);
		w2 = w->removeNode();
		if (w2) wireOK(p, w2);	// recursive call
		// remove nodeless wire
		netwire.Destroy(w);
		delete w;
		return 0;
	}
	return 1;
}
 
// wire released after clicked with the mouse
int XDeviceNet::releaseWire(QPainter *p, XWire *w)
{
	w->Wire::garbageCollection();

	// check, that wire has at leat two nodes
	if (!wireOK(p, w)) return 0;
	w->releaseNode();
	return 1;
}

// delete wire
void XDeviceNet::deleteWire(XWire *w)
{
	netwire.Destroy(w);
	delete w;
}

//*************************************************************************
//* calculation methods
//*************************************************************************

// set net to init-state (flush all calculation-fifos)
void XDeviceNet::flush()
{	list<XDevice> *l = netdev.First();

	while(l) {
		l->Get()->flush(0);
		l = l->Next();
	}
}

// calculate burst (normally bust is 1)
void XDeviceNet::Burst(int burst)
{	int burst_step;

	for (burst_step = 0; burst_step < burst; burst_step++) {
		switchInterface(IF_INPUT);
		Calculate(burst_step);
		Propagate(burst_step);
		switchInterface(IF_OUTPUT);
	}
}

// switch input- or output devices of this
void XDeviceNet::switchInterface(int if_type)
{	list<XDevice> *ld;
	list<XDeviceNet> *ln;
	Device *d;

	// set output-values of input-devices of this net
	// (they are part of this net and the interface from upper circuit to this net)
	if (if_type == IF_INPUT) {
		ld = netin.First();
		while(ld) {
			d = ld->Get();
			// set input device from the named input value
			d->setStaticInput(input(d->getID()));
			ld = ld->Next();
		}
	}

	// calculate output devices, set output values of named outputs
	if (if_type == IF_OUTPUT) {
		ld = netout.First();
		while (ld) {
			d = ld->Get();
			// set the named output value
			setStaticOutput(d->getID(), d->output());
			ld = ld->Next();
		}
	}

	ln = netnet.First();
	while(ln) {
		ln->Get()->switchInterface(if_type);
		ln = ln->Next();
	}
}

// calculate a step
void XDeviceNet::Calculate(int burst_step)
{	list<XDevice> *ld;
	XDevice *d;
	int type;

	// calculate all devices of this circuit and all sub circuits
	ld = netdev.First();
	while(ld) {
		d = ld->Get();
		type = d->type();
		if (type != fTXT && type != fIN) {
			if (type != fNET) d->Calculate(0, burst_step);
			else devIsNet(d)->Calculate(burst_step);	// recursive call
		}
		ld = ld->Next();
	}
}

// set output values of all devices
void XDeviceNet::Propagate(int burst_step)
{	list<XDevice> *ld;
	XDevice *d;
	int type;

	ld = netdev.First();
	while(ld) {
		d = ld->Get();
		type = d->type();
		if (type != fTXT && type != fIN) {
			if (d->type() != fNET) d->Calculate(1, burst_step);
			else devIsNet(d)->Propagate(burst_step);	// recursive call
		}
		ld = ld->Next();
	}
}

/***************************************************/
/* import/export operations on a net-widget        */
/***************************************************/
QString XDeviceNet::exportInstance()
{
	QString s = device2string(0, 0);
	return s;
}

// export all wires of this net
// to export only the selected objects, call with selonly=1
QString XDeviceNet::exportWires(int selonly, int dx = 0, int dy = 0)
{	list<XWire> *lw = netwire.First();
	QString s="";
	QString ws;

	while(lw) {
		if ((selonly && lw->Get()->isSelected()) || (!selonly)) {
			if (!s.isEmpty()) s += "\n";
			// name of wire
			ws.sprintf("\t%d%c", lw->Get()->getID(), DLM);
			// nodes of wire
			ws.append((const char *)lw->Get()->wire2string(selonly, dx, dy));
			s = s + ws;
		}
		lw = lw->Next();
	}
	return s;
}

// to export only the selected objects, call with selonly=1
QString XDeviceNet::exportDevices(int selonly, int dx = 0, int dy = 0)
{	list<XDevice> *ld = netdev.First();
	XDevice *d;
	QString s="";

	while(ld) {
		d = ld->Get();
		if ((selonly && d->isSelected()) || (!selonly)) {
			if (!s.isEmpty()) s.append("\n");
			s =  s + d->device2string(dx, dy);
		}
		ld = ld->Next();
	}
	return s;
}

// to export only the selected objects, call with selonly=1
QString XDeviceNet::exportWireConn(int selonly, int dx = 0, int dy = 0)
{	list<XWire> *lw = netwire.First();
	XWire *w;
	QPoint pi, po;
	QString s="";
	QString ws;

	while(lw) {
		w = lw->Get();
		if ((selonly && w->isSelected()) || (!selonly)) {
			pi = w->getIOPosition(2, selonly);
			po = w->getIOPosition(1, selonly);
			if ((!pi.isNull()) || (!po.isNull())) {
				if (!s.isEmpty()) s += "\n";
				s += "\t";
				ws.sprintf("%d", w->getID());
				s += ws;
				if (!pi.isNull()) {
					ws.sprintf("%c%d%c%d%c%d",
						DLM,
						w->ioWire(2)->getID(),
						DLM,
						pi.x() - dx,
						DLM,
						pi.y() - dy);
					s = s + ws;
				}
				if (!po.isNull()) {
					if (!pi.isNull()) {
						s += "\n\t";
						ws.sprintf("%d", w->getID());
						s += ws;
					}
					ws.sprintf("%c%d%c%d%c%d",
						DLM,
						w->ioWire(1)->getID(),
						DLM,
						po.x() - dx,
						DLM,
						po.y() - dy);
					s = s + ws;
				}
			}
		}
		lw = lw->Next();
	}
	return s;
}

// to export only the selected objects, call with selonly=1
QString XDeviceNet::exportDevConn(int selonly, int dx = 0, int dy = 0)
{	list<XWire> *lw = netwire.First();
	XDevice *d;
	XWire *w;
	QString s="", ds="";
	QPoint pi1, pi2, po;

	while(lw) {
		w = lw->Get();
		if ((selonly && w->isSelected()) || (!selonly)) {
			pi1 = w->getInputPosition(1, selonly);
			pi2 = w->getInputPosition(2, selonly);
			po = w->getOutputPosition(selonly);
			if (!pi1.isNull() || !pi2.isNull() || !po.isNull()) {
				if (!s.isEmpty()) s = s + "\n";
				s += "\t";
				ds.sprintf("%d", w->getID());
				s += ds;

				if (!pi1.isNull()) {
					d = w->inputDevice(1);
					ds.sprintf("%c%d%c%d%c%d%c%d",
							DLM,
							d->getID(),
							DLM,
							w->inputIsInverted(d),
							DLM,
							pi1.x() - dx,
							DLM,
							pi1.y() - dy);
					s = s + ds;
				}
				if (!pi2.isNull()) {
					d = w->inputDevice(2);
					ds.sprintf("%c%d%c%d%c%d%c%d",
							DLM,
							d->getID(),
							DLM,
							w->inputIsInverted(d),
							DLM,
							pi2.x() - dx,
							DLM,
							pi2.y() - dy);
					s = s + ds;
				}
				if (!po.isNull()) {
					d = w->outputDevice();
					ds.sprintf("%c%d%c%d%c%d%c%d",
							DLM,
							d->getID(),
							DLM,
							w->outputIsInverted(),
							DLM,
							po.x() - dx,
							DLM,
							po.y() - dy);
					s = s + ds;
				}
			}
		}
		lw = lw->Next();
	}
	return s;
}

void XDeviceNet::initImport()
{
	wiremap.init();
	devmap.init();
}

// reading a sub circuit from file: create a new sub circuit device and initialize it
int XDeviceNet::importInstance(QString s, int dx = 0, int dy = 0)
{
	if (hist_isHist(s))
		return hist_importInstance(s);

	importDevice(s, 0, dx, dy);

	return 0;
}

int XDeviceNet::importWire(QString s, int select, int dx = 0, int dy = 0)
{	int epos;
	QString sid;
	XWire *w;

	epos = s.find(DLM, 0);
	if (epos == -1) return -1;
	sid = s.mid(1, epos - 1);
	s = s.right(s.length() - epos);

	if (!(w = new XWire())) return -1;
	if (!w->string2wire(s, dx, dy)) {
		delete w;
		return -1;
	}
	netwire.Append(w, w->getID());

	wiremap.fileID(sid, w->getID());

	// select wire?
	if (select) w->select(1);

	return 0;
}

int XDeviceNet::importDevice(QString s, int select, int dx = 0, int dy = 0)
{	int apos, epos;
	int func;
	QString sid;
	QString devext;
	XDevice *d;

	if (hist_isHist(s))
		return hist_importDevice(s, select, dx, dy);

	apos = 1;
	epos = s.find(DLM, 0);
	if (epos == -1) return -1;
	sid = s.mid(apos, epos - apos);

	apos = epos + 1;
	epos = s.find(DLM, apos);
	if (epos == -1) return -1;
	func = s.mid(apos, epos - apos).toInt();

	// create the new device
	d = newDevice(func, 0, 0, 1);

	// call member method to import device parameter
	d->string2device(s, dx, dy);

	devmap.fileID(sid, d->getID());

	// add device to a selection? 
	if (select) d->select(1);

	if (d->type() == fIN || d->type() == fOUT) {
		// the text of the i/o-device is currently the old id ... (or -1 if it is an old format)
		list<value> *lv = NULL;
		list<opStack> *lop = NULL;
		if (d->type() == fIN) {
			lv = named_input.With(sid);			// access with the old id string
			if (lv) lv->setID1(d->getID());			// exchange the old id with the new one
		}
		if (d->type() == fOUT) {
			lop = named_output.With(sid);			// access with the old id string
			if (lop) lop->setID1(d->getID());		// exchange the old id with the new one
		}

		// now set the correct text
		setText(d->getID(), d->getText());
		setImage();
	}

	return 0;
}

void XDeviceNet::setText(int dev_id, QString new_text)
{	XDevice *d = netdev.Get(dev_id);

	if (d) {
		d->setText(new_text);
		if (new_text && strlen(new_text)) {
			if (d->type() == fIN) {
				list<value> *lv = named_input.With(dev_id);
				if (lv) lv->setText(new_text);
			}
			if (d->type() == fOUT) {
				list<opStack> *lop = named_output.With(dev_id);
				if (lop) lop->setText(new_text);
			}
		}
	}
}

int XDeviceNet::importWireConn(QString s, int, int dx = 0, int dy = 0)
{	int apos, epos;
	XWire *w1, *w2;
	QPoint pt;
	QString refWire, conWire;
	int refWireID, conWireID;

	epos = s.find(DLM, 0);
	if (epos == -1) return -1;
	refWire = s.mid(1, epos - 1);

	refWireID = wiremap.mapID(refWire);

	apos = epos + 1;
	epos = s.find(DLM, apos);
	if (apos == -1 || epos == -1) return -1;
	conWire = s.mid(apos, epos - apos);

	conWireID = wiremap.mapID(conWire);

	apos = epos + 1;
	epos = s.find(DLM, apos);
	if (apos == -1 || epos == -1) return -1;
	pt.setX(s.mid(apos, epos - apos).toInt() + dx);

	apos = epos + 1;
	epos = s.length();
	if (apos == -1) return -1;
	pt.setY(s.mid(apos, epos - apos).toInt() + dy);

	w1 = netwire.Get(refWireID);
	w2 = netwire.Get(conWireID);
	if (!w1 || !w2) return -1;
	if (Wire::NODE_NOLOCK == w1->lockNode(pt)) {
		if (Wire::NODE_NOLOCK == w2->lockNode(pt))
			return -1;
		else
			w1->checkConnection(w2);
	}
	else
		w2->checkConnection(w1);

	return 0;
}

int XDeviceNet::importDevConn(QString s, int, int dx = 0, int dy = 0)
{	int apos, epos;
	QString sid;
	QString dev1, dev2;
	int wire_id;
	int dev1_id = 0, dev2_id = 0;
	int inverted1 = 0, inverted2 = 0;
	QPoint p1, p2;
	XWire *w;

	epos = s.find(DLM, 0);
	if (epos == -1) return -1;
	sid = s.mid(1, epos - 1);

	wire_id = wiremap.mapID(sid);

	apos = epos + 1;
	epos = s.find(DLM, apos);
	if (apos == -1 || epos == -1) return -1;
	dev1 = s.mid(apos, epos - apos);
	dev1_id = devmap.mapID(dev1);

	apos = epos + 1;
	epos = s.find(DLM, apos);
	if (apos == -1 || epos == -1) return -1;
	inverted1 = s.mid(apos, epos - apos).toInt();
	if (inverted1 == 0) inverted1 = -1;

	apos = epos + 1;
	epos = s.find(DLM, apos);
	if (apos == -1 || epos == -1) return -1;
	p1.setX(s.mid(apos, epos - apos).toInt() + dx);

	apos = epos + 1;
	epos = s.find(DLM, apos);
	if (epos == -1)
		epos = s.length();
	if (apos == -1) return -1;
	p1.setY(s.mid(apos, epos - apos).toInt() + dy);

	apos = epos + 1;
	epos = s.find(DLM, apos);
	if (apos != -1 && epos != -1) {
		dev2 = s.mid(apos, epos - apos);
		dev2_id = devmap.mapID(dev2);

		apos = epos + 1;
		epos = s.find(DLM, apos);
		if (apos == -1 || epos == -1) return -1;
		inverted2 = s.mid(apos, epos - apos).toInt();
		if (inverted2 == 0) inverted2 = -1;

		apos = epos + 1;
		epos = s.find(DLM, apos);
		if (apos == -1 || epos == -1) return -1;
		p2.setX(s.mid(apos, epos - apos).toInt() + dx);

		apos = epos + 1;
		epos = s.length();
		if (apos == -1) return -1;
		p2.setY(s.mid(apos, epos - apos).toInt() + dy);
	}
	else {
		p2.setX(0);
		p2.setY(0);
	}
	// connect wire with devices
	w = netwire.Get(wire_id);
	if (!w) return -1;
	if (!w->contains(p1)) return -1;
	if (DCONN != checkConnection(dev1_id, inverted1, w)) return -1;
	if (!p2.isNull()) {
		if (!w->contains(p2)) return -1;
		if (!checkConnection(dev2_id, inverted2, w)) return -1;
	}
	return 0;
}

