/*************************************************/
/* methods for class klogicIO                    */
/*                                               */
/* load/save digital circuits                    */
/*                                               */
/* Andreas Rostin                                */
/* 15.03.99                                      */
/*************************************************/
#include <qfile.h>
#include <qstring.h>
#include <qtextstream.h>
#include <qregexp.h>
#include <xnet.h>

#include <klogicIO.h>

klogicIO::klogicIO(QString filename, XDeviceNet *_net)
{
	net = _net;
	fname = filename;
	if (-1 == fname.find(QRegExp("\\.circuit$")))
		fname.append(".circuit");
	net->initImport();
}

klogicIO::klogicIO(QString filename)
{
	net = (XDeviceNet *)NULL;
	fname = filename;
	if (-1 == fname.find(QRegExp("\\.circuit$")))
		fname.append(".circuit");
}

klogicIO::klogicIO(XDeviceNet *_net)
{
	net = _net;
	fname = "";
	net->initImport();
}

int klogicIO::checkSubNet()
{	QFile *f;
	QString s;

	f = new QFile((const char *)fname);
	if (!f->open(IO_ReadOnly)) return -1;

	QTextStream t(f);
	s = t.readLine();

	if (7 == type(s)) {
		f->close();
		return 1;
	}
	f->close();
	return 0;
}

void klogicIO::setNet(XDeviceNet *_net)
{
	net = _net;
	net->initImport();
}

void klogicIO::setSubFilename()
{
	if (-1 == fname.find(QRegExp("\\.sub\\.circuit$")))
		fname.replace( QRegExp("\\.circuit$"), ".sub.circuit" );
}

int klogicIO::writeNet(XDeviceNet *_net = (XDeviceNet *)NULL)
{	int ret;

	if (!net) return 0;
	if (fname.isEmpty()) return 0;

	QFile *f = new QFile((const char *)fname);
	if (!f->open(IO_WriteOnly|IO_Truncate))
		return 0;

	QTextStream *t = new QTextStream(f);
	ret =  writeNet(t, EXP_ALL, _net);
	f->flush();
	f->close();
	delete f;

	return ret;
}

int klogicIO::writeNet(QTextStream *ts, int selection, XDeviceNet *_net = (XDeviceNet *)NULL, int minx = 0, int miny = 0)
{	QString s;
	int sret = 0;
	int flag;
	int subnet = 0;

	if (!_net) {
		_net = net;
		// write net informations
		*ts << IO_B_NET << "\n";
		subnet = 0;
	} else {
		// write subnet informations
		*ts << IO_B_SUBNET << "\n";
		subnet = 1;
	}

	*ts << _net->exportInstance() << "\n";

	// write wire position information
	*ts << IO_B_WIRE << "\n";
	s = _net->exportWires(selection, minx, miny);
	if (!s.isEmpty()) *ts << s << "\n";
	*ts << IO_E_WIRE << "\n";

	// write device type/position information
	*ts << IO_B_DEVICE << "\n";
	s = _net->exportDevices(selection, minx, miny);
	if (!s.isEmpty()) *ts << s << "\n";
	*ts << IO_E_DEVICE << "\n";

	// write subnet information
	list<XDeviceNet> *ln = _net->subnetPointer();
	while(ln) {
		if (selection == EXP_SELECTION) {
			if (ln->Get()->isSelected()) flag = 1;
			else flag = 0;
		} else flag = 1;

		if (flag) {
			// (EXP_ALL) always write the whole content of sub circuits
			if (0 > (sret = writeNet(ts, EXP_ALL, ln->Get()))) {
				return sret;
			}
		}
		ln = ln->Next();
	}

	// write wire-wire connections
	*ts << IO_B_CONNWIRE << "\n";
	s = _net->exportWireConn(selection, minx, miny);
	if (!s.isEmpty()) *ts << s << "\n";
	*ts << IO_E_CONNWIRE << "\n";

	// write wire-device connections
	*ts << IO_B_CONNDEVICE << "\n";
	s = _net->exportDevConn(selection, minx, miny);
	if (!s.isEmpty()) *ts << s << "\n";
	*ts << IO_E_CONNDEVICE << "\n";

	if (!subnet) {
		// write subnet information
		*ts << IO_E_NET << "\n";
	} else {
		// write net information
		*ts << IO_E_SUBNET << "\n";
	}

	return 1;
}

int klogicIO::readNet(int create_sub)
{	QFile *f;
	int ret;

	if (!net) return 0;
	if (fname.isEmpty()) return 0;

	f = new QFile((const char *)fname);
	if (!f->open(IO_ReadOnly)) return 0;

	QTextStream *t = new QTextStream(f);
	ret = readNet(create_sub, t, NO_SELECTION);
	f->close();
	delete t;
	delete f;

	if (ret < 0) return 0;
	else return 1;
}

int klogicIO::readNet(int create_sub, QTextStream *t, int select, XDeviceNet *_net = (XDeviceNet *)NULL, int dx = 0, int dy = 0)
{	QString s;
	int told=0, tnew=0;
	int sub = 0;
	int sub_is_main = 0;
	int must_change = 0;
	XDeviceNet *__net;
	int ret = 0;
	char err[1024];
// temp: perf.tuning
	//QTime time;

	if (!_net) _net = net;
	else {
		sub = 1;
		told = 1;
	}

	while(!t->eof()) {
		s = t->readLine();
		if (s.isEmpty()) continue;

		// for debugging purposes
		strncpy(err, (const char *)s, 1023);
		err[1023] = 0;

		// change status
		if (0 != (tnew = type(s))) {
			must_change = 0;
			// avoid impossible status
			if ((tnew != told + 1) &&
			    (!(told == 1 && tnew == 3)) &&
			    (!(told == 6 && tnew == 9)) &&
			    (!(told == 8 && tnew == 7)) &&
			    (!(told == 0 && tnew == 7)) &&
			    (tnew != 8) && (tnew != 2)) {
				sprintf(err, "%s\n%s\ntold: %d tnew:%d\n", "corrupt file content", "impossible status change", told, tnew);
				warning(err);
				return -1;
			}
		}
		else tnew = told;

		// special case: subcircuit is the main circuit
		if (told == 0 && tnew == 7 && sub == 0) {
			tnew = 1;
			sub_is_main = 1;
		}

		// process current status
		if (told == tnew) {
			switch (tnew) {
			case 1:
				if (create_sub) {
					// create sub circuit
					ret = ret + _net->importInstance(s, dx, dy);
					__net = _net->subNet(s);
					if (!__net) {
						warning("corrupt content\ncannot find sub-circuit");
						return -1;
					}
					// (NO_SELECTION) do never select the content of a sub circuit
					ret = ret + readNet(0, t, NO_SELECTION, __net);
					told = 8;
				}
				break;
			case 3:
				ret = ret + _net->importWire(s, select, dx, dy);
				break;
			case 5:
				ret = ret + _net->importDevice(s, select, dx, dy);
				break;
			case 7:
				__net = _net->subNet(s);
				if (!__net) {
					warning("corrupt content\ncannot find sub-circuit");
					return -1;
				}
				// (NO_SELECTION) do never select the content of a sub circuit
				ret = ret + readNet(0, t, NO_SELECTION, __net);
				told = 8;
				break;
			case 9:
				ret = ret + _net->importWireConn(s, select, dx, dy);
				break;
			case 11:
				ret = ret + _net->importDevConn(s, select, dx, dy);
				break;
			default:
				warning("corrupt content\nunknown status while reading file");
				fprintf(stderr, "%s while in %d\n", (const char *)s, tnew);
				return -1;
				break;
			}
		}
		// current status actually changed
		else {
			told = tnew;
			if (tnew == 2 || (tnew == 8 && sub_is_main)) {
				return ret;
			}
			if (tnew == 8) return ret;
		}
	}
	return ret;
}

int klogicIO::type(QString s)
{
	if (s.contains(IO_B_NET)) return 1;
	if (s.contains(IO_E_NET)) return 2;
	if (s.contains(IO_B_WIRE)) return 3;
	if (s.contains(IO_E_WIRE)) return 4;
	if (s.contains(IO_B_DEVICE)) return 5;
	if (s.contains(IO_E_DEVICE)) return 6;
	if (s.contains(IO_B_SUBNET)) return 7;
	if (s.contains(IO_E_SUBNET)) return 8;
	if (s.contains(IO_B_CONNWIRE)) return 9;
	if (s.contains(IO_E_CONNWIRE)) return 10;
	if (s.contains(IO_B_CONNDEVICE)) return 11;
	if (s.contains(IO_E_CONNDEVICE)) return 12;
	return 0;
}

