#include <qframe.h>
#include <qpopupmenu.h>
#include <qtooltip.h>
#include <qpushbutton.h>
#include <qlist.h>
#include <qinputdialog.h>
#include <qlayout.h>
#include <qgroupbox.h>
#include <qlineedit.h>
#include <qlabel.h>
#include <qtextstream.h>
#include <qslider.h>

#include <kapp.h>
#include <kmessagebox.h>
#include <kglobal.h>
#include <klocale.h>
#include <kconfig.h>

#include "providerpref.h"
#include "prefdialog.h"


/************** \/ prefDialog \/ ***************/


prefDialog::prefDialog(QWidget* parent=0, const char* name=0,KConfig *conf,int maxS=100,int minS=10)
	: QTabDialog(parent,name), DMConf(conf), maxSize(maxS), minSize(minS),providerAddedOrRemoved(false)
{
	setCaption(i18n("DiaMon Settings"));
	setCancelButton(i18n("&Cancel"));
	setOkButton(i18n("&Ok"));

 	QObject::connect( this, SIGNAL(applyButtonPressed()), this, SLOT(slotOk()) );
 	QObject::connect( this, SIGNAL(cancelButtonPressed()), this, SLOT(slotCancel()) );

	/* list that stores all providerPrefs */
	TabsList=new QList <providerPref>;
	TabsList->clear();
	TabsList->setAutoDelete(true);

	/******* General Settings Tab *******/
	generalSettings = new QWidget(this);
	generalLayout = new QVBoxLayout(generalSettings, 5);
	generalLayout->setResizeMode(QLayout::Minimum);

	/* size settings */
	sizeBox = new QGroupBox(i18n("Size"),generalSettings);

	sizeGrid=new QGridLayout(sizeBox,2,3,10,5);
	sizeGrid->setResizeMode(QLayout::Minimum);

	sizeSlider=new QSlider(minSize, maxSize, 10, 100, Qt::Horizontal, sizeBox);
	sizeLabel=new QLabel(sizeBox);
	sizeLabel->setFixedSize(35,20);
	sizeLabel->setAlignment( AlignVCenter | AlignRight );
	QObject::connect( sizeSlider, SIGNAL(valueChanged(int)), sizeLabel, SLOT(setNum(int)) );
	QObject::connect( sizeSlider, SIGNAL(valueChanged(int)), this, SLOT(updateSize(int)) );

	sizeGrid->addRowSpacing(0,5);
	sizeGrid->addWidget(sizeLabel,1,0,AlignVCenter | AlignRight);
	sizeGrid->addWidget(lpx=new QLabel("px",sizeBox),1,1,AlignVCenter | AlignLeft);
	sizeGrid->addWidget(sizeSlider,1,2);

	/* color settings */
	colorsBox = new QGroupBox(i18n("Colors"),generalSettings);
	colorGrid=new QGridLayout(colorsBox,4,2,10,5);
	colorGrid->setResizeMode(QLayout::Minimum);

	int hb=23,wb=140;
	bOff =new KColorButton(QColor(  0,255,0),colorsBox);
	bOff->setMinimumSize(wb,hb);
	bDial=new KColorButton(QColor(255,255,0),colorsBox);
	bDial->setMinimumSize(wb,hb);
	bOn  =new KColorButton(QColor(255,  0,0),colorsBox);
	bOn->setMinimumSize(wb,hb);

	colorGrid->addRowSpacing(0,5);
	colorGrid->addWidget(lOff=new QLabel(i18n("Offline:"),colorsBox),1,0,AlignRight);
	colorGrid->addWidget(bOff,1,1);
	colorGrid->addWidget(lDial=new QLabel(i18n("Dialing:"),colorsBox),2,0,AlignRight);
	colorGrid->addWidget(bDial,2,1);
	colorGrid->addWidget(lOn=new QLabel(i18n("Online:"),colorsBox),3,0,AlignRight);
	colorGrid->addWidget(bOn,3,1);

	newButtonBox = new QGroupBox(i18n("New Provider"),generalSettings);
	newButtonBox->setMinimumSize(300,67);
	newButton=new QPushButton(i18n("&New..."),newButtonBox);
	newButton->setGeometry(15,25,70,23);

	/* scale settings */
	scaleBox = new QGroupBox(i18n("Scale in Bytes/sec"),generalSettings);
	scaleGrid=new QGridLayout(scaleBox,2,1,10,5);
	scaleGrid->setResizeMode(QLayout::Minimum);
	lineScale=new QLineEdit(QString::number(12000),scaleBox);
	scaleGrid->addRowSpacing(0,5);
	scaleGrid->addWidget(lineScale,1,0);

	/* timer settings */
	timerBox = new QGroupBox(i18n("Refresh time in msec"),generalSettings);
	timerGrid=new QGridLayout(timerBox,2,1,10,5);
	timerGrid->setResizeMode(QLayout::Minimum);
	lineTimer=new QLineEdit(QString::number(1000),timerBox);
	timerGrid->addRowSpacing(0,5);
	timerGrid->addWidget(lineTimer,1,0);

	scaleTimerLayout=new QHBoxLayout(generalSettings,0);
	scaleTimerLayout->setResizeMode(QLayout::Minimum);
	scaleTimerLayout->addWidget(scaleBox, 5);
	scaleTimerLayout->addSpacing(5);
	scaleTimerLayout->addWidget(timerBox, 5);

	/* groupbox -> layout */
	generalLayout->addWidget(colorsBox, 5);
	generalLayout->addWidget(sizeBox, 5);
	generalLayout->addLayout(scaleTimerLayout,5);
	generalLayout->addWidget(newButtonBox, 5);
	QObject::connect( newButton, SIGNAL(clicked()), this, SLOT(slotNew()) );
	addTab(generalSettings,i18n("&General"));

	/* load config & add providerPref tabs */
	getDMConfig();
}

prefDialog::~prefDialog()
{
	TabsList->setAutoDelete(true);
	TabsList->clear();
	delete newButton;
	delete scaleTimerLayout;
	delete lineScale;
	delete scaleGrid;
	delete scaleBox;
	delete newButtonBox;
	delete lOn;
	delete lDial;
	delete lOff;
	delete bOn;
	delete bDial;
	delete bOff;
	delete colorGrid;
	delete colorsBox;
	delete lpx;
	delete sizeLabel;
	delete sizeSlider;
	delete sizeGrid;
	delete sizeBox;
	delete generalLayout;
	delete generalSettings;
	delete TabsList;
}

void prefDialog::resizeEvent( QResizeEvent *r) {}

void prefDialog::updateSize(int newSize)
{
	/* update size when the slider is moved */
	size=newSize;
}

void prefDialog::changeSize(int newSize)
{
	/* update size & sizeSlider & sizeLabel when the size was changed with MMB */
	size=newSize;
	sizeSlider->setValue(size);
}

void prefDialog::slotCancel(void)
{
	showPage(generalSettings);

	// remove all providerPref tabs
	TabsList->first();
	while(TabsList->current() != 0)
	{
		TabsList->remove(TabsList->current());
		TabsList->next();
	}

	TabsList->clear();

	// restore config & tabs
	getDMConfig();
}

void prefDialog::slotOk(void)
{
	// set Config ..
	setDMConfig();
	// .. and save config to disk
	DMConf->sync();

	// tell DiaMon that it has to update Layout & Menu
	emit okPressed(providerAddedOrRemoved);
}

void prefDialog::providerRemoved(int ID)
{
	showPage(generalSettings);

	// remove page
	removePage(TabsList->at(ID));
	TabsList->remove(ID);

	// update ID's of the providerPref pages
	TabsList->first();
	while(TabsList->current() != 0)
	{
		(TabsList->current())->setNumber(TabsList->at());
		TabsList->next();
	}

	providerAddedOrRemoved=true;
}

void prefDialog::slotNew(void)
{
	bool ok=false;
	bool nameOk;
	bool nameLengthOk;
	QString text;

	do
	{
		nameLengthOk=nameOk=true;
		text = QInputDialog::getText( i18n("New Provider"), i18n("Name:"), QString::null, &ok, 0 );

		for(TabsList->first();TabsList->current() != 0;TabsList->next())
			if( text == ((TabsList->current())->getName()) ) nameOk=false;
		if(!nameOk && ok)
			KMessageBox::information(0, i18n("This provider already exists. Please choose another name."));
		if(text.length() > MAX_LEN && nameOk && ok)
		{
			nameLengthOk=false;
			KMessageBox::information(0, i18n("This provider name is too long. Please choose a shorter name."));
		}
	}
	while( ((!nameOk || !nameLengthOk) && ok) );

	if ( ok && !text.isEmpty() )
	{
		TabsList->append(new providerPref(this,text));
		(TabsList->current())->setNumber(TabsList->at());
		QObject::connect( TabsList->current(), SIGNAL(remove(int)), this, SLOT(providerRemoved(int)) );
		addTab(TabsList->current(),text);
		showPage(TabsList->current());
		providerAddedOrRemoved=true;
	}
}

void prefDialog::getDMConfig(void)
{
	QColor defin(255,255,255);
	QColor defout(0,0,0);
	DMConf->setGroup("Provider");
	int count=DMConf->readNumEntry("Number",0);
	for(int i=0;i<count;i++)
	{
		QString s;
		QTextOStream(&s) << "prov" << i;
		DMConf->setGroup(s);

		TabsList->append(new providerPref( this, DMConf->readEntry("Name","Default") ));
		(TabsList->current())->setNumber(TabsList->at());
		QObject::connect( TabsList->current(), SIGNAL(remove(int)), this, SLOT(providerRemoved(int)) );
		addTab(TabsList->current(),DMConf->readEntry("Name","Default"));

		(TabsList->current())->setDevice(DMConf->readEntry("Device"));
		(TabsList->current())->setComCon(DMConf->readEntry("CommandToConnect"));
		(TabsList->current())->setComDiscon(DMConf->readEntry("CommandToDisconnect"));
		(TabsList->current())->setInColor(DMConf->readColorEntry("InColor",&defin));
		(TabsList->current())->setOutColor(DMConf->readColorEntry("OutColor",&defout));

		(TabsList->current())->writeLineEdits();
	}

	DMConf->setGroup("General");
	size=DMConf->readNumEntry("Size",100);
	sizeSlider->setValue(size);

	QColor def(0,255,0);
	bOff->setColor(DMConf->readColorEntry("offlineColor",&def));
	def=QColor(255,0,0);
	bOn->setColor(DMConf->readColorEntry("onlineColor",&def));
	def=QColor(255,255,0);
	bDial->setColor(DMConf->readColorEntry("dialingColor",&def));

	int scale=DMConf->readNumEntry("Scale",12000);
	lineScale->setText(QString::number(scale));

	int timer=DMConf->readNumEntry("Timer",1000);
	lineTimer->setText(QString::number(timer));
}

void prefDialog::setDMConfig(void)
{
	int count=0;
	TabsList->first();
	while(TabsList->current() != 0)
	{
		QString s;
		QTextOStream(&s) << "prov" << count;
		DMConf->setGroup(s);
		DMConf->writeEntry("Name",(TabsList->current())->getName());
		DMConf->writeEntry("Device",(TabsList->current())->getDevice());
		DMConf->writeEntry("CommandToConnect",(TabsList->current())->getComCon());
		DMConf->writeEntry("CommandToDisconnect",(TabsList->current())->getComDiscon());
		DMConf->writeEntry("InColor",(TabsList->current())->getInColor());
		DMConf->writeEntry("OutColor",(TabsList->current())->getOutColor());
		TabsList->next();
		count++;
	}
	DMConf->setGroup("Provider");
	DMConf->writeEntry("Number",count);

	DMConf->setGroup("General");
	DMConf->writeEntry("Size",size);
	DMConf->writeEntry("Scale",lineScale->text().toInt());
	DMConf->writeEntry("Timer",lineTimer->text().toInt());
	DMConf->writeEntry("onlineColor",bOn->color());
	DMConf->writeEntry("offlineColor",bOff->color());
	DMConf->writeEntry("dialingColor",bDial->color());

}


