/* This file is part of the KDE libraries
    Copyright (c) 1998 Emmeran Seehuber (the_emmy@hotmail.com)
 
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include "klgroupbox.h"
#include "klgroupboxpriv.h"
#include "klhelper.h"
#include "kllabel.h"


KLGroupBox::KLGroupBox(const char *title,bool horiz) : QGroupBox((QWidget *)0,0), a_group(horiz)
{
  setKLWidget(this); 
  setTitle(title);
  a_group.parentChild = this;
}

void KLGroupBox::setupGrap()
{
  KLWidgetBase::setupGrap();
  a_group.setupGrap();
}

void KLGroupBox::cleanupGrap()
{
  a_group.cleanupGrap();
  KLWidgetBase::cleanupGrap();
}

bool KLGroupBox::klAskMinMax(KLMinMaxSizes *minMaxSizes)
{
  if( !KLWidgetBase::klAskMinMax(minMaxSizes) )
    return false;
  QFontMetrics fm = fontMetrics();
  ulong x = fm.height() + KLGroupBox_TopBorder + KLGroupBox_BottomBorder;
  ulong y = KLGroupBox_LeftBorder + KLGroupBox_RightBorder;
  ulong myminY = fm.width(title()) + KLGroupBox_TitleBorder;
  minMaxSizes->minX += x;
  minMaxSizes->minY += y;
  minMaxSizes->defX += x;
  minMaxSizes->defY += y;
  minMaxSizes->maxX += x;
  minMaxSizes->maxY += y;
  KLMinMaxSizes &mms = a_group.minMaxSizes();
  mms.init();
  a_group.klDoAskMinMax(&mms);
  *minMaxSizes += mms;
  if( mms.minY < myminY ) 
    minMaxSizes->minY += myminY - mms.minY;
  minMaxSizes->validate();
  return true;
}

bool KLGroupBox::klSetup( KLSetupInfo *setupInfo )
{
  if( !KLWidgetBase::klSetup(setupInfo) )
    return false;
  if( !a_group.klSetup(setupInfo) ) {
    KLWidgetBase::klCleanup();
    return false;
  }
  setFocusPolicy(QWidget::NoFocus);
  return true;
}

bool KLGroupBox::klShow(KLShowInfo *showInfo)
{
  if( !KLWidgetBase::klShow(showInfo) ) 
    return false;
  QFontMetrics fm = fontMetrics();
  ulong x = fm.height() + KLGroupBox_TopBorder;
  ulong y = KLGroupBox_LeftBorder;
  ulong myminY = fm.width(title()) + KLGroupBox_TitleBorder;
  KLShowInfo &si = a_group.showInfo();
  si = *showInfo;
  si.x += x;
  si.y += y;
  si.xSize -= x + KLGroupBox_BottomBorder;
  si.ySize -= y + KLGroupBox_RightBorder;

  // If the maximum of the group is smaller than our maximum,
  // just center the group
  if( a_group.minMaxSizes().maxY < myminY ) {
    ulong diff = myminY - a_group.minMaxSizes().maxY;
    si.ySize -= diff;
    si.y += diff / 2;
  }

  if( !a_group.klShow(&si) ) {
    KLWidgetBase::klHide();
    return false;
  }
  return true;
}

void KLGroupBox::klHide()
{
  if( a_group.a_state == SI_Showed )
    a_group.klHide();
  KLWidgetBase::klHide();
}

void KLGroupBox::klCleanup()
{
  if( a_group.a_state == SI_SetupDone )
    a_group.klCleanup();
  KLWidgetBase::klCleanup();
}


void KLGroupBox::setTitle(const char *text)
{
  QGroupBox::setTitle(text);
  doRelayout(false);
}

KLGroupBox &KLGroupBox::operator<<(KLChild *child)
{
  a_group.addChild(child);
  return *this;
}

KLGroupBox &KLGroupBox::operator<<(KLChild &child)
{
  a_group.addChild(&child);
  return *this;
}

KLGroupBox &KLGroupBox::operator<<(KLGroup::_grp_change_func func)
{
  func(a_group);
  return *this;
}

KLGroupBox &KLGroupBox::operator<<(KLChild::_child_change_func func)
{
  func(*this);
  return *this;
}

KLChild::KLStreamHelp &KLGroupBox::operator<<(KLChild::_child_change_p1_func func)
{
  static KLStreamHelp sh;
  sh.child = this;
  sh.func = func;
  return sh;
}

KLChild *KLGroupBox::findChild( ulong x, ulong y ) const
{
  if( !KLWidgetBase::findChild(x,y) )
    return 0;
  KLChild *child = a_group.findChild(x,y);
  if( child == 0 )
    child = (KLChild *)this; // Explizied discard const
  HASSERT(child);
  return child;
}

void KLGroupBox::addDropMarks( KLDropMarkList * dml ) const
{
  a_group.addDropMarks(dml);
}

bool KLGroupBox::isAChild( KLChild * _child ) const
{
  if( KLChild::isAChild(_child) )
    return true;
  return a_group.isAChild(_child);
}

IMPLEMENT_KLMETA_STANDALONE(KLGroupBox,KLWidgetBase,"GroupBox");

bool KLGroupBox::MyChildMeta::dumpObject(KLChild *_child, KLDumpDevice& dev) const
{
  QString helpStr;
  KLGroupBox *child = (KLGroupBox *)_child;
  dev.writeEntry("title",child->title());
  return KLChildMeta::dumpCompleteObject(&child->a_group,dev);
}

void KLGroupBox::MyChildMeta::restoreObject(KLChild *_child, KLDumpDevice& dev) const
{
  QString helpStr;
  KLGroupBox *child = (KLGroupBox *)_child;
  child->setTitle(dev.readEntry("title",child->title()));

  // Restore child group
  QString linestr;
  ulong origline = dev.linestart;
  ulong line = dev.linestart, level = 0;
  while( dev.readLine(linestr,line,level) ) {
    if( (level == 2) && (linestr[0] == '{') ) {
      dev.linestart = line - 1;
      KLChildMeta::restoreCompleteObject(dev,&child->a_group);
      break;
    }
  }
  dev.linestart = origline;
}

KLChild *KLGroupBox::MyChildMeta::createObjectEdit(KLChild *child,bool) const
{
  HASSERT(child->metaInherits("KLGroupBox"));
  KLGroupBox *group = (KLGroupBox *)child;
  KLGroupBoxEdit *edit = new KLGroupBoxEdit();
  edit->group = group;
  KLGridGroup *gridgroup = (KLGridGroup *)KLChildMeta::createObjectEdit(child,false);
  *gridgroup << klLabel("Title")
             << edit;
  
  QObject::connect(edit,SIGNAL(textChanged(const char *)),edit,SLOT(onTitleChange(const char *)));
  edit->setText(group->title());

  return gridgroup;
}

void KLGroupBoxEdit::onTitleChange(const char *text)
{
  group->setTitle(text);
}



#include "klgroupbox.moc"
#include "klgroupboxpriv.moc"
