/* arch-tag: 2168ecf0-9abc-45d2-ad33-d396361cca1d */

/*  eXperience GTK engine: utils.c
 *  
 *  Copyright (C) 2004-2005  Benjamin Berg <benjamin@sipsolutions.net>
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *
 *  Thanks to the Magic Chicken engine, where I found out, how I can modify
 *  the pixbufs for filtering.
 *    Magic Chicken author:
 *      James M. Cape <jcape@ignore-your.tv>
 *
*/

#include <gdk-pixbuf/gdk-pixbuf.h>

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <math.h>
#include <string.h>

#include "utils.h"
#include "experience.h"

#define USE_EVIL 1

gint
experience_round (eXperienceRoundingMethod method, gfloat value)
{
	switch (method) {
		case ROUND_CEIL:
			return (gint) ceil (value);
			break;
		case ROUND_FLOOR:
			return (gint) floor (value);
			break;
		case ROUND_NORMAL:
			return (gint) floor (value + 0.5);
			break;
		case ROUND_TO_ZERO:
			return (gint) value;
	}
	g_return_val_if_reached (0);
}

#if USE_EVIL

/* WARNING: TRUE EVIL FOLLOWING
 * 
 * The following is optimization insanity brought to you by Benjamin Berg
 * <benjamin@sipsolutions.net>. And all to save a bunch of GObject creation and
 * destructions. */
 
 
/* copied from gdk-pixbuf */
struct _GdkPixbuf {
        GObject parent_instance;

        /* Color space */
        GdkColorspace colorspace;

        /* Number of channels, alpha included */
        int n_channels;

        /* Bits per channel */
        int bits_per_sample;

        /* Size */
        int width, height;

        /* Offset between rows */
        int rowstride;

        /* The pixel array */
        guchar *pixels;

        /* Destroy notification function; it is supposed to free the pixel array */
        GdkPixbufDestroyNotify destroy_fn;

        /* User data for the destroy notification function */
        gpointer destroy_fn_data;

        /* Do we have an alpha channel? */
        guint has_alpha : 1;
};
typedef struct _GdkPixbuf RawGdkPixbuf;

static void
position_subpixbuf (GdkPixbuf *subpixbuf, GdkPixbuf *src, GdkRectangle *area)
{
	RawGdkPixbuf *rawdst = (RawGdkPixbuf*) subpixbuf;
	RawGdkPixbuf *rawsrc = (RawGdkPixbuf*) src;
	
	rawdst->n_channels      = rawsrc->n_channels;
	rawdst->has_alpha       = rawsrc->has_alpha;
	rawdst->bits_per_sample = rawsrc->bits_per_sample;
	rawdst->rowstride       = rawsrc->rowstride;
	rawdst->width           = area->width;
	rawdst->height          = area->height;
	
	rawdst->pixels = gdk_pixbuf_get_pixels (src) + area->y * rawsrc->rowstride + area->x * rawsrc->n_channels;
}

static void
modify_destination_pixbuf (GdkPixbuf *dst, GdkPixbuf *src, gint width, gint height)
{
	RawGdkPixbuf *rawdst = (RawGdkPixbuf*) dst;
	RawGdkPixbuf *rawsrc = (RawGdkPixbuf*) src;
	
	rawdst->n_channels      = rawsrc->n_channels;
	rawdst->has_alpha       = rawsrc->has_alpha;
	rawdst->bits_per_sample = rawsrc->bits_per_sample;
	rawdst->rowstride       = (rawdst->n_channels * width + 3) & ~3;
	rawdst->width           = width;
	rawdst->height          = height;
	
	/* in theory it should be possible to save this free/malloc, but how,
	 * and is it worth it?*/
	g_free (rawdst->pixels);
	rawdst->pixels = g_malloc (rawdst->rowstride * height);
}

#endif

static inline GdkPixbuf *
experience_create_subpixbuf (const GdkPixbuf *src, gint x, gint y, gint width, gint height)
{
	guchar * pixels;
	pixels = (gdk_pixbuf_get_pixels (src)
	          + y * gdk_pixbuf_get_rowstride (src)
	          + x * gdk_pixbuf_get_n_channels (src));

	return gdk_pixbuf_new_from_data (pixels,
	                                 gdk_pixbuf_get_colorspace (src),
	                                 gdk_pixbuf_get_has_alpha (src),
	                                 gdk_pixbuf_get_bits_per_sample (src),
	                                 width, height,
	                                 gdk_pixbuf_get_rowstride (src),
	                                 NULL, NULL);
}

GdkPixbuf *
experience_evil_pixbuf_scale_or_ref (GdkPixbuf * pixbuf, gboolean need_persistent, GdkRectangle * src, gint dest_width, gint dest_height, GdkInterpType interp_type, gboolean must_scale)
{
	static GdkPixbuf * subpixbuf = NULL;
	
	if ((src->x == 0) && (src->y == 0) && (src->width == dest_width) && (src->height == dest_height)/* &&
	    (dest_width  == gdk_pixbuf_get_width (pixbuf)) && (dest_height == gdk_pixbuf_get_height (pixbuf))*/) {
		/* we can just use the original pixbuf */
		/* it does not matter, if the returned pixbuf is larger, as long as
		 * the top left corner is correct :-D */
		g_object_ref (pixbuf);
		return pixbuf;
	} else {
#if USE_EVIL
		if (gdk_pixbuf_major_version == 2 && gdk_pixbuf_major_version <= 10) {
			/* EVIL codepath. */
			
			/* shortcut */
			if (need_persistent && src->width == dest_width && src->height == dest_height) {
				return experience_create_subpixbuf (pixbuf, src->x, src->y, src->width, src->height);
			}
			
			if (subpixbuf == NULL) {
				subpixbuf = gdk_pixbuf_new_from_data ((guchar*) 0x1, GDK_COLORSPACE_RGB, 0, 8, 10, 10, 10, NULL, NULL);
			}
			
			/* FAST subpixbuf creation. */
			position_subpixbuf (subpixbuf, pixbuf, src);
			
			/* did we just need to get the subpixbuf? */
			if (src->width == dest_width && src->height == dest_height) {
				g_object_ref (subpixbuf);
				return subpixbuf;
			}
			
			if (need_persistent) {
				/* just scale, can't do anything here ... */
				return gdk_pixbuf_scale_simple (subpixbuf, dest_width, dest_height, interp_type);
			} else {
				if (!must_scale) {
					/* ok, we know that it is compisited later on, and it will be scaled then. so just return. */
					g_object_ref (subpixbuf);
					return subpixbuf;
				} else {
					static GdkPixbuf *scaled_pixbuf = NULL;
					/* wheee, lets do stupid stuff again :) */
					if (scaled_pixbuf == NULL) {
						scaled_pixbuf = gdk_pixbuf_new_from_data ((guchar*) g_malloc (10), GDK_COLORSPACE_RGB, 0, 8, 10, 10, 10, NULL, NULL);
					}
					modify_destination_pixbuf (scaled_pixbuf, subpixbuf, dest_width, dest_height);
					
					gdk_pixbuf_scale (subpixbuf, scaled_pixbuf, 0, 0, dest_width, dest_height,
					                  0, 0, (float) dest_width / src->width, (float) dest_height / src->height,
					                  interp_type);
					g_object_ref (scaled_pixbuf);
					return scaled_pixbuf;
				}
			}
		} else {
#endif
			GdkPixbuf * result;
			
			subpixbuf = experience_create_subpixbuf (pixbuf, src->x, src->y, src->width, src->height);
			if (need_persistent || must_scale) {
				/* can't just ref ... */
				result = gdk_pixbuf_scale_simple (subpixbuf, dest_width, dest_height, interp_type);
				g_object_unref (subpixbuf);
				
				return result;
			} else {
				return subpixbuf;
			}
#if USE_EVIL
		}
#endif
	}
}

/* EVIL ENDS HERE */

/* and now the normal and healthy function */
GdkPixbuf *
experience_pixbuf_scale_simple_or_ref (GdkPixbuf * pixbuf, gint width, gint height, GdkInterpType interp_type)
{
	if ((width  == gdk_pixbuf_get_width (pixbuf)) && (height == gdk_pixbuf_get_height (pixbuf))) {
		g_object_ref (pixbuf);
		return pixbuf;
	} else {
		return gdk_pixbuf_scale_simple (pixbuf, width, height, interp_type);
	}
}


/*----------*/

GdkPixbuf *
experience_rotate (GdkPixbuf * pixbuf, eXperienceRotate rotation)
{
	GdkPixbuf * target = NULL;
	guint x, y, rowstride, height, width;
	guint target_rowstride, n_channels;
	guchar *row, *pixel, *target_row, *target_pixel;
	
	g_return_val_if_fail (pixbuf != NULL, NULL);
	g_return_val_if_fail (gdk_pixbuf_get_bits_per_sample (pixbuf) == 8, NULL); /* we can only handle one byte per sample */
	g_return_val_if_fail (gdk_pixbuf_get_colorspace (pixbuf) == GDK_COLORSPACE_RGB, NULL);
	n_channels = gdk_pixbuf_get_n_channels (pixbuf);
	g_return_val_if_fail ((n_channels == 3) || (n_channels == 4), NULL); /*RGB || RGBA*/
	
	width  = gdk_pixbuf_get_width  (pixbuf);
	height = gdk_pixbuf_get_height (pixbuf);
	rowstride = gdk_pixbuf_get_rowstride (pixbuf);
	
	target = pixbuf;
	
	if (rotation == ROTATE_AROUND) {
		target = gdk_pixbuf_new (GDK_COLORSPACE_RGB, gdk_pixbuf_get_has_alpha (pixbuf), 8,
		                         width, height);
		
		target_rowstride = gdk_pixbuf_get_rowstride (target);
		
		row = gdk_pixbuf_get_pixels (pixbuf);
		
		target_row = gdk_pixbuf_get_pixels (target);
		target_row = target_row + (height - 1) * target_rowstride;
		
		for (y = 0; y < height; y++) {
			pixel = row;
			target_pixel = target_row + (width - 1) * n_channels;
			
			for (x = 0; x < width; x++) {
				g_memmove (target_pixel, pixel, n_channels);
				
				pixel += n_channels;
				target_pixel -= n_channels;
			}
			row += rowstride;
			target_row -= target_rowstride;
		}
		
		g_object_unref (pixbuf);
	} else if (rotation == ROTATE_CW) {
		target = gdk_pixbuf_new (GDK_COLORSPACE_RGB, gdk_pixbuf_get_has_alpha (pixbuf), 8,
		                         height, width);

		target_rowstride = gdk_pixbuf_get_rowstride (target);
		
		row = gdk_pixbuf_get_pixels (pixbuf);
		
		target_row = gdk_pixbuf_get_pixels (target);
		target_row = target_row + (height - 1) * n_channels;
		
		for (y = 0; y < height; y++) {
			pixel = row;
			target_pixel = target_row;
			
			for (x = 0; x < width; x++) {
				g_memmove (target_pixel, pixel, n_channels);
				
				pixel += n_channels;
				target_pixel += target_rowstride;
			}
			row += rowstride;
			target_row -= n_channels;
		}
		
		g_object_unref (pixbuf);
	} else if (rotation == ROTATE_CCW) {
		target = gdk_pixbuf_new (GDK_COLORSPACE_RGB, gdk_pixbuf_get_has_alpha (pixbuf), 8,
		                         gdk_pixbuf_get_height (pixbuf), gdk_pixbuf_get_width (pixbuf));

		target_rowstride = gdk_pixbuf_get_rowstride (target);
		
		row = gdk_pixbuf_get_pixels (pixbuf);
		
		target_row = gdk_pixbuf_get_pixels (target);
		target_row = target_row + (width - 1) * target_rowstride;
		
		for (y = 0; y < height; y++) {
			pixel = row;
			target_pixel = target_row;
			
			for (x = 0; x < width; x++) {
				g_memmove (target_pixel, pixel, n_channels);
				
				pixel += n_channels;
				target_pixel -= target_rowstride;
			}
			row += rowstride;
			target_row += n_channels;
		}
		
		g_object_unref (pixbuf);
	}
	
	return target;
}

GdkPixbuf *
experience_mirror (GdkPixbuf * pixbuf, eXperienceRotate rotation)
{
	GdkPixbuf * source, * dest;
	guint x, y, rowstride, height, width;
	guint dest_rowstride, n_channels;
	guchar *row, *pixel, *dest_row, *dest_pixel;
	
	g_return_val_if_fail (pixbuf != NULL, NULL);
	g_return_val_if_fail (gdk_pixbuf_get_bits_per_sample (pixbuf) == 8, NULL); /* we can only handle one byte per sample */
	g_return_val_if_fail (gdk_pixbuf_get_colorspace (pixbuf) == GDK_COLORSPACE_RGB, NULL);
	n_channels = gdk_pixbuf_get_n_channels (pixbuf);
	g_return_val_if_fail ((n_channels == 3) || (n_channels == 4), NULL); /*RGB || RGBA*/
	
	source = pixbuf;
	
	width  = gdk_pixbuf_get_width  (source);
	height = gdk_pixbuf_get_height (source);
	
	if (rotation & ORIENTATION_HORIZONTAL) {
		rowstride = gdk_pixbuf_get_rowstride (source);
		
		dest = gdk_pixbuf_new (GDK_COLORSPACE_RGB, gdk_pixbuf_get_has_alpha (source), 8,
		                         width, height);
		dest_rowstride = gdk_pixbuf_get_rowstride (source);
		
		row = gdk_pixbuf_get_pixels (source);
		dest_row = gdk_pixbuf_get_pixels (dest);
		
		for (y = 0; y < height; y++) {
			pixel = row;
			dest_pixel = dest_row + (width - 1) * n_channels;
			
			for (x = 0; x < width; x++) {
				g_memmove (dest_pixel, pixel, n_channels);
				
				pixel += n_channels;
				dest_pixel -= n_channels;
			}
			
			row += rowstride;
			dest_row += dest_rowstride;
		}
		
		g_object_unref (source);
		source = dest;
	}
	
	if (rotation & ORIENTATION_VERTICAL) {
		rowstride = gdk_pixbuf_get_rowstride (source);
		
		dest = gdk_pixbuf_new (GDK_COLORSPACE_RGB, gdk_pixbuf_get_has_alpha (source), 8,
		                       width, height);
		dest_rowstride = gdk_pixbuf_get_rowstride (source);
		
		row = gdk_pixbuf_get_pixels (source);
		dest_row = gdk_pixbuf_get_pixels (dest);
		dest_row += (height - 1) * dest_rowstride;
		
		for (y = 0; y < height; y++) {
			pixel = row;
			dest_pixel = dest_row;
			
			for (x = 0; x < width; x++) {
				g_memmove (dest_pixel, pixel, n_channels);
				
				pixel += n_channels;
				dest_pixel += n_channels;
			}
			
			row += rowstride;
			dest_row -= dest_rowstride;
		}
		
		g_object_unref (source);
		source = dest;
	}
	
	return source;
}

GdkPixbuf *
experience_change_pixbuf_opacity (GdkPixbuf * pixbuf, gfloat opacity)
{
	GdkPixbuf * target;
	gint width, height, rowstride, n_channels;
	guchar *row, *pixel;
	gint x, y;
	
	g_return_val_if_fail (pixbuf != NULL, NULL);
	g_return_val_if_fail (gdk_pixbuf_get_bits_per_sample (pixbuf) == 8, NULL); /* we can only handle one byte per sample */
	g_return_val_if_fail (gdk_pixbuf_get_colorspace (pixbuf) == GDK_COLORSPACE_RGB, NULL);
	n_channels = gdk_pixbuf_get_n_channels (pixbuf);
	g_return_val_if_fail ((n_channels == 3) || (n_channels == 4), NULL); /*RGB || RGBA*/
	
	opacity = CLAMP (opacity, 0, 1);
	
	if (opacity == 1)
		return pixbuf;
	
	if (!gdk_pixbuf_get_has_alpha (pixbuf)) {
		target = gdk_pixbuf_add_alpha (pixbuf, FALSE, 0, 0, 0);
		g_object_unref (pixbuf);
		n_channels = gdk_pixbuf_get_n_channels (target);
	} else {
		target = pixbuf;
	}

	if (opacity == 0) {
		gdk_pixbuf_fill (pixbuf, 0x00000000);
		return pixbuf;
	}
	
	width  = gdk_pixbuf_get_width (target);
	height = gdk_pixbuf_get_height (target);
	
	rowstride = gdk_pixbuf_get_rowstride (target);
	row = gdk_pixbuf_get_pixels (target);
	
	for (y = 0; y < height; y++) {
		pixel = row;
		for (x = 0; x < width; x++) {
			pixel[ALPHA_OFFSET] = pixel[ALPHA_OFFSET] * opacity;
			
			pixel += n_channels;
		}
		row += rowstride;
	}
	
	return target;
}

void
experience_set_pixbuf_brightness (GdkPixbuf * pixbuf, gfloat brightness)
{
	gint width, height, rowstride, n_channels;
	guchar *row, *pixel;
	gint x, y;
	
	g_return_if_fail (pixbuf != NULL);
	g_return_if_fail (gdk_pixbuf_get_bits_per_sample (pixbuf) == 8); /* we can only handle one byte per sample */
	g_return_if_fail (gdk_pixbuf_get_colorspace (pixbuf) == GDK_COLORSPACE_RGB);
	n_channels = gdk_pixbuf_get_n_channels (pixbuf);
	g_return_if_fail ((n_channels == 3) || (n_channels == 4)); /*RGB || RGBA*/
	
	brightness = CLAMP (brightness, -1, 1);
	
	if (brightness == 0)
		return;
	
	width  = gdk_pixbuf_get_width (pixbuf);
	height = gdk_pixbuf_get_height (pixbuf);
	
	rowstride = gdk_pixbuf_get_rowstride (pixbuf);
	row = gdk_pixbuf_get_pixels (pixbuf);
	
	for (y = 0; y < height; y++) {
		pixel = row;
		for (x = 0; x < width; x++) {
			pixel[RED_OFFSET]   = CLAMP_UCHAR (pixel[RED_OFFSET] + brightness * 255);
			pixel[GREEN_OFFSET] = CLAMP_UCHAR (pixel[GREEN_OFFSET] + brightness * 255);
			pixel[BLUE_OFFSET]  = CLAMP_UCHAR (pixel[BLUE_OFFSET] + brightness * 255);
			
			pixel += n_channels;
		}
		row += rowstride;
	}
	
	return;
}


/*###########*/

void
experience_pixbuf_composite (eXperienceRenderContext * dest, GdkRectangle * dest_area, GdkRectangle * clip_area, GdkInterpType interp_type, experience_get_image_from_info get_image, gpointer info)
{
	GdkPixbuf * image;
	GdkRectangle tmp_area;
	
	gdk_rectangle_intersect (clip_area, dest_area, &tmp_area);
	
	/* only draw if size is bigger than 0. */
	if ((tmp_area.width > 0) && (tmp_area.height > 0)) {
		if (dest->type == TYPE_PIXBUF) {
			/* get the image */
			image = get_image (interp_type, info, FALSE);
			
			/* check whether anything was already painted on the area. Or if the image doesn't have an alpha chanel. */
			if (   (dest_area->width == gdk_pixbuf_get_width (image) && dest_area->height == gdk_pixbuf_get_height (image))
			    && ((dest->dirty_region && gdk_region_rect_in (dest->dirty_region, &tmp_area) == GDK_OVERLAP_RECTANGLE_OUT)
			     || (!gdk_pixbuf_get_has_alpha (image)))) { /* I am curious if the has_alpha check can actually gain any speed. Comments welcome. */
				/* We can just copy the area. */
				/* First calculate the offset on the image */
				gint x_offset = tmp_area.x - dest_area->x; /* ?_offset musst be bigger or equal than 0 after this. */
				gint y_offset = tmp_area.y - dest_area->y;
				
				/* Copy the data to the dest image. */
				gdk_pixbuf_copy_area (image,
				                      x_offset, y_offset,
				                      tmp_area.width, tmp_area.height,
				                      dest->pixbuf,
				                      tmp_area.x, tmp_area.y);
			} else {
				/* composite the data on the dest image. */
				gdk_pixbuf_composite (image, dest->pixbuf, tmp_area.x, tmp_area.y,
				                      tmp_area.width, tmp_area.height,
				                      dest_area->x, dest_area->y,
				                      (float) dest_area->width / gdk_pixbuf_get_width (image),
				                      (float) dest_area->height / gdk_pixbuf_get_height (image),
				                      interp_type, 255);
			}
			
			/* make dirty_region bigger ... */
			if (dest->dirty_region != NULL)
				gdk_region_union_with_rect (dest->dirty_region, &tmp_area);
			g_object_unref (image);
		} else {
			image = get_image (interp_type, info, TRUE);
			gdk_draw_pixbuf (dest->drawable, dest->clip_gc, image,
			                 tmp_area.x - dest_area->x, tmp_area.y - dest_area->y,
			                 tmp_area.x, tmp_area.y, tmp_area.width, tmp_area.height,
			                 GDK_RGB_DITHER_NORMAL, 0, 0);
			g_object_unref (image);
		}
	}
}

/*###########*/

guint
experience_hash_mem (gpointer start, guint count, guint init)
{
	guint result = init;
	const char * p;
	
	for (p = start; (gpointer) p < (gpointer) (start + count); p++) {
		result = (result << 5) - result + *p;
	}
	
	return result;
}
