/* arch-tag: 9d6b3fb9-d972-45ab-b09c-4065d4506be8 */

/*  eXperience GTK engine: parser.y
 *  
 *  Copyright (C) 2004-2005  Benjamin Berg <benjamin@sipsolutions.net>
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

%pure_parser

%{
#include "parser.h"
#include <gdk/gdk.h>
#include <glib.h>
#include <glib-object.h>
#include <stdio.h>
#include "parsetree.h"
#include "group.h"
#include "image.h"
#include "fill.h"
#include "match.h"
#include "filter.h"
#include "dynamic_color.h"

#define YYPARSE_PARAM pst
#define YYLEX_PARAM pst

eXperiencePosPercent pos_percent_tmp;
eXperienceImagePercent image_percent_tmp;

%}

%debug

%union {
  float v_float;
  int v_int;
  char* v_str;
  gboolean v_bool;
  GdkColor v_color;
  eXperienceDynamicColor v_dynamic_color;
  eXperiencePosPercent   v_pos_percent;
  eXperienceImagePercent v_image_percent;
  
  guint v_uint;
}

%destructor { g_free ($$); } STRING

%token <v_float>	FLOAT 1001
%token <v_int>		INT
%token				IMAGE_LITERAL
%token				GROUP_LITERAL
%token				ICONS_LITERAL
%token <v_str>		STRING

%token				FILE_LITERAL
%token				BORDER_LITERAL
%token				REPEAT_LITERAL
%token				STRETCH_LITERAL

%token				OVERLAY_FILE_LITERAL
%token				OVERLAY_BORDER_LITERAL
%token				OVERLAY_STRETCH_LITERAL

%token				STATE_LITERAL
%token				PADDING_LITERAL
%token				Y_POS_LITERAL
%token				X_POS_LITERAL
%token				POS_LITERAL
%token				TILE_LITERAL

%token				FUNCTION_LITERAL
%token				DETAIL_LITERAL
%token				HAS_DEFAULT_LITERAL
%token				HAS_FOCUS_LITERAL
%token				SHADOW_LITERAL
%token				ARROW_DIRECTION_LITERAL
%token				ORIENTATION_LITERAL
%token				GAP_SIZE_LITERAL
%token				GAP_SIDE_LITERAL
%token				EXPANDER_STYLE_LITERAL
%token				WINDOW_EDGE_LITERAL
%token				NONE_LITERAL

%token				BOX_GAP_LITERAL
%token				SHADOW_GAP_LITERAL

%token				SATURATION_LITERAL
%token				PIXELATE_LITERAL
%token				BRIGHTNESS_LITERAL
%token				OPACITY_LITERAL

%token				CLEAR_AREA_LITERAL

%token				DONT_DRAW_LITERAL
%token				DONT_CLIP_LITERAL
%token				DONT_INHERIT_LITERAL
%token				RECOLORABLE_LITERAL

%token				LINE_WIDTH_LITERAL

%token				RECOLOR_LITERAL
%token				RECOLOR_GRADIENT_LITERAL

%token				CENTER_LITERAL
%token				DRAW_COMPONENTS_LITERAL

%token				INTERPOLATION_TYPE_LITERAL

%token				ROUNDING_LITERAL

%token				MIRROR_LITERAL
%token				ROTATE_LITERAL

%token				WIDTH_LITERAL
%token				HEIGHT_LITERAL

%token				INNER_PADDING_LITERAL
%token				DRAW_ENTIRE_ONLY_LITERAL

%token				FILL_LITERAL
%token				COLOR_LITERAL

%token				WS_LITERAL
%token				IS_LITERAL
%token				PX_LITERAL


/* backward compatibilty */
%token				HLINE_LITERAL
%token				VLINE_LITERAL

%token	<v_uint>	FUNCTION_IDENTIFIER
%token	<v_uint>	NEW_FUNCTION_IDENTIFIER
%token	<v_uint>	STATE_IDENTIFIER
%token	<v_uint>	SHADOW_IDENTIFIER
%token	<v_uint>	ARROW_DIRECTION_IDENTIFIER
%token	<v_uint>	ORIENTATION_IDENTIFIER
%token	<v_uint>	POSITION_IDENTIFIER
%token	<v_uint>	EXPANDER_STYLE_IDENTIFIER
%token	<v_uint>	WINDOW_EDGE_IDENTIFIER
%token	<v_uint>	GAP_FILE_IDENTIFIER
%token	<v_uint>	GAP_BORDER_IDENTIFIER
%token	<v_uint>	DRAW_COMPONENTS_IDENTIFIER

%token	<v_uint>	INTERPOLATION_TYPE_IDENTIFIER

%token	<v_uint>	STYLE_COLOR_IDENTIFIER
%token	<v_uint>	ROUNDING_TYPE_IDENTIFIER

%token	<v_uint>	ROTATE_IDENTIFIER

%token <v_bool>		BOOLEAN
%token <v_uint>		LEFT_RIGHT

%token				YY_ALWAYS_KEEP_LAST

%type <v_uint>		function
%type <v_uint>		all_functions
%type <v_uint>		new_function
%type <v_uint>		states
%type <v_uint>		shadows
%type <v_uint>		shadow
%type <v_uint>		arrow_directions
%type <v_uint>		arrow_direction
%type <v_uint>		orientations
%type <v_uint>		positions
%type <v_uint>		position
%type <v_uint>		expander_styles
%type <v_uint>		window_edges
%type <v_uint>		draw_component
%type <v_uint>		draw_components
%type <v_uint>		filter_mirror

%type <v_float>		float
%type <v_float>		ufloat
%type <v_int>		uint
%type <v_int>		int
%type <v_color>		color
%type <v_dynamic_color>	dynamic_color

%type <v_pos_percent> pos_percent_calc
%type <v_pos_percent> pos_percent_number

%type <v_image_percent> image_percent_calc
%type <v_image_percent> image_percent_number

/* backward compatibility */
%type <v_uint>		bgw_functions
%type <v_uint>		bgw_function

%type <v_uint>		changed_function
%{

#include "symbols.h"

eXperienceDynamicColor dynamic_color_black = {GET_SET_COLOR, {0,0,0,0}, {0,0} };

static int experience_yylex (YYSTYPE *lvalp, parsestate* pst);
static int experience_yyerror (char*s);

#define PST ((parsestate*)pst)
#define CURRENT_DRAWABLE (PST->current_drawable)
#define CURRENT_IMAGE ((eXperienceImage*)CURRENT_DRAWABLE)
#define CURRENT_GROUP (PST->current_group)
#define CURRENT_FILTER (PST->current_filter)
#define CURRENT_FILL ((eXperienceFill*) PST->current_drawable)

%}

%% /* Grammar rules and actions follow */
start:		  definitions '}'			{ PST->should_end = 1; }
definitions:	  definition
				| definition definitions

definition:	  group
			| top_image
			| icons_def

/* group definition */
group:		  GROUP_LITERAL group_naming '{' group_defs '}'	{ parse_end_group (pst); }
			| GROUP_LITERAL group_naming_i					{ parse_end_group (pst); }

group_naming:								{ parse_begin_group (pst, NULL, NULL, FALSE); }
				| STRING					{ parse_begin_group (pst, $1, NULL, FALSE); }
				| group_naming_i
group_naming_i:	  STRING '=' STRING			{ parse_begin_group (pst, $1, $3, FALSE); }
				| '=' STRING				{ parse_begin_group (pst, NULL, $2, FALSE); }

group_defs:		  group_def
				| group_def group_defs
group_def:		  match_def
				| group_property
				| drawable
				| all_filter
				  //function is here because of backward compatibilty
				| FUNCTION_LITERAL       '=' all_functions			{ experience_match_add_functions (&CURRENT_GROUP->match, $3); }
				| GAP_SIZE_LITERAL '=' uint					{ experience_group_set_gap_size (CURRENT_GROUP, $3, $3); }
				| PADDING_LITERAL '=' '{' int ',' int ',' int ',' int '}'	{ experience_group_set_padding (CURRENT_GROUP, $4, $6, $8, $10); } 
				| DONT_CLIP_LITERAL '=' BOOLEAN				{ experience_group_set_dont_clip (CURRENT_GROUP, $3); }

/* group image */

drawable:			  group_image_begin group_image_defs '}'	{ parse_end_drawable (pst); }
					| fill_begin fill_defs '}'					{ parse_end_drawable (pst); }
group_image_begin:	  IMAGE_LITERAL uint '{'				{ parse_begin_drawable (pst, $2, experience_image_class); }
group_image_defs:	  group_image_def
					| group_image_def group_image_defs
group_image_def:	  drawable_property
					| image_property

fill_begin:			  FILL_LITERAL uint '{'				{ parse_begin_drawable (pst, $2, experience_fill_class); }
fill_defs:			  fill_def
					| fill_def fill_defs
fill_def:			  drawable_property
					| fill_property

/* toplevel image */

top_image:		  top_image_begin top_image_defs '}'
{
	parse_end_drawable (pst);
	parse_end_group (pst);
	
	PST->gap_group[0] = NULL;
	PST->gap_group[1] = NULL;
	PST->gap_group[2] = NULL;
	PST->gap_image[0] = NULL;
	PST->gap_image[1] = NULL;
	PST->gap_image[2] = NULL;
}
top_image_begin:  IMAGE_LITERAL '{'	
{
	gchar * name, * actual_name;
	
	parse_begin_group (pst, NULL, "RESERVED_BACKWARD_COMPAT", FALSE);
	actual_name = CURRENT_GROUP->name;
	name = actual_name;
	parse_end_group (pst);
	/*----*/
	parse_begin_group (pst, NULL, name, FALSE);
	name = CURRENT_GROUP->name;
	parse_begin_drawable (pst, 1, experience_image_class);
	experience_drawable_set_dont_inherit (CURRENT_DRAWABLE, TRUE);
	PST->gap_group[0] = CURRENT_GROUP;
	PST->gap_image[0] = CURRENT_IMAGE;
	parse_end_drawable (pst);
	parse_end_group (pst);
	
	parse_begin_group (pst, NULL, name, FALSE);
	parse_begin_drawable (pst, 1, experience_image_class);
	experience_drawable_set_dont_inherit (CURRENT_DRAWABLE, TRUE);
	PST->gap_group[1] = CURRENT_GROUP;
	PST->gap_image[1] = CURRENT_IMAGE;
	parse_end_drawable (pst);
	parse_end_group (pst);
	
	parse_begin_group (pst, NULL, name, FALSE);
	parse_begin_drawable (pst, 1, experience_image_class);
	experience_drawable_set_dont_inherit (CURRENT_DRAWABLE, TRUE);
	PST->gap_group[2] = CURRENT_GROUP;
	PST->gap_image[2] = CURRENT_IMAGE;
	parse_end_drawable (pst);
	parse_end_group (pst);
	
	/*----*/
	
	parse_begin_group(pst, actual_name, NULL, TRUE);
	parse_begin_drawable (pst, 2, experience_image_class);
	parse_end_drawable (pst); parse_begin_drawable (pst, 1, experience_image_class);
}

top_image_defs:	  top_image_def
				| top_image_def top_image_defs

top_image_def:	  overlay_def
				| match_def
				| gap_def
				| image_property
				| drawable_property
				| group_property
				  //function is here because of backward compatibilty
				| FUNCTION_LITERAL   '=' bgw_functions			{ experience_match_add_functions (&CURRENT_GROUP->match, $3); }
				| LINE_WIDTH_LITERAL '=' uint					{ experience_group_set_line_width (CURRENT_GROUP, $3); }
				| RECOLORABLE_LITERAL '=' BOOLEAN			/* ignore recolorable = boolean. This never did anything, but is often set in themes. */

/* icons */

icons_def:		  icons_begin icon_state_defs  '}'	{ parse_end_icons (pst); }
icons_begin:	  ICONS_LITERAL '{'					{ parse_begin_icons (pst); }

icon_state_defs:  icon_state_def
				| icon_state_def icon_state_defs
icon_state_def:		  icon_state_begin '{' all_filters '}'	{ parse_end_icons_state (pst); }
icon_state_begin:	  STATE_IDENTIFIER							{ parse_begin_icons_state (pst, $1); }

/* group properties */
group_property:	  CLEAR_AREA_LITERAL '=' BOOLEAN		{ experience_group_set_clear_area  (CURRENT_GROUP, $3); }

/* match definitions */

//function is not in here, because of backward compatibility ... (hline, vline vs. line)
match_def:	  STATE_LITERAL          '=' states				{ experience_match_add_states          (&CURRENT_GROUP->match, $3); }
			| DETAIL_LITERAL         '=' details			/* needs to be diffrent */
			| HAS_DEFAULT_LITERAL    '=' has_default
			| HAS_FOCUS_LITERAL      '=' has_focus
			| SHADOW_LITERAL         '=' shadows			{ experience_match_add_shadows         (&CURRENT_GROUP->match, $3); }
			| ARROW_DIRECTION_LITERAL'=' arrow_directions	{ experience_match_add_arrow_directions(&CURRENT_GROUP->match, $3); }
			| ORIENTATION_LITERAL    '=' orientations		{ experience_match_add_orientations    (&CURRENT_GROUP->match, $3); }
			| GAP_SIDE_LITERAL       '=' positions			{ experience_match_add_gap_sides       (&CURRENT_GROUP->match, $3); }
			| EXPANDER_STYLE_LITERAL '=' expander_styles	{ experience_match_add_expander_styles (&CURRENT_GROUP->match, $3); }
			| WINDOW_EDGE_LITERAL    '=' window_edges		{ experience_match_add_window_edges    (&CURRENT_GROUP->match, $3); }

/*--*/
details:		  detail
				| detail ',' details
detail:			  STRING										{ experience_match_add_detail (&CURRENT_GROUP->match, $1); }
/*--*/
states:			  STATE_IDENTIFIER								{ $$ = 1 << $1; }
				| STATE_IDENTIFIER ',' states					{ $$ = 1 << $1 | $3; }
/*--*/
function:		  FUNCTION_IDENTIFIER						{ $$ = $1; }
				| SHADOW_LITERAL							{ $$ = FUNCTION_SHADOW; }

changed_function:	  BOX_GAP_LITERAL					{ $$ = FUNCTION_BOX_GAP; }
					| SHADOW_GAP_LITERAL				{ $$ = FUNCTION_SHADOW_GAP; }

new_function:		  NEW_FUNCTION_IDENTIFIER			{ $$ = $1; }

all_functions:		  function								{ $$ = $1; }
					| new_function							{ $$ = $1; }
					| changed_function						{ $$ = $1; }
					| function ',' all_functions			{ $$ = $1 | $3; }
					| changed_function ',' all_functions	{ $$ = $1 | $3; }
					| new_function ',' all_functions		{ $$ = $1 | $3; }
/*--*/
shadows:		  shadow								{ $$ = 1 << $1; }
				| shadow ',' shadows					{ $$ = 1 << $1 | $3; }
shadow:			  SHADOW_IDENTIFIER						{ $$ = $1; }
				| NONE_LITERAL							{ $$ = GTK_SHADOW_NONE; }
/*--*/
arrow_directions: arrow_direction						{ $$ = $1; }
				| arrow_direction ',' arrow_directions	{ $$ = $1 | $3; }
arrow_direction:  ARROW_DIRECTION_IDENTIFIER	{ $$ = 1 << $1; }
				| LEFT_RIGHT			{ $$ = 1 << ($1 + GTK_ARROW_LEFT); }
/*--*/
orientations:	 ORIENTATION_IDENTIFIER							{ $$ = 1 << $1; }
				| ORIENTATION_IDENTIFIER ',' orientations		{ $$ = 1 << $1 | $3; }
/*--*/
positions:		  position							{ $$ = 1 << $1; }
				| position ',' positions				{ $$ = 1 << $1 | $3; }
position:	  POSITION_IDENTIFIER		{ $$ = $1; }
				| LEFT_RIGHT			{ $$ = $1 + GTK_POS_LEFT; }
				| NONE_LITERAL		{ $$ = POS_NONE; }
/*--*/
expander_styles:  EXPANDER_STYLE_IDENTIFIER						{ $$ = 1 << $1; }
				| EXPANDER_STYLE_IDENTIFIER ',' expander_styles	{ $$ = 1 << $1 | $3; }
/*--*/
window_edges:	  WINDOW_EDGE_IDENTIFIER						{ $$ = 1 << $1; }
				| WINDOW_EDGE_IDENTIFIER ',' window_edges		{ $$ = 1 << $1 | $3; }

/* image properties */
drawable_property: 	  padding_def
					| all_filter
					| DONT_INHERIT_LITERAL '=' BOOLEAN	{ experience_drawable_set_dont_inherit (CURRENT_DRAWABLE, $3); }
					| DONT_DRAW_LITERAL    '=' BOOLEAN	{ experience_drawable_set_dont_draw (CURRENT_DRAWABLE, $3); }
image_property:		  position_def
					| border_def
					| repeat_def
					| inner_padding_def
					| size_def
					| file_def
					| draw_components_def
					| interpolation_type_def
					| DONT_CLIP_LITERAL '=' BOOLEAN						{ experience_image_set_dont_clip (CURRENT_IMAGE, $3); }
					| DRAW_ENTIRE_ONLY_LITERAL '=' BOOLEAN				{ experience_image_set_draw_entire_only (CURRENT_IMAGE, $3); }
					| ROUNDING_LITERAL '=' ROUNDING_TYPE_IDENTIFIER		{ experience_image_set_rounding (CURRENT_IMAGE, $3); }

fill_property:		  COLOR_LITERAL '=' dynamic_color	{ experience_fill_set_color (CURRENT_FILL, $3); }

position_def:	  POS_LITERAL   '=' pos_percent_calc	{ experience_image_set_pos (CURRENT_IMAGE, $3, ORIENTATION_VERTICAL | ORIENTATION_HORIZONTAL); }
				| X_POS_LITERAL '=' pos_percent_calc	{ experience_image_set_pos (CURRENT_IMAGE, $3, ORIENTATION_HORIZONTAL); }
				| Y_POS_LITERAL '=' pos_percent_calc	{ experience_image_set_pos (CURRENT_IMAGE, $3, ORIENTATION_VERTICAL); }
				
				| POS_LITERAL   '=' TILE_LITERAL
				| X_POS_LITERAL '=' TILE_LITERAL
				| Y_POS_LITERAL '=' TILE_LITERAL

				| POS_LITERAL   '=' STRETCH_LITERAL
				| X_POS_LITERAL '=' STRETCH_LITERAL
				| Y_POS_LITERAL '=' STRETCH_LITERAL
				| stretch_def

border_def:		  BORDER_LITERAL '=' '{' uint ',' uint ',' uint ',' uint '}'	{ experience_image_set_border (CURRENT_IMAGE, $4, $6, $8, $10); }
repeat_def:		  REPEAT_LITERAL '=' '{' uint ',' uint ',' uint ',' uint '}'	{ experience_image_set_repeat (CURRENT_IMAGE, $4, $6, $8, $10); }
inner_padding_def: INNER_PADDING_LITERAL '=' '{' int ',' int ',' int ',' int '}'	{ experience_image_set_inner_padding (CURRENT_IMAGE, $4, $6, $8, $10); }
size_def:	  WIDTH_LITERAL  '=' image_percent_calc		{ experience_image_set_width  (CURRENT_IMAGE, $3); }
			| HEIGHT_LITERAL '=' image_percent_calc		{ experience_image_set_height (CURRENT_IMAGE, $3); }

padding_def:	  PADDING_LITERAL '=' '{' int ',' int ',' int ',' int '}'		{ experience_drawable_set_padding (CURRENT_DRAWABLE, $4, $6, $8, $10); }
file_def:		  FILE_LITERAL '=' STRING
				  { experience_image_set_file (CURRENT_IMAGE, gtk_rc_find_pixmap_in_path(PST->settings, PST->scanner, $3)); }

draw_component:		  WINDOW_EDGE_IDENTIFIER				{ $$ = 1 << $1; }
					| DRAW_COMPONENTS_IDENTIFIER			{ $$ = $1; }
					| BORDER_LITERAL						{ $$ = COMPONENT_BORDER; }
					| CENTER_LITERAL						{ $$ = COMPONENT_CENTER; }

draw_components:	  draw_component						{ $$ = $1; }
					| draw_components ',' draw_component	{ $$ = $1 | $3; }
					| NONE_LITERAL					{ $$ = COMPONENT_NONE; }


draw_components_def:
				  DRAW_COMPONENTS_LITERAL '=' draw_components	{ experience_image_set_draw_components (CURRENT_IMAGE, $3); }

interpolation_type_def:
				  INTERPOLATION_TYPE_LITERAL '=' INTERPOLATION_TYPE_IDENTIFIER { experience_image_set_interp_type (CURRENT_IMAGE, $3); }
/* filter */

all_filters:			  all_filter
					| all_filter all_filters

all_filter:		  SATURATION_LITERAL '=' ufloat				{ experience_filter_set_saturation (CURRENT_FILTER, $3); }
					| OPACITY_LITERAL    '=' ufloat				{ experience_filter_set_opacity    (CURRENT_FILTER, $3); }
					| BRIGHTNESS_LITERAL '=' float				{ experience_filter_set_brightness (CURRENT_FILTER, $3); }
					| PIXELATE_LITERAL   '=' BOOLEAN			{ experience_filter_set_pixelate   (CURRENT_FILTER, $3); }
					| ROTATE_LITERAL     '=' ROTATE_IDENTIFIER		{ experience_filter_set_rotation   (CURRENT_FILTER, $3); }
					| ROTATE_LITERAL     '=' NONE_LITERAL			{ experience_filter_set_rotation   (CURRENT_FILTER,  0); }
					| MIRROR_LITERAL     '=' filter_mirrors
					|  RECOLOR_LITERAL color '=' dynamic_color	{ experience_filter_add_recolor_color (CURRENT_FILTER, $2, $4); }
					| RECOLOR_GRADIENT_LITERAL '{' dynamic_color '}' 			{ experience_filter_set_recolor_gradient (CURRENT_FILTER, $3, dynamic_color_black, dynamic_color_black); }
					| RECOLOR_GRADIENT_LITERAL '{' dynamic_color ',' dynamic_color '}' 	{ experience_filter_set_recolor_gradient (CURRENT_FILTER, $3, $5, dynamic_color_black); }
					| RECOLOR_GRADIENT_LITERAL '{' dynamic_color ',' dynamic_color ',' dynamic_color '}' { experience_filter_set_recolor_gradient (CURRENT_FILTER, $3, $5, $7); }

/* backward compatibility */

stretch_def:	  STRETCH_LITERAL '=' BOOLEAN
{
	pos_percent_tmp.widget = 0;
	pos_percent_tmp.pixel = 0;
	experience_image_set_pos (CURRENT_IMAGE, pos_percent_tmp, ORIENTATION_VERTICAL | ORIENTATION_HORIZONTAL);
	
	image_percent_tmp.widget = 1;
	image_percent_tmp.image = 0;
	image_percent_tmp.pixel = 0;
	experience_image_set_width  (CURRENT_IMAGE, image_percent_tmp);
	experience_image_set_height (CURRENT_IMAGE, image_percent_tmp);
}

/* overlay definitions */
overlay_def:	  overlay_file_def
				| overlay_stretch_def
				| overlay_border_def

overlay_file_def: OVERLAY_FILE_LITERAL '=' STRING				{ parse_end_drawable (pst); parse_begin_drawable (pst, 2, experience_image_class);
																  experience_image_set_file (CURRENT_IMAGE, gtk_rc_find_pixmap_in_path(PST->settings, PST->scanner, $3));
				                          						  parse_end_drawable (pst); parse_begin_drawable (pst, 1, experience_image_class); }
overlay_stretch_def: OVERLAY_STRETCH_LITERAL '=' BOOLEAN			{ parse_end_drawable (pst); parse_begin_drawable (pst, 2, experience_image_class);
																	  pos_percent_tmp.widget = 0;
																	  pos_percent_tmp.pixel = 0;
																	  
																	  experience_image_set_pos (CURRENT_IMAGE, pos_percent_tmp, ORIENTATION_VERTICAL | ORIENTATION_HORIZONTAL);
																	  
																	  image_percent_tmp.widget = 0;
																	  image_percent_tmp.image  = 0;
																	  image_percent_tmp.pixel  = 0;
																	  
																	  if ($3) {
																	  	image_percent_tmp.widget = 1;
																	  } else {
																	  	image_percent_tmp.image = 1;
																	  	experience_group_set_dont_clip(CURRENT_GROUP, FALSE);
																	  }
																	  experience_image_set_width  (CURRENT_IMAGE, image_percent_tmp);
																	  experience_image_set_height (CURRENT_IMAGE, image_percent_tmp);
																  
				                          							  parse_end_drawable (pst); parse_begin_drawable (pst, 1, experience_image_class);
				                          							}
overlay_border_def: OVERLAY_BORDER_LITERAL '=' '{' uint ',' uint ',' uint ',' uint '}'
																{ parse_end_drawable (pst); parse_begin_drawable (pst, 2, experience_image_class);
																  experience_image_set_border(CURRENT_IMAGE, $4, $6, $8, $10);
				                          						  parse_end_drawable (pst); parse_begin_drawable (pst, 1, experience_image_class); }

/* gap stuff */
gap_def:  GAP_FILE_IDENTIFIER	'=' STRING
{
	gchar * file = gtk_rc_find_pixmap_in_path (PST->settings, PST->scanner, $3);
	GdkPixbuf * pixbuf;
	GError *err = NULL;
	
	if (file != NULL) {
		experience_image_set_file (PST->gap_image[$1], file);
		if ($3 == 0) {
			/* get image size here. */
			pixbuf = gdk_pixbuf_new_from_file (file, &err);
	
			if (!pixbuf)
			{
				experience_warning ("Cannot load pixmap file %s: %s\n",
				                file, err->message);
				g_error_free (err);
			} else {
				experience_group_set_gap_size (PST->gap_group[$1], gdk_pixbuf_get_width (pixbuf), gdk_pixbuf_get_height (pixbuf));
				g_object_unref (pixbuf);
			}
		}
	}
}
		| GAP_BORDER_IDENTIFIER '=' '{' uint ',' uint ',' uint ',' uint '}'	{ experience_image_set_border (PST->gap_image[$1], $4, $6, $8, $10); }

/* old function emulation */
bgw_functions:	    bgw_function				{ $$ = $1; }
				| bgw_function ',' bgw_functions			{ $$ = $1 | $3; }
bgw_function:	  function									{ $$ = $1; }
				| HLINE_LITERAL									{ $$ = FUNCTION_LINE; experience_match_add_orientations (&CURRENT_GROUP->match, 1 << GTK_ORIENTATION_HORIZONTAL); }
				| VLINE_LITERAL									{ $$ = FUNCTION_LINE; experience_match_add_orientations (&CURRENT_GROUP->match, 1 << GTK_ORIENTATION_VERTICAL); }
				| SHADOW_GAP_LITERAL
				{
					$$ = FUNCTION_SHADOW;
					
					g_free (CURRENT_GROUP->parent_name);
					CURRENT_GROUP->parent_name = g_strdup("RESERVED_BACKWARD_COMPAT_GAP_BG");
					
					experience_image_set_draw_components (CURRENT_IMAGE, COMPONENT_BORDER_LEFT | COMPONENT_BORDER_RIGHT | COMPONENT_BORDER_BOTTOM);
					
					experience_match_add_functions (& PST->gap_group[0]->match, FUNCTION_SHADOW_GAP_START);
					experience_match_add_functions (& PST->gap_group[1]->match, FUNCTION_SHADOW_GAP);
					experience_match_add_functions (& PST->gap_group[2]->match, FUNCTION_SHADOW_GAP_END);
				}
				| BOX_GAP_LITERAL
				{
					$$ = FUNCTION_BOX;
					
					g_free (CURRENT_GROUP->parent_name);
					CURRENT_GROUP->parent_name = g_strdup("RESERVED_BACKWARD_COMPAT_GAP_BG");
					
					experience_match_add_functions (& PST->gap_group[0]->match, FUNCTION_BOX_GAP_START);
					experience_match_add_functions (& PST->gap_group[1]->match, FUNCTION_BOX_GAP);
					experience_match_add_functions (& PST->gap_group[2]->match, FUNCTION_BOX_GAP_END);
				}
/* other */
ufloat:	  FLOAT				{ $$ = $1; }
		| INT				{ $$ = $1; }
float:	'-' float			{ $$ = - $2; }
		| ufloat			{ $$ = $1; }
		| CENTER_LITERAL	{ $$ = 0; }

uint:	  INT				{ $$ = $1; }
int:	  uint				{ $$ = $1; }
		| '-' int			{ $$ = - $2; }

color:	  '{' FLOAT ',' FLOAT ',' FLOAT '}'		{ $$.red = CLAMP_COLOR ($2 * GDK_COLOR_MAX); $$.green = CLAMP_COLOR ($4 * GDK_COLOR_MAX); $$.red = CLAMP_COLOR ($6 * GDK_COLOR_MAX); }
		| '{' uint  ',' uint  ',' uint  '}'		{ $$.red = CLAMP_COLOR ($2); $$.green = CLAMP_COLOR ($4); $$.red = CLAMP_COLOR($6); }
		| STRING								{ if (!gdk_color_parse ($1, &$$)) { YYERROR; } }

has_default:	  BOOLEAN			{ experience_match_add_has_default (& CURRENT_GROUP->match, $1); }
				| has_default ',' BOOLEAN	{ experience_match_add_has_default (& CURRENT_GROUP->match, $3); }

has_focus:	  BOOLEAN				{ experience_match_add_has_focus (& CURRENT_GROUP->match, $1); }
			| has_focus ',' BOOLEAN	{ experience_match_add_has_focus (& CURRENT_GROUP->match, $3); }

dynamic_color:	color										{ $$.source = GET_SET_COLOR; $$.output_color = $1; }
		| STYLE_COLOR_IDENTIFIER '[' STATE_IDENTIFIER ']'	{ $$.source = GET_STYLE_COLOR; $$.output_style_color.color_array = $1; $$.output_style_color.state = $3; }


filter_mirror:	  ORIENTATION_IDENTIFIER			{ $$ = $1; }
				| NONE_LITERAL						{ $$ = 0; }

filter_mirrors:	  filter_mirror						{ experience_filter_add_mirror (CURRENT_FILTER, 1 << $1); }
				| filter_mirror ',' filter_mirrors	{ experience_filter_add_mirror (CURRENT_FILTER, 1 << $1); }

pos_percent_number:
				  float			{ $$.pixel = 0;  $$.widget = $1; }
				| float WS_LITERAL	{ $$.pixel = 0;  $$.widget = $1; }
				| int PX_LITERAL	{ $$.pixel = $1; $$.widget = 0; }
pos_percent_calc: pos_percent_number						{ $$ = $1; }
				| pos_percent_number '+' pos_percent_calc	{ $$.widget = $1.widget + $3.widget; $$.pixel = $1.pixel + $3.pixel; }
				| pos_percent_number '-' pos_percent_calc 	{ $$.widget = $1.widget - $3.widget; $$.pixel = $1.pixel - $3.pixel; }

image_percent_number:
				  float	WS_LITERAL	{ $$.pixel = 0;  $$.widget = $1; $$.image = 0; }
				| float	IS_LITERAL	{ $$.pixel = 0;  $$.widget = 0;  $$.image = $1; }
				| int PX_LITERAL	{ $$.pixel = $1; $$.widget = 0;  $$.image = 0;}
				
image_percent_calc: image_percent_number						{ $$ = $1; }
				| image_percent_number '+' image_percent_calc	{ $$.widget = $1.widget + $3.widget; $$.pixel = $1.pixel + $3.pixel; $$.image = $1.image + $3.image; }
				| image_percent_number '-' image_percent_calc	{ $$.widget = $1.widget - $3.widget; $$.pixel = $1.pixel - $3.pixel; $$.image = $1.image - $3.image; }
;

%%

#include <ctype.h>
#include <stdio.h>

int
experience_yylex (YYSTYPE *lvalp, parsestate* pst)
{
	if (pst->should_end)
		return 0;
	
	GTokenType token = g_scanner_get_next_token (pst->scanner);
	switch (token) {
		case G_TOKEN_INT:
			lvalp->v_int = g_scanner_cur_value (pst->scanner).v_int;
			return INT;
		case G_TOKEN_FLOAT:
			lvalp->v_float = g_scanner_cur_value (pst->scanner).v_float;
			return FLOAT;
		case G_TOKEN_STRING:
			lvalp->v_str = g_strdup (g_scanner_cur_value (pst->scanner).v_string);
			return STRING;
		default:
			if ((token >= OTHER_SYMBOLS_START) && (token <= OTHER_SYMBOLS_END))
			{
				lvalp->v_uint = other_symbols[token - OTHER_SYMBOLS_START].value;
				return other_symbols[token - OTHER_SYMBOLS_START].identifier;
			}
			
			return token;
  }
}

void
experience_read_gtkrcstyle (parsestate * pst)
{
	guint old_scope;
	guint i;
	GScannerConfig old_config;

	g_assert(pst);
	g_assert(pst->scanner);
	
	old_config = *pst->scanner->config;
	
	old_scope = g_scanner_set_scope(pst->scanner, experience_g_quark);

	pst->scanner->config->case_sensitive = FALSE;

	if (!g_scanner_lookup_symbol(pst->scanner, theme_symbols[0].name))
	{
		for (i = 0; i < G_N_ELEMENTS (theme_symbols); i++)
			g_scanner_scope_add_symbol (pst->scanner, experience_g_quark, theme_symbols[i].name, GINT_TO_POINTER(theme_symbols[i].token));
		for (i = 0; i < G_N_ELEMENTS (other_symbols); i++)
			g_scanner_scope_add_symbol (pst->scanner, experience_g_quark, other_symbols[i].name, GINT_TO_POINTER(OTHER_SYMBOLS_START + i));
	}
	
	pst->should_end = 0;
	
	if (experience_yyparse (pst)) {
		/* There was an error ... */
		
		if (pst->expected_token == G_TOKEN_NONE) {
			/* ... so make sure that the user gets an error message for the correct error. */
			pst->expected_token = G_TOKEN_IDENTIFIER;
		}
	}
	
	g_scanner_set_scope (pst->scanner, old_scope);
	*pst->scanner->config = old_config;
}

#include <stdio.h>

int
experience_yyerror (char*s)  /* Called by experience_yyparse on error */
{
/*  printf ("%s\n", s); */
  return 0;
}
