/* arch-tag: 2168ecf0-9abc-45d2-ad33-d396361cca1d */

/*  eXperience GTK engine: utils.c
 *  
 *  Copyright (C) 2004-2005  Benjamin Berg <benjamin@sipsolutions.net>
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <gdk-pixbuf/gdk-pixbuf.h>

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <math.h>
#include <string.h>

#include "utils.h"
#include "experience.h"

gint
experience_round (eXperienceRoundingMethod method, gfloat value)
{
	switch (method) {
		case ROUND_CEIL:
			return (gint) ceilf (value);
			break;
		case ROUND_FLOOR:
			return (gint) floorf (value);
			break;
		case ROUND_NORMAL:
			return (gint) floorf (value + 0.5);
			break;
		case ROUND_TO_ZERO:
				return (gint) value;
	}
	g_assert_not_reached();
	return 0;
}

GdkPixbuf *
experience_gdk_pixbuf_scale_or_ref (GdkPixbuf * pixbuf, gint width, gint height, GdkInterpType interp_type)
{
	if ((width == 0) && (height == 0)) {
		g_object_ref (pixbuf);
		return pixbuf;
	} else {
		return gdk_pixbuf_scale_simple (pixbuf, width, height, interp_type);
	}
}

/*----------*/

GdkPixbuf *
experience_rotate (GdkPixbuf * pixbuf, eXperienceRotate rotation)
{
	GdkPixbuf * target = NULL;
	guint x, y, rowstride, height, width;
	guint target_rowstride;
	guchar *line, *pixel, *target_line, *target_pixel;
	guint pixel_size;
	
	g_assert (pixbuf != NULL);
	
	width  = gdk_pixbuf_get_width  (pixbuf);
	height = gdk_pixbuf_get_height (pixbuf);
	rowstride = gdk_pixbuf_get_rowstride (pixbuf);
	
	pixel_size = gdk_pixbuf_get_has_alpha (pixbuf) ? PIXEL_SIZE_WITH_ALPHA : PIXEL_SIZE;
	
	target = pixbuf;
	
	if (rotation == ROTATE_AROUND) {
		target = gdk_pixbuf_new (GDK_COLORSPACE_RGB, gdk_pixbuf_get_has_alpha (pixbuf), 8,
		                         width, height);
		
		target_rowstride = gdk_pixbuf_get_rowstride (target);
		
		line = gdk_pixbuf_get_pixels (pixbuf);
		
		target_line = gdk_pixbuf_get_pixels (target);
		target_line = target_line + (height - 1) * target_rowstride;
		
		for (y = 0; y < height; y++) {
			pixel = line;
			target_pixel = target_line + (width - 1) * pixel_size;
			
			for (x = 0; x < width; x++) {
				g_memmove (target_pixel, pixel, pixel_size);
				
				pixel += pixel_size;
				target_pixel -= pixel_size;
			}
			line += rowstride;
			target_line -= target_rowstride;
		}
		
		g_object_unref (pixbuf);
	} else if (rotation == ROTATE_CW) {
		target = gdk_pixbuf_new (GDK_COLORSPACE_RGB, gdk_pixbuf_get_has_alpha (pixbuf), 8,
		                         height, width);

		target_rowstride = gdk_pixbuf_get_rowstride (target);
		
		line = gdk_pixbuf_get_pixels (pixbuf);
		
		target_line = gdk_pixbuf_get_pixels (target);
		target_line = target_line + (height - 1) * pixel_size;
		
		for (y = 0; y < height; y++) {
			pixel = line;
			target_pixel = target_line;
			
			for (x = 0; x < width; x++) {
				g_memmove (target_pixel, pixel, pixel_size);
				
				pixel += pixel_size;
				target_pixel += target_rowstride;
			}
			line += rowstride;
			target_line -= pixel_size;
		}
		
		g_object_unref (pixbuf);
	} else if (rotation == ROTATE_CCW) {
		target = gdk_pixbuf_new (GDK_COLORSPACE_RGB, gdk_pixbuf_get_has_alpha (pixbuf), 8,
		                         gdk_pixbuf_get_height (pixbuf), gdk_pixbuf_get_width (pixbuf));

		target_rowstride = gdk_pixbuf_get_rowstride (target);
		
		line = gdk_pixbuf_get_pixels (pixbuf);
		
		target_line = gdk_pixbuf_get_pixels (target);
		target_line = target_line + (width - 1) * target_rowstride;
		
		for (y = 0; y < height; y++) {
			pixel = line;
			target_pixel = target_line;
			
			for (x = 0; x < width; x++) {
				g_memmove (target_pixel, pixel, pixel_size);
				
				pixel += pixel_size;
				target_pixel -= target_rowstride;
			}
			line += rowstride;
			target_line += pixel_size;
		}
		
		g_object_unref (pixbuf);
	}
	
	return target;
}

GdkPixbuf *
experience_mirror (GdkPixbuf * pixbuf, eXperienceRotate rotation)
{
	GdkPixbuf * source, * dest;
	guint x, y, rowstride, height, width;
	guint dest_rowstride;
	guchar *line, *pixel, *dest_line, *dest_pixel;
	guint pixel_size;
	
	g_assert (pixbuf != NULL);
	
	source = pixbuf;
	
	width  = gdk_pixbuf_get_width  (source);
	height = gdk_pixbuf_get_height (source);
	
	pixel_size = gdk_pixbuf_get_has_alpha (source) ? PIXEL_SIZE_WITH_ALPHA : PIXEL_SIZE;
	
	if (rotation & ORIENTATION_HORIZONTAL) {
		rowstride = gdk_pixbuf_get_rowstride (source);
		
		dest = gdk_pixbuf_new (GDK_COLORSPACE_RGB, gdk_pixbuf_get_has_alpha (source), 8,
		                         width, height);
		dest_rowstride = gdk_pixbuf_get_rowstride (source);
		
		line = gdk_pixbuf_get_pixels (source);
		dest_line = gdk_pixbuf_get_pixels (dest);
		
		for (y = 0; y < height; y++) {
			pixel = line;
			dest_pixel = dest_line + (width - 1) * pixel_size;
			
			for (x = 0; x < width; x++) {
				g_memmove (dest_pixel, pixel, pixel_size);
				
				pixel += pixel_size;
				dest_pixel -= pixel_size;
			}
			
			line += rowstride;
			dest_line += dest_rowstride;
		}
		
		g_object_unref (source);
		source = dest;
	}
	
	if (rotation & ORIENTATION_VERTICAL) {
		rowstride = gdk_pixbuf_get_rowstride (source);
		
		dest = gdk_pixbuf_new (GDK_COLORSPACE_RGB, gdk_pixbuf_get_has_alpha (source), 8,
		                       width, height);
		dest_rowstride = gdk_pixbuf_get_rowstride (source);
		
		line = gdk_pixbuf_get_pixels (source);
		dest_line = gdk_pixbuf_get_pixels (dest);
		dest_line += (height - 1) * dest_rowstride;
		
		for (y = 0; y < height; y++) {
			pixel = line;
			dest_pixel = dest_line;
			
			for (x = 0; x < width; x++) {
				g_memmove (dest_pixel, pixel, pixel_size);
				
				pixel += pixel_size;
				dest_pixel += pixel_size;
			}
			
			line += rowstride;
			dest_line -= dest_rowstride;
		}
		
		g_object_unref (source);
		source = dest;
	}
	
	return source;
}

GdkPixbuf *
experience_change_pixbuf_opacity (GdkPixbuf * pixbuf, gfloat opacity)
{
	GdkPixbuf * dest;
	
	guchar *line, *pixel;
	guint x, y, rowstride, height, width;
	
	g_assert (pixbuf != NULL);
	g_assert (GDK_IS_PIXBUF (pixbuf));
	
	if (gdk_pixbuf_get_has_alpha(pixbuf) == TRUE) {
		dest = pixbuf;
	} else {
		dest = gdk_pixbuf_add_alpha (pixbuf, FALSE, 0, 0, 0);
		g_object_unref (pixbuf);
	}
	
	/* make sure that opacity_percent has a sane value. */
	if (opacity >= 1.0)
		return dest;
	if (opacity < 0)
		opacity = 0; /* filling would do fine here ... */
	
	width  = gdk_pixbuf_get_width  (dest);
	height = gdk_pixbuf_get_height (dest);
	rowstride = gdk_pixbuf_get_rowstride (dest);

	line = gdk_pixbuf_get_pixels (dest);

	for (y = 0; y < height; y++) {
		pixel = line;
		for (x = 0; x < width; x++) {
			pixel[ALPHA_OFFSET] = pixel[ALPHA_OFFSET] * opacity;
			pixel += PIXEL_SIZE_WITH_ALPHA;
		}
		line += rowstride;
	}
	
	return dest;
}


void
experience_set_pixbuf_brightness (GdkPixbuf * pixbuf, gfloat level)
{
	guchar *pixel, *line;
	guint x, y, height, width;
	guint rowstride, bytes_per_pixel;
	gboolean has_alpha;

	g_assert (pixbuf != NULL);
	g_assert (GDK_IS_PIXBUF (pixbuf));
	g_assert (gdk_pixbuf_get_pixels (pixbuf) != NULL);

	if (level == 0.0)
		return;
	if (level < -1) level = -1;
	if (level > 1)  level = 1;

	width  = gdk_pixbuf_get_width (pixbuf);
	height = gdk_pixbuf_get_height (pixbuf);
	rowstride = gdk_pixbuf_get_rowstride (pixbuf);
	has_alpha = gdk_pixbuf_get_has_alpha (pixbuf);

	if (has_alpha)
		bytes_per_pixel = PIXEL_SIZE_WITH_ALPHA;
	else
		bytes_per_pixel = PIXEL_SIZE;

	line = gdk_pixbuf_get_pixels (pixbuf);

	for (y = 0; y < height; y++)
	{
		pixel = line;
		for (x = 0; x < width; x++)
		{
			pixel[RED_OFFSET] = CLAMP_UCHAR (pixel[RED_OFFSET] + level * 255);
			pixel[GREEN_OFFSET] = CLAMP_UCHAR (pixel[GREEN_OFFSET] + level * 255);
			pixel[BLUE_OFFSET] = CLAMP_UCHAR (pixel[BLUE_OFFSET] + level * 255);
			pixel += bytes_per_pixel;
		}
		line += rowstride;
	}
}


/*###########*/

void
experience_gdk_rectangle_union (GdkRectangle * src1, GdkRectangle * src2, GdkRectangle * dest)
{
	g_assert (src1 != NULL);
	g_assert (src2 != NULL);
	g_assert (dest != NULL);
	
	if ((src1->width <= 0) || (src1->height <= 0)) {
		*dest = *src2;
	} else if ((src2->width <= 0) || (src2->height <= 0)) {
		*dest = *src1;
	} else {
		gdk_rectangle_union (src1, src2, dest);
	}
}

guint
hash_mem (gpointer start, guint count, guint init)
{
	guint result = init;
	const char * p;
	
	for (p = start; (gpointer) p < (gpointer) (start + count); p++) {
		result = (result << 5) - result + *p;
	}
	
	return result;
}
