/* GStreamer
 * Copyright (C) <2001> Steve Baker <stevebaker_org@yahoo.co.uk>
 * Copyright (C) 2002, 2003 Andy Wingo <wingo at pobox dot com>
 *
 * int2float.c
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <gstint2float.h>
#include <gst/audio/audio.h>

/* elementfactory information */
static GstElementDetails int2float_details = {
  "Integer to Float effect",
  "Filter/Converter/Audio",
  "Convert from integer to floating point audio data",
  "Steve Baker <stevebaker_org@yahoo.co.uk>, Andy Wingo <wingo at pobox dot com>"
};

#define GST_INT2FLOAT_DEFAULT_BUFFER_FRAMES 256

#if 0
#define DEBUG(...) g_message (__VA_ARGS__)
#else
#define DEBUG(...)
#endif

static GstStaticPadTemplate int2float_src_factory =
GST_STATIC_PAD_TEMPLATE (
  "src%d",
  GST_PAD_SRC,
  GST_PAD_REQUEST,
  GST_STATIC_CAPS (GST_AUDIO_FLOAT_STANDARD_PAD_TEMPLATE_CAPS)
);

static GstStaticPadTemplate int2float_sink_factory =
GST_STATIC_PAD_TEMPLATE (
  "sink",
  GST_PAD_SINK,
  GST_PAD_ALWAYS,
  GST_STATIC_CAPS ( "audio/x-raw-int, "
    "rate = (int) [ 1, MAX ], "
    "channels = (int) [ 1, MAX ], "
    "endianness = (int) BYTE_ORDER, "
    "width = (int) 16, "
    "depth = (int) 16, "
    "signed = (boolean) true, "
    "buffer-frames = (int) [ 1, MAX]"
  )
);

static void                 gst_int2float_class_init (GstInt2FloatClass *klass);
static void                 gst_int2float_base_init (GstInt2FloatClass *klass);
static void                 gst_int2float_init (GstInt2Float *this);

static GstPadLinkReturn gst_int2float_sink_link (GstPad *pad, const GstCaps *caps);
static GstPadLinkReturn gst_int2float_src_link (GstPad *pad, const GstCaps *caps);

static GstPad*              gst_int2float_request_new_pad (GstElement *element, GstPadTemplate *temp, const gchar *unused);
static void                 gst_int2float_chain_gint16 (GstPad *pad, GstData *_data);
static GstElementStateReturn gst_int2float_change_state (GstElement *element);
static GstCaps * gst_int2float_sink_getcaps (GstPad *pad);
static GstCaps * gst_int2float_src_getcaps (GstPad *pad);

static GstElementClass *parent_class = NULL;

GType
gst_int2float_get_type(void) {
  static GType int2float_type = 0;

  if (!int2float_type) {
    static const GTypeInfo int2float_info = {
      sizeof(GstInt2FloatClass),
      (GBaseInitFunc)gst_int2float_base_init,
      NULL,
      (GClassInitFunc)gst_int2float_class_init,
      NULL,
      NULL,
      sizeof(GstInt2Float),
      0,
      (GInstanceInitFunc)gst_int2float_init,
    };
    int2float_type = g_type_register_static(GST_TYPE_ELEMENT, "GstInt2Float", &int2float_info, 0);
  }
  return int2float_type;
}

static void
gst_int2float_base_init (GstInt2FloatClass *klass)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (klass);

  gst_element_class_set_details (gstelement_class, &int2float_details);
  gst_element_class_add_pad_template (gstelement_class,
	gst_static_pad_template_get (&int2float_src_factory));
  gst_element_class_add_pad_template (gstelement_class,
	gst_static_pad_template_get (&int2float_sink_factory));
}

static void
gst_int2float_class_init (GstInt2FloatClass *klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass*)klass;
  gstelement_class = (GstElementClass*)klass;

  parent_class = g_type_class_ref(GST_TYPE_ELEMENT);

  gstelement_class->request_new_pad = gst_int2float_request_new_pad;
  gstelement_class->change_state = gst_int2float_change_state;
}

static void
gst_int2float_init (GstInt2Float *this)
{
  this->sinkpad = gst_pad_new_from_template(
			gst_static_pad_template_get (&int2float_sink_factory),"sink");

  gst_element_add_pad(GST_ELEMENT(this),this->sinkpad);
  gst_pad_set_chain_function(this->sinkpad,gst_int2float_chain_gint16);
  
  gst_pad_set_link_function(this->sinkpad, gst_int2float_sink_link);
  gst_pad_set_getcaps_function (this->sinkpad, gst_int2float_sink_getcaps);

  this->numsrcpads = 0;
  this->srcpads = NULL;
  
  this->channels = 0;
}

static GstCaps *
gst_int2float_get_allowed_srccaps (GstInt2Float *this, GstPad *ignorepad)
{
  GSList *pads;
  GstCaps *caps;
  GstCaps *allowed_caps;
  GstCaps *icaps;
  GstPad *srcpad;

  GST_DEBUG ("here");

  pads = this->srcpads;
  caps = gst_caps_from_string (GST_AUDIO_FLOAT_STANDARD_PAD_TEMPLATE_CAPS);
  while (pads) {
    srcpad = GST_PAD (pads->data);
    pads = g_slist_next (pads);
    if (srcpad == ignorepad) continue;

    allowed_caps = gst_pad_get_allowed_caps(srcpad);
    icaps = gst_caps_intersect (caps, allowed_caps);
    gst_caps_free (allowed_caps);
    gst_caps_free (caps);
    caps = icaps;

    pads = g_slist_next (pads);
  }

  GST_DEBUG ("allowed srcaps: %s", gst_caps_to_string (caps));

  return caps;
}

static void
gst_int2float_caps_remove_format (GstCaps *caps, const char *name)
{
  int i;
  GstStructure *structure;

  GST_DEBUG ("here");

  for(i=0;i<gst_caps_get_size(caps);i++){
    structure = gst_caps_get_structure(caps, i);
    gst_structure_set_name (structure, name);
    gst_structure_remove_field (structure, "width");
    gst_structure_remove_field (structure, "channels");
    gst_structure_remove_field (structure, "endianness");
    gst_structure_remove_field (structure, "depth");
    gst_structure_remove_field (structure, "signed");
    //gst_structure_remove_field (structure, "buffer-frames");
  }
}

static GstCaps *
gst_int2float_sink_getcaps (GstPad *pad)
{
  GstInt2Float *this;
  GstCaps *caps;
  GstCaps *icaps;
  int i;
  GstStructure *structure;

  this = GST_INT2FLOAT (gst_pad_get_parent (pad));

  caps = gst_int2float_get_allowed_srccaps (this, NULL);
  gst_int2float_caps_remove_format (caps, "audio/x-raw-int");

  GST_DEBUG ("allowed caps, no format %s", gst_caps_to_string (caps));

  if(this->numsrcpads > 0) {
    for(i=0;i<gst_caps_get_size(caps);i++){
      structure = gst_caps_get_structure(caps, i);
      gst_structure_set (structure, "channels", G_TYPE_INT,
          this->numsrcpads, NULL);
    }
  }

  GST_DEBUG ("allowed caps, no format %s", gst_caps_to_string (caps));
  GST_DEBUG ("pad template caps %s", gst_caps_to_string (gst_pad_get_pad_template_caps(pad)));

  icaps = gst_caps_intersect (caps, gst_pad_get_pad_template_caps(pad));
  gst_caps_free (caps);

  GST_DEBUG ("icaps %s", gst_caps_to_string (icaps));

  return icaps;
}

static GstCaps *
gst_int2float_src_getcaps (GstPad *pad)
{
  GstInt2Float *this;
  GstCaps *sinkcaps;
  GstCaps *srccaps;
  GstCaps *icaps;
  GstCaps *caps;
  int i;
  GstStructure *structure;

  this = GST_INT2FLOAT (gst_pad_get_parent (pad));

  GST_DEBUG ("here");

  srccaps = gst_int2float_get_allowed_srccaps (this, pad);
  gst_int2float_caps_remove_format (srccaps, "audio/x-raw-float");

  sinkcaps = gst_pad_get_allowed_caps (this->sinkpad);
  gst_int2float_caps_remove_format (sinkcaps, "audio/x-raw-float");

  icaps = gst_caps_intersect (srccaps, sinkcaps);
  gst_caps_free (sinkcaps);
  gst_caps_free (srccaps);

  for(i=0;i<gst_caps_get_size(icaps);i++){
    structure = gst_caps_get_structure(icaps, i);
    gst_structure_set (structure, "channels", G_TYPE_INT, 1, NULL);
  }

  caps = gst_caps_intersect (icaps, gst_pad_get_pad_template_caps(pad));
  gst_caps_free (icaps);

  return caps;
}

static GstPadLinkReturn
gst_int2float_sink_link (GstPad *pad, const GstCaps *caps)
{
  GstInt2Float *this;
  GstStructure *structure;
  GstCaps *fcaps;
  GstPad *srcpad;
  int rate;
  int channels;
  int buffer_frames;
  GSList *pads;
  GstPadLinkReturn ret;

  GST_DEBUG ("here");

  this = GST_INT2FLOAT (GST_PAD_PARENT (pad));

  structure = gst_caps_get_structure (caps, 0);

  gst_structure_get_int (structure, "rate", &rate);
  gst_structure_get_int (structure, "channels", &channels);
  gst_structure_get_int (structure, "buffer-frames", &buffer_frames);

  fcaps = gst_caps_from_string (GST_AUDIO_FLOAT_STANDARD_PAD_TEMPLATE_CAPS);
  gst_caps_set_simple (fcaps,
      "rate", G_TYPE_INT, rate,
      "buffer-frames", G_TYPE_INT, buffer_frames,
      NULL);

  pads = this->srcpads;
  while (pads) {
    srcpad = GST_PAD (pads->data);

    ret = gst_pad_try_set_caps (srcpad, fcaps);
    if(GST_PAD_LINK_FAILED (ret)) {
      /* FIXME we should revert the other changes */
      return ret;
    }

    pads = g_slist_next (pads);
  }

  /* ok, it worked */

  this->rate = rate;
  //this->channels = channels;
  this->buffer_frames = buffer_frames;
  this->channels = this->numsrcpads;

  return GST_PAD_LINK_OK;
}

static GstPadLinkReturn
gst_int2float_src_link (GstPad *pad, const GstCaps *caps)
{
  GstInt2Float *this;
  GstStructure *structure;
  GstCaps *icaps;
  GstPad *srcpad;
  int rate;
  int channels;
  int buffer_frames;
  GSList *pads;
  GstPadLinkReturn ret;

  GST_DEBUG ("here");

  this = GST_INT2FLOAT (GST_PAD_PARENT (pad));

  structure = gst_caps_get_structure (caps, 0);

  gst_structure_get_int (structure, "rate", &rate);
  gst_structure_get_int (structure, "channels", &channels);
  gst_structure_get_int (structure, "buffer-frames", &buffer_frames);

  icaps = gst_caps_copy (gst_pad_get_pad_template_caps (this->sinkpad));
  gst_caps_set_simple (icaps,
      "rate", G_TYPE_INT, rate,
      "buffer-frames", G_TYPE_INT, buffer_frames,
      "channels", G_TYPE_INT, this->numsrcpads,
      NULL);
  ret = gst_pad_try_set_caps (this->sinkpad, icaps);
  if(GST_PAD_LINK_FAILED (ret)) {
    return ret;
  }

  pads = this->srcpads;
  while (pads) {
    srcpad = GST_PAD (pads->data);
    pads = g_slist_next (pads);
    if (srcpad == pad) continue;

    ret = gst_pad_try_set_caps (srcpad, caps);
    if(GST_PAD_LINK_FAILED (ret)) {
      /* FIXME we should revert the other changes */
      return ret;
    }
  }

  /* ok, it worked */

  this->rate = rate;
  //this->channels = channels;
  this->buffer_frames = buffer_frames;
  this->channels = this->numsrcpads;

  return GST_PAD_LINK_OK;
}

static GstPad*
gst_int2float_request_new_pad (GstElement *element, GstPadTemplate *templ, const gchar *unused) 
{
  gchar *name;
  GstPad *srcpad;
  GstInt2Float *this;

  this = GST_INT2FLOAT (element);
  
  if (GST_STATE (element) == GST_STATE_PLAYING) {
    g_warning ("new pads can't be requested from gstint2float while it is PLAYING");
    return NULL;
  }

  name = g_strdup_printf ("src%d", this->numsrcpads);
  srcpad = gst_pad_new_from_template (templ, name);
  g_free (name);
  gst_element_add_pad (GST_ELEMENT (this), srcpad);
  gst_pad_set_link_function (srcpad, gst_int2float_src_link);
  gst_pad_set_getcaps_function (srcpad, gst_int2float_src_getcaps);
  
  this->srcpads = g_slist_append (this->srcpads, srcpad);
  this->numsrcpads++;
  
  return srcpad;
}

static inline void
fetch_buffers (GstInt2Float *this) 
{
  int i;

  g_return_if_fail (this->frames_to_write == 0);

  for (i=0; i<this->numsrcpads; i++) {
    /* FIXME the 1024 is arbitrary */
    this->buffers[i] = gst_buffer_new_and_alloc (1024);
    this->data_out[i] = (gfloat*)GST_BUFFER_DATA (this->buffers[i]);
    GST_BUFFER_TIMESTAMP (this->buffers[i]) = this->offset * GST_SECOND / this->rate;
  }
  this->offset += this->buffer_frames;
  this->frames_to_write = this->buffer_frames;
}

static void
gst_int2float_chain_gint16 (GstPad *pad, GstData *_data)
{
  GstBuffer *buf_in = GST_BUFFER (_data);
  GstInt2Float *this;
  gint16 *data_in;
  gfloat **data_out;
  gint i, j;
  GSList *srcpads;
  GstBuffer **buffers;
  gint to_process, channels;
  
  this = GST_INT2FLOAT (GST_OBJECT_PARENT (pad));

  if (!this->channels) {
    GST_ELEMENT_ERROR (this, CORE, NEGOTIATION, (NULL),
                       ("format wasn't negotiated before chain function"));
    return;
  }
  buffers = this->buffers;
  data_out = this->data_out;
  channels = this->channels;

  DEBUG ("chaining a buffer with %d frames",
         GST_BUFFER_SIZE (buf_in) / channels / sizeof (gint16));

  if (!this->frames_to_write)
    fetch_buffers (this);

  to_process = this->leftover_frames;
  if (to_process) {
    DEBUG ("dealing with %d leftover frames", to_process);

    /* we are guaranteed that if some frames are left over, they can't fill a
       whole buffer */
    data_in = (gint16*)GST_BUFFER_DATA (this->in_buffer);
    data_in += GST_BUFFER_SIZE (this->in_buffer) / sizeof (gint16) - channels * to_process;
    for (i=0; i<this->numsrcpads; i++) {
      for (j=0; j<to_process; j++)
        data_out[i][j] = ((gfloat)data_in[(j*channels) + (i%channels)]) / 32767.0;
      data_out[i] += to_process;
    }
    this->frames_to_write -= to_process;
    gst_buffer_unref (this->in_buffer);
    this->in_buffer = NULL;
  }
  
  this->in_buffer = buf_in;
  this->leftover_frames = GST_BUFFER_SIZE (buf_in) / sizeof (gint16) / channels;
  data_in = (gint16*)GST_BUFFER_DATA (this->in_buffer);
  
  while (1) {
    to_process = MIN (this->leftover_frames, this->frames_to_write);
    DEBUG ("processing %d frames", to_process);

    for (i=0, srcpads=this->srcpads ; i<this->numsrcpads; i++) {
      for (j=0; j<to_process; j++){
        data_out[i][j] = ((gfloat)data_in[(j*channels) + (i%channels)]) / 32767.0;
      }
      data_out[i] += to_process;

      DEBUG ("if %d == %d then I'm pushing...", this->frames_to_write, to_process);
      if (this->frames_to_write == to_process)
        gst_pad_push (GST_INT2FLOAT_SRCPAD (srcpads), GST_DATA (buffers[i]));

      srcpads = srcpads->next;
    }

    if (this->leftover_frames != to_process) {
      this->leftover_frames -= to_process;
      data_in += channels * to_process;
      this->frames_to_write = 0;

      DEBUG ("%d frames remaining in input buffer", this->leftover_frames);

      if (this->leftover_frames < this->buffer_frames)
        return;
    } else {
      gst_buffer_unref (this->in_buffer);
      this->in_buffer = NULL;
      this->frames_to_write -= to_process;
      this->leftover_frames = 0;

      DEBUG ("finished with input buffer, %d frames needed to fill output. returning.",
             this->frames_to_write);
      return;
    }
    
    fetch_buffers (this);
  }
}

static GstElementStateReturn
gst_int2float_change_state (GstElement *element)
{
  GstInt2Float *this;

  this = GST_INT2FLOAT (element);
  
  switch (GST_STATE_TRANSITION (element)) {
  case GST_STATE_PAUSED_TO_PLAYING:
    this->buffers = g_new0 (GstBuffer*, this->numsrcpads);
    this->data_out = g_new0 (gfloat*, this->numsrcpads);
    break;
    
  case GST_STATE_PLAYING_TO_PAUSED:
    g_free (this->buffers);
    this->buffers = NULL;
    g_free (this->data_out);
    this->data_out = NULL;
    break;
    
  case GST_STATE_PAUSED_TO_READY:
    break;

  default:
    break;
  }

  if (GST_ELEMENT_CLASS (parent_class)->change_state) 
    return GST_ELEMENT_CLASS (parent_class)->change_state (element);

  return GST_STATE_SUCCESS;

}
