/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  gpa-transport-selector.c: A print transport selector
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors :
 *    Lauris Kaplinski <lauris@ximian.com>
 *    Chema Celorio <chema@ximian.com>
 *
 *  Copyright (C) 2000-2003 Ximian, Inc. 
 *
 */

#include "config.h"

#include <string.h>
#include <gtk/gtk.h>

#include "gnome-print-i18n.h"
#include "gpa-transport-selector.h"
#include <libgnomeprint/private/gnome-print-config-private.h>
#include <libgnomeprint/gnome-print-config.h>
#include <libgnomeprint/private/gpa-node-private.h>
#include <libgnomeprint/private/gpa-key.h>

static void gpa_transport_selector_class_init (GPATransportSelectorClass *klass);
static void gpa_transport_selector_init (GPATransportSelector *selector);
static void gpa_transport_selector_finalize (GObject *object);
static gint gpa_transport_selector_construct (GPAWidget *widget);

static void gpa_transport_selector_file_button_clicked_cb       (GtkButton *button, GPATransportSelector *ts);
static void gpa_transport_selector_custom_entry_changed_cb (GtkEntry *entry, GPATransportSelector *ts);
static void gpa_transport_selector_node_modified_cb (GPANode *node, guint flags, GPATransportSelector *ts);

static GPAWidgetClass *parent_class;

GType
gpa_transport_selector_get_type (void)
{
	static GType type = 0;
	if (!type) {
		static const GTypeInfo info = {
			sizeof (GPATransportSelectorClass),
			NULL, NULL,
			(GClassInitFunc) gpa_transport_selector_class_init,
			NULL, NULL,
			sizeof (GPATransportSelector),
			0,
			(GInstanceInitFunc) gpa_transport_selector_init,
		};
		type = g_type_register_static (GPA_TYPE_WIDGET, "GPATransportSelector", &info, 0);
	}
	return type;
}

static void
gpa_transport_selector_class_init (GPATransportSelectorClass *klass)
{
	GObjectClass *object_class;
	GPAWidgetClass *gpa_class;

	object_class = (GObjectClass *) klass;
	gpa_class = (GPAWidgetClass *) klass;

	parent_class = gtk_type_class (GPA_TYPE_WIDGET);

	gpa_class->construct   = gpa_transport_selector_construct;
	object_class->finalize = gpa_transport_selector_finalize;
}

static void
gpa_transport_selector_init (GPATransportSelector *ts)
{
	GtkBox *hbox;

	hbox = (GtkBox *) gtk_hbox_new (FALSE, 4);

	ts->menu = gtk_option_menu_new ();
	ts->file_name   = g_strdup ("");
	ts->file_name_label   = gtk_label_new ("");
	ts->file_button   = gtk_button_new_from_stock (GTK_STOCK_SAVE_AS);
	ts->custom_entry = gtk_entry_new ();

	g_signal_connect (G_OBJECT (ts->file_button), "clicked", (GCallback)
			  gpa_transport_selector_file_button_clicked_cb, ts);
	g_signal_connect (G_OBJECT (ts->custom_entry), "changed", (GCallback)
			  gpa_transport_selector_custom_entry_changed_cb, ts);

	gtk_box_pack_start (hbox, ts->menu,         FALSE, FALSE, 0);
	gtk_box_pack_start (hbox, ts->file_button,   FALSE, FALSE, 0);
	gtk_box_pack_start (hbox, ts->file_name_label,  FALSE, FALSE, 0);
	gtk_box_pack_start (hbox, ts->custom_entry, FALSE, FALSE, 0);
	
	gtk_container_add (GTK_CONTAINER (ts), GTK_WIDGET (hbox));
	gtk_widget_show_all (GTK_WIDGET (hbox));
}

static void
gpa_transport_selector_disconnect (GPATransportSelector *ts)
{
	if (ts->handler) {
		g_signal_handler_disconnect (ts->node, ts->handler);
		ts->handler = 0;
	}
	
	if (ts->node) {
		gpa_node_unref (ts->node);
		ts->node = NULL;
	}
}

static void
gpa_transport_selector_finalize (GObject *object)
{
	GPATransportSelector *ts;

	ts = (GPATransportSelector *) object;

	if (ts->file_selector)
		gtk_widget_destroy (ts->file_selector);
	ts->file_selector = NULL;

	gpa_transport_selector_disconnect (ts);

	if (ts->handler_config)
		g_signal_handler_disconnect (ts->config, ts->handler_config);
	ts->handler_config = 0;
	ts->config = NULL;
	g_free (ts->file_name);
	ts->file_name = NULL;
	
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gpa_transport_selector_file_selector_destroyed_cb (GtkObject *object,
						   GPATransportSelector *ts)
{
	ts->file_selector = NULL;
}

static void
gpa_transport_selector_file_selected_cb (GtkWidget *button, GPATransportSelector *ts) 
{
	const gchar *selected_filename;
	gchar *filename;
	gsize bytes_read;
	gsize bytes_written;

	selected_filename = gtk_file_selection_get_filename 
		(GTK_FILE_SELECTION (ts->file_selector));
	
	filename = g_filename_to_utf8 (selected_filename, -1, 
                                       &bytes_read, &bytes_written,NULL);

	if (g_file_test (selected_filename, G_FILE_TEST_IS_DIR)) {
		GtkWidget *dialog;
		dialog = gtk_message_dialog_new 
			(GTK_WINDOW (ts->file_selector),
			 GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR,
			 GTK_BUTTONS_CLOSE,
			 _("The specified filename \"%s\" is an existing "
			   "directory."), filename);
		
		g_signal_connect_swapped (GTK_OBJECT (dialog), "response",
					  G_CALLBACK (gtk_widget_destroy),
					  GTK_OBJECT (dialog));
		gtk_widget_show (dialog);
		return;
	}

	if (g_file_test (selected_filename, G_FILE_TEST_EXISTS)) {
		GtkWidget *dialog;
		gint response;

		dialog = gtk_message_dialog_new 
			(GTK_WINDOW (ts->file_selector), 
			 GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION,
			 GTK_BUTTONS_YES_NO,
			 _("Should the file %s be overwritten?"), filename);
		
		response = gtk_dialog_run (GTK_DIALOG (dialog));

		gtk_widget_destroy (dialog);

		if (GTK_RESPONSE_YES != response)
			return;
	}
	/* FIXME: One of these should be enough... */
	gpa_node_set_path_value (ts->config, 
				 "Settings.Output.Job.FileName", 
				 filename);
	gpa_node_set_path_value (ts->config, 
				 "Settings.Transport.Backend.FileName", 
				 filename);
	gpa_node_set_path_value (ts->node, "FileName", filename);
	g_free (ts->file_name);
	ts->file_name = filename;
	gtk_label_set_text (GTK_LABEL (ts->file_name_label), ts->file_name);
}

static void
gpa_transport_selector_file_button_clicked_cb (GtkButton *button, GPATransportSelector *ts)
{
	gchar     *filename;
	gsize bytes_read;
	gsize bytes_written;
	
	/* Create the selector */
   
	ts->file_selector = gtk_file_selection_new 
		(_("Please specify the location and filename of the output file:"));
	
	filename = g_filename_from_utf8 (ts->file_name, -1,
					 &bytes_read, &bytes_written, NULL);
	if (filename != NULL) {
		gtk_file_selection_set_filename 
			(GTK_FILE_SELECTION(ts->file_selector),
			 filename);
		g_free (filename);
	}
   
	g_signal_connect 
		(GTK_OBJECT(GTK_FILE_SELECTION(ts->file_selector)->ok_button),
		 "clicked",
		 G_CALLBACK(gpa_transport_selector_file_selected_cb),
		 (gpointer) ts);
	g_signal_connect 
		(GTK_OBJECT(ts->file_selector),
		 "destroy",
		 G_CALLBACK(gpa_transport_selector_file_selector_destroyed_cb),
		 (gpointer) ts);
   			   
	/* Ensure that the dialog box is destroyed when the user clicks a button. */
	
	g_signal_connect_swapped 
		(GTK_OBJECT(GTK_FILE_SELECTION(ts->file_selector)
			    ->ok_button),
		 "clicked",
		 G_CALLBACK(gtk_widget_destroy), 
		 (gpointer) ts->file_selector); 
	
	g_signal_connect_swapped 
		(GTK_OBJECT(GTK_FILE_SELECTION(ts->file_selector)
			    ->cancel_button),
		 "clicked",
		 G_CALLBACK(gtk_widget_destroy),
		 (gpointer) ts->file_selector); 
    
	gtk_window_set_modal(GTK_WINDOW (ts->file_selector), TRUE);
	/* Display that dialog */
	gtk_widget_show (ts->file_selector);
	
}

static void
gpa_transport_selector_custom_entry_changed_cb (GtkEntry *entry, GPATransportSelector *ts)
{
	const guchar *text;

	if (ts->updating)
		return;

	text = gtk_entry_get_text (entry);
	ts->updating = TRUE;
	gpa_node_set_path_value (ts->node, "Command", text);
	ts->updating = FALSE;
}

static void
gpa_transport_selector_update_widgets (GPATransportSelector *ts)
{
	gchar *backend, *filename, *command;

	backend  = gpa_node_get_path_value (ts->config, "Settings.Transport.Backend");
	filename = gpa_node_get_path_value (ts->config, "Settings.Transport.Backend.FileName");
	command  = gpa_node_get_path_value (ts->config, "Settings.Transport.Backend.Command");
	
	gtk_widget_hide (ts->file_name_label);
	gtk_widget_hide (ts->file_button);
	gtk_widget_hide (ts->custom_entry);

	if (backend && !strcmp (backend, "file")) {
		ts->updating = TRUE;
		g_free (ts->file_name);
		ts->file_name = g_strdup (filename ? filename 
					  : "gnome-print.out");
		gtk_label_set_text (GTK_LABEL (ts->file_name_label), 
				    ts->file_name);
		ts->updating = FALSE;
		gtk_widget_show (ts->file_button);
		gtk_widget_show (ts->file_name_label);
	}
	
	if (backend && !strcmp (backend, "custom")) {
		ts->updating = TRUE;
		gtk_entry_set_text (GTK_ENTRY (ts->custom_entry), command ? command : "lpr %f");
		ts->updating = FALSE;
		gtk_widget_show (ts->custom_entry);
	}

	my_g_free (filename);
	my_g_free (command);
	my_g_free (backend);
}

static void
gpa_transport_selector_item_activate_cb (GtkWidget *item, GPATransportSelector *ts)
{
	GPANode *node;

	node =  g_object_get_data (G_OBJECT (item), "node");
	ts->updating = TRUE;
	gpa_node_set_value (ts->node, gpa_node_id (node));
	ts->updating = FALSE;

	gpa_transport_selector_update_widgets (ts);
}

static void
gpa_transport_selector_rebuild_menu (GPATransportSelector *ts)
{
	GtkWidget *menu, *item;
	GPANode *option, *child;
	guchar *def = NULL;
	gint pos = 0;
	gint sel = -1;

	menu = gtk_menu_new ();
	gtk_widget_show (menu);

	if (ts->node != NULL) {
		option = GPA_KEY (ts->node)->option;
		def = gpa_node_get_value (ts->node);
		
		child = gpa_node_get_child (option, NULL);
		while (child) {
			const guchar *id;
			guchar *name;
			
			name = gpa_node_get_value (child);
			item = gtk_menu_item_new_with_label (name);
			g_free (name);
			
			id = gpa_node_id (child);
			
			g_signal_connect 
				(G_OBJECT (item), "activate", 
				 (GCallback) 
				 gpa_transport_selector_item_activate_cb, ts);
			g_object_set_data_full 
				(G_OBJECT (item), "node", child, 
				 (GtkDestroyNotify) gpa_node_unref);
			gtk_widget_show (item);
			gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
			if (GPA_NODE_ID_COMPARE (child, def))
				sel = pos;
			
			pos++;
			child = gpa_node_get_child (option, child);
		}
		if (sel == -1) {
			g_warning ("rebuild_menu_cb, could not set value "
				   "of %s to %s",
				   gpa_node_id (option), def);
			sel = 0;
		}
	}
	if (pos < 1) {
		item = gtk_menu_item_new_with_label 
			(_("No options are defined"));
		gtk_widget_set_sensitive (item, FALSE);
		gtk_widget_show (item);
		gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
	}

	gtk_widget_show (menu);
	gtk_option_menu_set_menu (GTK_OPTION_MENU (ts->menu), menu);
	gtk_option_menu_set_history (GTK_OPTION_MENU (ts->menu), sel);
	if (def)
		g_free (def);

	gpa_transport_selector_update_widgets (ts);
}

static void
gpa_transport_selector_connect (GPATransportSelector *ts)
{
	ts->node = gpa_node_lookup (ts->config, "Settings.Transport.Backend");
	if (ts->node)
		ts->handler = g_signal_connect 
			(G_OBJECT (ts->node), 
			 "modified", (GCallback)
			 gpa_transport_selector_node_modified_cb, ts);
}

static void
gpa_transport_selector_config_modified_cb (GPANode *node, guint flags, GPATransportSelector *ts)
{
	gpa_transport_selector_disconnect (ts);
	gpa_transport_selector_connect (ts);
	gpa_transport_selector_rebuild_menu (ts);
}

static void
gpa_transport_selector_node_modified_cb (GPANode *node, guint flags, GPATransportSelector *ts)
{
	if (ts->updating)
		return;
	
	gpa_transport_selector_rebuild_menu (ts);
}

static gint
gpa_transport_selector_construct (GPAWidget *gpaw)
{
	GPATransportSelector *ts;

	ts = GPA_TRANSPORT_SELECTOR (gpaw);
	ts->config  = GNOME_PRINT_CONFIG_NODE (gpaw->config);
	ts->handler_config = g_signal_connect (G_OBJECT (ts->config), "modified", (GCallback)
					       gpa_transport_selector_config_modified_cb, ts);

	gpa_transport_selector_connect (ts);
	gpa_transport_selector_rebuild_menu (ts);

	return TRUE;
}

