/* Bug-buddy bug submitting program
 *
 * Copyright (C) 1999 - 2001 Jacob Berkman
 * Copyright 2000, 2001 Ximian, Inc.
 *
 * Author:  jacob berkman  <jacob@bug-buddy.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include "eds-buddy.h"

#include "bugzilla.h"
#include "bug-buddy.h"
#include "distribution.h"
#include "proccess.h"

#include <stdio.h>
#include <unistd.h>
#include <string.h>

#include <errno.h>

#include <gnome.h>
#include <libgnomeui/gnome-window-icon.h>
#include <glade/glade.h>
#include <glade/glade-build.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <libgnomecanvas/gnome-canvas-pixbuf.h>
#include <libgnome/libgnometypebuiltins.h>


#include <libxml/tree.h>
#include <libxml/parser.h>

#include <gconf/gconf-client.h>

#ifdef HAVE_NETWORKMANAGER
#include <libnm_glib.h>
#endif

#include <libsoup/soup.h>
#include <libsoup/soup-xmlrpc-message.h>

#include <sys/types.h>
#include <signal.h>

#define d(x)

#define USE_PROXY_KEY 	"/system/http_proxy/use_http_proxy"
#define PROXY_HOST_KEY	"/system/http_proxy/host"
#define PROXY_PORT_KEY	"/system/http_proxy/port"
#define USE_PROXY_AUTH	"/system/http_proxy/use_authentication"
#define PROXY_USER	"/system/http_proxy/authentication_user"
#define PROXY_PASSWORD	"/system/http_proxy/authentication_password"

static GOptionData gopt_data;
static int bug_count = 0;

static const GOptionEntry options[] = {
	{ "name",       '\0', 0, G_OPTION_ARG_STRING,   &gopt_data.name,        N_("Name of contact"),                    N_("NAME") },
	{ "email",      '\0', 0, G_OPTION_ARG_STRING,   &gopt_data.email,       N_("Email address of contact"),           N_("EMAIL") },
	{ "package",    '\0', 0, G_OPTION_ARG_STRING,   &gopt_data.package,     N_("Package containing the program"),     N_("PACKAGE") },
	{ "package-ver",'\0', 0, G_OPTION_ARG_STRING,   &gopt_data.package_ver, N_("Version of the package"),             N_("VERSION") },
	{ "appname",    '\0', 0, G_OPTION_ARG_FILENAME, &gopt_data.app_file,    N_("File name of crashed program"),       N_("FILE") },
	{ "pid",        '\0', 0, G_OPTION_ARG_INT,      &gopt_data.pid,         N_("PID of crashed program"),             N_("PID") },
	{ "core",       '\0', 0, G_OPTION_ARG_FILENAME, &gopt_data.core_file,   N_("Core file from program"),             N_("FILE") },
	{ "include",    '\0', 0, G_OPTION_ARG_FILENAME, &gopt_data.include_file,N_("Text file to include in the report"), N_("FILE") },
	{ "kill",       '\0', 0, G_OPTION_ARG_INT,      &gopt_data.kill,        N_("PID of the program to kill after the report"), N_("KILL") },
	{ NULL }
};

enum {
	NETWORK_CONNECTED,
	NETWORK_DISCONNECTED,
	NETWORK_UNKNOWN
};


static void
buddy_error (GtkWidget *parent, const char *msg, ...)
{
	GtkWidget *w;
	GtkDialog *d;
	gchar *s;
	va_list args;

	/* No va_list version of dialog_new, construct the string ourselves. */
	va_start (args, msg);
	s = g_strdup_vprintf (msg, args);
	va_end (args);

	w = gtk_message_dialog_new (GTK_WINDOW (parent),
				    0,
				    GTK_MESSAGE_ERROR,
				    GTK_BUTTONS_OK,
				    "%s",
				    s);
	d = GTK_DIALOG (w);
	gtk_dialog_set_default_response (d, GTK_RESPONSE_OK);
	gtk_dialog_run (d);
	gtk_widget_destroy (w);
	g_free (s);
}

static GnomeVersionInfo*
get_gnome_version_info (void)
{
	GnomeVersionInfo *version;
	xmlDoc *doc;
	char *xml_file;
	xmlNode *node;
	guchar *platform, *minor, *micro, *distributor, *date;

	version = g_new0 (GnomeVersionInfo, 1);

	xml_file = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_DATADIR,
					      "gnome-about/gnome-version.xml",
					      TRUE, NULL);
	if (!xml_file)
		return NULL;
	doc = xmlParseFile (xml_file);
	g_free (xml_file);

	if (!doc)
		return NULL;

	platform = minor = micro = distributor = NULL;
	
	for (node = xmlDocGetRootElement (doc)->children; node; node = node->next) {
		if (!strcmp ((char *)node->name, "platform"))
			platform = xmlNodeGetContent (node);
		else if (!strcmp ((char *)node->name, "minor"))
			minor = xmlNodeGetContent (node);
		else if (!strcmp ((char *)node->name, "micro"))
			micro = xmlNodeGetContent (node);
		else if (!strcmp ((char *)node->name, "distributor"))
			distributor = xmlNodeGetContent (node);
		else if (!strcmp ((char *)node->name, "date"))
			date = xmlNodeGetContent (node);
	}
	
	if (platform && minor && micro)
		version->gnome_platform = g_strdup_printf ("%s.%s.%s", platform, minor, micro);
  
	if (distributor && *distributor)
		version->gnome_distributor = g_strdup ((char *)distributor);
	
	if (date && *date)
		version->gnome_date = g_strdup ((char *)date);

	xmlFree (platform);
	xmlFree (minor);
	xmlFree (micro);
	xmlFree (distributor);
	xmlFree (date);
	
	xmlFreeDoc (doc);

	return version;
}

static gboolean
update_progress_bar (gpointer data)
{
	GtkProgressBar *pbar = GTK_PROGRESS_BAR (data);

	gtk_progress_bar_pulse (pbar);

	return TRUE;
}

void
save_email (const char *email)
{
	GConfClient *conf_client;

	conf_client = gconf_client_get_default ();
	gconf_client_set_string (conf_client, "/apps/bug-buddy/email_address", email, NULL);
	g_object_unref (conf_client);
}

void
link_callback (GtkLinkButton *button, gpointer user_data)
{
	const gchar *link = gtk_link_button_get_uri (button);

	if (gnome_vfs_url_show (link) != GNOME_VFS_OK) {
		char *text;

		text = g_markup_printf_escaped (_("Bug-buddy was unable to view the link \"%s\"\n"), link);
		buddy_error (NULL, text);
		g_free (text);
	}

	return;
}

static void
save_to_file (const gchar *filename, const gchar *text)
{
	FILE          *bugfile;

	if ((bugfile = fopen (filename, "w")) == NULL) {
		g_warning ("Unable to save document %s\n", filename);
		return;
	}

	fwrite (text, g_utf8_strlen (text, -1), 1, bugfile);

	fclose (bugfile);
}



static void
network_error (SoupMessage *msg, GladeXML *xml)
{

	GtkWidget *dialog;
	int res;
		
	gtk_widget_hide (glade_xml_get_widget (xml, "progressbar"));
	gtk_statusbar_push (GTK_STATUSBAR (glade_xml_get_widget (xml, "statusbar")),
		    	    DONE, _("Done"));


		
	dialog = gtk_message_dialog_new_with_markup (NULL,
						      GTK_DIALOG_MODAL,
						      GTK_MESSAGE_QUESTION,
						      GTK_BUTTONS_YES_NO,
						      "<span weight=\"bold\">Network Connection Error</span>\n"
						      "Maybe no Network Connection available.\n"
						      "Do you want to store this report until "
						      "a Network Connection is available?");
	res = gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);

	if (res == GTK_RESPONSE_YES) {
		gchar *dirname;
		gchar *filename;
		gchar *message_string;
	       
		dirname = g_strdup_printf ("%s/.gnome2/bug-buddy/pending_reports", g_get_home_dir ());
		if (!g_file_test (dirname, G_FILE_TEST_IS_DIR)) {
			g_mkdir_with_parents (dirname, 0755);
		}

		filename = g_strdup_printf ("%s/%ld", dirname, (long)time (NULL));
					
		message_string = soup_xmlrpc_message_to_string (SOUP_XMLRPC_MESSAGE (msg));

		save_to_file (filename, message_string);

		g_free (message_string);
		g_free (dirname);
		g_free (filename);
	}
		
	gtk_main_quit();
	return;
}
void
remove_pending_reports (void)
{
		GDir *dir;
		char *dirname;
		GError *error = NULL;

		dirname = g_strdup_printf ("%s/.gnome2/bug-buddy/pending_reports", g_get_home_dir ());
		dir = g_dir_open (dirname, 0, &error);
		if (dir) {
			const char *name = g_dir_read_name (dir);
			while (name) {
				char *path = g_strdup_printf ("%s/%s", dirname, name);
				g_remove (path);
				g_free (path);
				name = g_dir_read_name (dir);
			}
			g_dir_close (dir);
		}

		g_remove (dirname);
		g_free (dirname);
}

void
all_sent (GladeXML *xml)
{
	GtkWidget *close_button;

	/* hide the progressbar */
	gtk_widget_hide (glade_xml_get_widget (xml, "progressbar"));

	/* connect the closed button to gtk_main_quit() */
	close_button = glade_xml_get_widget (xml, "close-button");
	g_signal_connect (close_button, "clicked", gtk_main_quit, NULL);
	gtk_widget_show (close_button);
	gtk_statusbar_push (GTK_STATUSBAR (glade_xml_get_widget (xml, "statusbar")),
			    DONE, _("Done"));
}

void
previous_sent (SoupMessage *msg, GladeXML *xml)
{
	if (--bug_count == 0) {
		all_sent (xml);
	}
}



void
bug_sent (SoupMessage *msg, GladeXML *xml)
{
	char *text = NULL;
	char *errmsg = NULL;
	char *str = NULL;
	char *ptr = NULL;
	long bugid;
	GtkWidget *urlbutton;
	GtkRequisition requisition;
	GError *err = NULL;


	if (SOUP_STATUS_IS_TRANSPORT_ERROR (msg->status_code)) {
		network_error (msg, xml);
	} else {
		remove_pending_reports ();
	}

	/* parse the XML-RPC response from bugzilla */
	bugid = bugzilla_parse_response (msg, &err);

	if (bugid > 0) {
		GtkWidget *main_vbox;
		
		/* we need a reference to the vbox containing the text so that we
	 	* can add a GtkLinkButton to the bug report */
		main_vbox = glade_xml_get_widget (xml, "main-vbox");

		text = g_strdup_printf ("http://bugzilla.gnome.org/show_bug.cgi?id=%d", bugid);

		/* create a clickable link to the bug report */
		urlbutton = gtk_link_button_new (text);
		g_signal_connect (G_OBJECT (urlbutton), "clicked", G_CALLBACK (link_callback), NULL);
		gtk_box_pack_end (GTK_BOX (main_vbox), urlbutton, FALSE, FALSE, 0);

		gtk_widget_show (urlbutton);
		g_free (text);

		text = g_markup_printf_escaped (_("A bug report detailing your software crash has been sent to GNOME. "
		                                  "This information will allow the developers to understand the cause "
		                                  "of the crash and prepare a solution for it.\n\n"
		                                  "You may be contacted by a GNOME developer if more details are "
		                                  "required about the crash.\n\n"
		                                  "You can view your bug report and follow its progress with this URL:\n")) ;

		gtk_label_set_text (GTK_LABEL (glade_xml_get_widget (xml, "main-text")), text);
		g_free (text);
		save_email (gtk_entry_get_text (GTK_ENTRY (glade_xml_get_widget (xml, "email-entry"))));
	} else {
		errmsg = _("Bug Buddy has encountered an error while submitting your report "
		           "to the Bugzilla server.  Details of the error are included below.\n\n");

		if (err != NULL) {
			switch (err->code) {
			case BUGZILLA_ERROR_RECV_BAD_STATUS:
				text = g_strdup_printf (_("Server returned bad state.  This is most likely a server "
				                          "issue and should be reported to bugmaster@gnome.org\n\n%s"), 
				                        err->message);
				break;
			case BUGZILLA_ERROR_RECV_PARSE_FAILED:
				text = g_strdup_printf (_("Failed to parse the xml-rpc response.  Response follows:\n\n%s"), 
				                        err->message);
				break;
			case BUGZILLA_ERROR_RECV_FAULT:
				/* in this case, the error message returned is the faultCode and faultString from
				 * the XML-RPC response, separated by a colon.  We construct our error message
				 * based on the faultString */
				ptr = strstr (err->message, ":");
				if (ptr == NULL) {
					text = g_strdup_printf (_("Bugzilla reported an error when trying to process your "
					                          "request, but was unable to parse the response."));
					break;
				}

				/* skip the colon */
				ptr++;
				
				/* see http://cvs.gnome.org/viewcvs/bugzilla-newer/Bugzilla/RPC.pm?view=markup */
				if (g_str_equal (ptr, "invalid_username")) {
					text = g_strdup_printf (_("The email address you provided is not valid."));
				} else if (g_str_equal (ptr, "account_disabled")) {
					text = g_strdup_printf (_("The account associated with the email address "
				                                  "provided, has been disabled."));
				} else if (g_str_equal (ptr, "product_doesnt_exist")) {
					text = g_strdup_printf (_("The product specified doesn't exist or has been "
				                                  "renamed.  Please upgrade to the latest version."));
				} else if (g_str_equal (ptr, "component_not_valid")) {
					text = g_strdup_printf (_("The component specified doesn't exist or has been "
				                                  "renamed.  Please upgrade to the latest version."));
				} else if (g_str_equal (ptr, "require_summary")) {
					text = g_strdup_printf (_("The summary is required in your bug report. "
				                                  "This should not happen with the latest bug buddy."));
				} else if (g_str_equal (ptr, "description_required")) {
					text = g_strdup_printf (_("The description is required in your bug report. "
				                                  "This should not happen with the latest bug buddy."));
				} else {
					text = g_strdup_printf (_("The fault code returned by Bugzilla is not recognized. "
				                                  "Please report the following information to "
					                          "bugzilla.gnome.org manually:\n\n%s"), err->message);
				}

				break;
			default:
				text = g_strdup_printf (_("An unknown error occurred.  This is most likely a problem with "
				                          "bug-buddy. Please report this problem manually at bugzilla."
			                                  "gnome.org\n\n"));
				break;
			}

			str = g_strconcat (errmsg, text, NULL);
			gtk_label_set_text (GTK_LABEL (glade_xml_get_widget (xml, "main-text")), str);

			g_free (str);
			g_free (text);
			g_error_free (err);
		}
	}
	
	if (--bug_count == 0) {
		all_sent (xml);
	}

	gtk_widget_size_request (glade_xml_get_widget (xml, "main-window"), &requisition);
	gtk_window_resize (GTK_WINDOW (glade_xml_get_widget (xml, "main-window")),
			   requisition.width, requisition.height);
	
}


static void
set_proxy (SoupSession *session)
{
	GConfClient *gconf_client;
	char *host;
	int port;
	char *proxy_uri;
	SoupUri *uri;
	char *username = NULL;
	char *password = NULL;
	
	
	gconf_client = gconf_client_get_default ();

	if (gconf_client_get_bool (gconf_client, USE_PROXY_KEY, NULL) == FALSE) {
		g_object_unref (gconf_client);
		return;
	}

	host = gconf_client_get_string (gconf_client, PROXY_HOST_KEY, NULL);
	if (host == NULL) {
		g_object_unref (gconf_client);
		return;
	}
	port = gconf_client_get_int (gconf_client, PROXY_PORT_KEY, NULL);
	if (port == 0)
		port = 80;

	if (gconf_client_get_bool (gconf_client, USE_PROXY_AUTH, NULL)) {
		username = gconf_client_get_string (gconf_client, PROXY_HOST_KEY, NULL);
		password = gconf_client_get_string (gconf_client, PROXY_HOST_KEY, NULL);
	}

	if (username && password)
		proxy_uri = g_strdup_printf ("http://%s:%s@%s:%d", username, password, host, port);
	else
		proxy_uri = g_strdup_printf ("http://%s:%d", host, port);

	uri = soup_uri_new (proxy_uri);
	g_object_set (G_OBJECT (session), "proxy-uri", uri, NULL);
	
	g_free (host);
	g_free (username);
	g_free (password);
	g_free (proxy_uri);
	soup_uri_free (uri);
	g_object_unref (gconf_client);

}


static char*
create_report_title (BugzillaApplication *app, int type, const char *description)
{
	char *title;
	int limit;
	long size;
	char *tmp = NULL;

	if (description) {
		tmp = g_malloc0 (256); /* This should be safe enough for 24 UTF-8 chars.
					* anyway, I miss a g_utf8_strndup :) */ 
		size = g_utf8_strlen (description, -1);
		if (size > 24) {
			g_utf8_strncpy (tmp, description, 24);
		} else {
			g_utf8_strncpy (tmp, description, size);
		}
	}

	if (type == BUG_TYPE_CRASH) {
		title = g_strdup_printf ("crash in %s: %s%s", app->cname, 
				         tmp ? tmp : "empty description",
					 (tmp && size > 24) ? "..." : "");
	} else {
		title = g_strdup_printf ("%s: %s%s", app->cname,
					 tmp ? tmp : "empty description",
			       		 (tmp && size > 24) ? "..." : "");
	}

	g_free (tmp);

	return title;
}
static void
send_report (BugzillaApplication *app, GnomeVersionInfo *gnome_version, GladeXML *xml)
{
	GtkTextView *text_view;
	GtkTextBuffer *buffer;
	GtkTextIter start;
	GtkTextIter end;
	int type;
	char *gdb_text;
	char *details_text;
	char *title;
	char *final_text;
	const char *email;
	SoupSession *session;
        SoupXmlrpcMessage *message;
	SoupXmlrpcResponse *response;
	GError *err = NULL;

	gtk_widget_hide (glade_xml_get_widget (xml, "pending-reports-check"));
	
	text_view = GTK_TEXT_VIEW (glade_xml_get_widget (xml, "gdb-text"));
	buffer = gtk_text_view_get_buffer (text_view);
	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	gdb_text = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);

	text_view = GTK_TEXT_VIEW (glade_xml_get_widget (xml, "details-view"));
	buffer = gtk_text_view_get_buffer (text_view);
	gtk_text_buffer_get_start_iter (buffer, &start);
	gtk_text_buffer_get_end_iter (buffer, &end);
	details_text = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);

	final_text = g_strdup_printf ("What were you doing when the application crashed?\n%s\n\n\n%s", 
				      details_text != NULL ? details_text : "",
				      gdb_text != NULL ? gdb_text : "<empty backtrace>");
	
	email = gtk_entry_get_text (GTK_ENTRY (glade_xml_get_widget (xml, "email-entry")));
	type = GPOINTER_TO_INT (g_object_get_data (G_OBJECT(xml), "type"));
	title = create_report_title (app, type, details_text);

	message = bugzilla_create_report (app, gnome_version, email, title, final_text, xml, &err);
	if (message == NULL) {
		char *text;

		if (err != NULL) {
			text = g_strdup_printf (_("Unable to create the bug report: %s\n"), err->message);
		} else {
			text = g_strdup_printf (_("There was an error creating the bug report\n"));
		}
		
		buddy_error (NULL, text);
		g_free (text);
		g_free (gdb_text);
		g_free (details_text);
		g_free (title);
		g_free (final_text);
		gtk_main_quit();
		return;
	}

	session = soup_session_async_new ();
	set_proxy (session);


	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (glade_xml_get_widget (xml, "pending-reports-check")))) {
		GDir *dir;
		char *dirname;
		GError *error = NULL;

		dirname = g_strdup_printf ("%s/.gnome2/bug-buddy/pending_reports", g_get_home_dir ());
		dir = g_dir_open (dirname, 0, &error);
		if (dir) {
			const char *name = g_dir_read_name (dir);
			while (name != NULL) {
				char *path;
				char *contents;
				
				path = g_strdup_printf ("%s/%s", dirname, name);
				if (g_file_get_contents (path, &contents, NULL, NULL)) {
					SoupXmlrpcMessage *msg;
					msg = soup_xmlrpc_message_new ("http://bugzilla.gnome.org/bugbuddy.cgi");
					soup_xmlrpc_message_from_string (msg, contents);
					bug_count++;
					soup_xmlrpc_message_persist (msg);
        				soup_session_queue_message (session, SOUP_MESSAGE (msg),
               		                    			    previous_sent, xml);
					g_free (contents);
				}
				g_free (path);
				name = g_dir_read_name (dir);
			}

			g_dir_close (dir);
		}
	}

	bug_count++;

	soup_xmlrpc_message_persist (message);
        soup_session_queue_message (session, SOUP_MESSAGE (message),
                                    (SoupMessageCallbackFn)bug_sent, xml);
	g_free (gdb_text);
	g_free (details_text);
	g_free (title);
	g_free (final_text);

	gtk_statusbar_push (GTK_STATUSBAR (glade_xml_get_widget (xml, "statusbar")),
			    SENDING, _("Sending..."));

}



/* Check for *simple* email addresses of the form user@host, with checks
 * in characters used, and sanity checks on the form of host.
 */
/* FIXME: Should we provide a useful error message? */
static gboolean
email_is_valid (const char *addy)
{
	const char *tlds[] = {
		"com",	"org",	"net",	"edu",	"mil",	"gov",	"int",	"arpa",
		"aero",	"biz",	"coop",	"info",	"museum",	"name", "pro",
		NULL
	};

	gboolean is_dot_ok = FALSE;
	gboolean seen_at  = FALSE;
	const char *tld = NULL;
	const char *s;
	const char **t;

	for (s = addy; *s; s++) {
		/* Must be in acceptable ASCII range. */
		if ((unsigned char)*s <= 32 || (unsigned char)*s >= 128)
			return FALSE;
		if (*s == '.') {
			/* No dots at start, after at, or two in a row. */
			if (!is_dot_ok)
				return FALSE;
			is_dot_ok = FALSE;
			if (seen_at)
				tld = s+1;
		} else if (*s == '@') {
			/* Only one at, not allowed if a dot isn't. */
			if (!is_dot_ok || seen_at)
				return FALSE;
			seen_at = TRUE;
			is_dot_ok = FALSE;
		} else {
			/* Can put a dot or at after this character. */
			is_dot_ok = TRUE;
		}
	}
	if (!seen_at || tld == NULL)
		return FALSE;

	/* Make sure they're not root. */
	if (!strncmp (addy, "root@", 5))
		return FALSE;

	/* We'll believe any old ccTLD. */
	if (strlen (tld) == 2)
		return TRUE;
	/* Otherwise, check the TLD list. */
	for (t=tlds; *t; t++) {
		if (!g_ascii_strcasecmp (tld, *t))
			return TRUE;
	}
	/* Unrecognised. :( */
	return FALSE;
}


static void
check_email (GtkEditable *editable, gpointer data)
{
	const char *email;
	GladeXML *xml = (GladeXML*) data;

	email = gtk_entry_get_text (GTK_ENTRY (editable));
	gtk_widget_set_sensitive (glade_xml_get_widget (xml, "send-button"),
				  email_is_valid (email));
}
		
static void
on_send_clicked (GtkWidget *button, gpointer data)
{
	BugzillaApplication *app;
	GnomeVersionInfo *gnome_version;
	GtkRequisition requisition;
	GladeXML *xml = (GladeXML*) data;

	app = g_object_get_data (G_OBJECT (xml), "app");
	gnome_version = g_object_get_data (G_OBJECT (xml), "gnome-version");

	/* hide the send button immediately so that the user can't click
	 * it more than once (this will create multiple bugs) */
	gtk_widget_hide (glade_xml_get_widget (xml, "send-button"));

	gtk_widget_show (glade_xml_get_widget (xml, "progressbar"));
	gtk_widget_hide (glade_xml_get_widget (xml, "final-box"));
	gtk_widget_hide (glade_xml_get_widget (xml, "expander1"));

	gtk_widget_size_request (glade_xml_get_widget (xml, "main-window"), &requisition);
	gtk_window_resize (GTK_WINDOW (glade_xml_get_widget (xml, "main-window")),
			   requisition.width, requisition.height);
			   
	send_report (app, gnome_version, xml);
}

static void
gdb_insert_text (const gchar *stacktrace, GladeXML *xml)
{
	GtkTextView *text_view;
	GtkTextIter end;
	GtkTextBuffer *buffer;

	text_view = GTK_TEXT_VIEW (glade_xml_get_widget (xml, "gdb-text"));
	buffer = gtk_text_view_get_buffer (text_view);
	gtk_text_buffer_get_end_iter (buffer, &end);

	/* add the stacktrace to the GtkTextView */
	gtk_text_buffer_insert (buffer, &end, stacktrace, strlen (stacktrace));

}

static void
show_pending_checkbox_if_pending (GladeXML *xml)
{
	char *dirname;
	GtkWidget *check;

	dirname = g_strdup_printf ("%s/.gnome2/bug-buddy/pending_reports", g_get_home_dir ());
	if (g_file_test (dirname, G_FILE_TEST_IS_DIR)) {
		check = glade_xml_get_widget (xml, "pending-reports-check");
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), TRUE);
		gtk_widget_show (check);
	}
}

static void
known_app_finished (GladeXML *xml)
{
	GtkWidget *email_entry;
	char *default_email;
	
	gtk_widget_show (glade_xml_get_widget (xml, "final-box"));
	gtk_widget_show (glade_xml_get_widget (xml, "send-button"));
	gtk_widget_hide (glade_xml_get_widget (xml, "progressbar"));
	gtk_statusbar_pop (GTK_STATUSBAR (glade_xml_get_widget (xml, "statusbar")), DEBUGGING);


	gtk_label_set_text (GTK_LABEL (glade_xml_get_widget (xml, "main-text")),
			    _("The information about the crash has been successfully collected. "
			      "Please provide some background about what you were doing when "
			      "the application crashed.\n\n"
			      "A valid email address is required.  This will allow developers to "
			      "contact you for more information if necessary.\n\n" 
			      "Sensitive data may be present in the debug information, so please "
			      "review details below if you are concerned about transmitting "
	                      "passwords or other sensitive data.\n"));

	show_pending_checkbox_if_pending (xml);
	
	g_signal_connect (glade_xml_get_widget (xml, "send-button"), "clicked", 
	                  G_CALLBACK (on_send_clicked), xml);

	email_entry = glade_xml_get_widget (xml, "email-entry");
	g_signal_connect (email_entry, "changed", G_CALLBACK (check_email), xml);

	default_email = get_default_user_email ();

	if (default_email != NULL) {
		gtk_entry_set_text (GTK_ENTRY (email_entry), default_email);
		g_free (default_email);
	}

	gtk_widget_grab_focus (glade_xml_get_widget (xml, "details-view"));
}

static void
gdb_finished (const gchar *stacktrace, gpointer data)
{
	GladeXML *xml = (GladeXML*) data;

	gdb_insert_text (stacktrace, xml);
	known_app_finished (xml);
}



static void
on_save_clicked (GtkWidget *button, gpointer user_data)
{
	GladeXML *xml = (GladeXML *)user_data;
	GtkWidget *dialog;
	gchar *desktop;
	char *filename;
	gboolean saved = FALSE;

	dialog = gtk_file_chooser_dialog_new ("Save File",
	                                      GTK_WINDOW (glade_xml_get_widget (xml, "main-window")),
	                                      GTK_FILE_CHOOSER_ACTION_SAVE,
	                                      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	                                      GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
	                                      NULL);

	gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER (dialog), TRUE);

	desktop = g_strconcat (g_getenv ("HOME"), "/Desktop", NULL); 
	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), desktop);

	filename = g_strconcat (gopt_data.app_file, _("-bugreport.txt"), NULL);
	gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (dialog), filename);
	g_free (desktop);
	g_free (filename);

	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
		char *filename;
		GtkTextView   *text_view;
		GtkTextBuffer *buffer;
		GtkTextIter    start;
		GtkTextIter    end;
		gchar         *text;

		text_view = GTK_TEXT_VIEW (glade_xml_get_widget (xml, "gdb-text"));
		buffer = gtk_text_view_get_buffer (text_view);
		gtk_text_buffer_get_start_iter (buffer, &start);
		gtk_text_buffer_get_end_iter (buffer, &end);
		text = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);

		filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
		save_to_file (filename, text);
		g_free (filename);
		g_free (text);
		saved = TRUE;
	}

	gtk_widget_destroy (dialog);
	if (saved) {
		gtk_main_quit ();
	}
}

static void
unknown_app_finished (GladeXML *xml)
{
	GtkWidget *button;

	/* don't need user input, so hide these widgets */
	gtk_widget_hide (glade_xml_get_widget (xml, "final-box"));
	gtk_widget_hide (glade_xml_get_widget (xml, "progressbar"));
	
	/* make the send button into a save button :-) */
	button = glade_xml_get_widget (xml, "send-button");
	gtk_button_set_label (GTK_BUTTON (button), _("Save"));
	g_signal_connect (GTK_BUTTON (button), "clicked", G_CALLBACK (on_save_clicked), xml);
	gtk_widget_show (glade_xml_get_widget (xml, "send-button"));

	/* expand the details about the stack trace */
	gtk_expander_set_expanded (GTK_EXPANDER (glade_xml_get_widget (xml, "expander1")), TRUE);

	gtk_statusbar_pop (GTK_STATUSBAR (glade_xml_get_widget (xml, "statusbar")), DEBUGGING);

	gtk_label_set_text (GTK_LABEL (glade_xml_get_widget (xml, "main-text")),
			    _("The information about the crash has been successfully collected.\n\n"
	                      "The application that crashed is not known to bug-buddy, therefore the "
	                      "bug report can not be sent to the GNOME Bugzilla.  Please save the "
	                      "bug to a text file and report it to the appropriate bug tracker "
	                      "for this application."));

	gtk_widget_grab_focus (glade_xml_get_widget (xml, "details-view"));
}

static void
gdb_finished_unknown_app (const gchar *stacktrace, gpointer data)
{
	GladeXML *xml = (GladeXML*) data;

	gdb_insert_text (stacktrace, xml);
	unknown_app_finished (xml);
}




static void
close_callback (GtkWidget *widget, gpointer user_data)
{
	GladeXML *xml = (GladeXML *)user_data;
	gpointer data;

	g_return_if_fail (xml != NULL);

	data = g_object_get_data (G_OBJECT (xml), "sourceid");

	if (data != NULL) {
		guint source_id = GPOINTER_TO_UINT (data);

		/* removes the context from the main loop and kills any remaining
	 	* gdb process */
		if (source_id > 0) {
			g_source_remove (source_id);
			g_object_set_data (G_OBJECT (xml), "sourceid", GUINT_TO_POINTER (0));
		}
	}

	gtk_main_quit ();
}

static gboolean
delete_callback (GtkWidget *widget, GdkEvent *event, gpointer data)
{
	close_callback (NULL, data);
	return TRUE;
}

static void
fill_gnome_info (BugzillaApplication *app, GnomeVersionInfo *gnome_version, GladeXML *xml)
{
	char *version_info;
	char *distro;
	GtkTextView *text_view;
	GtkTextIter end;
	GtkTextBuffer *buffer;

	g_return_if_fail (app != NULL);
	g_return_if_fail (gnome_version != NULL);
	g_return_if_fail (xml != NULL);
	
	distro = get_distro_name ();
        version_info = g_strdup_printf ("Distribution: %s\n"
                                        "Gnome Release: %s %s (%s)\n"
                                        "BugBuddy Version: %s\n"
                                        "\n",
                                   	distro,
                                   	gnome_version->gnome_platform, gnome_version->gnome_date,
                                   	gnome_version->gnome_distributor, VERSION);

	g_free (distro);
	
	text_view = GTK_TEXT_VIEW (glade_xml_get_widget (xml, "gdb-text"));
	buffer = gtk_text_view_get_buffer (text_view);
	gtk_text_buffer_get_end_iter (buffer, &end);
	gtk_text_buffer_insert (buffer, &end, version_info, strlen (version_info));

	g_free (version_info);
}


static void
fill_proccess_info (pid_t pid, GladeXML *xml)
{
	GtkTextView *text_view;
	GtkTextIter end;
	GtkTextBuffer *buffer;
	char *mem;
	char *time;
	char *proccess_info;
	
	mem = proccess_get_mem_state (pid);
	time = proccess_get_time (pid);

        proccess_info = g_strdup_printf ("%s\n"
                                         "%s\n"
                                         "\n",
			 		 mem, time);

	g_free (mem);
	g_free (time);
	
	text_view = GTK_TEXT_VIEW (glade_xml_get_widget (xml, "gdb-text"));
	buffer = gtk_text_view_get_buffer (text_view);
	gtk_text_buffer_get_end_iter (buffer, &end);
	gtk_text_buffer_insert (buffer, &end, proccess_info, strlen (proccess_info));

	g_free (proccess_info);

}

static void
fill_include_file (char *filename, GladeXML *xml)
{
	GtkTextView *text_view;
	GtkTextIter end;
	GtkTextBuffer *buffer;
	char *text;
	GError *error = NULL;

	if (g_file_get_contents (filename, &text, NULL, &error) != TRUE) {
		buddy_error (NULL, error->message);
		g_error_free (error);
		return;
	}
	
	text_view = GTK_TEXT_VIEW (glade_xml_get_widget (xml, "gdb-text"));
	buffer = gtk_text_view_get_buffer (text_view);
	gtk_text_buffer_get_end_iter (buffer, &end);
	gtk_text_buffer_insert (buffer, &end, text, strlen (text));

	g_free (text);

}



int
main (int argc, char *argv[])
{
	GtkIconInfo *icon_info;
	GladeXML *xml;
	char *s;
	GHashTable *apps;
	char *bt;
	GtkTextView *text_view;
	BugzillaApplication *app;
	GnomeVersionInfo *gnome_version;
	guint progress;
	GtkWidget *main_window, *statusbar;
	GOptionContext *context;
	GnomeProgram *program;
	GError *err = NULL;
	guint source_id;
		
	memset (&gopt_data,  0, sizeof (gopt_data));

	bindtextdomain (PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset (PACKAGE, "UTF-8");
	textdomain (PACKAGE);

	context = g_option_context_new (_("\n\nBug Buddy is a utility that helps report debugging\n"
	                                  "information to the GNOME bugzilla when a program crashes."));

	g_option_context_add_main_entries (context, options, GETTEXT_PACKAGE);

	program = gnome_program_init (PACKAGE, VERSION,
	                              LIBGNOMEUI_MODULE,
	                              argc, argv,
	                              GNOME_PARAM_GOPTION_CONTEXT, context,
	                              GNOME_PARAM_APP_DATADIR, REAL_DATADIR,
	                              NULL);

	g_set_application_name (_("Bug-Buddy"));
	gtk_window_set_default_icon_name ("bug-buddy");

	s = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
				       "bug-buddy/bug-buddy.glade", TRUE, NULL);
	if (s)
		xml = glade_xml_new (s, NULL, GETTEXT_PACKAGE);
	else
		xml = NULL;

	if (!xml) {
		buddy_error (NULL, 
			     _("Bug Buddy could not load its user interface file (%s).\n"
			       "Please make sure Bug Buddy was installed correctly."), 
			     s);
		g_object_unref (program);
		return 0;
	}
	g_free (s);	

	main_window = glade_xml_get_widget (xml, "main-window");
	g_signal_connect (main_window, "delete-event", G_CALLBACK (delete_callback), xml);
	gtk_window_set_default_size (GTK_WINDOW (main_window), 400, 400);

	gtk_widget_hide (glade_xml_get_widget (xml, "final-box"));

	statusbar = glade_xml_get_widget (xml, "statusbar");
	gtk_statusbar_push (GTK_STATUSBAR (statusbar),
			    LOADING_INFO, _("Collecting info from your system..."));
	progress = g_timeout_add (100, update_progress_bar,
				  glade_xml_get_widget (xml, "progressbar"));

	if (gopt_data.app_file == NULL && gopt_data.package == NULL) {
		buddy_error (NULL, _("Either --appname or --package arguments are required.\n"));	
		g_object_unref (program);
		return 0;
	}	

	if (gopt_data.app_file && gopt_data.pid == 0 && gopt_data.include_file == NULL) {
		buddy_error (NULL, _("Either --pid or --include arguments are required.\n"));	
		g_object_unref (program);
		return 0;
	}	
	
	/* get some information about the gnome version */
	gnome_version = get_gnome_version_info ();
	if (gnome_version == NULL) {
		buddy_error (NULL, _("Bug Buddy was unable to retreive information regarding "
		                     "the version of GNOME you are running.  This is most likely "
		                     "due to a missing installation of gnome-desktop.\n"));
		g_object_unref (program);
		return 0;
	}
	
	g_object_set_data (G_OBJECT (xml), "gnome-version", gnome_version);

	gtk_widget_show (main_window);

	apps = load_applications ();
	
	/* If we have a binary file it is a crash */
	if (gopt_data.app_file) {
		app = g_hash_table_lookup (apps, gopt_data.app_file);

		/* we handle an unknown application (no .desktop file) differently */
		if (app != NULL) {
			s = g_markup_printf_escaped (_("The application %s has crashed.\n"
		                               		"We are collecting information about the crash to send it to the\n"
                                               		"developers in order to fix the problem."), app->name);	
			gtk_label_set_text (GTK_LABEL (glade_xml_get_widget (xml, "main-text")), s);
			g_free (s);
		
			g_object_set_data (G_OBJECT (xml), "app", app);
	
			if (app->pixbuf)
				gtk_image_set_from_pixbuf (GTK_IMAGE (glade_xml_get_widget (xml, "app-image")),
			                           		app->pixbuf);
			fill_gnome_info (app, gnome_version, xml);
		}

		gtk_statusbar_push (GTK_STATUSBAR (statusbar),
			    	    DEBUGGING, _("Collecting info from the crash..."));

		fill_proccess_info (gopt_data.pid, xml);

		if (gopt_data.pid > 0) {
			/* again, if this is an unknown application, we connect a different callback that
	 		* will allow the user to save the trace rather than sending it to the GNOME bugzilla */
			if (app == NULL) {
				source_id = gdb_get_trace (gopt_data.app_file, gopt_data.pid, xml, 
		                           		   gdb_finished_unknown_app, &err);
			} else {
				source_id = gdb_get_trace (gopt_data.app_file, gopt_data.pid, xml, gdb_finished, &err);
			}

			if (source_id == 0) {
				buddy_error (NULL, _("Bug buddy encountered the following error when trying "
		                     		     "to retrieve debugging information: %s\n"), err->message);
				g_error_free (err);
				g_object_unref (program);
				return 0;
			}

			/* connect the close button callback so that we can remove the source from 
	 		* the main loop (and kill gdb) if the user wants to quit before gdb is finished */
			g_object_set_data (G_OBJECT (xml), "sourceid", GUINT_TO_POINTER (source_id));
		} else {
			if (app == NULL) {
				unknown_app_finished (xml);
			} else {
				known_app_finished (xml);
			}
		}
		g_object_set_data (G_OBJECT (xml), "type", GINT_TO_POINTER(BUG_TYPE_CRASH));
	} else {
		/* No binary file, so this is a non-crashing bug. Look the application from the --package arg */
		GtkWidget *email_entry;
		char *default_email;
		
		app = g_hash_table_find (apps, bugzilla_search_for_package, gopt_data.package);
		if (app == NULL) {
			/* Fallback to binary name */
			app = g_hash_table_lookup (apps, gopt_data.package);
		}
		if (app == NULL) {
			buddy_error (NULL, _("Bug Buddy doesn't know how to send a suggestion for the application %s.\n"),
		                     	     gopt_data.package);
			return 0;
		}
		s = g_markup_printf_escaped (_("Thank you for helping us improving our software.\n"
				               "Please fill your suggestions/error information for %s application.\n"),
                                              	app->name);
		gtk_label_set_text (GTK_LABEL (glade_xml_get_widget (xml, "main-text")), s);
		g_free (s);
		
		g_object_set_data (G_OBJECT (xml), "app", app);
	
		if (app->pixbuf)
			gtk_image_set_from_pixbuf (GTK_IMAGE (glade_xml_get_widget (xml, "app-image")),
	                           		   app->pixbuf);
		fill_gnome_info (app, gnome_version, xml);
	
		gtk_widget_show (glade_xml_get_widget (xml, "final-box"));
		gtk_widget_show (glade_xml_get_widget (xml, "send-button"));
		gtk_widget_hide (glade_xml_get_widget (xml, "progressbar"));
		gtk_statusbar_pop (GTK_STATUSBAR (glade_xml_get_widget (xml, "statusbar")), DEBUGGING);

		s = g_markup_printf_escaped (_("Thank you for helping us improving our software.\n"
					        "Please fill your suggestions/error information for %s application.\n\n"
			      			"A valid email address is required.  This will allow developers to "
			      			"contact you for more information if necessary.\n\n"),
				             app->name);
		gtk_label_set_text (GTK_LABEL (glade_xml_get_widget (xml, "main-text")), s);
		g_free (s);

		s = g_markup_printf_escaped (_("<span weight=\"bold\">Suggestion / Error description:</span>"));
		gtk_label_set_markup (GTK_LABEL (glade_xml_get_widget (xml, "main-label")), s);
		g_free (s);
		
		show_pending_checkbox_if_pending (xml);
		g_signal_connect (glade_xml_get_widget (xml, "send-button"), "clicked", 
	                  	  G_CALLBACK (on_send_clicked), xml);

		email_entry = glade_xml_get_widget (xml, "email-entry");
		g_signal_connect (email_entry, "changed", G_CALLBACK (check_email), xml);

		default_email = get_default_user_email ();
		
		if (default_email != NULL) {
			gtk_entry_set_text (GTK_ENTRY (email_entry), default_email);
			g_free (default_email);
		}
		g_object_set_data (G_OBJECT (xml), "type", GINT_TO_POINTER(BUG_TYPE_REQUEST));
	}
	
	if (gopt_data.include_file != NULL) {
		fill_include_file (gopt_data.include_file, xml);
	}

	
	g_signal_connect (glade_xml_get_widget (xml, "close-button"), "clicked", 
	                  G_CALLBACK (close_callback), xml);
	
	gtk_main ();

	g_object_unref (program);
	return 0;
}
