/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */

/*
   eel-open-with-dialog.c: an open-with dialog

   Copyright (C) 2004 Novell, Inc.

   The Gnome Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The Gnome Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the Gnome Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Authors: Dave Camp <dave@novell.com>
*/

#include <config.h>
#include "eel-open-with-dialog.h"

#include "eel-mime-extensions.h"
#include "eel-stock-dialogs.h"

#include <string.h>
#include <glib/gi18n-lib.h>
#include <gtk/gtkalignment.h>
#include <gtk/gtkbox.h>
#include <gtk/gtkexpander.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkfilechooserdialog.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkicontheme.h>
#include <gtk/gtkiconfactory.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkstock.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtktreeselection.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkcellrendererpixbuf.h>
#include <gtk/gtkvbox.h>
#include <libgnome/gnome-desktop-item.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <gmenu-tree.h>

#define sure_string(s)                    ((const char *)((s)!=NULL?(s):""))
#define DESKTOP_ENTRY_GROUP		  "Desktop Entry"

struct _EelOpenWithDialogDetails {
	char *uri;

	char *mime_type;
	char *mime_description;

	gboolean new_mime_type;
	char *new_glob;
	char *desktop_file;

	GtkWidget *label;
	GtkWidget *entry;
	GtkWidget *button;

	GtkWidget *desc_label;

	GtkWidget *open_label;

	GtkWidget     *program_list;
	GtkListStore  *program_list_store;
	GSList	      *add_icon_paths;
	gint	       add_items_idle_id;
	gint	       add_icons_idle_id;
};

enum {
        COLUMN_ICON,
        COLUMN_ICON_FILE,
        COLUMN_NAME,
        COLUMN_COMMENT,
        COLUMN_PATH,
        COLUMN_EXEC,
        NUM_COLUMNS
};

enum {
	RESPONSE_OPEN
};

enum {
	APPLICATION_SELECTED,
	LAST_SIGNAL
};

static gpointer parent_class;
static guint signals[LAST_SIGNAL] = { 0 };

static void
eel_open_with_dialog_finalize (GObject *object)
{
	EelOpenWithDialog *dialog;

	dialog = EEL_OPEN_WITH_DIALOG (object);

	if (dialog->details->add_icons_idle_id) {
		g_source_remove (dialog->details->add_icons_idle_id);
	}

	if (dialog->details->add_items_idle_id) {
		g_source_remove (dialog->details->add_items_idle_id);
	}
	
	g_free (dialog->details->uri);
	g_free (dialog->details->mime_type);
	g_free (dialog->details->mime_description);
	g_free (dialog->details->new_glob);

	g_free (dialog->details);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
eel_open_with_dialog_destroy (GtkObject *object)
{
	GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

/* An application is valid if:
 *
 * 1) The file exists
 * 2) The user has permissions to run the file
 */
static gboolean
check_application (EelOpenWithDialog *dialog)
{
	char *command;
	char *path = NULL;
	char **argv = NULL;
	int argc;
	GError *error = NULL;
	gint retval = TRUE;

	command = NULL;
	if (dialog->details->desktop_file != NULL) {
		GKeyFile *keyfile;
		
		keyfile = g_key_file_new ();
		if (g_key_file_load_from_file (keyfile, dialog->details->desktop_file,
					       0, NULL)) {
			command = g_key_file_get_string (keyfile,
							 DESKTOP_ENTRY_GROUP,
				      			 GNOME_DESKTOP_ITEM_EXEC,
							 NULL);
		}
		g_key_file_free (keyfile);
	}

	if (command == NULL) {
		command = g_strdup (gtk_entry_get_text (GTK_ENTRY (dialog->details->entry)));
	}
	
	g_shell_parse_argv (command, &argc, &argv, &error);

	if (error) {
		eel_show_error_dialog (_("Could not run application"),
				       error->message,
				       _("Could not run application"),
				       GTK_WINDOW (dialog));
		g_error_free (error);
		retval = FALSE;
		goto cleanup;
	}

	path = g_find_program_in_path (argv[0]);
	if (!path) {
		char *error_message;

		error_message = g_strdup_printf (_("Could not find '%s'"),
						 argv[0]);

		eel_show_error_dialog (_("Could not find application"),
				       error_message,
				       _("Could not find application"),
				       GTK_WINDOW (dialog));
		g_free (error_message);
		retval = FALSE;
		goto cleanup;
	}

 cleanup:
	g_strfreev (argv);
	g_free (path);
	g_free (command);

	return retval;
}

/* Only called for non-desktop files */
static char *
get_app_name (EelOpenWithDialog *dialog)
{
	GError *error = NULL;
	char *basename;
	char *unquoted;
	char **argv;
	int argc;

	g_shell_parse_argv (gtk_entry_get_text (GTK_ENTRY (dialog->details->entry)),
			    &argc, &argv, &error);

	if (error) {
		eel_show_error_dialog (_("Could not run application"),
				       error->message,
				       _("Could not run application"),
				       GTK_WINDOW (dialog));
		g_error_free (error);

		return NULL;
	}

	unquoted = g_shell_unquote (argv[0], &error);

	if (!error) {
		basename = g_path_get_basename (unquoted);
	} else {
		basename = g_strdup (argv[0]);
		g_error_free (error);
	}

	g_free (unquoted);
	g_strfreev (argv);

	return basename;
}

/* This will check if the application the user wanted exists will return that
 * application.  If it doesn't exist, it will create one and return that.
 */
static GnomeVFSMimeApplication *
add_or_find_application (EelOpenWithDialog *dialog)
{
	GnomeVFSMimeApplication *app;
	char *app_name;

	if (dialog->details->new_mime_type) {
		eel_mime_add_glob_type (dialog->details->mime_type,
					dialog->details->mime_description,
					dialog->details->new_glob);

	}

	if (dialog->details->desktop_file != NULL) {
		app = eel_mime_check_for_desktop_duplicates (dialog->details->mime_type,
							     dialog->details->desktop_file);
		if (app != NULL)
			return app;
		
		app = eel_mime_add_custom_mime_type_for_desktop_file (dialog->details->desktop_file,
								      dialog->details->mime_type);
	} else {
		app = eel_mime_check_for_duplicates (dialog->details->mime_type,
						     gtk_entry_get_text (GTK_ENTRY (dialog->details->entry)));
		if (app != NULL)
			return app;

		app_name = get_app_name (dialog);

		if (app_name) {
			app = eel_mime_add_application (dialog->details->mime_type,
							gtk_entry_get_text (GTK_ENTRY (dialog->details->entry)),
							app_name,
							FALSE);
		} else {
			app = NULL;
		}
		g_free (app_name);
	}

	if (!app) {
		eel_show_error_dialog (_("Could not add application"),
				       _("Could not add application to the application database"),
				       _("Could not add application"),
				       GTK_WINDOW (dialog));
	}

	return app;
}

static void
emit_application_selected (EelOpenWithDialog *dialog,
			   GnomeVFSMimeApplication *application)
{
	g_signal_emit (G_OBJECT (dialog), signals[APPLICATION_SELECTED], 0,
		       application);
}

static void
response_cb (EelOpenWithDialog *dialog,
	     int response_id,
	     gpointer data)
{
	GnomeVFSMimeApplication *application;

	switch (response_id) {
	case RESPONSE_OPEN:
		if (check_application (dialog)) {
			application = add_or_find_application (dialog);

			if (application) {
				emit_application_selected (dialog, application);
				gnome_vfs_mime_application_free (application);

				gtk_widget_destroy (GTK_WIDGET (dialog));
			}
		}

		break;
	case GTK_RESPONSE_NONE:
	case GTK_RESPONSE_DELETE_EVENT:
	case GTK_RESPONSE_CANCEL:
		gtk_widget_destroy (GTK_WIDGET (dialog));
		break;
	default :
		g_assert_not_reached ();
	}

}


static void
eel_open_with_dialog_class_init (EelOpenWithDialogClass *class)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;

	parent_class = g_type_class_peek_parent (class);

	gobject_class = G_OBJECT_CLASS (class);
	gobject_class->finalize = eel_open_with_dialog_finalize;

	object_class = GTK_OBJECT_CLASS (class);
	object_class->destroy = eel_open_with_dialog_destroy;

	signals[APPLICATION_SELECTED] =
		g_signal_new ("application_selected",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (EelOpenWithDialogClass,
					       application_selected),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__POINTER,
			      G_TYPE_NONE,
			      1, G_TYPE_POINTER);
}

static void
chooser_response_cb (GtkFileChooser *chooser,
		     int response,
		     gpointer user_data)
{
	EelOpenWithDialog *dialog;

	dialog = EEL_OPEN_WITH_DIALOG (user_data);

	if (response == GTK_RESPONSE_OK) {
		char *filename;

		filename = gtk_file_chooser_get_filename (chooser);

		if (filename) {
			char *quoted_text;

			quoted_text = g_shell_quote (filename);

			gtk_entry_set_text (GTK_ENTRY (dialog->details->entry),
					    quoted_text);
			gtk_entry_set_position (GTK_ENTRY (dialog->details->entry), -1);
			g_free (quoted_text);
			g_free (filename);
		}
	}

	gtk_widget_destroy (GTK_WIDGET (chooser));
}

static void
browse_clicked_cb (GtkWidget *button,
		   gpointer user_data)
{
	EelOpenWithDialog *dialog;
	GtkWidget *chooser;

	dialog = EEL_OPEN_WITH_DIALOG (user_data);

	chooser = gtk_file_chooser_dialog_new (_("Select an Application"),
					       GTK_WINDOW (dialog),
					       GTK_FILE_CHOOSER_ACTION_OPEN,
					       GTK_STOCK_CANCEL,
					       GTK_RESPONSE_CANCEL,
					       GTK_STOCK_OPEN,
					       GTK_RESPONSE_OK,
					       NULL);
	gtk_window_set_destroy_with_parent (GTK_WINDOW (chooser), TRUE);
	g_signal_connect (chooser, "response",
			  G_CALLBACK (chooser_response_cb), dialog);
	gtk_dialog_set_default_response (GTK_DIALOG (chooser),
					 GTK_RESPONSE_OK);
	gtk_file_chooser_set_local_only (GTK_FILE_CHOOSER (chooser), TRUE);
	gtk_file_chooser_set_select_multiple (GTK_FILE_CHOOSER (chooser),
					      FALSE);
	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (chooser),
					     "/usr/bin");

	gtk_widget_show (chooser);
}

static void
entry_changed_cb (GtkWidget *entry,
		  EelOpenWithDialog *dialog)
{
	/* We are writing in the entry, so we are not using a known .dekstop
	 * file anymore */
	g_free (dialog->details->desktop_file);
	dialog->details->desktop_file = NULL;

	if (gtk_entry_get_text (GTK_ENTRY (dialog->details->entry))[0] == '\000') {
		gtk_widget_set_sensitive (dialog->details->button, FALSE);
	} else {
		gtk_widget_set_sensitive (dialog->details->button, TRUE);
	}
}

static gboolean
eel_open_with_dialog_add_icon_idle (EelOpenWithDialog *dialog)
{
	GtkTreeIter   iter;
	GtkTreePath  *path;
	GdkPixbuf    *pixbuf;
	char         *file;
	gboolean      long_operation;

	long_operation = FALSE;
	do {
		if (!dialog->details->add_icon_paths) {
			dialog->details->add_icons_idle_id = 0;
			return FALSE;
		}

		path = dialog->details->add_icon_paths->data;
		dialog->details->add_icon_paths->data = NULL;
		dialog->details->add_icon_paths = g_slist_delete_link (dialog->details->add_icon_paths,
								       dialog->details->add_icon_paths);

		if (!gtk_tree_model_get_iter (GTK_TREE_MODEL (dialog->details->program_list_store),
					      &iter, path)) {
			gtk_tree_path_free (path);
			continue;
		}
		
		gtk_tree_path_free (path);

		gtk_tree_model_get (GTK_TREE_MODEL (dialog->details->program_list_store), &iter,
				    COLUMN_ICON_FILE, &file, -1);

		if (file == NULL) {
			continue;
		}
		
		if (g_path_is_absolute (file)) {
			pixbuf = gdk_pixbuf_new_from_file_at_size (file, 24, 24, NULL);
			long_operation = TRUE;
		} else {
			char *icon_no_extension;
			char *p;

			icon_no_extension = g_strdup (file);
			p = strrchr (icon_no_extension, '.');
			if (p &&
			    (strcmp (p, ".png") == 0 ||
			     strcmp (p, ".xpm") == 0 ||
			     strcmp (p, ".svg") == 0)) {
				*p = 0;
			}
			pixbuf = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (),
							   icon_no_extension, 24, 0, NULL);
			g_free (icon_no_extension);
			long_operation = TRUE;
		}
		if (pixbuf) {
			gtk_list_store_set (dialog->details->program_list_store, &iter, COLUMN_ICON, pixbuf, -1);
			g_object_unref (pixbuf);
		}
		g_free (file);
		
	/* don't go back into the main loop if this wasn't very hard to do */
	} while (!long_operation);

	return TRUE;
}

static int
compare_applications (GMenuTreeEntry *a,
		      GMenuTreeEntry *b)
{
	return g_utf8_collate (gmenu_tree_entry_get_name (a),
			       gmenu_tree_entry_get_name (b));
}

static GSList *
get_all_applications_from_dir (GMenuTreeDirectory *directory,
			       GSList             *list)
{
	GSList *contents;
	GSList *l;

	contents = gmenu_tree_directory_get_contents (directory);

	for (l = contents; l; l = l->next) {
		GMenuTreeItem *item = l->data;

		switch (gmenu_tree_item_get_type (item)) {
			case GMENU_TREE_ITEM_DIRECTORY:
				list = get_all_applications_from_dir (GMENU_TREE_DIRECTORY (item), list);
				gmenu_tree_item_unref (item);
				break;

			case GMENU_TREE_ITEM_ENTRY:
				list = g_slist_prepend (list, item);
				break;

			default:
				break;
		}
	}

	g_slist_free (contents);

	return list;
}


static GSList *
get_all_applications (void)
{
	GMenuTree          *tree;
	GMenuTreeDirectory *root;
	GSList             *retval;

	tree = gmenu_tree_lookup ("applications.menu", GMENU_TREE_FLAGS_NONE);

	root = gmenu_tree_get_root_directory (tree);

	retval = get_all_applications_from_dir (root, NULL);

	gmenu_tree_item_unref (root);
	gmenu_tree_unref (tree);

	retval = g_slist_sort (retval,
			       (GCompareFunc) compare_applications);

	return retval;
}


static gboolean
eel_open_with_dialog_add_items_idle (EelOpenWithDialog *dialog)
{
	GtkCellRenderer   *renderer;
	GtkTreeViewColumn *column;
	GSList            *all_applications;
	GSList            *l;
	const char        *prev_name;

	/* create list store */
	dialog->details->program_list_store = gtk_list_store_new (NUM_COLUMNS,
								  GDK_TYPE_PIXBUF,
								  G_TYPE_STRING,
								  G_TYPE_STRING,
								  G_TYPE_STRING,
								  G_TYPE_STRING,
								  G_TYPE_STRING);

	all_applications = get_all_applications ();
	
	prev_name = NULL;
	for (l = all_applications; l; l = l->next) {
		GMenuTreeEntry *entry = l->data;
		GtkTreeIter     iter;
		GtkTreePath    *path;

		if (prev_name && strcmp (gmenu_tree_entry_get_name (entry), prev_name) == 0) {
			gmenu_tree_item_unref (entry);
			continue;
		}

		gtk_list_store_append (dialog->details->program_list_store, &iter);
		gtk_list_store_set (dialog->details->program_list_store, &iter,
				    COLUMN_ICON,      NULL,
				    COLUMN_ICON_FILE, gmenu_tree_entry_get_icon (entry),
				    COLUMN_NAME,      gmenu_tree_entry_get_name (entry),
				    COLUMN_COMMENT,   gmenu_tree_entry_get_comment (entry),
				    COLUMN_PATH,      gmenu_tree_entry_get_desktop_file_path (entry),
				    -1);

		path = gtk_tree_model_get_path (GTK_TREE_MODEL (dialog->details->program_list_store), &iter);
		if (path != NULL) {
			dialog->details->add_icon_paths = g_slist_prepend (dialog->details->add_icon_paths, path);
		}

		prev_name = gmenu_tree_entry_get_name (entry);
		gmenu_tree_item_unref (entry);
	}
	g_slist_free (all_applications);

	gtk_tree_view_set_model (GTK_TREE_VIEW (dialog->details->program_list), 
				 GTK_TREE_MODEL (dialog->details->program_list_store));

	renderer = gtk_cell_renderer_pixbuf_new ();
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_pack_start (column, renderer, FALSE);
	gtk_tree_view_column_set_attributes (column, renderer,
                                             "pixbuf", COLUMN_ICON,
                                             NULL);
        
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_attributes (column, renderer,
                                             "text", COLUMN_NAME,
                                             NULL);
					          
	gtk_tree_view_append_column (GTK_TREE_VIEW (dialog->details->program_list), column);

	dialog->details->add_icon_paths = g_slist_reverse (dialog->details->add_icon_paths);

	if (!dialog->details->add_icons_idle_id) {
		dialog->details->add_icons_idle_id =
			g_idle_add_full (G_PRIORITY_DEFAULT_IDLE, (GSourceFunc) eel_open_with_dialog_add_icon_idle,
					 dialog, NULL);
	}

	dialog->details->add_items_idle_id = 0;					 
	return FALSE;
}


static char *
remove_parameters (const char *exec)
{
	GString *str;
	char    *retval, *p;

	str = g_string_new (exec);

	while ((p = strstr (str->str, "%"))) {
		switch (p [1]) {
		case '%':
			g_string_erase (str, p - str->str, 1);
			break;
		case 'U':
		case 'F':
		case 'N':
		case 'D':
		case 'f':
		case 'u':
		case 'd':
		case 'n':
		case 'm':
		case 'i':
		case 'c':
		case 'k':
		case 'v':
			g_string_erase (str, p - str->str, 2);
			break;
		default:
			break;
		}
	}

	retval = str->str;
	g_string_free (str, FALSE);

	return retval;
}

static void
program_list_selection_changed (GtkTreeSelection  *selection,
				EelOpenWithDialog *dialog)
{
	GtkTreeModel     *model;
	GtkTreeIter       iter;
	char             *temp;
	char             *path, *stripped;
	GKeyFile         *keyfile;
		
	if (!gtk_tree_selection_get_selected (selection, &model, &iter)) {
		gtk_widget_set_sensitive (dialog->details->button, FALSE);
		return;
	}

	path = NULL;
	gtk_tree_model_get (model, &iter,
			    COLUMN_PATH, &path,
			    -1);
				  
	if (path == NULL)
		return;

	keyfile = g_key_file_new ();
	if (!g_key_file_load_from_file (keyfile, path, 0, NULL)) {
		g_free (path);
		g_key_file_free (keyfile);
		return;
	}

	temp = g_key_file_get_string (keyfile, DESKTOP_ENTRY_GROUP,
				      GNOME_DESKTOP_ITEM_EXEC, NULL);
			
	if (temp) {
		stripped = remove_parameters (temp);
		gtk_entry_set_text (GTK_ENTRY (dialog->details->entry), stripped);
		g_free (stripped);
	} else {
		temp = g_key_file_get_string (keyfile, DESKTOP_ENTRY_GROUP,
					      GNOME_DESKTOP_ITEM_URL, NULL);
		gtk_entry_set_text (GTK_ENTRY (dialog->details->entry), sure_string (temp));
	}
	g_free (temp);

	temp = g_key_file_get_locale_string (keyfile, DESKTOP_ENTRY_GROUP,
					     GNOME_DESKTOP_ITEM_COMMENT, NULL, NULL);
	gtk_label_set_text (GTK_LABEL (dialog->details->desc_label), sure_string (temp));
	g_free (temp);
	gtk_widget_set_sensitive (dialog->details->button, TRUE);

	g_free (dialog->details->desktop_file);
	dialog->details->desktop_file = g_strdup (path);
			
	g_free (path);
	g_key_file_free (keyfile);
}

static void
program_list_selection_activated (GtkTreeView       *view,
				  GtkTreePath       *path,
				  GtkTreeViewColumn *column,
				  EelOpenWithDialog *dialog)
{
	GtkTreeSelection *selection;

	/* update the entry with the info from the selection */
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dialog->details->program_list));	
	program_list_selection_changed (selection, dialog);
	
	gtk_dialog_response (GTK_DIALOG (&dialog->parent), RESPONSE_OPEN);
}

static void
expander_toggled (GtkWidget *expander, EelOpenWithDialog *dialog)
{
	if (gtk_expander_get_expanded (GTK_EXPANDER (expander)) == TRUE) {
		gtk_widget_grab_focus (dialog->details->entry);
		gtk_window_resize (GTK_WINDOW (dialog), 400, 1);
	}
	else {
		GtkTreeSelection *selection;

		gtk_widget_grab_focus (dialog->details->program_list);
		selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dialog->details->program_list));
		program_list_selection_changed (selection, dialog);
	}
}

static void
eel_open_with_dialog_instance_init (EelOpenWithDialog *dialog)
{
	GtkWidget *hbox;
	GtkWidget *vbox;
	GtkWidget *label;
	GtkWidget *align;
	GtkWidget *scrolled_window;
	GtkWidget *expander;
	GtkTreeSelection *selection;

	dialog->details = g_new0 (EelOpenWithDialogDetails, 1);

	gtk_window_set_title (GTK_WINDOW (dialog), _("Open With"));
	gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (dialog), 6);
	gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
	gtk_window_set_destroy_with_parent (GTK_WINDOW (dialog), TRUE);

	gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 2);

	vbox = gtk_vbox_new (FALSE, 6);
	dialog->details->label = gtk_label_new ("");
	gtk_misc_set_alignment (GTK_MISC (dialog->details->label), 0.0, 0.5);
	gtk_label_set_line_wrap (GTK_LABEL (dialog->details->label), TRUE);
	gtk_box_pack_start (GTK_BOX (vbox), dialog->details->label,
			    FALSE, FALSE, 0);
	gtk_widget_show (dialog->details->label);


	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_set_size_request (scrolled_window, 400, 400);
	
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled_window),
					     GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	dialog->details->program_list = gtk_tree_view_new ();
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (dialog->details->program_list),
					   FALSE);
	gtk_container_add (GTK_CONTAINER (scrolled_window), dialog->details->program_list);
	
	gtk_box_pack_start (GTK_BOX (vbox), scrolled_window, TRUE, TRUE, 0);

	dialog->details->desc_label = gtk_label_new (_("Select an application to view its description."));
	gtk_misc_set_alignment (GTK_MISC (dialog->details->desc_label), 0.0, 0.5);
	gtk_label_set_justify (GTK_LABEL (dialog->details->desc_label), GTK_JUSTIFY_LEFT);
	gtk_label_set_line_wrap (GTK_LABEL (dialog->details->desc_label), TRUE);
	gtk_label_set_single_line_mode (GTK_LABEL (dialog->details->desc_label), FALSE);
	gtk_box_pack_start (GTK_BOX (vbox), dialog->details->desc_label, FALSE, FALSE, 0);
	
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dialog->details->program_list));
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
	g_signal_connect (selection, "changed",
			  G_CALLBACK (program_list_selection_changed),
			  dialog);
	g_signal_connect (dialog->details->program_list, "row-activated",
			  G_CALLBACK (program_list_selection_activated),
			  dialog);

	dialog->details->add_items_idle_id = g_idle_add_full (G_PRIORITY_DEFAULT_IDLE,
						     (GSourceFunc) eel_open_with_dialog_add_items_idle,
						      dialog, NULL);

	
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), vbox, TRUE, TRUE, 0);
	gtk_widget_show_all (vbox);


	expander = gtk_expander_new_with_mnemonic (_("Use a _custom command"));
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), expander,
			    FALSE, FALSE, 0);
	g_signal_connect_after (expander, "activate", G_CALLBACK (expander_toggled), dialog);

	gtk_widget_show (expander);

	hbox = gtk_hbox_new (FALSE, 6);
	gtk_container_add (GTK_CONTAINER (expander), hbox);
	gtk_widget_show (hbox);

	dialog->details->entry = gtk_entry_new ();
	gtk_entry_set_activates_default (GTK_ENTRY (dialog->details->entry), TRUE);

	gtk_box_pack_start (GTK_BOX (hbox), dialog->details->entry,
			    TRUE, TRUE, 0);
	gtk_widget_show (dialog->details->entry);

	dialog->details->button = gtk_button_new_with_mnemonic (_("_Browse..."));
	g_signal_connect (dialog->details->button, "clicked",
			  G_CALLBACK (browse_clicked_cb), dialog);
	gtk_box_pack_start (GTK_BOX (hbox), dialog->details->button, FALSE, FALSE, 0);
	gtk_widget_show (dialog->details->button);

	gtk_dialog_add_button (GTK_DIALOG (dialog),
			       GTK_STOCK_CANCEL,
			       GTK_RESPONSE_CANCEL);


	/* Create a custom stock icon */
	dialog->details->button = gtk_button_new ();

	/* Hook up the entry to the button */
	gtk_widget_set_sensitive (dialog->details->button, FALSE);
	g_signal_connect (G_OBJECT (dialog->details->entry), "changed",
			  G_CALLBACK (entry_changed_cb), dialog);

	hbox = gtk_hbox_new (FALSE, 2);
	gtk_widget_show (hbox);

	label = gtk_label_new_with_mnemonic (_("_Open"));
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), GTK_WIDGET (dialog->details->button));
	gtk_widget_show (label);
	dialog->details->open_label = label;

	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

	align = gtk_alignment_new (0.5, 0.5, 0.0, 0.0);
	gtk_widget_show (align);

	gtk_widget_show (dialog->details->button);
	GTK_WIDGET_SET_FLAGS (dialog->details->button, GTK_CAN_DEFAULT);


	gtk_container_add (GTK_CONTAINER (align), hbox);
	gtk_container_add (GTK_CONTAINER (dialog->details->button), align);

	gtk_dialog_add_action_widget (GTK_DIALOG (dialog),
				      dialog->details->button, RESPONSE_OPEN);


	gtk_dialog_set_default_response (GTK_DIALOG (dialog),
					 RESPONSE_OPEN);

	g_signal_connect (dialog, "response",
			  G_CALLBACK (response_cb),
			  dialog);
}

static char *
get_extension (const char *basename)
{
	char *p;

	p = strrchr (basename, '.');

	if (p && *(p + 1) != '\0') {
		return g_strdup (p + 1);
	} else {
		return NULL;
	}
}

static void
set_uri_and_mime_type (EelOpenWithDialog *dialog,
		       const char *uri,
		       const char *mime_type)
{
	char *label;
	char *name;
	GnomeVFSURI *vfs_uri;

	dialog->details->uri = g_strdup (uri);

	vfs_uri = gnome_vfs_uri_new (uri);

	name = gnome_vfs_uri_extract_short_name (vfs_uri);

	if (!strcmp (mime_type, "application/octet-stream")) {
		char *extension;

		extension = get_extension (uri);

		if (!extension) {
			g_warning ("No extension, not implemented yet");
			return;
		}

		dialog->details->mime_type =
			g_strdup_printf ("application/x-extension-%s",
					 extension);
		/* the %s here is a file extension */
		dialog->details->mime_description =
			g_strdup_printf (_("%s document"), extension);
		dialog->details->new_glob = g_strdup_printf ("*.%s",
							     extension);
		dialog->details->new_mime_type = TRUE;

		g_free (extension);
	} else {
		char *description;

		dialog->details->mime_type = g_strdup (mime_type);
		description = g_strdup (gnome_vfs_mime_get_description (mime_type));

		if (description == NULL) {
			description = g_strdup (_("Unknown"));
		}

		dialog->details->mime_description = description;
	}

	label = g_strdup_printf (_("Open <i>%s</i> and other files of type \"%s\" with:"), name, dialog->details->mime_description);

	gtk_label_set_markup (GTK_LABEL (dialog->details->label), label);

	g_free (label);
	g_free (name);
	gnome_vfs_uri_unref (vfs_uri);
}

GtkWidget *
eel_open_with_dialog_new (const char *uri,
			  const char *mime_type)
{
	GtkWidget *dialog;

	dialog = gtk_widget_new (EEL_TYPE_OPEN_WITH_DIALOG, NULL);

	set_uri_and_mime_type (EEL_OPEN_WITH_DIALOG (dialog), uri, mime_type);

	return dialog;
}

GtkWidget* 
eel_add_application_dialog_new (const char *uri,
				const char *mime_type)
{
	EelOpenWithDialog *dialog;
	
	dialog = EEL_OPEN_WITH_DIALOG (eel_open_with_dialog_new (uri, mime_type));
	
	gtk_label_set_text_with_mnemonic (GTK_LABEL (dialog->details->open_label),
					  _("_Add"));
	gtk_window_set_title (GTK_WINDOW (dialog), _("Add Application"));

	return GTK_WIDGET (dialog);
}

GType
eel_open_with_dialog_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (EelOpenWithDialogClass),
			NULL,
			NULL,
			(GClassInitFunc)eel_open_with_dialog_class_init,
			NULL,
			NULL,
			sizeof (EelOpenWithDialog),
			0,
			(GInstanceInitFunc)eel_open_with_dialog_instance_init,
		};

		type = g_type_register_static (GTK_TYPE_DIALOG,
					       "EelOpenWithDialog",
					       &info, 0);
	}

	return type;
}
