/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  gpa-list.c: 
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors :
 *    Jose M. Celorio <chema@ximian.com>
 *    Lauris Kaplinski <lauris@ximian.com>
 *
 *  Copyright (C) 2000-2001 Ximian, Inc. and Jose M. Celorio
 *
 */


/*
 * A GPAList is a node to hold a list of nodes of the same type.
 *
 * Lists are used for
 *  1. Printer list globals->printer
 *  2. Vendors list globals->vendors
 *  3. Each printer has a Settings list printer->settings
 *  4. Each Vendor has a Models list vendor->models
 *  5. Model->options
 *
 */

#include "config.h"
#include <string.h>
#include "gpa-utils.h"
#include "gpa-reference.h"
#include "gpa-list.h"

static void gpa_list_class_init (GPAListClass *klass);
static void gpa_list_init (GPAList *list);

static void      gpa_list_finalize  (GObject *object);
static gboolean  gpa_list_verify    (GPANode *node);
static GPANode * gpa_list_lookup    (GPANode *node, const guchar *path);
static GPANode * gpa_list_get_child (GPANode *node, GPANode *previous_child);
static guchar  * gpa_list_get_value (GPANode *node);
static gboolean  gpa_list_set_value (GPANode *node, const guchar *value);
static void      gpa_list_modified  (GPANode *node, guint flags);

static GPANodeClass *parent_class = NULL;

GType
gpa_list_get_type (void) {
	static GType type = 0;
	if (!type) {
		static const GTypeInfo info = {
			sizeof (GPAListClass),
			NULL, NULL,
			(GClassInitFunc) gpa_list_class_init,
			NULL, NULL,
			sizeof (GPAList),
			0,
			(GInstanceInitFunc) gpa_list_init
		};
		type = g_type_register_static (GPA_TYPE_NODE, "GPAList", &info, 0);
	}
	return type;
}

static void
gpa_list_class_init (GPAListClass *klass)
{
	GObjectClass *object_class;
	GPANodeClass *node_class;

	object_class = (GObjectClass *) klass;
	node_class = (GPANodeClass *) klass;

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gpa_list_finalize;

	node_class->get_child = gpa_list_get_child;
	node_class->get_value = gpa_list_get_value;
	node_class->set_value = gpa_list_set_value;
	node_class->lookup    = gpa_list_lookup;
	node_class->verify    = gpa_list_verify;
	node_class->modified  = gpa_list_modified;
}

static void
gpa_list_init (GPAList *list)
{
	list->childtype = GPA_TYPE_NODE;
	list->children = NULL;
	list->can_have_default = FALSE;
	list->def = NULL;
}

static void
gpa_list_finalize (GObject *object)
{
	GPAList *list;

	list = GPA_LIST (object);

	if (list->def) {
		list->def = gpa_node_detach_unref (GPA_NODE (list), GPA_NODE (list->def));
	}

	while (list->children) {
		GPANode *child;
		child = list->children;
		list->children = child->next;
		child->next = NULL;
		child->parent = NULL;
		gpa_node_unref (child);
	}

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static gboolean
gpa_list_verify (GPANode *node)
{
	GPAList *list;
	
	list = GPA_LIST (node);

	gpa_return_false_if_fail (list != NULL);
	gpa_return_false_if_fail (GPA_NODE (node)->qid != 0);
	if (list->can_have_default) {
		gpa_return_false_if_fail (list->def != NULL);
		gpa_return_false_if_fail (gpa_node_verify (list->def));
	} else {
		gpa_return_false_if_fail (list->def == NULL);
	}

	return TRUE;
}

static gboolean
gpa_list_set_default_from_name (GPAList *list, const guchar *name)
{
	GPANode *child;
	gboolean ret;
	
	if (strchr (name, '.') != NULL) {
		g_warning ("Set default from name can't contain \".\"");
		return FALSE;
	}
	
	child = gpa_list_lookup (GPA_NODE (list), name);
	if (!child) {
		g_warning ("Can't find \"%s\" as a child of \"%s\". Default not set.",
			   name, gpa_node_id (GPA_NODE (list)));
		return FALSE;
	}

	ret = gpa_reference_set_reference (GPA_REFERENCE (list->def), child);

	gpa_node_request_modified (GPA_NODE (list), GPA_NODE_MODIFIED_FLAG);

	return ret;
}

static gboolean
gpa_list_set_value (GPANode *node, const guchar *value)
{
	g_return_val_if_fail (GPA_IS_LIST (node), FALSE);
	
	return gpa_list_set_default_from_name (GPA_LIST (node), value);
}

static guchar *
gpa_list_get_value (GPANode *node)
{
	GPAList *list;

	g_return_val_if_fail (GPA_IS_LIST (node), FALSE);
	
	list = GPA_LIST (node);

	if (!list->can_have_default)
		return g_strdup ("");
		
	if (gpa_reference_is_empty (list->def))
		return g_strdup ("");

	return g_strdup (gpa_node_id (GPA_REFERENCE (list->def)->ref));
}

static GPANode *
gpa_list_lookup (GPANode *node, const guchar *path)
{
	GPAList *list;
	GPANode *child = NULL;
	const guchar *dot, *next;
	gint len;

	list = GPA_LIST (node);

	/* First satisfy the .DefaultItem path if the list can have a default */
	if (list->can_have_default) {
		g_assert (list->def != NULL);
		if (gpa_node_lookup_helper (GPA_NODE (list->def), path, GPA_LIST_DEFAULT_ITEM_KEY, &child)) {
			g_assert (child != NULL);
			return child;
		}
	}

	dot = strchr (path, '.');
	if (dot != NULL) {
		len = dot - path;
		next = dot + 1;
	} else {
		len = strlen (path);
		next = NULL;
	}

	/* Scan the list for a match */
	child = list->children;
	while (child) {
		const guchar *id;
		id = gpa_node_id (child);
		if (id && (strncmp (id, path, len) == 0) && (strlen (id) == len)) {
			if (next == NULL) {
				gpa_node_ref (child);
				return child;
			} else {
				return gpa_node_lookup (child, next);
			}
		}
		child = child->next;
	}

	g_warning ("The GPAList \"%s\" does not have a child with id \"%s\".\n",
		   gpa_node_id (node), path);

	return NULL;
}

static void
gpa_list_modified (GPANode *node, guint flags)
{
	GPAList *list;
	GPANode *child;

	list = GPA_LIST (node);

	child = list->children;
	while (child) {
		GPANode *next;
		next = child->next;
		if (GPA_NODE_FLAGS (child) & GPA_NODE_MODIFIED_FLAG) {
			gpa_node_ref (child);
			gpa_node_emit_modified (child, 0);
			gpa_node_unref (child);
		}
		child = next;
	}

	if (list->can_have_default && list->def) {
		if (GPA_NODE_FLAGS (list->def) & GPA_NODE_MODIFIED_FLAG) {
			gpa_node_ref (GPA_NODE (list->def));
			gpa_node_emit_modified (GPA_NODE (list->def), 0);
			gpa_node_unref (GPA_NODE (list->def));
		}
	}
}

static gboolean
gpa_list_def_set_value_cb (GPAReference *reference, const guchar *value, gpointer data)
{

	g_return_val_if_fail (GPA_IS_LIST (data), FALSE);
	g_return_val_if_fail (value != NULL, FALSE);

	return gpa_list_set_default_from_name (GPA_LIST (data), value);
}

/**
 * gpa_list_set_default:
 * @list: 
 * @def: 
 * 
 * Sets the default child of the list
 * 
 * Return Value: TRUE on success, FALSE otherwise
 **/
gboolean
gpa_list_set_default (GPAList *list, GPANode *def)
{
	g_return_val_if_fail (list != NULL, FALSE);
	g_return_val_if_fail (def  != NULL, FALSE);
	g_return_val_if_fail (GPA_IS_LIST (list), FALSE);

	if (list->can_have_default == FALSE) {
		g_warning ("Trying to set the default of a GPAList which has ->can_have_default to FALSE\n");
		return FALSE;
	}
	g_assert (list->def != NULL);

	return gpa_reference_set_reference (GPA_REFERENCE (list->def), def);
}


/**
 * gpa_list_get_child:
 * @node: The list to get a child from
 * @previous_child: Used to get the next child after the fisrt one, if not NULL gets you the next child after this one
 * 
 * Gets a child of the list. previous_child is a parameter that allows us to get the next child, previous_child is a
 * previously gotten child* from this function, and if present returns the next child after it. If NULL it returns the
 * first child
 * 
 * Return Value: a reference child of the list
 **/
static GPANode *
gpa_list_get_child (GPANode *node, GPANode *previous_child)
{
	GPAList *list;

	list = GPA_LIST (node);

	if (previous_child) {
		if (previous_child->next)
			gpa_node_ref (previous_child->next);
		return previous_child->next;
	}
	
	gpa_node_ref (list->children);
	
	return list->children;
}


/**
 * gpa_list_new:
 * @childtype: The GType of the members of the list
 * @list_name: The list name
 * @parent: 
 * 
 * Creates a new GPAList
 * 
 * Return Value: the newly created list, NULL on error
 **/
GPAList *
gpa_list_new (GType childtype, const gchar *list_name, gboolean can_have_default)
{
	GPAList *list;

	g_return_val_if_fail (g_type_is_a (childtype, GPA_TYPE_NODE), NULL);
	g_return_val_if_fail (list_name != NULL, NULL);

	list = (GPAList *) gpa_node_new (GPA_TYPE_LIST, list_name);

	list->childtype = childtype;
	list->can_have_default = can_have_default ? TRUE : FALSE;
	
	if (can_have_default) {
		list->def = gpa_node_attach (GPA_NODE (list), gpa_reference_new_empty ());
		g_signal_connect (G_OBJECT (list->def), "set_value",
				  (GCallback) gpa_list_def_set_value_cb, list);
	}
	
	return list;
}



/**
 * gpa_list_prepend:
 * @list: 
 * @child: 
 * 
 * Prepends a child to a GPAList
 * 
 * Return Value: TRUE on success, FALSE otherwise
 **/
gboolean
gpa_list_prepend (GPAList *list, GPANode *child)
{
	g_return_val_if_fail (GPA_IS_LIST (list),  FALSE);
	g_return_val_if_fail (GPA_IS_NODE (child), FALSE);

	if (gpa_node_id (child) && (strcmp (gpa_node_id (child), GPA_LIST_DEFAULT_ITEM_KEY) == 0)) {
		g_warning ("Can't prepend a child to a GPAList with id \"%s\" because the path that "
			   "it would create clashes with the path used to get the Default list node",
			   GPA_LIST_DEFAULT_ITEM_KEY);
		return FALSE;
	}

	gpa_node_attach (GPA_NODE (list), child);
	
	child->parent = (GPANode *) list;
	child->next = ((GPAList *) list)->children;
	((GPAList *) list)->children = child;

	return TRUE;
}
