/* GStreamer VGA plugin
 * Copyright (C) 2001 Ronald Bultje <rbultje@ronald.bitfreak.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */
/* let's not forget to mention that all this was based on aasink ;-) */

#include <config.h>
#include <signal.h>
#include <string.h>
#include <stdio.h>
#include <sys/time.h>
#include <asm/vga.h>

#include "vgavideosink.h"

/* elementfactory information */
static GstElementDetails gst_vgavideosink_details = {
  "Video sink",
  "Sink/Video",
  "LGPL",
  "An vga-based videosink",
  VERSION,
  "Ronald Bultje <rbultje@ronald.bitfreak.net>",
  "(C) 2001",
};

/* vgavideosink signals and args */
enum {
  SIGNAL_FRAME_DISPLAYED,
  LAST_SIGNAL
};


enum {
  ARG_0,
  ARG_WIDTH,
  ARG_HEIGHT,
  ARG_DRIVER,
  ARG_FRAMES_DISPLAYED,
  ARG_FRAME_TIME,
};

GST_PAD_TEMPLATE_FACTORY (sink_template,
  "sink",
  GST_PAD_SINK,
  GST_PAD_ALWAYS,
  GST_CAPS_NEW (
    "vgavideosink_caps",
    "video/raw",
    "format", 	  GST_PROPS_FOURCC(GST_MAKE_FOURCC('R','G','B',' ')),   
      "bpp", 	    GST_PROPS_INT(16),
      "depth", 	    GST_PROPS_INT(16),
      "endianness", GST_PROPS_INT (G_BYTE_ORDER),
      "red_mask",   GST_PROPS_INT (0xf800),
      "green_mask", GST_PROPS_INT (0x07e0),
      "blue_mask",  GST_PROPS_INT (0x001f),
      "width", 	    GST_PROPS_INT_RANGE (0, G_MAXINT),
      "height",     GST_PROPS_INT_RANGE (0, G_MAXINT)
  )
)
    
static void	gst_vgavideosink_class_init		(GstVGAVideoSinkClass *klass);
static void	gst_vgavideosink_init			(GstVGAVideoSink *vgavideosink);

static void	gst_vgavideosink_chain			(GstPad *pad, GstBuffer *buf);

static void	gst_vgavideosink_set_arg		(GtkObject *object, GtkArg *arg, guint id);
static void	gst_vgavideosink_get_arg		(GtkObject *object, GtkArg *arg, guint id);
static void	gst_vgavideosink_close			(GstVGAVideoSink *vgavideosink);
static GstElementStateReturn gst_vgavideosink_change_state (GstElement *element);

static GstElementClass *parent_class = NULL;
static guint gst_vgavideosink_signals[LAST_SIGNAL] = { 0 };

GtkType
gst_vgavideosink_get_type (void)
{
  static GtkType vgavideosink_type = 0;

  if (!vgavideosink_type) {
    static const GtkTypeInfo vgavideosink_info = {
      "GstVGAVideoSink",
      sizeof(GstVGAVideoSink),
      sizeof(GstVGAVideoSinkClass),
      (GtkClassInitFunc)gst_vgavideosink_class_init,
      (GtkObjectInitFunc)gst_vgavideosink_init,
      (GtkArgSetFunc)gst_vgavideosink_set_arg,
      (GtkArgGetFunc)gst_vgavideosink_get_arg,
      (GtkClassInitFunc)NULL,
    };
    vgavideosink_type = gtk_type_unique(GST_TYPE_ELEMENT,&vgavideosink_info);
  }
  return vgavideosink_type;
}

static void
gst_vgavideosink_class_init (GstVGAVideoSinkClass *klass)
{
  GtkObjectClass *gtkobject_class;
  GstElementClass *gstelement_class;

  gtkobject_class = (GtkObjectClass*)klass;
  gstelement_class = (GstElementClass*)klass;

  parent_class = gtk_type_class (GST_TYPE_ELEMENT);

  gtk_object_add_arg_type ("GstVGAVideoSink::width", GTK_TYPE_INT,
                           GTK_ARG_READABLE, ARG_WIDTH);
  gtk_object_add_arg_type ("GstVGAVideoSink::height", GTK_TYPE_INT,
                           GTK_ARG_READABLE, ARG_HEIGHT);
  gtk_object_add_arg_type ("GstVGAVideoSink::frames_displayed", GTK_TYPE_INT,
                           GTK_ARG_READABLE, ARG_FRAMES_DISPLAYED);
  gtk_object_add_arg_type ("GstVGAVideoSink::frame_time", GTK_TYPE_INT,
                           GTK_ARG_READABLE, ARG_FRAME_TIME);

  gtkobject_class->set_arg = gst_vgavideosink_set_arg;
  gtkobject_class->get_arg = gst_vgavideosink_get_arg;

  gst_vgavideosink_signals[SIGNAL_FRAME_DISPLAYED] =
    gtk_signal_new ("frame_displayed", GTK_RUN_LAST, gtkobject_class->type,
                    GTK_SIGNAL_OFFSET (GstVGAVideoSinkClass, frame_displayed),
                    gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);
  gtk_object_class_add_signals (gtkobject_class, gst_vgavideosink_signals,
                                  LAST_SIGNAL);

  gstelement_class->change_state = gst_vgavideosink_change_state;
}

static GstPadConnecReturn
gst_vgavideosink_sinkconnect (GstPad *pad, GstCaps *caps)
{
  GstVGAVideoSink *vgavideosink;

  vgavideosink = GST_VGAVIDEOSINK (gst_pad_get_parent (pad));

  if (!GST_CAPS_IS_FIXED (caps))
    return GST_PAD_CONNECT_DELAYED;

  vgavideosink->width = gst_caps_get_int (caps, "width");
  vgavideosink->height = gst_caps_get_int (caps, "height");

  return GST_PAD_CONNECT_OK;
}

static void
gst_vgavideosink_init (GstVGAVideoSink *vgavideosink)
{
  vgavideosink->sinkpad = gst_pad_new_from_template (
		  GST_PAD_TEMPLATE_GET (sink_template), "sink");
  gst_element_add_pad (GST_ELEMENT (vgavideosink), vgavideosink->sinkpad);
  gst_pad_set_chain_function (vgavideosink->sinkpad, gst_vgavideosink_chain);
  gst_pad_set_connect_function (vgavideosink->sinkpad, gst_vgavideosink_sinkconnect);

  vgavideosink->clock = gst_clock_get_system();
  gst_clock_register(vgavideosink->clock, GST_OBJECT(vgavideosink));

  vgavideosink->width = -1;
  vgavideosink->height = -1;

  GST_FLAG_SET(vgavideosink, GST_ELEMENT_THREAD_SUGGESTED);
}

static void
gst_vgavideosink_chain (GstPad *pad, GstBuffer *buf)
{
  GstVGAVideoSink *vgavideosink;
  GstClockTimeDiff jitter;
  gint v;
  guint16 *data = (guint16 *)GST_BUFFER_DATA(buf);
  guint16 *mem = (guint16 *)vga_getgraphmem();
  gint size;

  g_return_if_fail (pad != NULL);
  g_return_if_fail (GST_IS_PAD (pad));
  g_return_if_fail (buf != NULL);

  vgavideosink = GST_VGAVIDEOSINK (gst_pad_get_parent (pad));

  GST_DEBUG (0,"videosink: clock wait: %llu", GST_BUFFER_TIMESTAMP(buf));

  jitter = gst_clock_current_diff(vgavideosink->clock, GST_BUFFER_TIMESTAMP (buf));

  if (jitter > 500000 || jitter < -500000)
  {
    GST_DEBUG (0, "jitter: %lld", jitter);
    gst_clock_set (vgavideosink->clock, GST_BUFFER_TIMESTAMP (buf));
  }
  else {
    gst_clock_wait(vgavideosink->clock, GST_BUFFER_TIMESTAMP(buf), GST_OBJECT(vgavideosink));
  }

  size = MIN (GST_BUFFER_SIZE (buf), vgavideosink->width * vgavideosink->height*2);

  /* FIXME copy over the buffer in chuncks of 64K using vga_setpage(gint p) */
  memcpy (mem, data, 64000);

  gtk_signal_emit(GTK_OBJECT(vgavideosink),gst_vgavideosink_signals[SIGNAL_FRAME_DISPLAYED]);

  gst_buffer_unref(buf);
}


static void
gst_vgavideosink_set_arg (GtkObject *object, GtkArg *arg, guint id)
{
  GstVGAVideoSink *vgavideosink;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail (GST_IS_VGAVIDEOSINK (object));

  vgavideosink = GST_VGAVIDEOSINK (object);
}

static void
gst_vgavideosink_get_arg (GtkObject *object, GtkArg *arg, guint id)
{
  GstVGAVideoSink *vgavideosink;

  /* it's not null if we got it, but it might not be ours */
  vgavideosink = GST_VGAVIDEOSINK(object);

  switch(id) {
    case ARG_FRAMES_DISPLAYED: {
      GTK_VALUE_INT(*arg) = vgavideosink->frames_displayed;
      break;
    }
    case ARG_FRAME_TIME: {
      GTK_VALUE_INT(*arg) = vgavideosink->frame_time/1000000;
      break;
    }
    default: {
      arg->type = GTK_TYPE_INVALID;
      break;
    }
  }
}

static gboolean
gst_vgavideosink_open (GstVGAVideoSink *vgavideosink)
{
  g_return_val_if_fail (!GST_FLAG_IS_SET (vgavideosink ,GST_VGAVIDEOSINK_OPEN), FALSE);

  /* Initialize the SVGA  library */

  vga_init(); 
  /*vga_setmode(G640x400x64K); */
  vga_setmode(G640x480x64K);
  GST_FLAG_SET (vgavideosink, GST_VGAVIDEOSINK_OPEN);

  return TRUE;
}

static void
gst_vgavideosink_close (GstVGAVideoSink *vgavideosink)
{
  g_return_if_fail (GST_FLAG_IS_SET (vgavideosink ,GST_VGAVIDEOSINK_OPEN));

  vga_setmode(0);
    
  GST_FLAG_UNSET (vgavideosink, GST_VGAVIDEOSINK_OPEN);
}

static GstElementStateReturn
gst_vgavideosink_change_state (GstElement *element)
{
  g_return_val_if_fail (GST_IS_VGAVIDEOSINK (element), GST_STATE_FAILURE);

  if (GST_STATE_PENDING (element) == GST_STATE_NULL) {
    if (GST_FLAG_IS_SET (element, GST_VGAVIDEOSINK_OPEN))
      gst_vgavideosink_close (GST_VGAVIDEOSINK (element));
  } else {
    if (!GST_FLAG_IS_SET (element, GST_VGAVIDEOSINK_OPEN)) {
      if (!gst_vgavideosink_open (GST_VGAVIDEOSINK (element)))
        return GST_STATE_FAILURE;
    }
  }

  if (GST_ELEMENT_CLASS (parent_class)->change_state)
    return GST_ELEMENT_CLASS (parent_class)->change_state (element);

  return GST_STATE_SUCCESS;
}

static gboolean
plugin_init (GModule *module, GstPlugin *plugin)
{
  GstElementFactory *factory;

  /* create an elementfactory for the vgavideosink element */
  factory = gst_element_factory_new("vgavideosink",GST_TYPE_VGAVIDEOSINK,
                                   &gst_vgavideosink_details);
  g_return_val_if_fail(factory != NULL, FALSE);

  gst_element_factory_add_pad_template (factory, 
		  GST_PAD_TEMPLATE_GET (sink_template));

  gst_plugin_add_feature (plugin, GST_PLUGIN_FEATURE (factory));

  return TRUE;
}

GstPluginDesc plugin_desc = {
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "vgavideosink",
  plugin_init
};
