/*
 *  Copyright (C) 2002  Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "window-recent-history-menu.h"
#include "gul-gobject-misc.h"
#include "gul-string.h"
#include "gul-bonobo-extensions.h"
#include "galeon-marshal.h"
#include <string.h>
#include <stdlib.h>

#define NOT_IMPLEMENTED g_warning ("not implemented: " G_STRLOC);
//#define DEBUG_MSG(x) g_print x
#define DEBUG_MSG(x)

#define MAX_LABEL_LENGTH 30

/**
 * Private data
 */
struct _GaleonWindowRecentHistoryMenuPrivate 
{
	GaleonWindowRecentHistory *wrh;
	BonoboUIComponent *uic;
	gchar *path;
};

/**
 * Private functions, only availble from this file
 */
static void	galeon_window_recent_history_menu_class_init	(GaleonWindowRecentHistoryMenuClass *klass);
static void	galeon_window_recent_history_menu_init		(GaleonWindowRecentHistoryMenu *wrhm);
static void	galeon_window_recent_history_menu_finalize_impl	(GObject *o);
static void	galeon_window_recent_history_menu_rebuild	(GaleonWindowRecentHistoryMenu *wrhm);
static void	galeon_window_recent_history_menu_changed_cb	(GaleonWindowRecentHistory *wrh,
								 GaleonWindowRecentHistoryMenu *wrhm);
static void 	galeon_window_recent_history_menu_verb_cb	(BonoboUIComponent *uic,
								 gpointer user_data,
								 const char *cname);

static gpointer g_object_class;

enum GaleonWindowRecentHistoryMenuSignalsEnum {
	GALEON_WINDOW_RECENT_HISTORY_MENU_ACTIVATED,
	GALEON_WINDOW_RECENT_HISTORY_MENU_LAST_SIGNAL
};
static gint GaleonWindowRecentHistoryMenuSignals[GALEON_WINDOW_RECENT_HISTORY_MENU_LAST_SIGNAL];


/**
 * GaleonWindowRecentHistoryMenu object
 */
MAKE_GET_TYPE (galeon_window_recent_history_menu,
	       "GaleonWindowRecentHistoryMenu", GaleonWindowRecentHistoryMenu, 
	       galeon_window_recent_history_menu_class_init, galeon_window_recent_history_menu_init, 
	       G_TYPE_OBJECT);


static void
galeon_window_recent_history_menu_class_init (GaleonWindowRecentHistoryMenuClass *klass)
{
	G_OBJECT_CLASS (klass)->finalize = galeon_window_recent_history_menu_finalize_impl;
	g_object_class = g_type_class_peek_parent (klass);

	GaleonWindowRecentHistoryMenuSignals[GALEON_WINDOW_RECENT_HISTORY_MENU_ACTIVATED] = g_signal_new (
		"activated", G_OBJECT_CLASS_TYPE (klass),  
		G_SIGNAL_RUN_FIRST | G_SIGNAL_RUN_LAST | G_SIGNAL_RUN_CLEANUP,
                G_STRUCT_OFFSET (GaleonWindowRecentHistoryMenuClass, activated), 
		NULL, NULL, 
		galeon_marshal_VOID__STRING_STRING,
		G_TYPE_NONE, 2, G_TYPE_STRING, G_TYPE_STRING);
	
}

static void 
galeon_window_recent_history_menu_init (GaleonWindowRecentHistoryMenu *wrhm)
{
	GaleonWindowRecentHistoryMenuPrivate *p = g_new0 (GaleonWindowRecentHistoryMenuPrivate, 1);
	wrhm->priv = p;
}

static void
galeon_window_recent_history_menu_finalize_impl (GObject *o)
{
	GaleonWindowRecentHistoryMenu *wrhm = GALEON_WINDOW_RECENT_HISTORY_MENU (o);
	GaleonWindowRecentHistoryMenuPrivate *p = wrhm->priv;
	
	if (p->wrh)
	{
		g_signal_handlers_disconnect_matched (p->wrh, G_SIGNAL_MATCH_DATA, 
						      0, 0, NULL, NULL, wrhm);
		g_object_unref (p->wrh);
	}
	if (p->path)
	{
		g_free (p->path);
	}
	if (p->uic)
	{
		g_object_unref (p->uic);
	}
	g_free (p);
	
	G_OBJECT_CLASS (g_object_class)->finalize (o);
}

GaleonWindowRecentHistoryMenu *
galeon_window_recent_history_menu_new (void)
{
	GaleonWindowRecentHistoryMenu *ret = g_object_new (GALEON_TYPE_WINDOW_RECENT_HISTORY_MENU, NULL);
	return ret;
}

void
galeon_window_recent_history_menu_set_history (GaleonWindowRecentHistoryMenu *wrhm,
					       GaleonWindowRecentHistory *wrh)
{
	GaleonWindowRecentHistoryMenuPrivate *p;
	
	g_return_if_fail (GALEON_IS_WINDOW_RECENT_HISTORY_MENU (wrhm));
	g_return_if_fail (GALEON_IS_WINDOW_RECENT_HISTORY (wrh));

	p = wrhm->priv;

	if (p->wrh)
	{
		g_signal_handlers_disconnect_matched (p->wrh, G_SIGNAL_MATCH_DATA, 0, 0, NULL, NULL, wrhm);
		g_object_unref (p->wrh);
	}
	
	p->wrh = g_object_ref (wrh);
	g_signal_connect (p->wrh, "changed", 
			  G_CALLBACK (galeon_window_recent_history_menu_changed_cb), wrhm);

	galeon_window_recent_history_menu_rebuild (wrhm);
}

void
galeon_window_recent_history_menu_set_path (GaleonWindowRecentHistoryMenu *wrhm,
					    BonoboUIComponent *uic, const gchar *path)
{
	GaleonWindowRecentHistoryMenuPrivate *p;
	
	g_return_if_fail (GALEON_IS_WINDOW_RECENT_HISTORY_MENU (wrhm));
	g_return_if_fail (BONOBO_IS_UI_COMPONENT (uic));
	g_return_if_fail (path != NULL);

	p = wrhm->priv;

	if (p->path)
	{
		g_free (p->path);
	}
	p->path = g_strdup (path);

	if (p->uic)
	{
		g_object_unref (p->uic);
	}
	p->uic = g_object_ref (uic);

	galeon_window_recent_history_menu_rebuild (wrhm);
}

static void
galeon_window_recent_history_menu_rebuild (GaleonWindowRecentHistoryMenu *wrhm)
{
	GaleonWindowRecentHistoryMenuPrivate *p = wrhm->priv;
	GString *xml;
	gint i;
	gint num_items;

	if (!(p->uic && p->path))
	{
		return;
	}

	gul_bonobo_clear_path (p->uic, p->path);

	if (!p->wrh)
	{
		return;
	}

	DEBUG_MSG (("Rebuilding recent history menu\n"));

	xml = g_string_new (NULL);
	g_string_append_printf (xml, "<placeholder name=\"wrhm%x\">\n", (guint) wrhm);
	galeon_window_recent_history_get_num_items (p->wrh, &num_items);
	for (i = 0; i < num_items; ++i)
	{
		gchar *verb = g_strdup_printf ("Wrhm%xn%d", (guint) wrhm, i);
		gchar *title_s;
		gchar *title;
		gchar *label;
		gchar *label_x;
		galeon_window_recent_history_get_item (p->wrh, i, NULL, &title);
		title_s = gul_string_shorten (title, MAX_LABEL_LENGTH);
		label = gul_string_new_num_accel (i, title_s, TRUE);
		label_x = gul_string_strdup_replace (label, "\"", "&quot;");
		
		g_string_append (xml, "<menuitem name=\"");
		g_string_append (xml, verb);
		g_string_append (xml, "\" label=\"");
		g_string_append (xml, label_x);
		g_string_append (xml, "\" verb=\"");
		g_string_append (xml, verb);
		g_string_append (xml, "\"/>\n");
		bonobo_ui_component_add_verb (p->uic, verb, 
					      galeon_window_recent_history_menu_verb_cb, wrhm);
		g_free (title);
		g_free (label_x);
		g_free (label);
		g_free (title_s);
		g_free (verb);
	}
	g_string_append (xml, "</placeholder>\n");
	DEBUG_MSG (("\n%s\n", xml->str));
	if (num_items > 0)
	{
		bonobo_ui_component_set (p->uic, p->path, 
					 xml->str, NULL);
	}
	g_string_free (xml, TRUE);
}

static void
galeon_window_recent_history_menu_changed_cb (GaleonWindowRecentHistory *wrh, 
					      GaleonWindowRecentHistoryMenu *wrhm)
{
	galeon_window_recent_history_menu_rebuild (wrhm);
}

static void 
galeon_window_recent_history_menu_verb_cb (BonoboUIComponent *uic,
					   gpointer user_data,
					   const char *cname)
{
	gchar *url;
	gchar *title;
	const gchar *t;
	gint i;
	GaleonWindowRecentHistoryMenu *wrhm = user_data;
	GaleonWindowRecentHistoryMenuPrivate *p;

	g_return_if_fail (GALEON_IS_WINDOW_RECENT_HISTORY_MENU (wrhm));
	
	p = wrhm->priv;
	t = rindex (cname, 'n');
	i = atoi (t + 1);
	
	galeon_window_recent_history_get_item (p->wrh, i, &url, &title);

	DEBUG_MSG (("Going to item %d in recent history:\n%s\n%s\n", i, url, title));

	if (url)
	{
		g_signal_emit 
			(wrhm,
			 GaleonWindowRecentHistoryMenuSignals[GALEON_WINDOW_RECENT_HISTORY_MENU_ACTIVATED],
			 0, url, title);
	}
	
	g_free (url);
	g_free (title);
}

